/*  Glimmer - bindings.c
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "declarations.h"
#include "bindings.h"
#include "widgets/gdsfile.h"
#include "build.h"
#include "edit.h"
#include "file-io.h"
#include "main.h"
#include "misc.h"
#include "searches.h"
#include "settings.h"
#include "syntax-highlight.h"
#include "macro-language.h"
#include "commandbar.h"
#include "windows.h"

int get_file_number(void)
{
   int rval;
   rval = g_list_index(files_list, (gpointer)cur_file);
   return(rval);
}

int change(int file_no)
{
   GdsFile *file;
   int success = FALSE;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   if(file)
   {
      file_change(NULL, file);
      success = TRUE;
   }
   return(success);
}

int get_files(void)
{
   return(total_files);
}

char *get_default_term_command(void)
{
   return(g_strdup(general_preferences.terminal));
}

char *get_default_make_command(void)
{
   return(g_strdup(general_preferences.make));
}

int get_tab_stops(void)
{
   return(general_preferences.tab_stops);
}

int get_file_length(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(file != NULL, -1);
   g_return_val_if_fail(GDS_IS_FILE(file), -1);
   return(GTK_EXTEXT(file->text)->length);
}

int get_file_changed(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(file != NULL, -1);
   g_return_val_if_fail(GDS_IS_FILE(file), -1);
   if(GDS_EDITOR(file->text)->changed) return(TRUE);
   return(FALSE);
}

int get_file_tab_width(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(file != NULL, -1);
   g_return_val_if_fail(GDS_IS_FILE(file), -1);
   return(gtk_extext_get_tab_width(GTK_EXTEXT(file->text)));
}

int get_file_use_spaces(int file_no)
{
   GdsFile *file;
   int value = FALSE;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(file != NULL, -1);
   g_return_val_if_fail(GDS_IS_FILE(file), -1);
   if(file->props.over_ride &&  file->props.use_spaces) value = TRUE;
   else if(general_preferences.use_spaces && !file->props.over_ride) value = TRUE;
   return(value);
}

/* These will probably leak a great deal */
char *get_full_filename(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(file != NULL, NULL);
   g_return_val_if_fail(GDS_IS_FILE(file), NULL);
   return(g_strdup(file->filename));
}

char *get_filename(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(file != NULL, NULL);
   g_return_val_if_fail(GDS_IS_FILE(file), NULL);
   return(get_file_from_filename(file->filename));
}

char *get_path(int file_no)
{
   GdsFile *file;
   gchar *path;
   gchar *tmp;

   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(file != NULL, NULL);
   g_return_val_if_fail(GDS_IS_FILE(file), NULL);
   path = get_path_from_filename(file->filename);
   if((tmp = strstr(path, "://")))
   {
      tmp += 3;
      tmp = g_strdup(tmp);
      g_free(path);
      path = tmp;
   }
   return(path);
}

char *get_file(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(file != NULL, NULL);
   g_return_val_if_fail(GDS_IS_FILE(file), NULL);
   return(get_file_wo_extension(file->filename));
}

char *get_file_compile_string(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(file != NULL, NULL);
   g_return_val_if_fail(GDS_IS_FILE(file), NULL);
   return(g_strdup(file->props.compiler));
}

char *get_file_debug_string(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(file != NULL, NULL);
   g_return_val_if_fail(GDS_IS_FILE(file), NULL);
   return(g_strdup(file->props.debugger));
}

char *get_file_execute_string(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(file != NULL, NULL);
   g_return_val_if_fail(GDS_IS_FILE(file), NULL);
   return(g_strdup(file->props.execution));
}

void set_file_done_compiling(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_if_fail(file != NULL);
   g_return_if_fail(GDS_IS_FILE(file));
   if(file->compile_pid == -1) file->compile_pid = 0;
   if(file == cur_file)
   {
      set_build_sensitivity();
   }
}

void set_file_done_debugging(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_if_fail(file != NULL);
   g_return_if_fail(GDS_IS_FILE(file));
   if(file->debug_pid == -1) file->debug_pid = 0;
   if(file == cur_file)
   {
      set_build_sensitivity();
   }
}

void set_file_done_executing(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_if_fail(file != NULL);
   g_return_if_fail(GDS_IS_FILE(file));
   if(file->exec_pid == -1) file->exec_pid = 0;
   if(file == cur_file)
   {
      set_build_sensitivity();
   }
}

int has_selection(void)
{
   gint value;
   GtkExText *text;
   g_return_val_if_fail(cur_file, 0);
   text = GTK_EXTEXT(cur_file->text);
   value = (int)GTK_EXTEXT(cur_file->text)->has_selection;
   return(value);   
}

int selection_start(void)
{
   int value;
   GtkExText *text;
   g_return_val_if_fail(cur_file, -1);
   text = GTK_EXTEXT(cur_file->text);
   value = (int)text->selection_start_pos;
   return(value);
}

int selection_end(void)
{
   int value;
   GtkExText *text;
   g_return_val_if_fail(cur_file, -1);
   text = GTK_EXTEXT(cur_file->text);
   value = (int)text->selection_end_pos;
   return(value);
}

void move(int relative_movement)
{
   int index;
   g_return_if_fail(cur_file);
   index = gtk_extext_get_position(GTK_EXTEXT(cur_file->text));
   index += relative_movement;
   g_return_if_fail(index >= 0);
   gtk_extext_set_position(GTK_EXTEXT(cur_file->text), index);
}

void move_to(int index)
{
   g_return_if_fail(cur_file);
   g_return_if_fail(index >= 0);
   gtk_extext_set_position(GTK_EXTEXT(cur_file->text), index);
}

int current_position(void)
{
   g_return_val_if_fail(cur_file, -1);
   return gtk_extext_get_position(GTK_EXTEXT(cur_file->text));
}

void move_to_line (int index)
{
   g_return_if_fail(cur_file);
   g_return_if_fail(index >= 1);
   gtk_extext_set_line(GTK_EXTEXT(cur_file->text), index-1);
}

void move_lines(int relative_movement)
{
   int index;
   g_return_if_fail(cur_file);
   index = gtk_extext_get_line(GTK_EXTEXT(cur_file->text));
   gtk_extext_set_line(GTK_EXTEXT(cur_file->text),index+relative_movement);
}

int current_line(void)
{
   g_return_val_if_fail(cur_file, -1);
   return GTK_EXTEXT(cur_file->text)->line_number + 1;
}

int line_start(void)
{
   GtkExText *text;
   GtkExTextLineData *linedata;
   gint start = 0;
   g_return_val_if_fail(cur_file, -1);
   text = GTK_EXTEXT(cur_file->text);
   linedata = gtk_extext_get_line_data(text, text->line_number, text->scroll_line_start);
   if(linedata) start = linedata->startpos;
   g_free(linedata);
   return(start);
}

int line_end(void)
{
   GtkExText *text;
   GtkExTextLineData *linedata;
   gint end;
   g_return_val_if_fail(cur_file, -1);
   text = GTK_EXTEXT(cur_file->text);
   end = text->length;
   linedata = gtk_extext_get_line_data(text, text->line_number, text->scroll_line_start);
   if(linedata) end = linedata->endpos;
   g_free(linedata);
   return(end);
}

int freeze(void)
{
   GtkExText *text;
   g_return_val_if_fail(cur_file, FALSE);
   text = GTK_EXTEXT(cur_file->text);
   gtk_extext_freeze(text);
   return(TRUE);
}

int thaw(void)
{
   GtkExText *text;
   g_return_val_if_fail(cur_file, FALSE);
   text = GTK_EXTEXT(cur_file->text);
   gtk_extext_thaw(text);
   return(TRUE);
}

int line_by_pos(int pos)
{
   GtkExTextLineData *linedata;
   g_return_val_if_fail(cur_file, -1);
   linedata = gtk_extext_get_line_by_char_pos(GTK_EXTEXT(cur_file->text), pos, NULL);
   pos = linedata ? linedata->line_number+1 : -1;
   return(pos);
}

void move_to_col(int index)
{
   g_return_if_fail(cur_file);
   g_return_if_fail(index >= 0);
   gtk_extext_set_column(GTK_EXTEXT(cur_file->text), index);
}

char* get_text(int begin, int end)
{
   g_return_val_if_fail(cur_file, NULL);
   return gtk_extext_get_chars(GTK_EXTEXT(cur_file->text), begin, end);
}

char get_current_char(void)
{
   int index;
   g_return_val_if_fail(cur_file, -1);
   index = gtk_extext_get_position(GTK_EXTEXT(cur_file->text));
   return GTK_EXTEXT_INDEX(GTK_EXTEXT(cur_file->text), index);
}

char get_next_char(void)
{
   char c;
   int index;
   g_return_val_if_fail(cur_file, -1);
   index = gtk_extext_get_position(GTK_EXTEXT(cur_file->text));
   c = GTK_EXTEXT_INDEX(GTK_EXTEXT(cur_file->text),index);
   index++;
   gtk_extext_set_position(GTK_EXTEXT(cur_file->text), index);
   return(c);
}

int buffer_size(void)
{
   g_return_val_if_fail(cur_file, -1);
   return gtk_extext_get_length(GTK_EXTEXT(cur_file->text));
}

void insert_in(int file_no, int pos, char* text)
{
   GdsEditor *editor;
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_if_fail(file != NULL);
   g_return_if_fail(text != NULL);
   editor = GDS_EDITOR(file->text);
   gds_editor_insert_text(editor, text, strlen(text), (gint *)&pos);
}

void insert(char* text)
{
   GdsEditor *editor;
   GtkExText *extext;
   g_return_if_fail(text != NULL);
   editor = GDS_EDITOR(cur_file->text);
   extext = GTK_EXTEXT(cur_file->text);
   gds_editor_insert_text(editor, text, strlen(text), (gint *)&extext->current_pos);
}

void delete_text(int start, int end)
{
   g_return_if_fail(cur_file);
   delete_text_in(get_file_number(), start, end);
}

void delete_text_in(int file_no, int start, int end)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_if_fail(file != NULL);
   gds_editor_delete_text(GDS_EDITOR(file->text), start, end);
}

void forward_delete(int n)
{
   int cur_pos;
   g_return_if_fail(cur_file);
   cur_pos = current_position();
   gds_editor_delete_text(GDS_EDITOR(cur_file->text), cur_pos, cur_pos + n);
}

void backward_delete(int n)
{
   int cur_pos;
   g_return_if_fail(cur_file);
   cur_pos = current_position();
   gds_editor_delete_text(GDS_EDITOR(cur_file->text), cur_pos-n, cur_pos);
}

void undo(void)
{
   g_return_if_fail(cur_file);
   undo_last_real(cur_file);
}

void undo_last(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   undo_last_real(file);
}

void redo(void)
{
   g_return_if_fail(cur_file);
   redo_last_real(cur_file);
}

void redo_last(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   redo_last_real(file);
}

void cut(void)
{
   g_return_if_fail(cur_file);
   cut_text_real(cur_file);
}

void cut_text(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   cut_text_real(file);
}

void copy(void)
{
   g_return_if_fail(cur_file);
   copy_text_real(cur_file);
}

void copy_text(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   copy_text_real(file);
}

void paste(void)
{
   g_return_if_fail(cur_file);
   paste_text_real(cur_file);
}

void paste_text(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   paste_text_real(cur_file);
}

void clear(void)
{
   g_return_if_fail(cur_file);
   clear_text_real(cur_file);
}

void clear_text(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   clear_text_real(file);
}

void clear_entry(void)
{
   clear_entry_real();
}

void select_area(int from, int to)
{
   g_return_if_fail(cur_file);
   gtk_extext_set_position(GTK_EXTEXT(cur_file->text), to);
   gtk_extext_select_region(GTK_EXTEXT(cur_file->text), from, to);
   gtk_widget_grab_focus(cur_file->text);
}

void select_text(int file_no, int from, int to)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_if_fail(file);
   gtk_extext_set_position(GTK_EXTEXT(cur_file->text), to);
   gtk_extext_select_region(GTK_EXTEXT(file->text), from, to);
   gtk_widget_grab_focus(cur_file->text);
}

void select_current(void)
{
   g_return_if_fail(cur_file);
   select_current_real(cur_file);
}

void select_all(void)
{
   g_return_if_fail(cur_file);
   select_all_text_real(cur_file);
}

void select_all_text(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   select_all_text_real(file);
}

void highlight_syntax(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   highlight_syntax_real(file);
}

void new_window(void)
{
   new_window_cb(NULL, NULL);
}

void new_file(char *filename)
{
   new_file_real(filename);
}

void open_file(char *filename)
{
   open_file_real(filename);
}

void close_file(int file_no)
{
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   close_file_real(file);
}

void save_all_files(void)
{
   save_all_files_cb(NULL, NULL);
}

void close_all_files(void)
{
   close_all_files_cb(NULL, NULL);
}

int find(char *text)
{
	g_return_val_if_fail(cur_file, FALSE);
   return(find_in(get_file_number(), text));
}

int find_in(int file_no, char *text)
{
   int success = FALSE;
   GtkExTextMatch m;
   GtkExText *extext;
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(cur_file, FALSE);
   extext = GTK_EXTEXT(file->text);
   success = (int)find_text(extext, extext->current_pos, extext->length, text, FALSE, FALSE, TRUE, &m);
#ifdef WITH_PYTHON
   gtk_widget_grab_focus(macro_entry);
#endif
   return(success);
}

int find_regex(char *text)
{
   g_return_val_if_fail(cur_file, FALSE);
   return(find_regex_in(get_file_number(), text));
}

int find_regex_in(int file_no, char *text)
{
   int success = FALSE;
   GtkExTextMatch m;
   GtkExText *extext;
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(cur_file, FALSE);
   extext = GTK_EXTEXT(file->text);
   success = (int)find_text(extext, extext->current_pos, extext->length, text, FALSE, TRUE, FALSE, &m);
#ifdef WITH_PYTHON
   gtk_widget_grab_focus(macro_entry);
#endif
   return(success);
}

int replace(char *search_text, char *replacer)
{
   g_return_val_if_fail(cur_file, FALSE);
   return(replace_in(get_file_number(), search_text, replacer));
}

int replace_in(int file_no, char *search_text, char *replacer)
{
   int success = FALSE;
   GtkExTextMatch m;
   GtkExText *text;
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
	g_return_val_if_fail(cur_file, FALSE);
   text = GTK_EXTEXT(file->text);
   if(find_text(text, text->current_pos, text->length, search_text, TRUE, FALSE, TRUE, &m))
   {
      replace_text(text, replacer);
      success = TRUE;
   }
#ifdef WITH_PYTHON
   gtk_widget_grab_focus(macro_entry);
#endif
   return(success);
}

int replace_all(char *search_text, char *replacer)
{
   g_return_val_if_fail(cur_file, FALSE);
   return(replace_all_in(get_file_number(), search_text, replacer));
}

int replace_all_in(int file_no, char *search_text, char *replacer)
{
   int success = FALSE;
   GdsFile *file;
   file = (GdsFile *)g_list_nth_data(files_list, file_no);
   g_return_val_if_fail(cur_file, FALSE);
   success = (int)replace_all_text(GTK_EXTEXT(file->text), search_text, replacer, TRUE, FALSE, GTK_EXTEXT(file->text)->length);
   return(success);
}

/*
 * End of regular bindings, start of syntax highlighting bindings for language definitions
*/

void add_syntax_entry(char *name, char *start, char *end, char *color, char *cclass)
{
   gchar *start_string;
   gchar *end_string;
   HighlightInfo *info;
   if(!parsing_language) return;
   start_string = g_str_replace_bad_chars(start);
   end_string = g_str_replace_bad_chars(end);
   info = get_highlight_info(color, cclass);
   syntax_list = gtk_extext_syntax_entry_new(syntax_list, name, start_string, end_string, info->color, NULL, info->font, info->flags, GINT_TO_POINTER(info->type));
   g_free(start_string);
   g_free(end_string);
   g_free(info);
}

void add_pattern_entry(char *name, char *words, char *color, char *cclass)
{
   gchar *words_string;
   HighlightInfo *info;
   if(!parsing_language) return;
   words_string = g_str_replace_bad_chars(words);
   info = get_highlight_info(color, cclass);
   pattern_list = gtk_extext_pattern_entry_new(pattern_list, name, words_string, info->color, NULL, info->font, info->flags, GINT_TO_POINTER(info->type));
   g_free(words_string);
   g_free(info);
}

void add_embedded_entry(char *name, char *outside, char *inside, char *color, char *cclass)
{
   gchar *outside_string;
   gchar *inside_string;
   HighlightInfo *info;
   if(!parsing_language) return;
   outside_string = g_str_replace_bad_chars(outside);
   inside_string = g_str_replace_bad_chars(inside);
   info = get_highlight_info(color, cclass);
   embedded_list = gtk_extext_embedded_entry_new(embedded_list, name, outside_string, inside_string, info->color, NULL, info->font, info->flags, GINT_TO_POINTER(info->type));
   g_free(outside_string);
   g_free(inside_string);
   g_free(info);
}
