/* source.c

 * Copyright (C) 2002 Claudio Girardi
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include <stdlib.h>
#include <stdio.h>
#include <gtk/gtk.h>
#include "source.h"
#include "audio.h"
#include "fft.h"
#include "mtm.h"
#include "hparma.h"
#include "util.h"
#include "glfer.h"
#include "wav_fmt.h"
#include "g_main.h"

#ifdef HAVE_DMALLOC_H
#include <dmalloc.h>		/* dmalloc.h should be included last */
#endif


static int audio_fd;
static int psd_n;
static char *fname = NULL;
static float *psdbuf = NULL;
static gint input_tag = -1;
static int file_ended = 0;

perf_timer_t perf_timer, proc_timer;

extern opt_t opt;
extern glfer_t glfer;
extern fft_params_t fft_gram_par;
extern mtm_params_t mtm_gram_par;
extern hparma_params_t hparma_gram_par;

static float getclock(void)
{

  clock_t nclock;
  float nsec;

  nclock = clock();
  nsec = (float) nclock / CLOCKS_PER_SEC;

  return nsec;
}


static void timefn(void)
{
#ifdef HAVE_GETTIMEOFDAY
  struct timeval tv;
  double newtime;
  static double oldtime;

  if (gettimeofday(&tv, NULL))
    perror("gettimeofday");
  newtime = (double) tv.tv_sec + (double) tv.tv_usec / 1.0e06;

  printf("%e %e %e\n", newtime, oldtime, newtime - oldtime);

  oldtime = newtime;
#endif /* HAVE_GETTIMEOFDAY */
}


/* reads the audio data, calls the fft processing function and the drawing
 * function
 * (called from within gtk_main whenever data is available */
static void audio_available(gpointer data, gint source, GdkInputCondition condition)
{
  float *audio_buf;
  int i, n;
  int n_eff = opt.fft_n * (1.0 - opt.fft_overlap);

  if (fname) {
    wav_read(&audio_buf, &n);
    if (n == 0) {
      file_ended = 1;
      stop_reading_audio();
      close_wav_file();
    }
  } else {
    /* fill audio_buf */
    audio_read(&audio_buf, &n);
    /* n is the number of blocks of requested size read from the audio device */ }

  for (i = 0; i < n; i++) {
    /* get current time */
    perf_timer.stop = tc_time();
    /* elapsed time since last call */
    perf_timer.delta = perf_timer.stop - perf_timer.start;
    glfer.cpu_usage = proc_timer.delta / perf_timer.delta;
    /* start new period */
    perf_timer.start = perf_timer.stop;
    /* start new data processing period */
    proc_timer.start = perf_timer.stop;

    switch (glfer.current_mode) {
    case MODE_FFT :
      fft_do(audio_buf + i * n_eff, &fft_gram_par);
      fft_psd(psdbuf, NULL, &fft_gram_par);
      break;
    case MODE_MTM :
      mtm_do(audio_buf + i * n_eff, psdbuf, NULL, &mtm_gram_par);
      break;
    case MODE_HPARMA :
      hparma_do(audio_buf + i * n_eff, psdbuf, NULL, &hparma_gram_par);
      break;
    default:
      g_print("audio_available: MODE_UNKNOWN\n");
    }
    /* jason_do(audio_buf + i * opt.fft_n, opt.fft_n); */
    main_window_draw(psdbuf);
    
    /* end of data processing */
    proc_timer.stop = tc_time();
    proc_timer.delta = proc_timer.stop - proc_timer.start;
  }
}


void init_audio(char *filename)
{
  int n_eff = opt.fft_n * (1.0 - opt.fft_overlap);

  fname = filename;

  if (fname) {
    audio_fd = open_wav_file(fname, n_eff);
    file_ended = 0;
  } else {
    audio_fd = audio_init(opt.audio_device, &opt.sample_rate, n_eff);
  }
}


void close_audio(void)
{
  /* close current source */
  if (fname == NULL) {
    audio_close();
  } else {
    close_wav_file();
  }
}


void toggle_stop_start(GtkWidget * widget, gpointer data)
{
  if (input_tag == -1) {
    start_reading_audio();
  } else {
    stop_reading_audio();
  }
}


void start_reading_audio()
{
  int n_eff = opt.fft_n * (1.0 - opt.fft_overlap);

  if (file_ended == 1) {
    audio_fd = open_wav_file(fname, n_eff);
    file_ended = 0;
  }
  /* audio_available() has to be called whenever there's data on audio_fd */
  input_tag = gdk_input_add(audio_fd, GDK_INPUT_READ, audio_available, NULL);
  //input_tag = g_idle_add((GSourceFunc)audio_available, NULL);
  stop_start_button_set_label("Stop");
}


void stop_reading_audio()
{
  if (input_tag != -1) {
    gtk_input_remove(input_tag);
    input_tag = -1;
    stop_start_button_set_label("Start");
  }
}


void change_params(int mode_sel)
{
  /* check if this is the first call at program start-up */
  if (glfer.current_mode != MODE_NONE) {
    /* stop reading from the current data source */
    stop_reading_audio();
    /* close current mode */
    switch (glfer.current_mode) {
    case MODE_FFT:
      fft_close(&fft_gram_par);
      break;
    case MODE_MTM:
      mtm_close(&mtm_gram_par);
      break;
    case MODE_HPARMA:
      hparma_close(&hparma_gram_par);
      break;
    default:
      g_print("change_params: MODE_UNKNOWN\n");
    }

    FREE_MAYBE(psdbuf);
  }

  /* initialize mode */
  switch (mode_sel) {
  case MODE_FFT :
    if ((fft_gram_par.n != opt.fft_n) || (fft_gram_par.overlap != opt.fft_overlap)) {
      close_audio();
      init_audio(fname);
    }

    psd_n = opt.fft_n / 2 + 1;
    psdbuf = calloc(psd_n, sizeof(float));
    
    fft_gram_par.n = opt.fft_n;
    fft_gram_par.window_type = opt.window_type;
    fft_gram_par.overlap = opt.fft_overlap;
    fft_gram_par.a = opt.fft_a;
    fft_gram_par.limiter = opt.fft_limiter;
    fft_init(&fft_gram_par);
    break;
  case MODE_MTM :
    if ((mtm_gram_par.fft.n != opt.fft_n) || (mtm_gram_par.fft.overlap != opt.fft_overlap)) {
      close_audio();
      init_audio(fname);
    }

    psd_n = opt.mtm_n / 2 + 1;
    psdbuf = calloc(psd_n, sizeof(float));

    mtm_gram_par.fft.n = opt.mtm_n;
    mtm_gram_par.fft.window_type = NO_WINDOW;
    mtm_gram_par.fft.overlap = opt.mtm_overlap;
    mtm_gram_par.fft.a = opt.fft_a;
    mtm_gram_par.fft.limiter = opt.fft_limiter;
    mtm_gram_par.w = opt.mtm_w;
    mtm_gram_par.kmax = opt.mtm_k;
    mtm_init(&mtm_gram_par);
    break;
  case MODE_HPARMA :
    if ((hparma_gram_par.fft.n != opt.fft_n) || (hparma_gram_par.fft.overlap != opt.fft_overlap)) {
      close_audio();
      init_audio(fname);
    }

    psd_n = opt.hparma_n / 2 + 1;
    psdbuf = calloc(psd_n, sizeof(float));

    hparma_gram_par.fft.n = opt.hparma_n;
    hparma_gram_par.fft.window_type = NO_WINDOW;
    hparma_gram_par.fft.overlap = opt.hparma_overlap;
    hparma_gram_par.fft.a = opt.fft_a;
    hparma_gram_par.fft.limiter = opt.fft_limiter;
    hparma_gram_par.t = opt.hparma_t;
    hparma_gram_par.p_e = opt.hparma_p_e;
    hparma_gram_par.q_e = -1;
    hparma_init(&hparma_gram_par);
    break;
  default:
    g_print("change_params: MODE_UNKNOWN\n");
  }

  glfer.current_mode = mode_sel;

  main_window_init(psd_n, 0, (float) opt.sample_rate / 2);

  start_reading_audio();
}
