// $Id: glade--.cc,v 1.38 2002/03/01 21:50:23 christof Exp $
/*  glade--: C++ frontend for glade (Gtk+ User Interface Builder)
 *  Copyright (C) 1998  Christof Petig
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "TagStream.hh"
#include "Cxx.hh"
#include "Configuration.hh"
#include <unistd.h>
#include <cstdlib>
#include <sys/stat.h> // for makedir
#include <getopt.h>
#include <cstdio>

#ifndef HAVE_GETOPT_LONG
#include "./getopt.h"
#endif

struct Configuration Configuration;

enum longopts { GTKVERSION=0x100, GTKMMVERSION, AUTOCONF,
	GTK_OSTREAM,FORCE_ACCEL,FORCE_NOACCEL, TEMPL_POSTFIX, OM_STRINGS,
	OM_CREATE_ENUMS, CASE_CONV, UPPERCASE_ENUMS, MIXEDCASE_DEFINES,
	OM_NO_CREATE_ENUMS };

const static struct option options[]=
{ { "directory",	required_argument, NULL, 'd' },
  { "name",		required_argument, NULL, 'm' },
  { "version",		no_argument,	   NULL, 'V' },
  { "debug",		no_argument,	   NULL, 'g' },
  { "samplecode",       no_argument,       NULL, 's' },
  { "barebones",        no_argument,       NULL, 'r' },
  { "gladeonly",        no_argument,       NULL, 'r' },
  { "noautoconf",       no_argument,       NULL, 'A' },
  { "autoconf",         no_argument,       NULL, AUTOCONF },
  { "help",		no_argument,       NULL, '?' },
  { "gtkversion",	required_argument, NULL, GTKVERSION },
  { "gtkmmversion",	required_argument, NULL, GTKMMVERSION },
  { "gtkostream",	no_argument,	   NULL, GTK_OSTREAM },
  // compatibility with old versions
  { "widgettable",      no_argument,       NULL, 'w' },
  { "fulltable",        no_argument,       NULL, 'w' },

  { "libglade",		no_argument,	   NULL, 'l' },
  { "accelerators",	no_argument,	   NULL, FORCE_ACCEL },
  { "noaccelerators",	no_argument,	   NULL, FORCE_NOACCEL },
  { "template-postfix",	required_argument, NULL, TEMPL_POSTFIX },
  { "optionmenu-strings", no_argument,	   NULL, OM_STRINGS },
  { "optionmenu-no-create-enums", no_argument,NULL, OM_NO_CREATE_ENUMS },
  { "optionmenu-create-enums", no_argument,NULL, OM_CREATE_ENUMS },
  { "do-case-conversion", no_argument,	   NULL, CASE_CONV },
  { "uppercase-enums",	no_argument,	   NULL, UPPERCASE_ENUMS },
  { "mixedcase-defines", no_argument,	   NULL, MIXEDCASE_DEFINES },
  
  { "header-suffix",	required_argument, NULL, 'h' },
  { "source_suffix",	required_argument, NULL, 'c' },
  
  { NULL,		0,		   NULL, 0 }
};

static bool parse_version(const char *arg,Gtk_Version &v)
{  int mymajor=0,myminor=0,mymicro=0;
   /* const */ char *nextp;
   mymajor=strtol(arg,&nextp,10);
   if (!*nextp) { v=Gtk_Version(mymajor,myminor,mymicro); return true; }
   if (*nextp!='.' && *nextp!=',')
   {  std::cerr << "error parsing version std::string '" << arg << "\n";
      return false;
   }
   myminor=strtol(nextp+1,&nextp,10);
   if (!*nextp) { v=Gtk_Version(mymajor,myminor,mymicro); return true; }
   if (*nextp!='.' && *nextp!=',')
   {  std::cerr << "error parsing version std::string '" << arg << "\n";
      return false;
   }
   mymicro=strtol(nextp+1,&nextp,10);
   if (!*nextp) { v=Gtk_Version(mymajor,myminor,mymicro); return true; }
   std::cerr << "error parsing version std::string '" << arg << "\n";
   return false;
}

static void call_gtkmm_config()
{  char buf[80];
   FILE *f=popen("gtk-config --version","r");
   if (f)
   {  if (fgets(buf,sizeof(buf),f))
      {  for (unsigned char *s=(unsigned char*)buf;*s;s++)
            if (isspace(*s)) 
            {  *s=0; break; }
         Gtk_Version v;
         if (parse_version(buf,v))
            Configuration.gtk_version=v;
         else std::cerr << "gtk-config --version: strange result '" << buf << "'\n";
      }
      pclose(f);
   }
   else perror("gtk-config");
   f=popen("gtkmm-config --version","r");
   if (f)
   {  if (fgets(buf,sizeof(buf),f))
      {  for (unsigned char *s=(unsigned char*)buf;*s;s++)
            if (isspace(*s)) 
            {  *s=0; break; }
         Gtk_Version v;
         if (parse_version(buf,v))
            Configuration.gtkmm_version=v;
         else std::cerr << "gtkmm-config --version: strange result '" << buf << "'\n";
      }
      pclose(f);
   }
   else perror("gtkmm-config");
}

static void ApplyProject(const Tag &top)
{  Tag::const_iterator t=top.begin();
   if (t!=top.end() && t->Type()=="?xml") ++t; 
   if (t!=top.end() && !(t->Type().size())) ++t; 
   if (t==top.end())
   {  std::cerr << "Empty .glade file, exiting\n"; exit(1); }
   if (t->Type()!="GTK-Interface") 
   {  std::cerr << "Can't find GTK-Interface tag\n"; return; }
   t=t->begin();
   if (t->Type()!="project") 
   {  std::cerr << "Can't find project tag\n"; return; }
   if (!Configuration.main_filename.size() && t->hasTag("program_name"))
      Configuration.main_filename=t->getString("program_name","error");
   
   Configuration.source_directory="/"+t->getString("source_directory","src");
   if (Configuration.source_directory[1]=='/')
   {  std::cerr << "I can't handle an absolute source directory ("
   	<< (Configuration.source_directory.c_str()+1) << ")\n";
      exit(2);
   }
   
   Configuration.destination_directory=t->getString("directory",".");
   if (!Configuration.destination_directory.size())
   	Configuration.destination_directory=".";
   if (Configuration.destination_directory!=".")
   {  std::cerr << "I'm confused about the directory to write to, \n"
           "by default I'm writing relative to the .glade file.\n"
           "\nPlease report this incident to christof.petig@wtal.de for future fixing.\n\n";
      Configuration.destination_directory=".";
   }
   
   if (access(Configuration.destination_directory.c_str(),F_OK))
      if (mkdir(Configuration.destination_directory.c_str(),0755))
      {  perror(Configuration.destination_directory.c_str());
         exit(2);
      }
   const std::string srcdir(Configuration.destination_directory+Configuration.source_directory);
   if (access(srcdir.c_str(),F_OK))
      if (mkdir(srcdir.c_str(),0755))
      {  perror(srcdir.c_str());
         exit(2);
      }
   
   Configuration.pixmap_dir=t->getString("pixmaps_directory","pixmaps");
   // XXX make pixmaps directory relative to source
   if ((!Configuration.pixmap_dir.size() || Configuration.pixmap_dir[0]!='/'))
      // blind guess, but better than nothing
   {  if (Configuration.source_directory.size())
         Configuration.pixmap_dir_relative_to_src="../"+Configuration.pixmap_dir;
      else // src should be current (?)
         Configuration.pixmap_dir_relative_to_src=Configuration.pixmap_dir;
   }
   else 
      Configuration.pixmap_dir_relative_to_src=Configuration.pixmap_dir;
   if ((!Configuration.pixmap_dir.size() || Configuration.pixmap_dir[0]!='/'))
   // relative
   {  Configuration.pixmap_dir=Configuration.destination_directory+'/'+Configuration.pixmap_dir;
   }
   
   Configuration.gnome_support=t->getBool("gnome_support");
   Configuration.gettext_support=t->getBool("gettext_support");
   Configuration.widget_names=t->getBool("use_widget_names");
//   Configuration.output_main_file=t->getBool("output_main_file");
// ...   
   std::string podir(Configuration.destination_directory+"/po");
   if (!Configuration.destination_directory.size()) podir="po";
   if (Configuration.gettext_support && access(podir.c_str(),F_OK))
      if (mkdir(podir.c_str(),0755))
      {  perror(podir.c_str());
         exit(2);
      }
}

int main(int argc,char **argv)
{  int opt;
   
   call_gtkmm_config();
   for (int i=0;i<argc;i++)
   {  if (i) Configuration.commandline+=' ';
      Configuration.commandline+=argv[i];
   }
   while ((opt=getopt_long(argc,argv,"d:m:h:c:Vgrs1AwlN",options,NULL))!=EOF) 
    switch(opt)
   {  case 'd': Configuration.destination_directory=optarg;
         break;
      case 'm': Configuration.main_filename=optarg;
         break;
      case 'c': Configuration.source_suffix=optarg;
      	 break;
      case 'h': Configuration.header_suffix=optarg;
         break;
      case 'V': std::cout<< "glademm V"VERSION" (glade to Gtk-- converter)\n";
         return 0;
         break;
      case 'g': Configuration.debug=true;
         break;
      case 's': Configuration.sample_code=true;
         break;
      case 'r': Configuration.bare_bones=true;
         break;
      case 'A': Configuration.no_autoconf=true;
         break;
      case 'w': Configuration.lookup_table_compat=true;
         break;
      case 'l': Configuration.use_libglade=true;
         break;
      case GTKVERSION: 
         {  Gtk_Version v;
            if (parse_version(optarg,v))
               Configuration.gtk_version=v;
            else return 1;
         }
         break;
      case GTKMMVERSION: 
         {  Gtkmm_Version v;
            if (parse_version(optarg,v))
               Configuration.gtkmm_version=v;
            else return 1;
         }
         break;
      case GTK_OSTREAM: Configuration.gtk_ostream=true;
         break;
      case FORCE_ACCEL: Configuration.has_accelerators=true;
         break;
      case FORCE_NOACCEL: Configuration.has_accelerators=false;
         break;
      case TEMPL_POSTFIX: Configuration.template_postfix=optarg;
         break;
      case OM_STRINGS: Configuration.optionmenu_strings=true;
         break;
      case OM_CREATE_ENUMS: Configuration.optionmenu_create_enum=true;
         break;
      case OM_NO_CREATE_ENUMS: Configuration.optionmenu_create_enum=false;
         break;
      case UPPERCASE_ENUMS: Configuration.uppercase_enums=true;
         break;
      case CASE_CONV: Configuration.do_case_conversion=true;
         break;
      case MIXEDCASE_DEFINES: Configuration.mixedcase_defines=true;
         break;
      case '?': usage:
         std::cerr << "Since glade-- now honors glade's project settings, there should be no need\n"
         	 "to pass any options. Simply create the code within glade.\n";
         std::cerr << "\n"
         	 "But if you plan to use experimental/compatibility features, the appropriate options are:\n"
         	 "\t--[no]accelerators\tforce accelerator code support on/off\n"
         	 "\t\t\tBEWARE: SWITCHING THIS BREAKS CODE COMPATIBILITY.\n"
         	 "\t\t\t(accelerator enabled code might get default in the future).\n"
         	 "\t--do-case-conversion\trestore old glade-- setting:\n"
         	 "\t\t\tlowercase filenames, variables (instances)\n"
         	 "\t\t\tmixedcase class names\n"
         	 "\n"
         	 "\t--gladeonly\tonly generate *_glade.?? files\n"
         	 "\t--gtk[mm]version\tgenerate code for specific version\n"
         	 "\t--template-postfix <extension>\tif you don't like 'foo.cc_new'\n"
         	 "\t--optionmenu-strings\tpass label (char*) as user_data to optionmenu items\n"
         	 "\t--optionmenu-create-enums\tcreate enum for optionmenu labels\n"
         	 "\t--uppercase-enums\tuppercase enum labels\n"
         	 "\t--mixedcase-defines\tdo not uppercase preprocessor symbols (e.g. GMM_Foo)\n"
         	 "\t--libglade\tgenerate code skeleton for a libglade-- application.\n"
         	 "\t\t\tTHIS DOES NOT WORK, YET.\n"
         	 "\t--noautoconf\tdisable autoconf/make. You are kidding ;-)\n"
         	 "\t--version\tprints 'glademm V"VERSION"'\n";
         return 1;
   }
   if (argc-optind!=1) goto usage;
   Configuration.in_filename=argv[optind];
   // there should be a way to do this with STL
   {  char buf[10240];
      strncpy(buf,argv[optind],sizeof buf);
      buf[sizeof(buf)-1]=0;
      char *s=strrchr(buf,'/');
      if (s)
      {  Configuration.in_filename=s+1;
         *s=0;
         if (*buf) 
         {  if (!chdir(buf)) std::cout << "Changed dir to "<<buf<<'\n';
            else { perror(buf); exit(2); }
         }
      }
   }
   
   std::cout << "Generating code for gtk "
        << Configuration.gtk_version.mymajor << '.'
	<< Configuration.gtk_version.myminor << '.'
	<< Configuration.gtk_version.mymicro <<
	", gtkmm " << Configuration.gtkmm_version.mymajor << '.'
	<< Configuration.gtkmm_version.myminor << '.'
	<< Configuration.gtkmm_version.mymicro << '\n';
   TagStream top(Configuration.in_filename);
   ApplyProject(top);
   if (Configuration.gettext_support && !Configuration.gnome_support)
   {  std::cout << "Gettext support is not implemented without gnome.\n"
   	"Disabling gettext\n";
      Configuration.gettext_support=false;
   }
   Cxx *cxx=new Cxx(&top);
   cxx->WriteTags();
   delete cxx;
}
