/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <dlfcn.h>
#include "gkrellm.h"
#include "gkrellm_private_proto.h"

#include <dirent.h>


/* ======================================================================= */
/* Plugin interface to GKrellM.
*/

gchar *
gkrellm_get_theme_path(void)
	{
	return GK.theme_path;
	}

Krell *
gkrellm_krell_new0(void)
	{
	Krell *k;

	k = g_new0(Krell, 1);
	return k;
	}

Decal *
gkrellm_decal_new0(void)
	{
	Decal	*d;

	d = g_new0(Decal, 1);
	return d;
	}


Label *
gkrellm_label_new0(void)
	{
	Label *l;

	l = g_new0(Label, 1);
	return l;
	}


Style *
gkrellm_style_new0(void)
	{
	Style	*s;

	s = g_new0(Style, 1);
	return s;
	}

Style *
gkrellm_copy_style(Style *style)
	{
	Style *s = NULL;

	if (style)
		{
		s = gkrellm_style_new0();
		*s = *style;
		}
	return s;
	}

void
gkrellm_copy_style_values(Style *src, Style *dst)
	{
	if (src && dst)
		*src = *dst;
	}

gboolean
gkrellm_style_is_themed(Style *style)
	{
	return style->themed;
	}

TextStyle *
gkrellm_textstyle_new0(void)
	{
	TextStyle	*t;

	t = g_new0(TextStyle, 1);
	return t;
	}

TextStyle *
gkrellm_copy_textstyle(TextStyle *ts)
	{
	TextStyle *t	= gkrellm_textstyle_new0();

	*t = *ts;
	return t;
	}

Chart *
gkrellm_chart_new0(void)
	{
	Chart	*c;

	c = g_new0(Chart, 1);
	return c;
	}

ChartConfig *
gkrellm_chartconfig_new0(void)
	{
	ChartConfig	*config;

	config = g_new0(ChartConfig, 1);
	return config;
	}

Panel *
gkrellm_panel_new0(void)
	{
	Panel *p;

	p = g_new0(Panel, 1);
	p->label = gkrellm_label_new0();
	return p;
	}


static Style *
get_style_from_list(GList *list, gint n)
	{
	GList	*l;
	Style	*style;

	l = g_list_nth(list, n);
	if (l == NULL || l->data == NULL)
		l = list;
	if (l->data == NULL)
		{
		printf("Warning: NULL style returned %d\n", n);
		abort();
		}
	style = (Style *) l->data;
	return style;
	}

Style *
gkrellm_meter_style(gint n)
	{
	return get_style_from_list(GK.meter_style_list, n);
	}

Style *
gkrellm_panel_style(gint n)
	{
	return get_style_from_list(GK.panel_style_list, n);
	}

Style *
gkrellm_chart_style(gint n)
	{
	return get_style_from_list(GK.chart_style_list, n);
	}

static Style *
get_style_from_list_by_name(GList *name_list, GList *style_list, gchar *name)
	{
	GList	*list;
	gchar	*p, buf[128];
	gint	n;

	strncpy(buf, name, sizeof(buf));
	buf[sizeof(buf) - 1] = '\0';
	if (   (p = strchr(buf, '.')) != NULL
		&& (n = string_position_in_list(GK.custom_name_list, name)) >= 0
	   )
		list = GK.custom_style_list;
	else
		{
		if (p)
			*p = '\0';
		if ((n = string_position_in_list(name_list, buf)) < 0)
			n = 0;
		list = style_list;
		}
	return get_style_from_list(list, n);
	}

Style *
gkrellm_meter_style_by_name(gchar *name)
	{
	return get_style_from_list_by_name(GK.meter_name_list,
				GK.meter_style_list, name);
	}

Style *
gkrellm_panel_style_by_name(gchar *name)
	{
	return get_style_from_list_by_name(GK.chart_name_list,
				GK.panel_style_list, name);
	}

Style *
gkrellm_chart_style_by_name(gchar *name)
	{
	return get_style_from_list_by_name(GK.chart_name_list,
				GK.chart_style_list, name);
	}

gint
gkrellm_lookup_chart_style_id(gchar *name)
	{
	GList	*list;
	gint	i;

	for (list = GK.chart_name_list, i = 0; list; list = list->next, ++i)
		if (name && !strcmp(name, (gchar *)list->data))
			return i;
	return 0;
	}

gint
gkrellm_lookup_meter_style_id(gchar *name)
	{
	GList	*list;
	gint	i;

	for (list = GK.meter_name_list, i = 0; list; list = list->next, ++i)
		if (name && !strcmp(name, (gchar *)list->data))
			return i;
	return 0;
	}

static GdkImlibImage *
get_image_from_list(GList *list, gint n)
	{
	GList	*l;

	l = g_list_nth(list, n);
	if (l == NULL || l->data == NULL)
		l = list;
	if (l->data == NULL)
		{
		printf("Warning: NULL image returned %d\n", n);
		abort();
		}
	return (GdkImlibImage *) l->data;
	}

GdkImlibImage *
gkrellm_bg_chart_image(gint n)
	{
	return get_image_from_list(GK.bg_chart_image_list, n);
	}

GdkImlibImage *
gkrellm_bg_grid_image(gint n)
	{
	return get_image_from_list(GK.bg_grid_image_list, n);
	}

GdkImlibImage *
gkrellm_bg_panel_image(gint n)
	{
	return get_image_from_list(GK.bg_panel_image_list, n);
	}

GdkImlibImage *
gkrellm_bg_meter_image(gint n)
	{
	return get_image_from_list(GK.bg_meter_image_list, n);
	}

GdkImlibImage *
gkrellm_krell_panel_image(gint n)
	{
	return get_image_from_list(GK.krell_panel_image_list, n);
	}

GdkImlibImage *
gkrellm_krell_meter_image(gint n)
	{
	return get_image_from_list(GK.krell_meter_image_list, n);
	}

void
gkrellm_get_decal_alarm_image(GdkImlibImage **im, gint *frames)
	{
	if (im)
		*im = GK.decal_alarm_image;
	if (frames)
		*frames = GK.decal_alarm_frames;
	}

void
gkrellm_get_decal_warn_image(GdkImlibImage **im, gint *frames)
	{
	if (im)
		*im = GK.decal_warn_image;
	if (frames)
		*frames = GK.decal_warn_frames;
	}

void
gkrellm_monitor_height_adjust(gint h)
	{
	GK.monitor_height += h;
	}

GdkPixmap *
gkrellm_decal_misc_pixmap(void)
	{
	return GK.decal_misc_pixmap;
	}

GdkBitmap *
gkrellm_decal_misc_mask(void)
	{
	return GK.decal_misc_mask;
	}

GdkPixmap **
gkrellm_data_in_pixmap(void)
	{
	return &GK.data_in_pixmap;
	}

GdkPixmap *
gkrellm_data_in_grid_pixmap(void)
	{
	return GK.data_in_grid_pixmap;
	}

GdkPixmap **
gkrellm_data_out_pixmap(void)
	{
	return &GK.data_out_pixmap;
	}

GdkPixmap *
gkrellm_data_out_grid_pixmap(void)
	{
	return GK.data_out_grid_pixmap;
	}


GdkImlibImage *
gkrellm_krell_slider_image(void)
	{
	return GK.krell_slider_image;
	}

Style *
gkrellm_krell_slider_style(void)
	{
	return GK.krell_slider_style;
	}

GdkImlibImage *
gkrellm_krell_mini_image(void)
	{
	return GK.krell_mini_image;
	}

Style *
gkrellm_krell_mini_style(void)
	{
	return GK.krell_mini_style;
	}

GdkImlibImage *
gkrellm_bg_slider_image(gint n)
	{
	if (n < 0 || n > N_PANEL_TYPES)
		n = 0;
	return GK.bg_slider_image[n];
	}

GdkImlibBorder *
gkrellm_bg_slider_border(gint n)
	{
	if (n < 0 || n > N_PANEL_TYPES)
		n = 0;
	return &GK.bg_slider_border[n];
	}

GdkGC *
gkrellm_draw_GC(gint n)
	{
	GdkGC	*gc;

	if (n == 0)
		return GK.draw_stencil_GC;
	else if (n == 1)
		gc = GK.draw1_GC;
	else if (n == 2)
		gc = GK.draw2_GC;
	else
		gc = GK.draw3_GC;
	return gc;
	}

GdkGC *
gkrellm_bit_GC(gint n)
	{
	if (n == 0)
		return GK.bit0_GC;
	else
		return GK.bit1_GC;
	}

TextStyle *
gkrellm_chart_textstyle(gint n)
	{
	Style		*style;
	TextStyle	*ts;

	style = get_style_from_list(GK.chart_style_list, n);
	ts	= &style->label_tsA;
	ts->font = *(ts->font_seed);
	return ts;
	}

TextStyle *
gkrellm_panel_textstyle(gint n)
	{
	Style		*style;
	TextStyle *ts;

	style = get_style_from_list(GK.panel_style_list, n);
	ts	= &style->label_tsA;
	ts->font = *(ts->font_seed);
	return ts;
	}

TextStyle *
gkrellm_meter_textstyle(gint n)
	{
	Style		*style;
	TextStyle *ts;

	style = get_style_from_list(GK.meter_style_list, n);
	ts	= &style->label_tsA;
	ts->font = *(ts->font_seed);
	return ts;
	}


TextStyle *
gkrellm_chart_alt_textstyle(gint n)
	{
	Style		*style;
	TextStyle *ts;

	style = get_style_from_list(GK.chart_style_list, n);
	ts	= &style->label_tsB;
	ts->font = *(ts->font_seed);
	return ts;
	}

TextStyle *
gkrellm_panel_alt_textstyle(gint n)
	{
	Style		*style;
	TextStyle *ts;

	style = get_style_from_list(GK.panel_style_list, n);
	ts	= &style->label_tsB;
	ts->font = *(ts->font_seed);
	return ts;
	}

TextStyle *
gkrellm_meter_alt_textstyle(gint n)
	{
	Style		*style;
	TextStyle *ts;

	style = get_style_from_list(GK.meter_style_list, n);
	ts	= &style->label_tsB;
	ts->font = *(ts->font_seed);
	return ts;
	}

GdkFont *
gkrellm_default_font(gint n)
	{
	if (n == 0)
		return GK.small_font;
	else if (n == 1)
		return GK.normal_font;
	else
		return GK.large_font;
	}

GdkColor *
gkrellm_white_color(void)
	{
	return &GK.white_color;
	}

GdkColor *
gkrellm_black_color(void)
	{
	return &GK.background_color;
	}

GdkColor *
gkrellm_in_color(void)
	{
	return &GK.in_color;
	}

GdkColor *
gkrellm_out_color(void)
	{
	return &GK.out_color;
	}

gboolean
gkrellm_demo_mode(void)
	{
	return GK.demo;
	}

gint
gkrellm_update_HZ(void)
	{
	return GK.update_HZ;
	}

gint
gkrellm_chart_width(void)
	{
	return GK.chart_width;
	}

void
gkrellm_allow_scaling(gboolean *allow, gint *width_ref)
	{
	if (allow)
		*allow = GK.allow_scaling;
	if (width_ref)
		*width_ref = GK.chart_width_ref;
	}

/* ======================================================================= */

static Monitor		*place_plugin;

static GtkWidget	*enable_clist;
static GtkWidget	*enable_button,
					*place_label,
					*place_button,
					*after_button,
					*before_button,
					*gravity_spin_button,
					*builtin_button[N_BUILTIN_MONITORS];
static GtkWidget	*text_log;

static gint			selected_row	= -1;

static GList		*plugins_list,
					*plugins_enable_list,
					*plugins_place_list;

static gboolean		plugin_enable_list_modified,
					plugin_placement_modified;

gchar				*plugin_install_log;


Monitor *
lookup_monitor_from_id(gint id)
	{
	GList	*list;
	Monitor	*mon;

	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (mon->id == id)
			return mon;
		}
	return NULL;
	}

Monitor *
lookup_monitor_from_name(gchar *name)
	{
	Monitor	*mon;
	GList	*list;

	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (!mon->name)
			continue;
		if (!strcmp(name, mon->name))
			return mon;
		}
	return NULL;
	}

Monitor *
lookup_monitor_from_style_name(gchar *style_name)
	{
	Monitor	*mon;
	GList	*list;

	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (!mon->privat || !mon->privat->style_name)
			continue;
		if (!strcmp(style_name, mon->privat->style_name))
			return mon;
		}
	return NULL;
	}

static void
plugin_log(gchar *string1, ...)
	{
	va_list	args;
	gchar	*s, *old_log;

	if (!string1)
		return;
	va_start(args, string1);
	s = string1;
	while (s)
		{
		old_log = plugin_install_log;
		if (plugin_install_log)
			plugin_install_log = g_strconcat(plugin_install_log, s, NULL);
		else
			plugin_install_log = g_strconcat(s, NULL);
		g_free(old_log);
		s = va_arg(args, gchar *);
		}
	va_end(args);
	}

static gboolean
user_placement(Monitor *plugin)
	{
	GList	*list;
	Monitor	*mon;
	gchar	*line, *s, *plugin_name, *mon_name;
	gint	n, after, gravity;

	if (!plugin->name)
		return FALSE;
	for (list = plugins_place_list; list; list = list->next)
		{
		line = g_strconcat((gchar *) list->data, NULL);
		plugin_name = cut_quoted_string(line, &s);
		mon_name = cut_quoted_string(s, &s);
		n = sscanf(s, "%d %d", &after, &gravity);
		if (n == 2)
			{
			if (   !strcmp(plugin_name, plugin->name)
				&& (mon = lookup_monitor_from_name(mon_name)) != NULL
			   )
				{
				plugin->privat->insert_before_id = mon->id;
				plugin->privat->gravity = gravity;
				plugin->privat->insert_after = after;
				g_free(line);
				return TRUE;
				}
			}
		g_free(line);
		}
	return FALSE;
	}

void
gkrellm_place_plugin(GList **monitor_list, Monitor *plugin)
	{
	Monitor		*mon;
	GList		*list, *plist;
	gint		n, gravity, after_flag;
	gchar		buf[120];

	if (plugin->create_monitor && plugin->privat->main_vbox == NULL)
		{
		plugin->privat->main_vbox = gtk_vbox_new(FALSE, 0);
		plugin->privat->spacer_top_vbox = gtk_vbox_new(FALSE, 0);
		plugin->privat->vbox = gtk_vbox_new(FALSE, 0);
		plugin->privat->spacer_bottom_vbox = gtk_vbox_new(FALSE, 0);
		}
	for (plist = NULL, list = *monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;

		/* Save list position as plugins are encountered so we can later
		|  walk through them looking at gravity.
		*/
		if (MONITOR_ID(mon) == MON_PLUGIN && plist == NULL)
			plist = list;
		if (MONITOR_ID(mon) == plugin->privat->insert_before_id)
			{
			after_flag = plugin->privat->insert_after;
			gravity = plugin->privat->gravity;
			snprintf(buf, sizeof(buf), _("\t%s: placement is %s %s  G:%d\n"),
					plugin->name,
					after_flag ? _("after") : _("before"), mon->name, gravity);
			plugin_log(buf, NULL);
			if (after_flag)
				{
				if ((n = g_list_position(*monitor_list, list)) < 0)
					n = 0;
				*monitor_list = g_list_insert(*monitor_list, plugin, n + 1);
				}
			else
				{
				/* If there are plugins already above this builtin, then place
				|  based on gravity.  Insert above the first plugin found that
				|  has greater gravity than the current placing plugin.
				*/
				if (plist)
					{
					for ( ; plist != list; plist = plist->next)
						{
						mon = (Monitor *) plist->data;	/* looking at plugins*/
						if (mon->privat->gravity > gravity)
							break;
						}
					list = plist;
					}
				if (list == *monitor_list)
					*monitor_list = g_list_prepend(list, plugin);
				else
					g_list_prepend(list, plugin);
				}
			return;
			}
		else if (MONITOR_ID(mon) != MON_PLUGIN)
			plist = NULL;
		}
	if (plugin->privat->insert_before_id != MON_UPTIME)
		{
		plugin->privat->insert_before_id = MON_UPTIME;
		gkrellm_place_plugin(monitor_list, plugin);
		return;
		}
	*monitor_list = g_list_append(*monitor_list, plugin);
	}

Monitor *
install_plugin(gchar *plugin_name)
	{
	GList		*list;
	Monitor		*m, *mm;
	void		*handle;
	Monitor		*(*init_plugin)();
	gchar		*error;
	gchar		buf[256];

#ifdef RTLD_NOW
	handle = dlopen(plugin_name, RTLD_NOW);
#else
	handle = dlopen(plugin_name, 0);
#endif

	plugin_log(plugin_name, "\n", NULL);
	if (! handle)
		{
		snprintf(buf, sizeof(buf), _("\tError: %s\n"), dlerror());
		plugin_log(buf, NULL);
		return NULL;
		}
	init_plugin = dlsym(handle, "init_plugin");
	if ((error = dlerror()) != NULL)
		{
		snprintf(buf, sizeof(buf), _("\tError: %s\n"), error);
		plugin_log(buf, NULL);
		dlclose(handle);
		return NULL;
		}
	m = (*init_plugin)();
	if (m == NULL)
		{
		plugin_log(_("\tOoops! plugin returned NULL, aborting\n"), NULL);
		dlclose(handle);
		return NULL;
		}
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mm = (Monitor *) list->data;
		if (   !mm->privat || !mm->privat->style_name
			|| !m->privat  || !m->privat->style_name
			|| strcmp(mm->privat->style_name, m->privat->style_name)
		   )
			continue;
		plugin_log(_("\tWarning: style name \""), m->privat->style_name,
			_("\" already used by:\n\t\t"), mm->path, "\n", NULL);
		}
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mm = (Monitor *) list->data;
		if (   !mm->config_keyword || !m->config_keyword
			|| strcmp(mm->config_keyword, m->config_keyword)
		   )
			continue;
		plugin_log(_("\tWarning: config keyword \""), m->config_keyword,
			_("\" already used by:\n\t\t"), mm->path, "\n", NULL);
		}
	m->handle = handle;
	m->path = plugin_name;
	if (!m->name)
		m->name = g_strdup(g_basename(m->path));
	if (m->privat == NULL)		/* Won't be null if style was added */
		m->privat = g_new0(MonPrivate, 1);
	m->privat->enabled = TRUE;

	/* Enforce some id fields.
	*/
	m->id &= ~(MON_ID_MASK | MON_CONFIG_MASK);
	m->id |= MON_PLUGIN;
	if (PLUGIN_INSERT_BEFORE_ID(m) >= N_BUILTIN_MONITORS)
		m->insert_before_id = MON_UPTIME;
	if (!user_placement(m))
		{
		m->privat->insert_before_id = PLUGIN_INSERT_BEFORE_ID(m);
		m->privat->gravity = PLUGIN_GRAVITY(m);
		m->privat->insert_after = PLUGIN_INSERT_AFTER(m);
		}
	gkrellm_place_plugin(&gkrellm_monitor_list, m);
	return m;
	}

pid_t
gkrellm_get_pid(void)
	{
	return getpid();
	}

void
gkrellm_disable_plugin_connect(Monitor *mon, void (*cb_func)())
	{
	if (!mon || !mon->privat || !cb_func)
		return;
	mon->privat->cb_disable_plugin = cb_func;
	}

static void
disable_plugin(Monitor *plugin)
	{
	MonPrivate	*mp = plugin->privat;

	if (!mp->enabled)
		return;
	if (mp->top_pixmapwid)
		gtk_widget_destroy(mp->top_pixmapwid);
	mp->top_pixmapwid = NULL;
	if (mp->bottom_pixmapwid)
		gtk_widget_destroy(mp->bottom_pixmapwid);
	mp->bottom_pixmapwid = NULL;

	/* It is not safe to destroy a plugin widget tree.
	|  They have already done one time first_create stuff.
	*/
	gtk_widget_hide(mp->vbox);
	gtk_widget_hide(mp->main_vbox);
	mp->enabled = FALSE;
	if (mp->cb_disable_plugin)
		(*mp->cb_disable_plugin)();
	build_gkrellm();
	remove_plugin_config_page(plugin);
	plugin_enable_list_modified = TRUE;
	}

static void
enable_plugin(Monitor *plugin)
	{
	MonPrivate	*mp = plugin->privat;

	if (mp->enabled)
		return;
	gtk_widget_show(mp->vbox);
	gtk_widget_show(mp->main_vbox);
	mp->enabled = TRUE;
	build_gkrellm();
	add_plugin_config_page(plugin);
	plugin_enable_list_modified = TRUE;
	}

static void
load_plugins_placement_file(void)
	{
	FILE	*f;
	gchar	*path, *s, buf[256];

	free_glist_and_data(&plugins_place_list);
	path = gkrellm_make_config_file_name(gkrellm_homedir(),
				PLUGINS_PLACEMENT_FILE);
	if ((f = fopen(path, "r")) != NULL)
		{
		while ((fgets(buf, sizeof(buf), f)) != NULL)
			{
			if ((s = strchr(buf, '\n')) != NULL)
				*s = '\0';
			s = g_strdup(buf);
			plugins_place_list = g_list_append(plugins_place_list, s);
			}
		fclose(f);
		}
	g_free(path);
	}

static void
save_plugins_placement_file(void)
	{
	FILE		*f;
	GList		*list;
	Monitor		*builtin, *plugin;
	MonPrivate	*mp;
	gchar		*path;
	
	if (!plugin_placement_modified || GK.demo)
		return;
	path = gkrellm_make_config_file_name(gkrellm_homedir(),
				PLUGINS_PLACEMENT_FILE);
	if ((f = fopen(path, "w")) != NULL)
		{
		for (list = plugins_list; list; list = list->next)
			{
			plugin = (Monitor *) list->data;
			mp  = plugin->privat;
			if (   mp->enabled && !mp->from_command_line
				&& (   mp->insert_before_id != PLUGIN_INSERT_BEFORE_ID(plugin)
					|| mp->insert_after != PLUGIN_INSERT_AFTER(plugin)
					|| mp->gravity != PLUGIN_GRAVITY(plugin)
				   )
			   )
				{
				builtin = lookup_monitor_from_id(mp->insert_before_id);
				if (!builtin)
					continue;
				fprintf(f, "\"%s\" \"%s\" %d %d\n", plugin->name,
						builtin->name, mp->insert_after, mp->gravity);
				}
			}
		fclose(f);
		}
	plugin_placement_modified = FALSE;
	g_free(path);
	}

static void
load_plugins_enable_file(void)
	{
	FILE	*f;
	gchar	*path, *s, buf[256];

	free_glist_and_data(&plugins_enable_list);
	path = gkrellm_make_config_file_name(gkrellm_homedir(),
				PLUGINS_INSTALL_FILE);
	if ((f = fopen(path, "r")) != NULL)
		{
		while ((fgets(buf, sizeof(buf), f)) != NULL)
			{
			if ((s = strchr(buf, '\n')) != NULL)
				*s = '\0';
			if (!buf[0])
				continue;
			s = g_strdup(g_basename(buf));
			plugins_enable_list = g_list_append(plugins_enable_list, s);
			}
		fclose(f);
		}
	g_free(path);
	}

static void
save_plugins_enable_file(void)
	{
	FILE	*f;
	GList	*list;
	Monitor	*m;
	gchar	*path;
	
	if (!plugin_enable_list_modified || GK.demo)
		return;
	path = gkrellm_make_config_file_name(gkrellm_homedir(),
			PLUGINS_INSTALL_FILE);
	if ((f = fopen(path, "w")) != NULL)
		{
		for (list = plugins_list; list; list = list->next)
			{
			m = (Monitor *) list->data;
			if (m->privat->enabled && !m->privat->from_command_line)
				fprintf(f, "%s\n", g_basename(m->path));
			}
		fclose(f);
		}
	plugin_enable_list_modified = FALSE;
	g_free(path);
	}

static void
scan_for_plugins(gchar *path)
	{
    DIR				*dir;
    struct dirent	*dentry;
	GList			*list;
	Monitor			*m = NULL;
	gchar			*s, *so;
	
	if ((dir = opendir(path)) == NULL)
		return;
	while ((dentry = readdir(dir)) != NULL)
		{
		if (   dentry->d_name[0] != '.'
			&& dentry->d_ino > 0
			&& (so = strrchr(dentry->d_name, '.')) != NULL
			&& strcmp(so, ".so") == 0
		   )
			{
			for (list = plugins_list; list; list = list->next)
				{
				m = (Monitor *) list->data;
				if (!strcmp(g_basename(m->path), dentry->d_name))
					break;
				m = NULL;
				}
			s = g_strdup_printf("%s/%s", path, dentry->d_name);
			if (m)
				{
				plugin_log(_("Ignoring duplicate plugin "), s, "\n", NULL);
				g_free(s);
				continue;
				}
			m = install_plugin(s);
			if (m)
				{
				plugins_list = g_list_append(plugins_list, m);
				if (! string_in_list(plugins_enable_list, g_basename(m->path)))
					m->privat->enabled = FALSE;
				}
			else
				g_free(s);
			}
		}
	closedir(dir);
	}

void
load_plugin_monitors(void)
	{
	Monitor			*m;
	gchar			*path;

	if (GK.command_line_plugin && strstr(GK.command_line_plugin, ".so"))
		{
		if (   *GK.command_line_plugin != '.'
			&& !strchr(GK.command_line_plugin, '/')
		   )
			path = g_strdup_printf("./%s", GK.command_line_plugin);
		else
			path = g_strdup(GK.command_line_plugin);
		plugin_log(_("*** Command line plugin:\n"), NULL);
		if ((m = install_plugin(path)) == NULL)
			g_free(path);
		else
			{
			m->privat->from_command_line = TRUE;
			plugins_list = g_list_append(plugins_list, m);
			}
		plugin_log("\n", NULL);
		}
	load_plugins_enable_file();
	load_plugins_placement_file();
	path = g_strdup_printf("%s/%s", gkrellm_homedir(), GKRELLM_PLUGINS_DIR);
	scan_for_plugins(path);
	g_free(path);
	scan_for_plugins(LOCAL_PLUGINS_DIR);
	scan_for_plugins(SYSTEM_PLUGINS_DIR);
	scan_for_plugins(OLD_LOCAL_PLUGINS_DIR);
	scan_for_plugins(OLD_SYSTEM_PLUGINS_DIR);
	}

static void
apply_place()
	{
	GList		*new_monitor_list, *list;
	Monitor		*mon;
	extern GtkWidget *gkrellm_monitor_vbox();

	new_monitor_list = NULL;
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (mon->privat->main_vbox)
			{
			gtk_widget_ref(mon->privat->main_vbox);
			gtk_container_remove(GTK_CONTAINER(gkrellm_monitor_vbox()),
					mon->privat->main_vbox);
			}
		if (MONITOR_ID(mon) != MON_PLUGIN)
			new_monitor_list = g_list_append(new_monitor_list, mon);
		}
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (MONITOR_ID(mon) == MON_PLUGIN)
			gkrellm_place_plugin(&new_monitor_list, mon);
		}
	g_list_free(gkrellm_monitor_list);
	gkrellm_monitor_list = new_monitor_list;
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (mon->privat->main_vbox)
			{
			gtk_box_pack_start(GTK_BOX(gkrellm_monitor_vbox()),
					mon->privat->main_vbox, FALSE, FALSE, 0);
			gtk_widget_unref(mon->privat->main_vbox);
			}
		}
	}


static GtkWidget	*place_plugin_window;
static GtkWidget	*place_plugin_vbox;

static void
cb_apply_place(GtkWidget *widget, gpointer data)
	{
	GList	*list;
	Monitor	*mon;
	gint	i;

	for (i = 0; i < N_BUILTIN_MONITORS; ++i)
		if (GTK_TOGGLE_BUTTON(builtin_button[i])->active)
			break;
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (MONITOR_ID(mon) == MON_PLUGIN || i != mon->privat->button_id)
			continue;
		place_plugin->privat->insert_before_id = mon->id;
		place_plugin->privat->insert_after =
				GTK_TOGGLE_BUTTON(after_button)->active;
		place_plugin->privat->gravity =  gtk_spin_button_get_value_as_int(
				GTK_SPIN_BUTTON(gravity_spin_button));

		apply_place();
		plugin_placement_modified = TRUE;
		break;
		}
	}

static void
cb_close_place(GtkWidget *widget, gpointer data)
	{
	if (place_plugin_window)
		gtk_widget_destroy(place_plugin_window);
	place_plugin_window = NULL;
	place_plugin_vbox = NULL;
	}

static gint
place_plugin_window_delete_event(GtkWidget *widget, GdkEvent *ev,gpointer data)
	{
	cb_close_place(NULL, NULL);
	return FALSE;
	}

static void
cb_place_default(GtkWidget *widget, gpointer data)
	{
	Monitor	*mon;

	if (!place_plugin)
		return;
	if (PLUGIN_INSERT_AFTER(place_plugin))
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(after_button),
				TRUE);
	else
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(before_button),
				TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(gravity_spin_button),
			(gfloat) PLUGIN_GRAVITY(place_plugin));
	mon = lookup_monitor_from_id(PLUGIN_INSERT_BEFORE_ID(place_plugin));
	if (mon && mon->privat->button_id >= 0)
		gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(builtin_button[mon->privat->button_id]), TRUE);
	}

static void
place_button_sensitivity(Monitor *plugin, gboolean state)
	{
	Monitor	*mon = NULL;

	place_plugin = plugin;
	if (state && plugin->create_monitor)
		{
		gtk_widget_set_sensitive(place_button, TRUE);
		if (place_plugin_vbox)
			gtk_widget_set_sensitive(place_plugin_vbox, TRUE);
		}
	else
		{
		gtk_widget_set_sensitive(place_button, FALSE);
		if (place_plugin_vbox)
			gtk_widget_set_sensitive(place_plugin_vbox, FALSE);
		return;
		}
	if (!place_plugin_window)
		return;

	gtk_label_set_text(GTK_LABEL(place_label), plugin->name);
	if (plugin->privat->insert_after)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(after_button), TRUE);
	else
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(before_button), TRUE);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(gravity_spin_button),
			(gfloat) plugin->privat->gravity);
	mon = lookup_monitor_from_id(plugin->privat->insert_before_id);
	if (mon && mon->privat->button_id >= 0)
		gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(builtin_button[mon->privat->button_id]), TRUE);
	}

static void
cb_place_button(GtkWidget *widget, gpointer data)
	{
	GtkWidget	*main_vbox, *vbox, *vbox1, *vbox2, *vbox3, *hbox;
	GtkWidget	*button;
	GSList		*group;
	GList		*list;
	Monitor		*mon;
	MonPrivate	*mp;
	gint		i;

	if (gkrellm_demo_mode())
		return;
	if (!place_plugin_window)
		{
		place_plugin_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_signal_connect(GTK_OBJECT(place_plugin_window), "delete_event",
				(GtkSignalFunc) place_plugin_window_delete_event, NULL);
		gtk_window_set_policy(GTK_WINDOW(place_plugin_window),
				FALSE, FALSE, TRUE);
		gtk_window_set_title(GTK_WINDOW(place_plugin_window),
				_("GKrellM Place Plugin"));
		main_vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(place_plugin_window), main_vbox);
		place_plugin_vbox = main_vbox;
		vbox = gkrellm_framed_vbox(main_vbox, NULL, 3, FALSE, 4, 3);
		hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

		vbox1 = gkrellm_framed_vbox(hbox, _("Builtin Monitors"),
				3, FALSE, 4, 3);
		group = NULL;
		for (i = 0, list = gkrellm_monitor_list; list; list = list->next)
			{
			mon = (Monitor *) list->data;
			mp  = mon->privat;
			mon->privat->button_id = -1;
			if (MONITOR_ID(mon) != MON_PLUGIN)
				{
				if (!mon->name || !mon->create_monitor)
					continue;
				button = gtk_radio_button_new_with_label(group, mon->name);
				gtk_box_pack_start(GTK_BOX(vbox1), button, FALSE, FALSE, 0);
				if (i < N_BUILTIN_MONITORS)
					{
					builtin_button[i] = button;
					mon->privat->button_id = i++;
					}
				group = gtk_radio_button_group(GTK_RADIO_BUTTON(button));
				}
			}
		vbox1 = gtk_vbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);
		place_label = gtk_label_new("");
		gtk_box_pack_start(GTK_BOX(vbox1), place_label, FALSE, FALSE, 10);
		vbox2 = gkrellm_framed_vbox(vbox1, _("Place Plugin"),
				3, FALSE, 4, 3);

		group = NULL;
		vbox3 = gkrellm_framed_vbox(vbox2, NULL, 3, FALSE, 4, 3);
		before_button = gtk_radio_button_new_with_label(group,
				_("Before selected builtin monitor"));
		gtk_box_pack_start(GTK_BOX(vbox3), before_button, FALSE, FALSE, 0);
		gkrellm_spin_button(vbox3, &gravity_spin_button, 8,
				0.0, 15.0, 1, 1, 0, 50, NULL, NULL, FALSE,
				_("With gravity"));

		vbox3 = gkrellm_framed_vbox(vbox2, NULL, 3, FALSE, 4, 3);
		group = gtk_radio_button_group(GTK_RADIO_BUTTON(before_button));
		after_button = gtk_radio_button_new_with_label(group,
				_("After selected builtin monitor"));
		gtk_box_pack_start(GTK_BOX(vbox3), after_button, FALSE, FALSE, 0);

		hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
		gkrellm_button_connected(hbox, NULL, TRUE, FALSE, 0,
				cb_place_default, NULL, _("Plugin Defaults"));

		hbox = gtk_hbutton_box_new();
		gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
		gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);
		gtk_box_pack_end(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);

		button = gtk_button_new_with_label(_("Apply"));
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) cb_apply_place, NULL);
		gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
		gtk_widget_grab_default(button);

		button = gtk_button_new_with_label(_("Close"));
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) cb_close_place, NULL);
		gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

		gtk_widget_show_all(place_plugin_window);
		}
	else
		gdk_window_raise(place_plugin_window->window);

	if (selected_row < 0)
		return;
	mon = (Monitor *) gtk_clist_get_row_data(GTK_CLIST(enable_clist),
			selected_row);
	place_button_sensitivity(mon, TRUE);
	}

static void
cb_enable_button_clicked(GtkWidget *widget, gpointer data)
	{
	Monitor	*m;
	gint	state;

	if (selected_row < 0)
		return;
	m = (Monitor *) gtk_clist_get_row_data(GTK_CLIST(enable_clist),
			selected_row);
	if (m->privat->from_command_line)
		return;
	state = GTK_TOGGLE_BUTTON(enable_button)->active;

	gtk_clist_set_text(GTK_CLIST(enable_clist), selected_row, 1, 
			state ? _("enabled") : "");

	if (state)
		enable_plugin(m);
	else
		disable_plugin(m);
	place_button_sensitivity(m, state);
	}

static void
cb_clist_unselected(GtkWidget *clist, gint row, gint column,
		GdkEventButton *bevent, gpointer data)
	{
	selected_row = -1;
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_button), FALSE);
	}

static void
cb_clist_selected(GtkWidget *clist, gint row, gint column,
		GdkEventButton *bevent, gpointer data)
	{
	Monitor	*m;

	selected_row = row;
	m = (Monitor *) gtk_clist_get_row_data(GTK_CLIST(clist), selected_row);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_button),
			m->privat->enabled);
	place_button_sensitivity(m, m->privat->enabled);
	gtk_widget_set_sensitive(enable_button,
			m->privat->from_command_line ? FALSE : TRUE);
	}

void
close_plugins_config(void)
	{
	save_plugins_enable_file();
	save_plugins_placement_file();
	cb_close_place(NULL, NULL);
	}

static gchar	*scroll_title[3] = { N_("Plugin"), N_("Status") };

void
create_plugin_config(GtkWidget *tab_vbox)
	{
	GtkWidget	*tabs;
	GtkWidget	*vbox;
	GtkWidget	*hbox;
	GtkWidget	*clist;
	GtkWidget	*scrolled;
	GList		*list;
	Monitor		*mon;
	gchar		*buf[3];
	gint		row;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);


/* -- Plugins tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Plugins"));
	if (!plugins_list)
		return;

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 2);
	scroll_title[0] = _(scroll_title[0]);
	scroll_title[1] = _(scroll_title[1]);
	clist = gtk_clist_new_with_titles(3, scroll_title);
	enable_clist = clist;
	gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_OUT);
	gtk_clist_set_column_width(GTK_CLIST(clist), 0, 180);
	gtk_clist_set_column_width(GTK_CLIST(clist), 1, 100);

	gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			(GtkSignalFunc) cb_clist_selected, NULL);
	gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
			(GtkSignalFunc) cb_clist_unselected, NULL);
	gtk_container_add(GTK_CONTAINER(scrolled), clist);
	for (row = 0, list = plugins_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		buf[0] = mon->name;
		if (mon->privat->from_command_line)
			buf[1] = _("from command line");
		else
			buf[1] = mon->privat->enabled ? _("enabled") : "";
		buf[2] = NULL;
		gtk_clist_append(GTK_CLIST(clist), buf);
		gtk_clist_set_row_data(GTK_CLIST(clist), row++, mon);
		}
	gtk_clist_sort(GTK_CLIST(enable_clist));
	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_START);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);

	enable_button = gtk_check_button_new_with_label(_("Enable"));
	gtk_box_pack_start(GTK_BOX(hbox), enable_button, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(GTK_BUTTON(enable_button)), "clicked",
			GTK_SIGNAL_FUNC (cb_enable_button_clicked), NULL);

	place_button = gtk_button_new_with_label(_("Place"));
	gtk_box_pack_start(GTK_BOX(hbox), place_button, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(GTK_BUTTON(place_button)), "clicked",
			GTK_SIGNAL_FUNC (cb_place_button), NULL);
	gtk_widget_set_sensitive(place_button, FALSE);

/* --Plugins detect log tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Install Log"));
	text_log = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_text_insert(GTK_TEXT(text_log), NULL, NULL, NULL,
			plugin_install_log, -1);
	}
