// MM1MDL.H : data storage for "all atoms"-models (molecular mechanics).

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "config.h"	// this is target-dependent...

#ifndef MM1MDL_H
#define MM1MDL_H

class mm1_mdl;

class mm1_readpdb_mdata;
struct mm1_readpdb_mdata_chain;
struct mm1_readpdb_data_atom;
struct mm1_readpdb_data_ssbond;

/*################################################################################################*/

#include "model.h"
#include "mm1util.h"
#include "mm1alg.h"

#include "interface.h"

#include <list>
#include <vector>
using namespace std;

#define NOT_FOUND 0x7fffffff	// numeric_limits<i32s>::max()?!?!?!

// OpenBabel includes...

#include "mol.h"
//#include "oeutil.h"
//#include "data.h"
//#include "typer.h"
using namespace OpenBabel;

/*################################################################################################*/

#define ENG_MM1_NORMAL		0x100		// mm1_eng_exp9
#define ENG_MM1_PERIODIC	0x200		// mm1_eng_exp1_mim

/**	A "##model" class for "##all-atoms" molecular mechanics.
	
	This class will store atoms and bonds of a MM model. The class will also keep track 
	of molecules, and there are additional classes for detecting chains and residues of 
	macromolecules (amino acids & nucleic acids at the moment). Also, the atom objects 
	will carry information about the bond objects they are related to; this information 
	is available for algorithms that need to traverse the bond networks quickly.
	
	The atom and bond information is stored in STL lists. The advantage of STL lists are 
	that the pointers to atoms and bonds remain valid in all insertion/removal/sorting 
	operations of the lists. The bond object is dependent on this, since the bond objects 
	store the information about bonds as atom pointers. When the model is organized into 
	molecules, chains and residues, in practice the atom and bond lists are sorted using 
	mol/chn/res/atm ID numbers as sorting criteria. Once the lists are sorted, one can 
	query a range of atoms that cover a molecule, a chain or a residue.
	
	This class has many similarities with qm1_mdl, which is a class for QM models:
		- The atom and bond objects are basically similar, althought the QM atoms 
		and bonds are more simple.
		- In qm1_mdl, the same principle of storing the atom and bond objects to 
		STL lists is used.
	
	Try to keep the mm1_mdl and qm1_mdl classes synchronized; there is even a special 
	base class all_atoms_interface for this purpose.
	
	If the qm1_mdl class is so similar, why not fuse the classes? There are differences:
		- There are big differences in principles of MM and QM. While MM is entirely 
		based on bonds and bond typing, the bonds in QM are just a (simplified) way 
		to interpret the result, and (in strong contrast to MM) the bonds are variable 
		(for example in a transition state study) instead of static. Because the basis 
		of MM and QM models are so different, is seems safest to keep them separate.
		- There are also technical differences; the plotting functions (like ESP) in MM 
		models are very simple, and they are computed directly, but in QM models the 
		plotting functions (ESP, electron density, MO, MO-density) are complicated and 
		are tightly coupled to the model itself. Therefore the plotting (and the use of 
		"##engine" classes in general) is quite different in qm1_mdl.
	
	TODO:
		- Keep track of similarities with mm1_mdl, and document them.
		- Keep the focus here in data storage; keep things simple, and offer interfaces 
		to data (like all_atoms_interface) instead of adding more complicated stuff.
*/

class mm1_mdl : public model_extended, public all_atoms_interface, public trajectory_interface
{
	protected:
	
	i32s nmol;
	vector<mm1_chn_info> * ref_civ;
	
	list<mm1_atom> atom_list;
	list<mm1_bond> bond_list;
	
// default engine settings begin!!!
// default engine settings begin!!!
// default engine settings begin!!!
	
	public:
	
	static const char * engtab1[];
	static const i32s engtab2[];
	
	i32s default_eng;
	
	//protected:	// the dialog boxes need these...
	
	bool periodic;
	f64 box_hdim[3];
	f64 box_fdim[3];
	
// default engine settings end!!!
// default engine settings end!!!
// default engine settings end!!!
	
	friend class mm1_eng;
	friend class mm1_eng_pbc;
	friend class mm1_sequencebuilder;
	friend class mm1_geomopt;
	friend class mm1_moldyn;
	friend class mm1_ribbon;
	
	friend class mm2_mdl;
	
	friend class qm1_mdl;
	friend class gnome_qm1_docv;	// temporary, for MM->QM conversion, will move into all_atoms_interface???
	
	friend class file_trans;
	
	friend void CopyCRD(mm1_mdl *, mm1_eng *, i32u);
	friend void CopyCRD(mm1_eng *, mm1_mdl *, i32u);
	
	friend fGL mm1_GetESPValue(fGL *, mm1_mdl *, fGL *);
	friend fGL mm1_GetVDWSValue(fGL *, mm1_mdl *, fGL *);
	
	friend void DefineSecondaryStructure(mm1_mdl *);
	friend f64 HBondEnergy(mm1_mdl *, i32s *, i32s *);
	
	public:
	
	static mm1_sequencebuilder amino_builder;
	static mm1_sequencebuilder nucleic_builder;
	
	public:
	
	mm1_mdl(ostream *, class_factory &);
	~mm1_mdl(void);
	
	const char * GetProjectFileNameExtension(void);		// virtual
	
	mm1_eng * CreateDefaultEngine(void);
	
	void PushCRDSets(i32u);			// virtual
	void PopCRDSets(i32u);			// virtual
	
	void CopyCRDSet(i32u, i32u);		// virtual
	void SwapCRDSets(i32u, i32u);		// virtual
	
	void CenterCRDSet(i32u);		// virtual
	void ReserveCRDSets(i32u);		// virtual
	
	// methods for adding new atoms and bonds:
	// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
/**	This will just push new atom to the atom list and update colors if necessary.
		Mark for updated connectivity (for GUI's and "save")???
*/
	void AddAtom(mm1_atom &);
	
/// This will delete all bonds associated with this atom, and erase atom from the list...
	void RemoveAtom(iter_mm1al);
	
/// This will add neighbor infos for both atoms and add the new bond into the bond list.
	void AddBond(mm1_bond &);
	
/// This will remove infos from the atoms and erase bond from the bond list.
	void RemoveBond(iter_mm1bl);
	
/// This will remove all atoms and bonds.
	void Clear(void);
	
	// methods for accessing atom/bond lists:
	// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	iter_mm1al GetAtomsBegin(void) { return atom_list.begin(); }	//const;
	iter_mm1al GetAtomsEnd(void) { return atom_list.end(); }	//const;
	
	iter_mm1bl GetBondsBegin(void) { return bond_list.begin(); }	//const;
	iter_mm1bl GetBondsEnd(void) { return bond_list.end(); }	//const;
	
	// methods for file I/O...
	// ^^^^^^^^^^^^^^^^^^^^^^^
	
	bool ReadStream_OLD(istream &);		// this is the very old version...
//	bool ReadStream_v100(istream &);	// this will be the version 1.00, etc...

	bool ReadStream(istream &, bool = false);	// virtual
	
	/// This is v1.00 output function for the mm1gp file format. The "!Section" idea is used here.
	void WriteStream(ostream &);			// virtual

	bool ImportFile(const char *filename, int index = 0);	// virtual

	// add or remove hydrogen atoms...
	
	void AddHydrogens(void);
	void RemoveHydrogens(void);
	
/// This will return atom_list.size().
	i32s GetAtomCount(void) { return atom_list.size(); }
	
/// This will return bond_list.size().
	i32s GetBondCount(void) { return bond_list.size(); }
	
/// This will return nmol (see GatherGroups() for more info).
	i32s GetMoleculeCount(void) { return nmol; }

/// This will return whether the molecule is empty
	bool IsEmpty(void) { return (bond_list.empty() && atom_list.empty()); }
	
	iter_mm1al FindAtomByIndex(i32s);
	
/**	This will group the atom list so that molecules (and chains/residues if defined) 
	will form continuous groups. Will validate mm1_model::nmol and permit the use 
	of mm1_model::GetRange()-functions.
*/
	void GatherGroups(void);
	
/// This is just a default version of GetRange() using the full range of atom list iterators...
	void GetRange(i32s, i32s, iter_mm1al *);
	
/**	This will reduce the initial range of two atom list iterators to some subrange
	with certain values in certain mm1_atom::id[]-fields. Before using this you MUST
	call mm1_model::GatherGroups() to arrange the atom list !!!
	
	What the above explanation really tries to say, is that using this function you 
	can pick up a certain part of the model; for example a molecule, or a chain in a 
	macromolecule, or a range of residue in a chain.
*/
	void GetRange(i32s, iter_mm1al *, i32s, iter_mm1al *);
	
/**	Adding or removing atoms or bonds will generally scramble the grouping, 
	and when this happens one should call this function to discard all grouping information...
*/
	void InvalidateGroups(void);
	
	i32s FindPath(mm1_atom *, mm1_atom *, i32s, i32s, i32s = 0);
	bool FindRing(mm1_atom *, mm1_atom *, signed char *, i32s, i32s, i32s = 0);
	
	void UpdateIndex(void);
	
	private:
	
/**	This will set molecule numbers quickly using a recursive search algorithm. 
	This is private because only mm1_model::GatherGroups() should use this...
*/
	void GatherAtoms(mm1_atom *, i32s);
	
	public:
	
	// the "readpdb" functions here are used to import PDB files as correctly as possible.
	// the PDB files represent experimental results, and in many cases the structures in files
	// have gapped and/or incomplete sequences, incomplete residues, and so on...
	
	// the "readpdb" functions do the import in two stages: in first stage read in the "metadata"
	// (all headers and remarks about the data including the original sequence), and in second stage
	// read in the data as correctly as possible. later, results from these two can be compared, for
	// example to evaluate quality of the data or to match the data with records in other databases.
	
	mm1_readpdb_mdata * readpdb_ReadMData(const char *);
	
	void readpdb_ReadData(const char *, mm1_readpdb_mdata *, i32s);
	i32s readpdb_ReadData_sub1(vector<mm1_readpdb_data_atom> &, i32s *, const char *, bool);
	void readpdb_ReadData_sub2(vector<mm1_readpdb_data_atom> &, i32s *, const char *, const char *, char);
	
	// the methods that implement all_atoms_interface are here!!!
	// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	protected:
	
	iter_mm1al aai_FindA(i32s); //const;	// these are used internally by aai...
	iter_mm1bl aai_FindB(i32s); //const;	// these are used internally by aai...
	
	public:
	
	i32u aai_GetAtomCount(void); //const;			// virtual
	i32u aai_GetBondCount(void); //const;			// virtual
	
	void * aai_FindAtomByInd(i32s); //const;		// virtual
	i32s aai_FindAtomByPtr(void *); //const;		// virtual
	i32s aai_FindBond(i32s, i32s); //const;			// virtual
	
	void * aai_AddAtom(element, fGL *);			// virtual
	bool aai_RemoveAtom(i32s);				// virtual
	
	bool aai_AddBond(i32s, i32s, bondtype);			// virtual
	bool aai_RemoveBond(i32s);				// virtual
	
	bool aai_AtomGetElement(i32s, element &); //const;	// virtual
	bool aai_AtomSetElement(i32s, element);			// virtual
	
	bool aai_AtomGetXYZ(i32s, fGL *); //const;		// virtual
	bool aai_AtomSetXYZ(i32s, const fGL *);			// virtual
	
	bool aai_BondGetBondType(i32s, bondtype &); //const;	// virtual
	bool aai_BondSetBondType(i32s, bondtype);		// virtual
	
	bool aai_BondGetAtomIndices(i32s, i32s *); //const;	// virtual
	
	// here we have a set of Do???()-functions. the idea is that there is a set
	// of features that we wish to behave EXACTLY same way in each target/platform.
	// we then create a Do???()-function for the feature, and hide the details of
	// the user interface in a set of virtual functions.
	
/// This will perform an energy calculation, and report the result.
	void DoEnergy(void);
	
/// This will perform geometry optimization.
	void DoGeomOpt(mm1_geomopt_param &);
	
/// This is used to ask user the GO options; this default function will just silently accept the defaults.
	virtual void GeomOptGetParam(mm1_geomopt_param &);
	
/// This will perform molecular dynamics.
	void DoMolDyn(mm1_moldyn_param &);
	
/// This is used to ask user the MD options; this default function will just silently accept the defaults.
	virtual void MolDynGetParam(mm1_moldyn_param &);
	
/// This will perform a random search using torsions as variables. AN INITIAL VERSION...
	void DoRandomSearch(void);
	
/// This will print the molecular formula and weight.
	void DoFormula(void);
	
	// these come from trajectory_interface...
	// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	void OpenTrajectory(const char *);	// virtual
	void CloseTrajectory(void);		// virtual
	void ReadFrame(void);			// virtual
};

/*################################################################################################*/

// define struct mm1_readpdb_mdata_chain before class mm1_readpdb_mdata, since the latter uses
// former in some inline functions...

// how to best relate mm1_readpdb_mdata_chain and mm1_chn_info !??!?!?!?!?
// maybe just by storing the alpha-carbon pointers here...

struct mm1_readpdb_mdata_chain
{
	char chn_id;
	char * seqres;
	
	vector<i32s> missing_residues;
	vector<mm1_atom *> alpha_carbons;
};

// class mm1_readpdb_mdata is a class just to make the memory management easier. the data members in
// the class are filled in mm1_mdl::readpdb_ReadMData(), and at end the object is just to be deleted.

class mm1_readpdb_mdata
{
	public:
	
	vector<mm1_readpdb_mdata_chain *> chn_vector;
	
	public:
	
	mm1_readpdb_mdata(void)
	{
	}
	
	~mm1_readpdb_mdata(void)
	{
		for (i32u n1 = 0;n1 < chn_vector.size();n1++)
		{
			delete[] chn_vector[n1]->seqres;	// delete the sequence...
			delete chn_vector[n1];			// delete the whole record...
		}
	}
};

// READPDB_MAX_CRDSETS is relevant only if READPDB_ENABLE_MULTIPLE_CRDSETS is defined...
// READPDB_MAX_CRDSETS is relevant only if READPDB_ENABLE_MULTIPLE_CRDSETS is defined...
// READPDB_MAX_CRDSETS is relevant only if READPDB_ENABLE_MULTIPLE_CRDSETS is defined...

#define READPDB_MAX_CRDSETS 10

struct mm1_readpdb_data_atom
{
	char chn_id;
	
	i32s res_num;
	char res_name[5];
	char atm_name[5];
	fGL crd[READPDB_MAX_CRDSETS][3];
	
	mm1_atom * ref;
};

struct mm1_readpdb_data_ssbond
{
	char chn_id;
	i32s res_num;
	
	mm1_atom * ref;
};

/*################################################################################################*/

fGL mm1_GetESPValue(fGL *, mm1_mdl *, fGL *);
fGL mm1_GetVDWSValue(fGL *, mm1_mdl *, fGL *);

/*################################################################################################*/

class mm1_superimpose : public superimpose
{
	private:
	
	mm1_mdl * mdl;
	
	public:
	
	mm1_superimpose(mm1_mdl *, i32s, i32s);
	~mm1_superimpose(void);
	
	f64 GetValue(void);		// virtual
	f64 GetGradient(void);		// virtual
	
	void Transform(void);		// virtual
};

/*################################################################################################*/

#endif	// MM1MDL_H

// eof
