{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to section 5.2 (Selection) of the OpenGL 1.2.1 specs.
-}

module GL_Selection (
   SelectionName(..), initNames, popName, pushName, loadName,
   RenderMode(..),
   unmarshalRenderMode,   -- internal use only
   renderMode,
   SelectionBuffer, newSelectionBuffer, freeSelectionBuffer,
   withSelectionBuffer, selectBuffer,
   SelectionHit(..), getSelectionHits
) where

import Foreign          (Ptr, mallocArray, allocaArray, free)
import Monad            (liftM, liftM3)

import GL_Constants     (gl_RENDER, gl_SELECT, gl_FEEDBACK)
import GL_BasicTypes    (GLenum, GLint, GLuint, GLsizei, GLfloat)
import GL_Marshal       (Marshal, evalMarshal, peekMarshal)

---------------------------------------------------------------------------

newtype SelectionName = SelectionName GLuint deriving (Eq,Ord)

foreign import "glInitNames" unsafe initNames :: IO ()
foreign import "glPopName" unsafe popName :: IO ()

foreign import "glPushName" unsafe pushName :: SelectionName -> IO ()

foreign import "glLoadName" unsafe loadName :: SelectionName -> IO ()

data RenderMode =
     Render
   | Select
   | Feedback
   deriving (Eq,Ord)

marshalRenderMode :: RenderMode -> GLenum
marshalRenderMode Render    = gl_RENDER
marshalRenderMode Select    = gl_SELECT
marshalRenderMode Feedback  = gl_FEEDBACK

unmarshalRenderMode :: GLenum -> RenderMode
unmarshalRenderMode mode
   | mode == gl_RENDER   = Render
   | mode == gl_SELECT   = Select
   | mode == gl_FEEDBACK = Feedback
   | otherwise           =  error "unmarshalRenderMode"

renderMode :: RenderMode -> IO GLint
renderMode = glRenderMode . marshalRenderMode

foreign import "glRenderMode" unsafe glRenderMode :: GLenum -> IO GLint

data SelectionBuffer =
   SelectionBuffer GLsizei        -- size of buffer GLuint units
                   (Ptr GLuint)   -- the malloced buffer itself

newSelectionBuffer :: Int -> IO SelectionBuffer
newSelectionBuffer n = liftM (SelectionBuffer (fromIntegral n)) $ mallocArray n

freeSelectionBuffer :: SelectionBuffer -> IO ()
freeSelectionBuffer (SelectionBuffer _ buf) = free buf

selectBuffer :: SelectionBuffer -> IO ()
selectBuffer (SelectionBuffer n buf) = glSelectBuffer n buf

foreign import "glSelectBuffer" unsafe glSelectBuffer :: GLsizei -> Ptr GLuint -> IO ()

-- the recommended way...
withSelectionBuffer :: Int -> IO a -> IO a
withSelectionBuffer n cont =
   allocaArray n $ \buf -> do
   glSelectBuffer (fromIntegral n) buf
   cont

data SelectionHit =
   SelectionHit GLfloat           -- minimum z
                GLfloat           -- maximum z
                [SelectionName]   -- name stack
   deriving (Eq, Ord)

nTimes :: Integral a => a -> Marshal b -> Marshal [b]
nTimes n = sequence . replicate (fromIntegral n)

peekUInt :: Marshal GLuint
peekUInt = peekMarshal

peekSelectionName :: Marshal SelectionName
peekSelectionName = liftM SelectionName peekUInt

peekUIF :: Marshal GLfloat
peekUIF = liftM (\x -> fromIntegral x / 0xffffffff) peekUInt

peekSelectionHit :: Marshal SelectionHit
peekSelectionHit = do
   nameStackDepth <- peekUInt
   liftM3 SelectionHit peekUIF peekUIF (nTimes nameStackDepth peekSelectionName)

getSelectionHits :: GLint -> SelectionBuffer -> IO (Maybe [SelectionHit])
getSelectionHits numHits (SelectionBuffer _ buf)
   | numHits < 0 = return Nothing
   | otherwise   = liftM Just $ evalMarshal (nTimes numHits peekSelectionHit) buf
