{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to table 6.6 (Vertex Array data) of the OpenGL 1.2.1 specs.
-}

module GL_QueryVertexArray (
   VarArraySizes(..),
   VarArrayTypes(..),
   VarArrayStrides(..),
   VarArrayPointers(..)
) where

import Foreign          ( Ptr )
import Monad            ( liftM )

import GL_Constants     ( gl_VERTEX_ARRAY_SIZE, gl_VERTEX_ARRAY_TYPE,
                          gl_VERTEX_ARRAY_STRIDE, gl_VERTEX_ARRAY_POINTER,
                          gl_NORMAL_ARRAY_TYPE, gl_NORMAL_ARRAY_STRIDE,
                          gl_NORMAL_ARRAY_POINTER, gl_COLOR_ARRAY_SIZE,
                          gl_COLOR_ARRAY_TYPE, gl_COLOR_ARRAY_STRIDE,
                          gl_COLOR_ARRAY_POINTER, gl_INDEX_ARRAY_TYPE,
                          gl_INDEX_ARRAY_STRIDE, gl_INDEX_ARRAY_POINTER,
                          gl_TEXTURE_COORD_ARRAY_SIZE, gl_TEXTURE_COORD_ARRAY_TYPE,
                          gl_TEXTURE_COORD_ARRAY_STRIDE, gl_TEXTURE_COORD_ARRAY_POINTER,
                          gl_EDGE_FLAG_ARRAY_STRIDE, gl_EDGE_FLAG_ARRAY_POINTER)
import GL_BasicTypes    ( GLint, Gettable(..) )
import GL_QueryUtils    ( getGLint, getGLenum, getPointer )
import GL_VertexArray   ( Type, unmarshalType, Stride )

---------------------------------------------------------------------------

data VarArraySizes =
     VarVertexArraySize
   | VarColorArraySize
   | VarTextureCoordArraySize
   deriving (Eq,Ord)

instance Gettable VarArraySizes GLint where
   get VarVertexArraySize       = getGLint gl_VERTEX_ARRAY_SIZE
   get VarColorArraySize        = getGLint gl_COLOR_ARRAY_SIZE
   get VarTextureCoordArraySize = getGLint gl_TEXTURE_COORD_ARRAY_SIZE

data VarArrayTypes =
     VarVertexArrayType
   | VarNormalArrayType
   | VarColorArrayType
   | VarIndexArrayType
   | VarTextureCoordArrayType
   deriving (Eq,Ord)

instance Gettable VarArrayTypes Type where
   get VarVertexArrayType       = liftM unmarshalType $ getGLenum gl_VERTEX_ARRAY_TYPE
   get VarNormalArrayType       = liftM unmarshalType $ getGLenum gl_NORMAL_ARRAY_TYPE
   get VarColorArrayType        = liftM unmarshalType $ getGLenum gl_COLOR_ARRAY_TYPE
   get VarIndexArrayType        = liftM unmarshalType $ getGLenum gl_INDEX_ARRAY_TYPE
   get VarTextureCoordArrayType = liftM unmarshalType $ getGLenum gl_TEXTURE_COORD_ARRAY_TYPE

data VarArrayStrides =
     VarVertexArrayStride
   | VarNormalArrayStride
   | VarColorArrayStride
   | VarIndexArrayStride
   | VarTextureCoordArrayStride
   | VarEdgeFlagArrayStride
   deriving (Eq,Ord)

instance Gettable VarArrayStrides Stride where
   get VarVertexArrayStride       = liftM fromIntegral $ getGLint gl_VERTEX_ARRAY_STRIDE
   get VarNormalArrayStride       = liftM fromIntegral $ getGLint gl_NORMAL_ARRAY_STRIDE
   get VarColorArrayStride        = liftM fromIntegral $ getGLint gl_COLOR_ARRAY_STRIDE
   get VarIndexArrayStride        = liftM fromIntegral $ getGLint gl_INDEX_ARRAY_STRIDE
   get VarTextureCoordArrayStride = liftM fromIntegral $ getGLint gl_TEXTURE_COORD_ARRAY_STRIDE
   get VarEdgeFlagArrayStride     = liftM fromIntegral $ getGLint gl_EDGE_FLAG_ARRAY_STRIDE

data VarArrayPointers =
     VarVertexArrayPointer
   | VarNormalArrayPointer
   | VarColorArrayPointer
   | VarIndexArrayPointer
   | VarTextureCoordArrayPointer
   | VarEdgeFlagArrayPointer
   deriving (Eq,Ord)

instance Gettable VarArrayPointers (Ptr a) where
   get VarVertexArrayPointer       = getPointer gl_VERTEX_ARRAY_POINTER
   get VarNormalArrayPointer       = getPointer gl_NORMAL_ARRAY_POINTER
   get VarColorArrayPointer        = getPointer gl_COLOR_ARRAY_POINTER
   get VarIndexArrayPointer        = getPointer gl_INDEX_ARRAY_POINTER
   get VarTextureCoordArrayPointer = getPointer gl_TEXTURE_COORD_ARRAY_POINTER
   get VarEdgeFlagArrayPointer     = getPointer gl_EDGE_FLAG_ARRAY_POINTER
