{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to table 6.7 (Transformation state) of the OpenGL 1.2.1
specs.
-}

module GL_QueryTransform (
   VarMatrices(..),
   VarViewport(..),
   VarDepthRange(..),
   VarMatrixStackDepths(..),
   VarMatrixMode(..),
   VarClipPlane(..)
) where

import Monad            ( liftM )

import GL_Constants     ( gl_COLOR_MATRIX, gl_MODELVIEW_MATRIX,
                          gl_PROJECTION_MATRIX, gl_TEXTURE_MATRIX,
                          gl_VIEWPORT, gl_DEPTH_RANGE,
                          gl_COLOR_MATRIX_STACK_DEPTH, gl_MODELVIEW_STACK_DEPTH,
                          gl_PROJECTION_STACK_DEPTH, gl_TEXTURE_STACK_DEPTH,
                          gl_MATRIX_MODE )
import GL_BasicTypes    ( GLfloat, GLdouble, GLclampd, GLint, Viewport,
                          Gettable(..), Capability(..) )
import GL_CoordTrans    ( GLmatrix, MatrixMode, unmarshalMatrixMode, Plane )
import GL_Clipping      ( ClipPlane )
import GL_QueryUtils    ( getGLmatrixf, getGLmatrixd, getViewport, getDepthRange,
                          getGLint, getGLenum, getClipPlaned )

---------------------------------------------------------------------------

data VarMatrices =
     VarColorMatrix
   | VarModelviewMatrix
   | VarProjectionMatrix
   | VarTextureMatrix
   deriving (Eq,Ord)

instance Gettable VarMatrices (GLmatrix GLfloat) where
   get VarColorMatrix      = getGLmatrixf gl_COLOR_MATRIX
   get VarModelviewMatrix  = getGLmatrixf gl_MODELVIEW_MATRIX
   get VarProjectionMatrix = getGLmatrixf gl_PROJECTION_MATRIX
   get VarTextureMatrix    = getGLmatrixf gl_TEXTURE_MATRIX

instance Gettable VarMatrices (GLmatrix GLdouble) where
   get VarColorMatrix      = getGLmatrixd gl_COLOR_MATRIX
   get VarModelviewMatrix  = getGLmatrixd gl_MODELVIEW_MATRIX
   get VarProjectionMatrix = getGLmatrixd gl_PROJECTION_MATRIX
   get VarTextureMatrix    = getGLmatrixd gl_TEXTURE_MATRIX

data VarViewport = VarViewport deriving (Eq,Ord)

instance Gettable VarViewport Viewport where
   get VarViewport = getViewport gl_VIEWPORT

data VarDepthRange = VarDepthRange deriving (Eq,Ord)

instance Gettable VarDepthRange (GLclampd, GLclampd) where
   get VarDepthRange = getDepthRange gl_DEPTH_RANGE

data VarMatrixStackDepths =
     VarColorMatrixStackDepth
   | VarModelviewStackDepth
   | VarProjectionStackDepth
   | VarTextureStackDepth
   deriving (Eq,Ord)

instance Gettable VarMatrixStackDepths GLint where
   get VarColorMatrixStackDepth = getGLint gl_COLOR_MATRIX_STACK_DEPTH
   get VarModelviewStackDepth   = getGLint gl_MODELVIEW_STACK_DEPTH
   get VarProjectionStackDepth  = getGLint gl_PROJECTION_STACK_DEPTH
   get VarTextureStackDepth     = getGLint gl_TEXTURE_STACK_DEPTH

data VarMatrixMode = VarMatrixMode deriving (Eq,Ord)

instance Gettable VarMatrixMode MatrixMode where
   get VarMatrixMode = liftM unmarshalMatrixMode $ getGLenum gl_MATRIX_MODE

data VarClipPlane = VarClipPlane ClipPlane

instance Gettable VarClipPlane (Plane GLdouble) where
   get (VarClipPlane cp) = getClipPlaned (marshalCapability cp)
