{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to table 6.9 (Lighting) of the OpenGL 1.2.1 specs.
-}

module GL_QueryLighting1 (
   VarColorMaterialParameter(..),
   VarColorMaterialFace(..),
   VarMaterialColor(..),
   VarShininess(..),
   VarLightModelAmbient(..),
   VarLightModelFlags(..),
   VarLightModelColorControl(..)
) where

import Monad            ( liftM )

import GL_Constants     ( gl_COLOR_MATERIAL_PARAMETER, gl_COLOR_MATERIAL_FACE,
                          gl_SHININESS, gl_LIGHT_MODEL_AMBIENT,
                          gl_LIGHT_MODEL_LOCAL_VIEWER, gl_LIGHT_MODEL_TWO_SIDE,
                          gl_LIGHT_MODEL_COLOR_CONTROL )
import GL_BasicTypes    ( GLboolean, GLfloat, Gettable(..) )
import GL_Colors        ( Face, marshalFace,
                          LightType, marshalLightType, unmarshalLightType,
                          ColorControl, unmarshalColorControl )
import GL_QueryUtils    ( getGLenum, getGLint, getColor4f, getGLboolean,
                          getMaterialColor4f, getMaterialf, getFace )
import GL_VertexSpec    ( Color4 )

---------------------------------------------------------------------------

data VarColorMaterialParameter = VarColorMaterialParameter deriving (Eq,Ord)

instance Gettable VarColorMaterialParameter LightType where
   get VarColorMaterialParameter = liftM unmarshalLightType $
                                   getGLenum gl_COLOR_MATERIAL_PARAMETER

data VarColorMaterialFace = VarColorMaterialFace deriving (Eq,Ord)

instance Gettable VarColorMaterialFace Face where
   get VarColorMaterialFace = getFace gl_COLOR_MATERIAL_FACE

data VarMaterialColor = VarMaterialColor Face LightType deriving (Eq,Ord)

instance Gettable VarMaterialColor (Color4 GLfloat) where
   get (VarMaterialColor f l) = getMaterialColor4f (marshalFace f) (marshalLightType l)

data VarShininess = VarShininess Face deriving (Eq,Ord)

instance Gettable VarShininess GLfloat where
   get (VarShininess f) = getMaterialf (marshalFace f) gl_SHININESS

data VarLightModelAmbient = VarLightModelAmbient deriving (Eq,Ord)

instance Gettable VarLightModelAmbient (Color4 GLfloat) where
   get VarLightModelAmbient = getColor4f gl_LIGHT_MODEL_AMBIENT

data VarLightModelFlags =
     VarLightModelLocalViewer
   | VarLightModelTwoSide
   deriving (Eq,Ord)

instance Gettable VarLightModelFlags GLboolean where
   get VarLightModelLocalViewer = getGLboolean gl_LIGHT_MODEL_LOCAL_VIEWER
   get VarLightModelTwoSide     = getGLboolean gl_LIGHT_MODEL_TWO_SIDE

data VarLightModelColorControl = VarLightModelColorControl deriving (Eq,Ord)

instance Gettable VarLightModelColorControl ColorControl where
   get VarLightModelColorControl = liftM unmarshalColorControl $
                                   getGLint gl_LIGHT_MODEL_COLOR_CONTROL
