{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to section 4.2 (Whole Framebuffer Operations)
of the OpenGL 1.2.1 specs.
-}

module GL_Framebuffer (
   DrawBuffer(..),
   marshalDrawBuffer, unmarshalDrawBuffer,   -- internal use only
   drawBuffer,
   indexMask, colorMask, depthMask, stencilMask,
   Buffer(..), clear, clearColor, clearIndex, clearDepth, clearStencil, clearAccum,
   AccumOp(..), accum
) where

import GL_BasicTypes    (GLboolean, GLboolean_, marshalGLboolean,
                         GLbitfield, toBitfield,
                         GLint, GLuint, GLenum, GLfloat, GLclampf, GLclampd)
import GL_Constants     (gl_NONE, gl_FRONT_LEFT, gl_FRONT_RIGHT, gl_BACK_LEFT,
                         gl_BACK_RIGHT, gl_FRONT, gl_BACK, gl_LEFT, gl_RIGHT,
                         gl_FRONT_AND_BACK, gl_AUX0, gl_COLOR_BUFFER_BIT,
                         gl_DEPTH_BUFFER_BIT, gl_ACCUM_BUFFER_BIT,
                         gl_STENCIL_BUFFER_BIT, gl_ACCUM, gl_LOAD,gl_RETURN,
                         gl_MULT, gl_ADD)
import GL_VertexSpec    (Color4(..), ColorIndex(..))

---------------------------------------------------------------------------
-- Section 4.2.1 (Selecting a Buffer for Writing)

-- GL_FRONT:          Collision with GL_Colors.Face (resolved here)
-- GL_BACK:           Collision with GL_Colors.Face (resolved here)
-- GL_FRONT_AND_BACK: Collision with GL_Colors.Face (resolved here)
data DrawBuffer = 
     None
   | FrontLeft
   | FrontRight
   | BackLeft
   | BackRight
   | Front'
   | Back'
   | Left
   | Right
   | FrontAndBack'
   | Aux Int
   deriving (Eq,Ord)

marshalDrawBuffer :: DrawBuffer -> GLenum
marshalDrawBuffer None                 = gl_NONE
marshalDrawBuffer FrontLeft            = gl_FRONT_LEFT
marshalDrawBuffer FrontRight           = gl_FRONT_RIGHT
marshalDrawBuffer BackLeft             = gl_BACK_LEFT
marshalDrawBuffer BackRight            = gl_BACK_RIGHT
marshalDrawBuffer Front'               = gl_FRONT
marshalDrawBuffer Back'                = gl_BACK
marshalDrawBuffer GL_Framebuffer.Left  = gl_LEFT
marshalDrawBuffer GL_Framebuffer.Right = gl_RIGHT
marshalDrawBuffer FrontAndBack'        = gl_FRONT_AND_BACK
marshalDrawBuffer (Aux n)              = gl_AUX0 + fromIntegral n

unmarshalDrawBuffer :: GLenum -> DrawBuffer
unmarshalDrawBuffer buffer
   | buffer == gl_NONE           = None
   | buffer == gl_FRONT_LEFT     = FrontLeft
   | buffer == gl_FRONT_RIGHT    = FrontRight
   | buffer == gl_BACK_LEFT      = BackLeft
   | buffer == gl_BACK_RIGHT     = BackRight
   | buffer == gl_FRONT          = Front'
   | buffer == gl_BACK           = Back'
   | buffer == gl_LEFT           = GL_Framebuffer.Left
   | buffer == gl_RIGHT          = GL_Framebuffer.Right
   | buffer == gl_FRONT_AND_BACK = FrontAndBack'
   | otherwise                   = Aux (fromIntegral (buffer - gl_AUX0)) -- TODO: Add check

drawBuffer :: DrawBuffer -> IO ()
drawBuffer = glDrawBuffer . marshalDrawBuffer

foreign import "glDrawBuffer" unsafe glDrawBuffer :: GLenum -> IO ()

---------------------------------------------------------------------------
-- Section 4.2.2 (Fine Control of Buffer Updates)

foreign import "glIndexMask" unsafe indexMask :: GLuint -> IO ()

colorMask :: Color4 GLboolean -> IO ()
colorMask (Color4 r g b a) = glColorMask (marshalGLboolean r) (marshalGLboolean g)
                                         (marshalGLboolean b) (marshalGLboolean a)

foreign import "glColorMask" unsafe glColorMask :: GLboolean_ -> GLboolean_ -> GLboolean_ -> GLboolean_ -> IO ()

depthMask :: GLboolean -> IO ()
depthMask = glDepthMask . marshalGLboolean

foreign import "glDepthMask" unsafe glDepthMask :: GLboolean_ -> IO ()

foreign import "glStencilMask" unsafe stencilMask :: GLuint -> IO ()

---------------------------------------------------------------------------
-- Section 4.2.3 (Clearing the Buffers)

-- GL_COLOR_BUFFER_BIT:   Collision with GL_SavResState.AttributeGroup (resolved there)
-- GL_DEPTH_BUFFER_BIT:   Collision with GL_SavResState.AttributeGroup (resolved there)
-- GL_ACCUM_BUFFER_BIT:   Collision with GL_SavResState.AttributeGroup (resolved there)
-- GL_STENCIL_BUFFER_BIT: Collision with GL_SavResState.AttributeGroup (resolved there)
data Buffer =
     ColorBufferBit
   | DepthBufferBit
   | AccumBufferBit
   | StencilBufferBit
   deriving (Eq,Ord)

marshalBuffer :: Buffer -> GLbitfield
marshalBuffer ColorBufferBit   = gl_COLOR_BUFFER_BIT
marshalBuffer DepthBufferBit   = gl_DEPTH_BUFFER_BIT
marshalBuffer AccumBufferBit   = gl_ACCUM_BUFFER_BIT
marshalBuffer StencilBufferBit = gl_STENCIL_BUFFER_BIT

clear :: [Buffer] -> IO ()
clear = glClear . toBitfield marshalBuffer

foreign import "glClear" unsafe glClear :: GLbitfield -> IO ()

clearColor :: Color4 GLclampf -> IO ()
clearColor (Color4 r g b a) = glClearColor r g b a

foreign import "glClearColor" unsafe glClearColor :: GLclampf -> GLclampf -> GLclampf -> GLclampf -> IO ()

foreign import "glClearIndex" unsafe clearIndex :: ColorIndex GLfloat -> IO ()

foreign import "glClearDepth" unsafe clearDepth :: GLclampd -> IO ()

foreign import "glClearStencil" unsafe clearStencil :: GLint -> IO ()

clearAccum :: Color4 GLfloat -> IO ()
clearAccum (Color4 r g b a) = glClearAccum r g b a

foreign import "glClearAccum" unsafe glClearAccum :: GLfloat -> GLfloat -> GLfloat -> GLfloat -> IO ()

---------------------------------------------------------------------------
-- Section 4.2.4 (The Accumulation Buffer)

data AccumOp =
     Accum
   | Load
   | Return
   | Mult
   | Add
   deriving (Eq,Ord)

marshalAccumOp :: AccumOp -> GLenum
marshalAccumOp Accum  = gl_ACCUM
marshalAccumOp Load   = gl_LOAD
marshalAccumOp Return = gl_RETURN
marshalAccumOp Mult   = gl_MULT
marshalAccumOp Add    = gl_ADD

accum :: AccumOp -> GLfloat -> IO ()
accum = glAccum . marshalAccumOp

foreign import "glAccum" unsafe glAccum :: GLenum -> GLfloat -> IO ()
