{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to chapter 7 (Callback Registration) of the
GLUT3 API docs.
-}

module GLUT_CBWindow (
   marshalMouseButton,  -- internal use only (see GLUT_Menu)
   DisplayAction,         displayFunc,
   overlayDisplayFunc,                                         -- @glut_api_geq3@
   ReshapeAction,         reshapeFunc,
   KeyboardAction,        keyboardFunc,
   keyboardUpFunc,                                             -- @glut_geq_4_13@
   MouseAction,           mouseFunc, MouseButton(..), ButtonState(..),
   MotionAction,          motionFunc, passiveMotionFunc,
   VisibilityAction,      visibilityFunc, Visibility(..),
   WindowStatusAction,    windowStatusFunc, WindowState(..),   -- @glut_geq_4_9@
   EntryAction,           entryFunc, EntryExit(..),
   SpecialAction,         specialFunc,
   specialUpFunc,                                              -- @glut_geq_4_13@
   SpecialKey(..),
   SpaceballMotionAction, spaceballMotionFunc,
   SpaceballRotateAction, spaceballRotateFunc,
   SpaceballButtonAction, spaceballButtonFunc,
   ButtonBoxAction,       buttonBoxFunc,
   DialsAction,           dialsFunc,
   TabletMotionAction,    tabletMotionFunc,
   TabletButtonAction,    tabletButtonFunc,
   JoystickAction,        joystickFunc, JoystickButton(..)     -- @glut_geq_4_13@
) where

import Prelude            hiding (Either(Left))
import Char               (chr)
import CForeign           (CUChar, CInt, CUInt)
import Foreign            (FunPtr)

import GL_BasicTypes      (WindowPosition, windowPosition,
                           WindowSize, windowSize, fromBitfield)
import GLUT_Constants     (glut_LEFT_BUTTON, glut_RIGHT_BUTTON,
                           glut_MIDDLE_BUTTON, glut_DOWN, glut_UP, glut_LEFT,
                           glut_ENTERED, glut_NOT_VISIBLE, glut_VISIBLE,
                           glut_HIDDEN, glut_FULLY_RETAINED,
                           glut_PARTIALLY_RETAINED, glut_FULLY_COVERED,
                           glut_KEY_F1, glut_KEY_F2, glut_KEY_F3, glut_KEY_F4,
                           glut_KEY_F5, glut_KEY_F6, glut_KEY_F7, glut_KEY_F8,
                           glut_KEY_F9, glut_KEY_F10, glut_KEY_F11, glut_KEY_F12,
                           glut_KEY_LEFT, glut_KEY_UP, glut_KEY_RIGHT,
                           glut_KEY_DOWN, glut_KEY_PAGE_UP, glut_KEY_PAGE_DOWN,
                           glut_KEY_HOME, glut_KEY_END, glut_KEY_INSERT,
                           glut_JOYSTICK_BUTTON_A, glut_JOYSTICK_BUTTON_B,
                           glut_JOYSTICK_BUTTON_C, glut_JOYSTICK_BUTTON_D)
import GLUT_CallbackUtils (PerWindowCallback(..), wFunc)

---------------------------------------------------------------------------
-- callbacks for current window

type DisplayAction = IO ()

-- Note: No Maybe here, because it's illegal to unregister a display callback.
displayFunc :: DisplayAction -> IO ()
displayFunc = wFunc DisplayCB mkDisplayFunc displayFunc_ . Just

foreign export dynamic mkDisplayFunc :: DisplayAction -> IO (FunPtr DisplayAction)
foreign import "glutDisplayFunc" displayFunc_ :: FunPtr DisplayAction -> IO ()

--------------------

-- @glut_api_geq3@
overlayDisplayFunc :: Maybe DisplayAction -> IO ()
overlayDisplayFunc = wFunc OverlayDisplayCB mkDisplayFunc overlayDisplayFunc_

foreign import "glutOverlayDisplayFunc" overlayDisplayFunc_ :: FunPtr DisplayAction -> IO ()

--------------------

type ReshapeAction = WindowSize -> IO ()

reshapeFunc :: Maybe ReshapeAction -> IO ()
reshapeFunc = wFunc ReshapeCB (\act -> mkReshapeFunc (\w h -> act (windowSize w h))) reshapeFunc_

foreign export dynamic mkReshapeFunc :: (CInt -> CInt -> IO ()) -> IO (FunPtr (CInt -> CInt -> IO ()))
foreign import "glutReshapeFunc" reshapeFunc_ :: FunPtr (CInt -> CInt -> IO ()) -> IO ()

--------------------

type KeyboardAction = Char -> WindowPosition -> IO ()

keyboardFunc :: Maybe KeyboardAction -> IO ()
keyboardFunc = wFunc KeyboardCB (\act -> mkKeyboardFunc (\c x y -> act (chr (fromIntegral c)) (windowPosition x y))) keyboardFunc_

foreign export dynamic mkKeyboardFunc :: (CUChar -> CInt -> CInt -> IO ()) -> IO (FunPtr (CUChar -> CInt -> CInt -> IO ()))
foreign import "glutKeyboardFunc" keyboardFunc_ :: FunPtr (CUChar -> CInt -> CInt -> IO ()) -> IO ()

--------------------

-- @glut_geq_4_13@
keyboardUpFunc :: Maybe KeyboardAction -> IO ()
keyboardUpFunc = wFunc KeyboardUpCB (\act -> mkKeyboardFunc (\c x y -> act (chr (fromIntegral c)) (windowPosition x y))) keyboardUpFunc_

foreign import "glutKeyboardUpFunc" keyboardUpFunc_ :: FunPtr (CUChar -> CInt -> CInt -> IO ()) -> IO ()

--------------------

data MouseButton =
     LeftButton
   | MiddleButton
   | RightButton
   deriving (Eq,Ord)

marshalMouseButton :: MouseButton -> CInt
marshalMouseButton LeftButton   = glut_LEFT_BUTTON
marshalMouseButton MiddleButton = glut_MIDDLE_BUTTON
marshalMouseButton RightButton  = glut_RIGHT_BUTTON

unmarshalMouseButton :: CInt -> MouseButton
unmarshalMouseButton button
   | button == glut_LEFT_BUTTON   = LeftButton
   | button == glut_MIDDLE_BUTTON = MiddleButton
   | button == glut_RIGHT_BUTTON  = RightButton
   | otherwise                    = error "unmarshalMouseButton"

data ButtonState =
     Down
   | Up
   deriving (Eq,Ord)

unmarshalButtonState :: CInt -> ButtonState
unmarshalButtonState buttonState
   | buttonState == glut_DOWN = Down
   | buttonState == glut_UP   = Up
   | otherwise                = error "unmarshalButtonState"

type MouseAction = MouseButton -> ButtonState -> WindowPosition -> IO ()

mouseFunc :: Maybe MouseAction -> IO ()
mouseFunc = wFunc MouseCB (\act -> mkMouseFunc (\b s x y -> act (unmarshalMouseButton b)
                                                                (unmarshalButtonState s)
                                                                (windowPosition x y)))
                  mouseFunc_

foreign export dynamic mkMouseFunc :: (CInt -> CInt -> CInt -> CInt -> IO ()) -> IO (FunPtr (CInt -> CInt -> CInt -> CInt -> IO ()))
foreign import "glutMouseFunc" mouseFunc_ :: FunPtr (CInt -> CInt -> CInt -> CInt -> IO ()) -> IO ()

--------------------

type MotionAction = WindowPosition -> IO ()

motionFunc :: Maybe MotionAction -> IO ()
motionFunc = wFunc MotionCB (\act -> mkMotionFunc (\x y -> act (windowPosition x y))) motionFunc_

foreign export dynamic mkMotionFunc :: (Int -> Int -> IO ()) -> IO (FunPtr (Int -> Int -> IO ()))
foreign import "glutMotionFunc" motionFunc_ :: FunPtr (Int -> Int -> IO ()) -> IO ()

--------------------

passiveMotionFunc :: Maybe MotionAction -> IO ()
passiveMotionFunc = wFunc PassiveMotionCB (\act -> mkMotionFunc (\x y -> act (windowPosition x y))) passiveMotionFunc_

foreign import "glutPassiveMotionFunc" passiveMotionFunc_ :: FunPtr (Int -> Int -> IO ()) -> IO ()

--------------------

data EntryExit =
     Left
   | Entered
   deriving (Eq,Ord)

unmarshalEntryExit :: CInt -> EntryExit
unmarshalEntryExit entryExit
   | entryExit == glut_LEFT    = Left
   | entryExit == glut_ENTERED = Entered
   | otherwise                 = error "unmarshalEntryExit" 

type EntryAction = EntryExit -> IO ()

entryFunc :: Maybe EntryAction -> IO ()
entryFunc = wFunc EntryExitCB (\act -> mkEntryFunc (\e -> act (unmarshalEntryExit e))) entryFunc_

foreign export dynamic mkEntryFunc :: (CInt -> IO ()) -> IO (FunPtr (CInt -> IO ()))
foreign import "glutEntryFunc" entryFunc_ :: FunPtr (CInt -> IO ()) -> IO ()

--------------------

data Visibility =
     NotVisible
   | Visible
   deriving (Eq,Ord)

unmarshalVisibility :: CInt -> Visibility
unmarshalVisibility visibility
   | visibility == glut_NOT_VISIBLE = NotVisible
   | visibility == glut_VISIBLE     = Visible
   | otherwise                      = error "unmarshalVisibility"

type VisibilityAction = Visibility -> IO ()

visibilityFunc :: Maybe VisibilityAction -> IO ()
visibilityFunc = wFunc VisibilityCB (\act -> mkVisibilityFunc (\v -> act (unmarshalVisibility v))) visibilityFunc_

foreign export dynamic mkVisibilityFunc :: (CInt -> IO ()) -> IO (FunPtr (CInt -> IO ()))
foreign import "glutVisibilityFunc" visibilityFunc_ :: FunPtr (CInt -> IO ()) -> IO ()

--------------------

-- @glut_geq_4_9@
data WindowState =
     Hidden
   | FullyRetained
   | PartiallyRetained
   | FullyCovered
   deriving (Eq,Ord)

unmarshalWindowState :: CInt -> WindowState
unmarshalWindowState windowState
   | windowState == glut_HIDDEN             = Hidden
   | windowState == glut_FULLY_RETAINED     = FullyRetained
   | windowState == glut_PARTIALLY_RETAINED = PartiallyRetained
   | windowState == glut_FULLY_COVERED      = FullyCovered
   | otherwise                              = error "unmarshalWindowState"

type WindowStatusAction = WindowState -> IO ()

windowStatusFunc :: Maybe WindowStatusAction -> IO ()
windowStatusFunc = wFunc WindowStatusCB (\act -> mkWindowStatusFunc (\v -> act (unmarshalWindowState v))) windowStatusFunc_

foreign export dynamic mkWindowStatusFunc :: (CInt -> IO ()) -> IO (FunPtr (CInt -> IO ()))
foreign import "glutWindowStatusFunc" windowStatusFunc_ :: FunPtr (CInt -> IO ()) -> IO ()

--------------------

data SpecialKey =
     KeyF1
   | KeyF2
   | KeyF3
   | KeyF4
   | KeyF5
   | KeyF6
   | KeyF7
   | KeyF8
   | KeyF9
   | KeyF10
   | KeyF11
   | KeyF12
   | KeyLeft
   | KeyUp
   | KeyRight
   | KeyDown
   | KeyPageUp
   | KeyPageDown
   | KeyHome
   | KeyEnd
   | KeyInsert
   deriving (Eq,Ord)

unmarshalSpecialKey :: CInt -> SpecialKey
unmarshalSpecialKey key
   | key == glut_KEY_F1        = KeyF1
   | key == glut_KEY_F2        = KeyF2
   | key == glut_KEY_F3        = KeyF3
   | key == glut_KEY_F4        = KeyF4
   | key == glut_KEY_F5        = KeyF5
   | key == glut_KEY_F6        = KeyF6
   | key == glut_KEY_F7        = KeyF7
   | key == glut_KEY_F8        = KeyF8
   | key == glut_KEY_F9        = KeyF9
   | key == glut_KEY_F10       = KeyF10
   | key == glut_KEY_F11       = KeyF11
   | key == glut_KEY_F12       = KeyF12
   | key == glut_KEY_LEFT      = KeyLeft
   | key == glut_KEY_UP        = KeyUp
   | key == glut_KEY_RIGHT     = KeyRight
   | key == glut_KEY_DOWN      = KeyDown
   | key == glut_KEY_PAGE_UP   = KeyPageUp
   | key == glut_KEY_PAGE_DOWN = KeyPageDown
   | key == glut_KEY_HOME      = KeyHome
   | key == glut_KEY_END       = KeyEnd
   | key == glut_KEY_INSERT    = KeyInsert
   | otherwise                 = error "unmarshalSpecialKey"

type SpecialAction = SpecialKey -> WindowPosition -> IO ()

specialFunc :: Maybe SpecialAction -> IO ()
specialFunc = wFunc SpecialCB (\act -> mkSpecialFunc (\k x y -> act (unmarshalSpecialKey k)
                                                                    (windowPosition x y)))
                    specialFunc_

foreign export dynamic mkSpecialFunc :: (CInt -> CInt -> CInt -> IO ()) -> IO (FunPtr (CInt -> CInt -> CInt -> IO ()))
foreign import "glutSpecialFunc" specialFunc_ :: FunPtr (CInt -> CInt -> CInt -> IO ()) -> IO ()

--------------------

-- @glut_geq_4_13@
specialUpFunc :: Maybe SpecialAction -> IO ()
specialUpFunc = wFunc SpecialUpCB (\act -> mkSpecialFunc (\k x y -> act (unmarshalSpecialKey k)
                                                                        (windowPosition x y)))
                      specialUpFunc_

foreign import "glutSpecialUpFunc" specialUpFunc_ :: FunPtr (CInt -> CInt -> CInt -> IO ()) -> IO ()

--------------------

type SpaceballMotionAction = (Int, Int, Int) -> IO ()

spaceballMotionFunc :: Maybe SpaceballMotionAction -> IO ()
spaceballMotionFunc = wFunc SpaceballMotionCB (\act -> mkSpaceballMotionFunc (\x y z -> act (fromIntegral x, fromIntegral y, fromIntegral z))) spaceballMotionFunc_

foreign export dynamic mkSpaceballMotionFunc :: (CInt -> CInt -> CInt -> IO ()) -> IO (FunPtr (CInt -> CInt -> CInt -> IO ()))
foreign import "glutSpaceballMotionFunc" spaceballMotionFunc_ :: FunPtr (CInt -> CInt -> CInt -> IO ()) -> IO ()

--------------------

type SpaceballRotateAction = (Int, Int, Int) -> IO ()

spaceballRotateFunc :: Maybe SpaceballRotateAction -> IO ()
spaceballRotateFunc = wFunc SpaceballRotateCB (\act -> mkSpaceballRotateFunc (\x y z -> act (fromIntegral x, fromIntegral y, fromIntegral z))) spaceballRotateFunc_

foreign export dynamic mkSpaceballRotateFunc :: (CInt -> CInt -> CInt -> IO ()) -> IO (FunPtr (CInt -> CInt -> CInt -> IO ()))
foreign import "glutSpaceballRotateFunc" spaceballRotateFunc_ :: FunPtr (CInt -> CInt -> CInt -> IO ()) -> IO ()

--------------------

type SpaceballButtonAction = Int -> ButtonState -> IO ()

spaceballButtonFunc :: Maybe SpaceballButtonAction -> IO ()
spaceballButtonFunc = wFunc SpaceballButtonCB (\act -> mkSpaceballButtonFunc (\b s -> act (fromIntegral b) (unmarshalButtonState s))) spaceballButtonFunc_

foreign export dynamic mkSpaceballButtonFunc :: (CInt -> CInt -> IO ()) -> IO (FunPtr (CInt -> CInt -> IO ()))
foreign import "glutSpaceballButtonFunc" spaceballButtonFunc_ :: FunPtr (CInt -> CInt -> IO ()) -> IO ()

--------------------

type ButtonBoxAction = Int -> ButtonState -> IO ()

buttonBoxFunc :: Maybe ButtonBoxAction -> IO ()
buttonBoxFunc = wFunc ButtonBoxCB (\act -> mkButtonBoxFunc (\b s -> act (fromIntegral b) (unmarshalButtonState s))) buttonBoxFunc_

foreign export dynamic mkButtonBoxFunc :: (CInt -> CInt -> IO ()) -> IO (FunPtr (CInt -> CInt -> IO ()))
foreign import "glutButtonBoxFunc" buttonBoxFunc_ :: FunPtr (CInt -> CInt -> IO ()) -> IO ()

--------------------

type DialsAction = Int -> Int -> IO ()

dialsFunc :: Maybe DialsAction -> IO ()
dialsFunc = wFunc DialsCB (\act -> mkDialsFunc (\d v -> act (fromIntegral d) (fromIntegral v))) dialsFunc_

foreign export dynamic mkDialsFunc :: (CInt -> CInt -> IO ()) -> IO (FunPtr (CInt -> CInt -> IO ()))
foreign import "glutDialsFunc" dialsFunc_ :: FunPtr (CInt -> CInt -> IO ()) -> IO ()

--------------------

type TabletMotionAction = (Int, Int) -> IO ()

tabletMotionFunc :: Maybe TabletMotionAction -> IO ()
tabletMotionFunc = wFunc TabletMotionCB (\act -> mkTabletMotionFunc (\x y -> act (fromIntegral x, fromIntegral y))) tabletMotionFunc_

foreign export dynamic mkTabletMotionFunc :: (CInt -> CInt -> IO ()) -> IO (FunPtr (CInt -> CInt -> IO ()))
foreign import "glutTabletMotionFunc" tabletMotionFunc_ :: FunPtr (CInt -> CInt -> IO ()) -> IO ()

--------------------

type TabletButtonAction = Int -> ButtonState -> (Int, Int) -> IO ()

tabletButtonFunc :: Maybe TabletButtonAction -> IO ()
tabletButtonFunc = wFunc TabletButtonCB (\act -> mkTabletButtonFunc (\b s x y -> act (fromIntegral b) (unmarshalButtonState s) (fromIntegral x, fromIntegral y))) tabletButtonFunc_

foreign export dynamic mkTabletButtonFunc :: (CInt -> CInt -> CInt -> CInt -> IO ()) -> IO (FunPtr (CInt -> CInt -> CInt -> CInt -> IO ()))
foreign import "glutTabletButtonFunc" tabletButtonFunc_ :: FunPtr (CInt -> CInt -> CInt -> CInt -> IO ()) -> IO ()

--------------------

-- @glut_geq_4_13@
data JoystickButton =
     JoystickButtonA
   | JoystickButtonB
   | JoystickButtonC
   | JoystickButtonD
   deriving (Eq,Ord,Enum,Bounded)

marshalJoystickButton :: JoystickButton -> CUInt
marshalJoystickButton JoystickButtonA = glut_JOYSTICK_BUTTON_A
marshalJoystickButton JoystickButtonB = glut_JOYSTICK_BUTTON_B
marshalJoystickButton JoystickButtonC = glut_JOYSTICK_BUTTON_C
marshalJoystickButton JoystickButtonD = glut_JOYSTICK_BUTTON_D

uintToButtons :: CUInt -> [JoystickButton]
uintToButtons = fromBitfield marshalJoystickButton

type JoystickAction = [JoystickButton] -> (Int, Int, Int) -> IO ()

joystickFunc :: Maybe JoystickAction -> Int -> IO ()
joystickFunc mbAct interv = wFunc JoystickCB (\act -> mkJoystickFunc (\b x y z -> act (uintToButtons b) (fromIntegral x, fromIntegral y, fromIntegral z))) (\a -> joystickFunc_ a (fromIntegral interv)) mbAct

foreign export dynamic mkJoystickFunc :: (CUInt -> CInt -> CInt -> CInt -> IO ()) -> IO (FunPtr (CUInt -> CInt -> CInt -> CInt -> IO ()))
foreign import "glutJoystickFunc" joystickFunc_ :: FunPtr (CUInt -> CInt -> CInt -> CInt -> IO ()) -> CInt -> IO ()
