/* 
 * File: messages.h
 * Author: Jason Short
 * Project: GGZCards Server
 * Date: 06/20/2001
 * Desc: Functions and data for messaging system
 * $Id: message.c 6077 2004-07-11 04:28:48Z jdorje $
 *
 * Right now GGZCards uses a super-generic "messaging" system.  Text
 * messages are generated by the server and sent to the client for
 * direct display.  This makes i18n basically impossible.  It also
 * makes really cool client displays basically impossible.  However,
 * it makes it really easy to add new games - you don't even have to
 * change the client!
 *
 * A more complicated alternate messaging system is in planning.
 *
 * Copyright (C) 2001-2002 Brent Hendricks.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-130
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>		/* Site-specific config */
#endif

#include <math.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ggz.h>

#include "net_common.h"

#include "common.h"
#include "message.h"
#include "net.h"

#include "message.h"

struct global_message_list_t {
	char *mark;
	char *message;
	struct global_message_list_t *next;
};

/* NOTE ON "MESSAGES" - Each seat has a message.  The client should display
   this near to the player.  It may contain information such as bid, etc.  -
   There is a global message.  The client should display this prominently. It 
   may contain information such as contract, trump, etc.  - There are
   additional "tagged" global messages.  These are not implemented. */


/* 
 * PLAYER MESSAGES
 */

/* send_player_message() sends seat s's message to player p fails silently... 
 */
void send_player_message(seat_t s, player_t p)
{
	if (game.seats[s].pmessage == NULL)
		return; /* ??? */
		
	net_send_player_text_message(p, s, game.seats[s].pmessage);
}

static void doput_player_message(seat_t s, char *msg)
{
	if (game.seats[s].pmessage != NULL)
		ggz_free(game.seats[s].pmessage);
	game.seats[s].pmessage = ggz_strdup(msg);
}

void clear_player_message(seat_t s)
{
	doput_player_message(s, "");
}

void put_player_message(seat_t s, char *fmt, ...)
{
	char buf[4096];

	va_list ap;
	va_start(ap, fmt);
	vsnprintf(buf, sizeof(buf), fmt, ap);
	va_end(ap);

	doput_player_message(s, buf);
}

void add_player_message(seat_t s, char *fmt, ...)
{
	va_list ap;
	char buf[4096] = "";
	int len;

	if (game.seats[s].pmessage != NULL)
		strcpy(buf, game.seats[s].pmessage);
	len = strlen(buf);

	va_start(ap, fmt);
	vsnprintf(buf + len, sizeof(buf) - len, fmt, ap);
	va_end(ap);

	doput_player_message(s, buf);
}

/* sends seat s's message to all players; fails silently... */
void broadcast_player_message(seat_t s)
{
	assert(s >= 0 && s < game.num_seats);

	allplayers_iterate(p) {
		send_player_message(s, p);
	} allplayers_iterate_end;
}

void set_player_message(player_t p)
{
	if (game.data == NULL) {
		/* silently fail; it's easier to check here than elsewhere */
		return;
	}

	ggz_debug(DBG_MISC, "Setting player %d/%s's message.", p,
		    get_player_name(p));
	if (p < 0 || p >= game.num_players)
		ggz_error_msg("set_player_message(%d) called.", p);
	game.data->set_player_message(p);
	broadcast_player_message(game.players[p].seat);
}

void set_all_player_messages()
{
	player_t p;
	for (p = 0; p < game.num_players; p++)
		set_player_message(p);
}


/*
 * GLOBAL MESSAGES
 */

/* send_global_message sends a global message to player p fails silently... */
void send_global_message(const char *mark, player_t p)
{
	net_send_global_text_message(p, mark, get_global_message(mark));
}

void send_all_global_messages(player_t p)
{
	global_message_list_t *gml;

	for (gml = game.message_head; gml != NULL; gml = gml->next)
		net_send_global_text_message(p, gml->mark, gml->message);
}

const char *get_global_message(const char *mark)
{
	global_message_list_t *gml;

	for (gml = game.message_head; gml != NULL; gml = gml->next)
		if (!strcmp(mark, gml->mark))
			return gml->message;
	return "";
}

static void put_global_message(char *mark, char *msg)
{
	global_message_list_t *gml;
	ggz_debug(DBG_MISC,
		    "Setting global message for '%s'.  Length is %d.", mark,
		    strlen(msg));

	if (!mark || !msg)
		ggz_error_msg("put_global_message called on NULL string.");

	for (gml = game.message_head; gml != NULL; gml = gml->next) {
		if (!strcmp(mark, gml->mark)) {
			ggz_free(gml->message);
			gml->message = ggz_strdup(msg);
			return;
		}
	}

	gml = ggz_malloc(sizeof(global_message_list_t));
	gml->mark = ggz_strdup(mark);
	gml->message = ggz_strdup(msg);
	if (game.message_tail)
		game.message_tail->next = gml;
	game.message_tail = gml;
	if (!game.message_head)
		game.message_head = gml;
}

/* set_global_message */
void set_global_message(char *mark, char *message, ...)
{
	char buf[4096];

	va_list ap;
	va_start(ap, message);
	vsnprintf(buf, sizeof(buf), message, ap);
	va_end(ap);

	put_global_message(mark, buf);

	net_broadcast_global_text_message(mark, buf);
}


/* 
 * AUTOMATED PLAYER MESSAGES
 */

void add_player_score_message(player_t p)
{
	seat_t s = game.players[p].seat;
	add_player_message(s, "Score: %d\n", game.players[p].score);
}

void add_player_bid_message(player_t p)
{
	seat_t s = game.players[p].seat;
	if (game.players[p].bid_count > 0) {
		char bid_text[512];
		game.data->get_bid_text(bid_text, sizeof(bid_text),
					 game.players[p].bid);
		if (*bid_text)
			add_player_message(s, "Bid: %s\n", bid_text);
	}
}

void add_player_tricks_message(player_t p)
{
	seat_t s = game.players[p].seat;

	if (game.state != STATE_WAIT_FOR_PLAY
	    && game.state != STATE_NEXT_TRICK
	    && game.state != STATE_NEXT_PLAY)
		return;

	if (game.players[p].team >= 0) {
		int tricks = 0;
		player_t p2;

		for (p2 = 0; p2 < game.num_players; p2++)
			if (game.players[p2].team == game.players[p].team)
				tricks += game.players[p2].tricks;

		add_player_message(s, "Tricks: %d (%d)\n",
				   game.players[p].tricks, tricks);
	} else {
		add_player_message(s, "Tricks: %d\n", game.players[p].tricks);
	}
}

void add_player_action_message(player_t p)
{
	seat_t s = game.players[p].seat;
	if (game.players[p].bid_data.is_bidding)
		add_player_message(s, "Bidding...");
	if (game.state == STATE_WAIT_FOR_PLAY && game.players[p].is_playing)
		add_player_message(s, "Playing...");
}

void add_player_rating_message(player_t p)
{
	/* TODO: use the correct interface for this. */
#if 0
	if (game.rated && game.stats) {
		seat_t s = game.players[p].seat;
		int rating, wins, losses, ties;
		
		if (ggzstats_get_record(game.stats, p, &wins, &losses, &ties) < 0)
			return;
			
		if (ggzstats_get_rating(game.stats, p, &rating) < 0)
			return;
		
		if (ties > 0)
			add_player_message(s, "Rating: %d (%d-%d-%d)\n",
			                   rating, wins, losses, ties);
		else
			add_player_message(s, "Rating: %d (%d-%d)\n",
			                   rating, wins, losses);
	}
#endif
}


/* 
 * AUTOMATED GLOBAL MESSAGES
 */

void send_last_hand(void)
{
	int *lengths = ggz_malloc(game.num_seats * sizeof(*lengths));
	card_t **cardlist = ggz_malloc(game.num_seats * sizeof(*cardlist));
	hand_t *hand;
	seat_t s;

	for (s = 0; s < game.num_seats; s++) {
		hand = &game.seats[s].hand;
		hand->hand_size = hand->full_hand_size;
		cards_sort_hand(hand);
		lengths[s] = hand->full_hand_size;
		cardlist[s] = hand->cards;
	}

	net_broadcast_global_cardlist_message("Previous Hand", lengths, cardlist);

	ggz_free(lengths);
	ggz_free(cardlist);
}

void send_last_trick(void)
{
	int *lengths = ggz_malloc(game.num_seats * sizeof(*lengths));
	card_t **cardlist = ggz_malloc(game.num_seats * sizeof(*cardlist));
	card_t *cardlist2 = ggz_malloc(game.num_seats * sizeof(*cardlist2));
	seat_t s;

	for (s = 0; s < game.num_seats; s++) {
		if (game.seats[s].player >= 0) {
			lengths[s] = 1;
			cardlist[s] = &cardlist2[s];
			cardlist[s][0] = game.seats[s].table;
		} else
			lengths[s] = 0;
	}

	net_broadcast_global_cardlist_message("Last Trick", lengths, cardlist);

	ggz_free(lengths);
	ggz_free(cardlist);
	ggz_free(cardlist2);
}

static int **cumulative_scores = NULL;
static int c_score_count = 0;
static int c_score_size = 0;

/* Surely there's some library function to do this... */
static int int_len(int x)
{
	int len = x < 0 ? 1 : 0;

	for (x = abs(x); x > 0; x /= 10)
		len++;

	return len;
}

/* This function is extra-ordinarily overcomplicated.  I hate string
   handling... */
static void send_cumulative_scores(void)
{
	char buf[4096];
	int buf_len = 0, r;
	int widths[game.num_players];

	if (!game.cumulative_scores)
		return;

	/* First, put up the list of players/teams and calculate the string
	   width of each.  This is a pretty complicated process... */
	if (game.num_teams == 0) {
		player_t p;
		for (p = 0; p < game.num_players; p++) {
			widths[p] = strlen(get_player_name(p));
			if (widths[p] < 4)
				widths[p] = 4;
			buf_len +=
				snprintf(buf + buf_len, sizeof(buf) - buf_len,
					 "%*s%s", widths[p],
					 get_player_name(p),
					 p ==
					 game.num_players - 1 ? "" : " ");
		}
	} else {
		team_t t;

		memset(widths, 0, sizeof(widths));

		for (t = 0; t < game.num_teams; t++) {
			int num = 0;
			player_t p;
			char *orig = buf + buf_len;

			for (p = 0; p < game.num_players; p++) {
				if (game.players[p].team == t) {
					if (num != 0)
						buf_len +=
							snprintf(buf +
								 buf_len,
								 sizeof(buf) -
								 buf_len,
								 "/");

					buf_len +=
						snprintf(buf + buf_len,
							 sizeof(buf) -
							 buf_len, "%s",
							 get_player_name(p));

					num++;
				}
			}

			widths[t] = strlen(orig);

			if (t != game.num_teams - 1) {
				buf_len +=
					snprintf(buf + buf_len,
						 sizeof(buf) - buf_len, " ");
			}
		}
	}
	buf_len += snprintf(buf + buf_len, sizeof(buf) - buf_len, "\n");

	for (r = 0; r < c_score_count; r++) {
		int max =
			game.num_teams ==
			0 ? game.num_players : game.num_teams;
		int x;		/* player or team... */

		for (x = 0; x < max; x++) {
			int score = cumulative_scores[r][x];
			int iw = int_len(score);
			int extra = widths[x] - iw;

			/* this overcomplicated bit of hackery is intended to
			   center the score under the name. Unfortunately, it
			   assumes the number isn't longer than 3 characters. */
			buf_len +=
				snprintf(buf + buf_len, sizeof(buf) - buf_len,
					 "%*s%*d%*s%s", extra / 2, "", iw,
					 score, (extra + 1) / 2, "",
					 (x == max - 1) ? "" : " ");
		}
		buf_len +=
			snprintf(buf + buf_len, sizeof(buf) - buf_len, "\n");
	}

	set_global_message("Scores", "%s", buf);
}

void init_cumulative_scores()
{
	int i;

	if (!game.cumulative_scores)
		return;

	/* Free the array. */
	if (cumulative_scores) {
		for (i = 0; i < c_score_count; i++)
			ggz_free(cumulative_scores[i]);
		ggz_free(cumulative_scores);
	}

	cumulative_scores = NULL;
	c_score_count = c_score_size = 0;

	send_cumulative_scores();
}

/* add on this hand's scores */
void update_cumulative_scores()
{
	int num = game.num_teams > 0 ? game.num_teams : game.num_players;
	int *score_round;

	if (!game.cumulative_scores)
		return;
	c_score_count++;
	if (c_score_count > c_score_size) {
		c_score_size += 10;
		c_score_size = c_score_size * 3 / 2;
		cumulative_scores =
			ggz_realloc(cumulative_scores,
				    c_score_size *
				    sizeof(*cumulative_scores));
	}
	score_round = ggz_malloc(num * sizeof(**cumulative_scores));
	cumulative_scores[c_score_count - 1] = score_round;

	if (game.num_teams == 0) {
		player_t p;
		for (p = 0; p < game.num_players; p++)
			cumulative_scores[c_score_count - 1][p] =
				game.players[p].score;
	} else {
		player_t p;
		for (p = 0; p < game.num_players; p++) {
			cumulative_scores[c_score_count -
					  1][game.players[p].team] =
				game.players[p].score;
		}
	}

	send_cumulative_scores();
}

void send_bid_history()
{
	int r, buf_len = 0, widths[game.num_players];
	player_t p;
	char buf[4096];
	char buf2[512];

	if (!game.bid_history)
		return;

	for (p = 0; p < game.num_players; p++) {
		widths[p] = strlen(get_player_name(p));
		/* TODO if (widths[p] < game.max_bid_length) widths[p] =
		   game.max_bid_length; */
		buf_len +=
			snprintf(buf + buf_len, sizeof(buf) - buf_len,
				 "%*s%s", widths[p], get_player_name(p),
				 p == game.num_players - 1 ? "" : " ");
	}
	buf_len += snprintf(buf + buf_len, sizeof(buf) - buf_len, "\n");

	for (r = 0; r <= game.bid_rounds; r++) {
		for (p = 0; p < game.num_players; p++) {
			game.data->get_bid_text(buf2, sizeof(buf2),
						 game.players[p].allbids[r]);
			buf_len +=
				snprintf(buf + buf_len, sizeof(buf) - buf_len,
					 "%*s%s", widths[p], buf2,
					 p ==
					 game.num_players - 1 ? "" : " ");

		}
		buf_len +=
			snprintf(buf + buf_len, sizeof(buf) - buf_len, "\n");
	}

	set_global_message("Bid History", "%s", buf);
}
