/* transform_ui.c */
/*
 * ggobi
 * Copyright (C) AT&T, Duncan Temple Lang, Dianne Cook 1999-2005
 *
 * ggobi is free software; you may use, redistribute, and/or modify it
 * under the terms of the Common Public License, which is distributed
 * with the source code and displayed on the ggobi web site, 
 * www.ggobi.org.  For more information, contact the authors:
 *
 *   Deborah F. Swayne   dfs@research.att.com
 *   Di Cook             dicook@iastate.edu
 *   Duncan Temple Lang  duncan@wald.ucdavis.edu
 *   Andreas Buja        andreas.buja@wharton.upenn.edu
*/

#include <string.h>
#include <stdlib.h>

#include <gtk/gtk.h>
#include "vars.h"
#include "externs.h"

static void tf0_combo_box_set_value(vartabled *vt, gboolean, ggobid *gg);
static void tf1_combo_box_set_value(vartabled *vt, gboolean, ggobid *gg);
static void tf2_combo_box_set_value(vartabled *vt, gboolean, ggobid *gg);


/*-- called when closed from the close button --*/
static void close_btn_cb (GtkWidget *w, ggobid *gg) {
  gtk_widget_hide (gg->tform_ui.window);
}
/*-- called when closed from the window manager --*/
static void close_wmgr_cb (GtkWidget *w, GdkEvent *event, ggobid *gg) {
  gtk_widget_hide (gg->tform_ui.window);
}

static gchar *stage0_lbl[] = {"No transformation",
                              "Raise minimum to 0",
                              "Raise minimum to 1",
                              "Negative"};
static void stage0_cb (GtkWidget *w, ggobid *gg)
{
  gint indx = gtk_combo_box_get_active (GTK_COMBO_BOX(w));
  GtkWidget *tree_view = get_tree_view_from_object (G_OBJECT(gg->tform_ui.window));
  GGobiData *d = (GGobiData *) g_object_get_data(G_OBJECT (tree_view), "datad");
  gint *vars;// = (gint *) g_malloc (d->ncols * sizeof(gint));
  gint nvars;
  
  vars = get_selections_from_tree_view (tree_view, &nvars);

  if (nvars) {
    transform (0, indx, -99., vars, nvars, d, gg);
    g_free (vars);
  }
}

static gchar *stage1_lbl[] = {"No transformation",
                              "Box-Cox",
                              "Log base 10",
                              "Inverse",
                              "Absolute value",
                              "Scale to [a,b]",
                              };
static void
stage1_cb (GtkWidget *w, ggobid *gg)
{
  gint indx = gtk_combo_box_get_active (GTK_COMBO_BOX(w));
  GtkWidget *tree_view = get_tree_view_from_object (G_OBJECT(gg->tform_ui.window));
  GGobiData *d = (GGobiData *) g_object_get_data(G_OBJECT (tree_view), "datad");
  gint *vars;// = (gint *) g_malloc (d->ncols * sizeof(gint));
  gint nvars;

  vars = get_selections_from_tree_view (tree_view, &nvars);

  if (nvars) {
    transform (1, indx, gg->tform_ui.boxcox_adj->value, vars, nvars, d, gg);
    g_free (vars);
  }
}

/*
 * Set the spin widget's adjustment->step_increment to adj->value
*/
void boxcox_cb (GtkAdjustment *adj, ggobid *gg)
{
  GtkWidget *tree_view = get_tree_view_from_object (G_OBJECT(gg->tform_ui.window));
  GGobiData *d = (GGobiData *) g_object_get_data(G_OBJECT (tree_view), "datad");
  gint *vars; // = (gint *) g_malloc (d->ncols * sizeof(gint));
  gint nvars;
  
  vars = get_selections_from_tree_view (tree_view, &nvars);
  
  if (nvars) {
    transform (1, BOXCOX, adj->value, vars, nvars, d, gg);
    g_free (vars);
  }
}

gfloat
scale_get_a (ggobid *gg) {
  gchar *val_str;
  gfloat val = 0;  /*-- default value --*/
  GtkWidget *entry_a;
  entry_a = widget_find_by_name (gg->tform_ui.window, "TFORM:entry_a");

  if (entry_a) {
    val_str = gtk_editable_get_chars (GTK_EDITABLE (entry_a), 0, -1);
    if (val_str != NULL && strlen (val_str) > 0) {
      val = (gfloat) atof (val_str);
      g_free (val_str);
    }
  } else {
    g_printerr ("Failed to locate the entry widget\n");
  }

  return val;
}
gfloat
scale_get_b (ggobid *gg) {
  gchar *val_str;
  gfloat val = 1;  /*-- default value --*/
  GtkWidget *entry_b;
  entry_b = widget_find_by_name (gg->tform_ui.window, "TFORM:entry_b");

  if (entry_b) {
    val_str = gtk_editable_get_chars (GTK_EDITABLE (entry_b), 0, -1);
    if (val_str != NULL && strlen (val_str) > 0) {
      val = (gfloat) atof (val_str);
      g_free (val_str);
    }
  } else {
    g_printerr ("Failed to locate the entry widget\n");
  }

  return val;
}

static gchar *stage2_lbl[] = {"No transformation",
                              "Standardize",
                              "Sort",
                              "Rank",
                              "Normal score",
                              "Z-score",
                              "Discretize: 2 levels"
                              };
static void stage2_cb (GtkWidget *w, ggobid *gg)
{
  GtkWidget *tree_view = get_tree_view_from_object (G_OBJECT(gg->tform_ui.window));
  GGobiData *d = (GGobiData *) g_object_get_data(G_OBJECT (tree_view), "datad");
  gint *vars;// = (gint *) g_malloc (d->ncols * sizeof(gint));
  gint nvars;
  gint indx = gtk_combo_box_get_active (GTK_COMBO_BOX(w));

  vars = get_selections_from_tree_view (tree_view, &nvars);
  
  if (nvars) {
    transform (2, indx, -99, vars, nvars, d, gg);
    g_free (vars);
  }
}

static void tform_reset_cb (GtkWidget *w, ggobid *gg)
{
  gint j;
  GtkWidget *tree_view = get_tree_view_from_object (G_OBJECT(gg->tform_ui.window));
  GGobiData *d = (GGobiData *) g_object_get_data(G_OBJECT (tree_view), "datad");

  for (j=0; j<d->ncols; j++) {
    transform0_values_set (NO_TFORM0, j, d, gg);
    transform1_values_set (NO_TFORM1, 1.0, j, d, gg);
    transform2_values_set (NO_TFORM2, j, d, gg);

    transform1_apply (j, d, gg);
    transform2_apply (j, d, gg);

    tform_label_update (j, d);
  }

  limits_set (d, true, true, gg->lims_use_visible);  
  vartable_limits_set (d);
  vartable_stats_set (d);

  tform_to_world (d, gg);
  displays_tailpipe (FULL, gg);
}

void
tfvar_selection_made_cb (GtkTreeSelection *tree_sel, ggobid *gg)
{
  GtkTreeView *tree_view = gtk_tree_selection_get_tree_view(tree_sel);
  GGobiData *d = (GGobiData *) g_object_get_data(G_OBJECT (tree_view), "datad");
  gint j, nvars, *rows;  // allocated in function

  /* Parameters of transformation */
  vartabled *vt0, *vtnew;

  if (!d)
    return;

  rows = get_selections_from_tree_view(GTK_WIDGET(tree_view), &nvars);
  if (nvars < 0)
    return;

  vtnew = (vartabled *) g_malloc(sizeof(vartabled));
  vt0 = vartable_element_get (rows[0], d);
  vt_copy (vt0, vtnew);

  /* If there's only one selected variable, we're ready to reset the
   * variable selection panel.  If there are more, we have to find out
   * if all the selected variables share the same transformation.
   */
  if (nvars > 1) {
    gboolean same = true;
    for (j=1; j<nvars; j++) {
      same = same && transform_values_compare(0, j, d);
      if (!same) 
        break;
    }
    /* If they aren't all the same, use the default values */
    if (!same) {
      vt_init(vtnew);
    }
  }

  /* This causes an endless loop, somehow -- resetting the widgets
   * calls transform() .. */
  /* Reset the widgets on the panel */
  tf0_combo_box_set_value(vtnew, false/*don't transform*/, gg);
  tf1_combo_box_set_value(vtnew, false, gg);
  tf2_combo_box_set_value(vtnew, false, gg);

  g_free(rows);
  g_free (vtnew);
}

void
transform_window_open (ggobid *gg) 
{
  GtkWidget *vbox, *frame, *notebook, *hb, *vb, *btn;
  GtkWidget *stage0_option_menu, *stage1_option_menu, *stage2_option_menu;
  GtkWidget *lbl, *entry_a, *entry_b;
  GtkWidget *spinner;

  /*-- if used before we have data, bail out --*/
  if (gg->d == NULL || g_slist_length (gg->d) == 0) 
    return;

  if (gg->tform_ui.window == NULL) {
    gg->tform_ui.window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (gg->tform_ui.window),
                          "Transform Variables");
    gtk_container_set_border_width (GTK_CONTAINER (gg->tform_ui.window), 10);

    g_signal_connect (G_OBJECT (gg->tform_ui.window),
                        "delete_event",
                        G_CALLBACK (close_wmgr_cb),
                        (gpointer) gg);

/*
 * Transformations
*/
    vbox = gtk_vbox_new (false, 2);
    gtk_container_add (GTK_CONTAINER (gg->tform_ui.window), vbox);

    /* Create a notebook, set the position of the tabs */
    notebook = create_variable_notebook (vbox,
      GTK_SELECTION_MULTIPLE, all_vartypes, all_datatypes,
      G_CALLBACK(tfvar_selection_made_cb), NULL, gg);

    /*
     * Stage 0: Domain adjustment
    */
    frame = gtk_frame_new ("Stage 0");
    //gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
    gtk_box_pack_start (GTK_BOX (vbox), frame, false, false, 1);

    stage0_option_menu = gtk_combo_box_new_text ();
    gtk_widget_set_name (stage0_option_menu, "TFORM:stage0_options");
    //gtk_container_set_border_width (GTK_CONTAINER (stage0_option_menu), 4);
    gtk_tooltips_set_tip (GTK_TOOLTIPS (gg->tips), stage0_option_menu,
      "Stage 0: Adjust the domain of the variables",
      NULL);
    populate_combo_box (stage0_option_menu, stage0_lbl, 
      G_N_ELEMENTS(stage0_lbl), G_CALLBACK(stage0_cb), gg);
    gtk_container_add (GTK_CONTAINER (frame), stage0_option_menu);

    /*
     * Stage 1: Power transformations et al
    */
    frame = gtk_frame_new ("Stage 1");
    //gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
    gtk_box_pack_start (GTK_BOX (vbox), frame, false, false, 1);

    vb = gtk_vbox_new (false, 5);
    gtk_container_set_border_width (GTK_CONTAINER (vb), 5);
    gtk_container_add (GTK_CONTAINER (frame), vb);

    stage1_option_menu = gtk_combo_box_new_text ();
    gtk_widget_set_name (stage1_option_menu, "TFORM:stage1_options");
    gtk_tooltips_set_tip (GTK_TOOLTIPS (gg->tips), stage1_option_menu,
      "Stage 1: Data-independent transformations, preserving user-defined limits",
      NULL);
    populate_combo_box (stage1_option_menu, stage1_lbl, 
      G_N_ELEMENTS(stage1_lbl), G_CALLBACK(stage1_cb), gg);
    gtk_box_pack_start (GTK_BOX (vb), stage1_option_menu, true, false, 1);

    /*-- label and spin button for Box-Cox parameter --*/
    hb = gtk_hbox_new (false, 2);
    gtk_box_pack_start (GTK_BOX (vb), hb, false, false, 2);
    
    lbl = gtk_label_new_with_mnemonic ("Box-Cox _param:");
    gtk_misc_set_alignment (GTK_MISC (lbl), 0, 0.5);
    gtk_box_pack_start (GTK_BOX (hb), lbl, false, false, 0);
    gg->tform_ui.boxcox_adj = (GtkAdjustment *) gtk_adjustment_new (1.0,
                          -4, 5, 0.05, .5, 0.0);
    spinner = gtk_spin_button_new (gg->tform_ui.boxcox_adj, 0, 3);
	gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), spinner);

    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), false);
    gtk_tooltips_set_tip (GTK_TOOLTIPS (gg->tips), spinner,
      "Set the Box-Cox power function parameter", NULL);
    gtk_box_pack_end (GTK_BOX (hb), spinner, true, true, 0);
    g_signal_connect (G_OBJECT (gg->tform_ui.boxcox_adj), "value_changed",
                        G_CALLBACK (boxcox_cb),
                        (gpointer) gg);

    /*-- labels and entries for scaling limits --*/
    /*style = gtk_widget_get_style (spinner);
    gdk_text_extents (
      gtk_style_get_font (style),
      "999999999", strlen ("999999999"),
      &lbearing, &rbearing, &width, &ascent, &descent);*/

    hb = gtk_hbox_new (false, 2);
    gtk_box_pack_start (GTK_BOX (vb), hb, false, false, 2);

    lbl = gtk_label_new_with_mnemonic ("_a:");
    gtk_misc_set_alignment (GTK_MISC (lbl), 0, 0.5);
    gtk_box_pack_start (GTK_BOX (hb), lbl, false, false, 0);

    entry_a = gtk_entry_new ();
	gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry_a);
    gtk_widget_set_name (entry_a, "TFORM:entry_a");
    gtk_entry_set_text (GTK_ENTRY (entry_a), "0");
	gtk_entry_set_width_chars(GTK_ENTRY(entry_a), 9);
    //gtk_widget_set_usize (entry_a, width, -1);
    gtk_box_pack_start (GTK_BOX (hb), entry_a, false, false, 0);

    lbl = gtk_label_new_with_mnemonic ("_b:");
    gtk_misc_set_alignment (GTK_MISC (lbl), 0, 0.5);
    gtk_box_pack_start (GTK_BOX (hb), lbl, false, false, 0);

    entry_b = gtk_entry_new ();
	gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry_b);
    gtk_widget_set_name (entry_b, "TFORM:entry_b");
    gtk_entry_set_text (GTK_ENTRY (entry_b), "1");
	gtk_entry_set_width_chars(GTK_ENTRY(entry_b), 9);
    //gtk_widget_set_usize (entry_b, width, -1);
    gtk_box_pack_start (GTK_BOX (hb), entry_b, false, false, 0);

    /*
     * Stage 2: Another standardization step
    */
    frame = gtk_frame_new ("Stage 2");
    //gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
    gtk_box_pack_start (GTK_BOX (vbox), frame, false, false, 1);

    stage2_option_menu = gtk_combo_box_new_text ();
    gtk_widget_set_name (stage2_option_menu, "TFORM:stage2_options");
    //gtk_container_set_border_width (GTK_CONTAINER (stage2_option_menu), 4);
    gtk_tooltips_set_tip (GTK_TOOLTIPS (gg->tips), stage2_option_menu,
      "Stage 2: Data-dependent transformations, ignoring user-defined limits",
      NULL);
    populate_combo_box (stage2_option_menu, stage2_lbl, 
      G_N_ELEMENTS(stage2_lbl), G_CALLBACK(stage2_cb), gg);
    gtk_container_add (GTK_CONTAINER (frame), stage2_option_menu);

    /*
     * A button or two
    */

    btn = gtk_button_new_with_mnemonic ("_Reset all");
    gtk_box_pack_start (GTK_BOX (vbox), btn, false, false, 0);
    gtk_tooltips_set_tip (GTK_TOOLTIPS (gg->tips), btn,
      "Set all transformation stages to 'no transformation' for the selected variables",
      NULL);
    g_signal_connect (G_OBJECT (btn), "clicked",
                        G_CALLBACK (tform_reset_cb), gg);

    /*-- add a close button --*/
    gtk_box_pack_start (GTK_BOX (vbox), gtk_hseparator_new(), false, true, 2);
    hb = gtk_hbox_new (false, 2);
    gtk_box_pack_start (GTK_BOX (vbox), hb, false, false, 1);

    btn = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
    gtk_tooltips_set_tip (GTK_TOOLTIPS (gg->tips), btn,
      "Close the window", NULL);
    gtk_box_pack_start (GTK_BOX (hb), btn, true, false, 1);
    g_signal_connect (G_OBJECT (btn), "clicked",
                        G_CALLBACK (close_btn_cb), gg);

    g_object_set_data(G_OBJECT (gg->tform_ui.window),
      "notebook", notebook);
  } 

  gtk_widget_show_all (gg->tform_ui.window);
  gdk_window_raise (gg->tform_ui.window->window);
}

/*
 * These routines are used to set the values of the option menus.
 * They're used when the transformations are set from somewhere
 * other than those option menus, such as the reset button.
*/
/*----------------------------------------------------------------*/
static void
tf0_combo_box_set_value (vartabled *vt, gboolean force, ggobid *gg)
{
  GtkWidget *cbox;
  GCallback func = G_CALLBACK(stage0_cb);
  gint n;

  cbox = widget_find_by_name (gg->tform_ui.window, "TFORM:stage0_options");
  if (cbox != NULL) {
    if (!force) { // remove callback: responding to variable selection
      n = g_signal_handlers_block_by_func(G_OBJECT(cbox), func, gg);
    }
    // Set the combo box selection
    gtk_combo_box_set_active (GTK_COMBO_BOX (cbox), vt->tform0);
    if (!force) { // restore callback
      n = g_signal_handlers_unblock_by_func(G_OBJECT(cbox), func, gg);
    }
  }
}
void
transform0_combo_box_set_value (gint j, gboolean force, GGobiData *d, ggobid *gg)
{
  vartabled *vt = vartable_element_get (j, d);
  if (vt != NULL)
    tf0_combo_box_set_value(vt, force, gg);
}
/*-------------------------------------------------------------------*/
static void
tf1_combo_box_set_value(vartabled *vt, gboolean force, ggobid *gg)
{
  gint n = 0;
  GtkWidget *cbox;
  GCallback func = G_CALLBACK(stage1_cb);

  cbox = widget_find_by_name (gg->tform_ui.window, "TFORM:stage1_options");
  if (cbox) {
    if (!force) { // remove callback: responding to variable selection
      n = g_signal_handlers_block_by_func(G_OBJECT(cbox), func, gg);
    }
    // Set the combo box selection
    gtk_combo_box_set_active (GTK_COMBO_BOX (cbox), vt->tform1);
    if (!force) { // restore callback
      n = g_signal_handlers_unblock_by_func(G_OBJECT(cbox), func, gg);
    }
  }
}
void
transform1_combo_box_set_value (gint j, gboolean force, GGobiData *d, ggobid *gg)
{
  vartabled *vt = vartable_element_get (j, d);
  if (vt != NULL)
    tf1_combo_box_set_value(vt, force, gg);
}
/*-------------------------------------------------------------------*/
static void
tf2_combo_box_set_value(vartabled *vt, gboolean force, ggobid *gg)
{
  gint n;
  GtkWidget *cbox;
  cbox = widget_find_by_name (gg->tform_ui.window, "TFORM:stage2_options");
  GCallback func = G_CALLBACK(stage2_cb);

  if (cbox != NULL)
    if (!force) { // remove callback: responding to variable selection
      n = g_signal_handlers_block_by_func(G_OBJECT(cbox), func, gg);
    }
    // Set the combo box selection
    gtk_combo_box_set_active (GTK_COMBO_BOX (cbox), vt->tform2);
    if (!force) { // restore callback
      n = g_signal_handlers_unblock_by_func(G_OBJECT(cbox), func, gg);
    }
}
void
transform2_combo_box_set_value (gint j, gboolean force, GGobiData *d, ggobid *gg)
{
  vartabled *vt = vartable_element_get (j, d);
  if (vt != NULL)
    tf2_combo_box_set_value(vt, force, gg);
}
