/* gEDA - GPL Electronic Design Automation
 * gschem - gEDA Schematic Capture
 * Copyright (C) 1998-2000 Ales V. Hvezda
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 */


/* DO NOT read or edit this file ! Use ../noweb/o_misc.nw instead */

#include <config.h>
#include <stdio.h>
#include <string.h>

#include <libgeda/libgeda.h>

#include "../include/x_states.h"
#include "../include/prototype.h"

/* break with the tradition here and input a list */
/* TODO: probably should go back and do the same for o_copy o_move
   o_delete... */
void o_edit(TOPLEVEL * w_current, SELECTION * list)
{
  char *equal_ptr;
  OBJECT *o_current;
#ifdef HAS_LIBGTKEXTRA
  SELECTION *s_current;
  int object_count = 0;
#endif

  if (list == NULL) {
    w_current->event_state = SELECT;
    i_update_status(w_current, "Select Mode");
    w_current->inside_action = 0;
    return;
  }

  o_current = list->selected_object;
  if (o_current == NULL) {
    fprintf(stderr, "Got an unexpected NULL in o_edit\n");
    exit(-1);
  }
#ifdef HAS_LIBGTKEXTRA
  /* count up how many non-text objects exist in the selection */
  /* list.  Why?  Because if there are multiple objects, invoke the */
  /* multi_multi_edit dialog box */
  s_current = list;
  while (s_current != NULL) {
    if (s_current->selected_object) {
      if (s_current->selected_object->type != OBJ_TEXT) {
	object_count++;
      }
    }
    s_current = s_current->next;
  }

  /* now decide what we want to do, either single edit or */
  /* multi multi edit */
  if (object_count == 1 && o_current->type != OBJ_TEXT) {
    multi_attrib_edit(w_current, list);
    return;
  } else if (object_count > 1) {
    multi_multi_edit(w_current, list);
    return;
  }
#endif


  /* for now deal with only the first item */
  switch (o_current->type) {

    /* also add the ability to multi attrib edit: nets, busses, pins */
  case (OBJ_COMPLEX):
  case (OBJ_NET):
  case (OBJ_PIN):
  case (OBJ_BUS):
    multi_attrib_edit(w_current, list);
    break;

  case (OBJ_TEXT):
    if (strchr(o_current->text->string, '=')) {

      /* now really make sure it's an attribute by 
       * checking that there are NO spaces around the ='s 
       */
      equal_ptr = strchr(o_current->text->string, '=');

      /* there is a possiblity for core dump yes? */
      /* by exceeding the size of the text_string? */
      /* or maybe not, since if the ='s is at the end of */
      /* the string, there better be a null after it! */
      if ((*(equal_ptr + 1) != ' ') && (*(equal_ptr - 1) != ' ')) {
	attrib_edit_dialog(w_current, o_current, FROM_MENU);
      } else {
	o_text_edit(w_current, o_current);
      }
    } else {
      o_text_edit(w_current, o_current);
    }
    break;
  }

  /* has to be more extensive in the future */
  /* some sort of redrawing? */
}


/* This locks the entire selected list.  It does lock components, but does NOT
 * change the color (of primatives of the components) though 
 * this cannot be called recursively */
void o_lock(TOPLEVEL * w_current)
{
  OBJECT *object = NULL;
  SELECTION *s_current = NULL;

  /* skip over head */
  s_current = w_current->page_current->selection2_head->next;

  while (s_current != NULL) {
    object = s_current->selected_object;
    if (object) {
      /* check to see if locked_color is already being used */
      if (object->locked_color == -1) {
	object->sel_func = NULL;
	object->locked_color = object->color;
	object->color = w_current->lock_color;
	w_current->page_current->CHANGED = 1;
      } else {
	s_log_message("Object alreadly locked\n");
      }
    }

    s_current = s_current->next;
  }

  o_unselect_all(w_current);
  o_undo_savestate(w_current, UNDO_ALL);
}

/* You can unlock something by selecting it with a bounding box... */
/* this will probably change in the future, but for now it's a
   something.. :-) */
/* this cannot be called recursively */
void o_unlock(TOPLEVEL * w_current)
{
  OBJECT *object = NULL;
  SELECTION *s_current = NULL;

  s_current = w_current->page_current->selection2_head->next;

  while (s_current != NULL) {
    object = s_current->selected_object;
    if (object) {
      /* only unlock if sel_func is not set to something */
      if (object->sel_func == NULL) {
	object->sel_func = select_func;
	object->color = object->locked_color;
	object->locked_color = -1;
	w_current->page_current->CHANGED = 1;
      } else {
	s_log_message("Object alreadly unlocked\n");
      }
    }

    s_current = s_current->next;
  }
  o_undo_savestate(w_current, UNDO_ALL);
}


void
o_rotate_90(TOPLEVEL * w_current, SELECTION * list, int centerx,
	    int centery)
{
  OBJECT *object;
  SELECTION *s_current;
  int new_angle;
  GList *other_objects = NULL;
  GList *connected_objects = NULL;
  OBJECT *o_current;

  /* this is okay if you just hit rotate and have nothing selected */
  if (list == NULL) {
    w_current->event_state = SELECT;
    i_update_status(w_current, "Select Mode");
    w_current->inside_action = 0;
    return;
  }

  s_current = list;

  while (s_current != NULL) {
    object = s_current->selected_object;

    if (!object) {
      fprintf(stderr, "ERROR: NULL object in o_rotate_90!\n");
      return;
    }

    g_list_free(other_objects);
    other_objects = NULL;
    g_list_free(connected_objects);
    connected_objects = NULL;

    switch (object->type) {


    case (OBJ_NET):
      o_cue_undraw(w_current, object);
      o_net_erase(w_current, object);
      o_line_erase_grips(w_current, object);

      /* save the other objects */
      other_objects = s_conn_return_others(other_objects, object);
      s_conn_remove(w_current, object);

      o_net_rotate(w_current, centerx, centery, 90, object);
      s_conn_update_object(w_current, object);
      o_net_draw(w_current, object);

      /* draw the other objects */
      o_cue_undraw_list(w_current, other_objects);
      o_cue_draw_list(w_current, other_objects);

      /* get other connected objects and redraw */
      connected_objects = s_conn_return_others(connected_objects, object);
      o_cue_undraw_list(w_current, connected_objects);
      o_cue_draw_list(w_current, connected_objects);

      /* finally redraw the cues on the current object */
      o_cue_draw_single(w_current, object);
      break;

    case (OBJ_BUS):
      o_cue_undraw(w_current, object);
      o_bus_erase(w_current, object);
      o_line_erase_grips(w_current, object);

      other_objects = s_conn_return_others(other_objects, object);
      s_conn_remove(w_current, object);

      o_bus_rotate(w_current, centerx, centery, 90, object);
      s_conn_update_object(w_current, object);
      o_bus_draw(w_current, object);

      /* draw the other objects */
      o_cue_undraw_list(w_current, other_objects);
      o_cue_draw_list(w_current, other_objects);

      /* get other connected objects and redraw */
      connected_objects = s_conn_return_others(connected_objects, object);
      o_cue_undraw_list(w_current, connected_objects);
      o_cue_draw_list(w_current, connected_objects);

      /* finally redraw the cues on the current object */
      o_cue_draw_single(w_current, object);
      break;

    case (OBJ_PIN):
      o_cue_undraw(w_current, object);
      o_pin_erase(w_current, object);
      o_line_erase_grips(w_current, object);

      other_objects = s_conn_return_others(other_objects, object);
      s_conn_remove(w_current, object);

      o_pin_rotate(w_current, centerx, centery, 90, object);
      s_conn_update_object(w_current, object);
      o_pin_draw(w_current, object);

      /* draw the other objects */
      o_cue_undraw_list(w_current, other_objects);
      o_cue_draw_list(w_current, other_objects);

      /* get other connected objects and redraw */
      connected_objects = s_conn_return_others(connected_objects, object);
      o_cue_undraw_list(w_current, connected_objects);
      o_cue_draw_list(w_current, connected_objects);

      /* finally redraw the cues on the current object */
      o_cue_draw_single(w_current, object);
      break;

    case (OBJ_COMPLEX):
      o_cue_undraw_objects(w_current, object->complex->prim_objs);
      /* erase the current selection */
      o_complex_erase(w_current, object);

      other_objects = s_conn_return_complex_others(other_objects, object);

      /* remove all conn references */
      o_current = object->complex->prim_objs;
      while (o_current != NULL) {
	s_conn_remove(w_current, o_current);
	o_current = o_current->next;
      }

      /* do the rotate */
      /*w_current->ADDING_SEL=1; NEWSEL: needed? */
      new_angle = (object->complex->angle + 90) % 360;
      o_complex_rotate(w_current, centerx, centery, new_angle, 90, object);
      /*w_current->ADDING_SEL = 0; NEWSEL: needed? */
      s_conn_update_complex(w_current, object->complex->prim_objs);
      o_complex_draw(w_current, object);

      o_cue_undraw_list(w_current, other_objects);
      o_cue_draw_list(w_current, other_objects);

      /* now draw the newly connected objects */
      connected_objects = s_conn_return_complex_others(connected_objects,
						       object);
      o_cue_undraw_list(w_current, connected_objects);
      o_cue_draw_list(w_current, connected_objects);
      break;

    case (OBJ_LINE):
      o_line_erase_grips(w_current, object);
      o_line_erase(w_current, object);

      o_line_rotate(w_current, centerx, centery, 90, object);

      o_line_draw(w_current, object);
      break;

    case (OBJ_BOX):
      /* erase the current selection */
      o_box_erase_grips(w_current, object);
      o_box_erase(w_current, object);

      o_box_rotate(w_current, centerx, centery, 90, object);

      o_box_draw(w_current, object);
      break;

    case (OBJ_CIRCLE):
      o_circle_erase_grips(w_current, object);
      o_circle_erase(w_current, object);

      o_circle_rotate(w_current, centerx, centery, 90, object);

      o_circle_draw(w_current, object);
      break;

    case (OBJ_ARC):
      o_arc_erase(w_current, object);

#if 0				/* not needed anymore */
      SCREENtoWORLD(w_current, centerx, centery,
		    &world_centerx, &world_centery);
      o_arc_rotate_world(w_current, world_centerx, world_centery, 90,
			 object);
#endif

      o_arc_rotate(w_current, centerx, centery, 90, object);
      o_arc_draw(w_current, object);
      break;

    case (OBJ_TEXT):
      /* erase the current selection */
      o_text_erase(w_current, object);

      new_angle = (object->text->angle + 90) % 360;
      o_text_rotate(w_current, centerx, centery, new_angle, 90, object);

      o_text_draw(w_current, object);
      break;
    }
    s_current = s_current->next;
  }

  w_current->page_current->CHANGED = 1;
  o_undo_savestate(w_current, UNDO_ALL);
}


void o_embed(TOPLEVEL * w_current)
{
  OBJECT *object = NULL;
  char *new_basename;
  SELECTION *s_current = NULL;

  s_current = w_current->page_current->selection2_head->next;

  while (s_current != NULL) {
    object = s_current->selected_object;

    if (!object) {
      fprintf(stderr, "ERROR: NULL object in o_embed!\n");
      return;
    }

    if (object->type == OBJ_COMPLEX) {
      if (strncmp(object->complex_clib, "EMBEDDED", 8) != 0) {

	if (object->complex_clib) {
	  free(object->complex_clib);
	}

	object->complex_clib = u_basic_strdup("EMBEDDED");
	new_basename = u_basic_strdup_multiple("EMBEDDED",
					       object->complex_basename,
					       NULL);

	free(object->complex_basename);

	object->complex_basename = u_basic_strdup(new_basename);

	free(new_basename);

	w_current->page_current->CHANGED = 1;
      }
    }
    s_current = s_current->next;
  }
}


void o_unembed(TOPLEVEL * w_current)
{
  OBJECT *object = NULL;
  char *new_basename;
  char *new_clib;
  SELECTION *s_current = NULL;

  s_current = w_current->page_current->selection2_head->next;

  while (s_current != NULL) {
    object = s_current->selected_object;

    if (!object) {
      fprintf(stderr, "ERROR: NULL object in o_embed!\n");
      return;
    }

    if (object->type == OBJ_COMPLEX) {
      if (strncmp(object->complex_clib, "EMBEDDED", 8) == 0) {

	new_basename =
	    (char *) malloc(sizeof(char) *
			    (strlen(object->complex_basename) + 1));

	sprintf(new_basename, "%s", (object->complex_basename + 8));

	new_clib = (char *) s_clib_search(new_basename);

	if (!new_clib) {
	  fprintf(stderr,
		  "Could not find component [%s], while trying to unembed.\n",
		  object->complex_basename);
	  fprintf(stderr, "Component is still embedded\n");
	} else {
	  free(object->complex_basename);

	  object->complex_basename = new_basename;

	  free(object->complex_clib);

	  object->complex_clib = new_clib;

	  w_current->page_current->CHANGED = 1;
	}
      }
    }

    s_current = s_current->next;
  }
}


void
o_mirror(TOPLEVEL * w_current, SELECTION * list, int centerx, int centery)
{
  OBJECT *object;
  SELECTION *s_current;
  OBJECT *o_current = NULL;
  GList *other_objects = NULL;
  GList *connected_objects = NULL;

  if (list == NULL) {
    w_current->event_state = SELECT;
    i_update_status(w_current, "Select Mode");
    w_current->inside_action = 0;
    return;
  }

  s_current = list;

  while (s_current != NULL) {

    object = s_current->selected_object;

    if (!object) {
      fprintf(stderr, "ERROR: NULL object in o_mirror!\n");
      return;
    }

    g_list_free(other_objects);
    other_objects = NULL;
    g_list_free(connected_objects);
    connected_objects = NULL;

    switch (object->type) {


    case (OBJ_NET):
      o_cue_undraw(w_current, object);
      o_net_erase(w_current, object);
      o_line_erase_grips(w_current, object);

      other_objects = s_conn_return_others(other_objects, object);
      s_conn_remove(w_current, object);

      o_net_mirror(w_current, centerx, centery, object);
      s_conn_update_object(w_current, object);
      o_net_draw(w_current, object);

      /* draw the other objects */
      o_cue_undraw_list(w_current, other_objects);
      o_cue_draw_list(w_current, other_objects);

      /* get other connected objects and redraw */
      connected_objects = s_conn_return_others(connected_objects, object);
      o_cue_undraw_list(w_current, connected_objects);
      o_cue_draw_list(w_current, connected_objects);

      /* finally redraw the cues on the current object */
      o_cue_draw_single(w_current, object);
      break;

    case (OBJ_PIN):
      o_cue_undraw(w_current, object);
      o_pin_erase(w_current, object);
      o_line_erase_grips(w_current, object);

      other_objects = s_conn_return_others(other_objects, object);
      s_conn_remove(w_current, object);

      o_pin_mirror(w_current, centerx, centery, object);
      s_conn_update_object(w_current, object);
      o_pin_draw(w_current, object);

      /* draw the other objects */
      o_cue_undraw_list(w_current, other_objects);
      o_cue_draw_list(w_current, other_objects);

      /* get other connected objects and redraw */
      connected_objects = s_conn_return_others(connected_objects, object);
      o_cue_undraw_list(w_current, connected_objects);
      o_cue_draw_list(w_current, connected_objects);

      /* finally redraw the cues on the current object */
      o_cue_draw_single(w_current, object);
      break;

    case (OBJ_BUS):
      o_bus_erase(w_current, object);
      o_line_erase_grips(w_current, object);

      other_objects = s_conn_return_others(other_objects, object);
      s_conn_remove(w_current, object);

      o_bus_mirror(w_current, centerx, centery, object);
      s_conn_update_object(w_current, object);
      o_bus_draw(w_current, object);

      /* draw the other objects */
      o_cue_undraw_list(w_current, other_objects);
      o_cue_draw_list(w_current, other_objects);

      /* get other connected objects and redraw */
      connected_objects = s_conn_return_others(connected_objects, object);
      o_cue_undraw_list(w_current, connected_objects);
      o_cue_draw_list(w_current, connected_objects);

      /* finally redraw the cues on the current object */
      o_cue_draw_single(w_current, object);
      break;

    case (OBJ_COMPLEX):
      o_cue_undraw_objects(w_current, object->complex->prim_objs);
      /* erase the current selection */
      o_complex_erase(w_current, object);

      other_objects = s_conn_return_complex_others(other_objects, object);

      /* remove all conn references */
      o_current = object->complex->prim_objs;
      while (o_current != NULL) {
	s_conn_remove(w_current, o_current);
	o_current = o_current->next;
      }

      o_complex_mirror(w_current, centerx, centery, object);
      s_conn_update_complex(w_current, object->complex->prim_objs);
      o_complex_draw(w_current, object);

      o_cue_undraw_list(w_current, other_objects);
      o_cue_draw_list(w_current, other_objects);

      /* now draw the newly connected objects */
      connected_objects = s_conn_return_complex_others(connected_objects,
						       object);
      o_cue_undraw_list(w_current, connected_objects);
      o_cue_draw_list(w_current, connected_objects);
      break;

    case (OBJ_LINE):
      o_line_erase_grips(w_current, object);
      o_line_erase(w_current, object);
      o_line_mirror(w_current, centerx, centery, object);
      o_line_draw(w_current, object);
      break;

    case (OBJ_BOX):
      o_box_erase_grips(w_current, object);
      o_box_erase(w_current, object);
      o_box_mirror(w_current, centerx, centery, object);
      o_box_draw(w_current, object);
      break;

    case (OBJ_CIRCLE):
      o_circle_erase_grips(w_current, object);
      o_circle_erase(w_current, object);
      o_circle_mirror(w_current, centerx, centery, object);
      o_circle_draw(w_current, object);
      break;

    case (OBJ_ARC):
      o_arc_erase(w_current, object);
#if 0				/* not needed anymore */
      SCREENtoWORLD(w_current, centerx, centery,
		    &world_centerx, &world_centery);
#endif
      o_arc_mirror(w_current, centerx, centery, object);
      o_arc_draw(w_current, object);
      break;

    case (OBJ_TEXT):
      o_text_erase(w_current, object);
      o_text_mirror(w_current, centerx, centery, object);
      o_text_draw(w_current, object);
      break;

    }

    s_current = s_current->next;

  }

  w_current->page_current->CHANGED = 1;
  o_undo_savestate(w_current, UNDO_ALL);
}


void o_edit_show_hidden(TOPLEVEL * w_current, OBJECT * o_list)
{
  OBJECT *o_current = NULL;

  if (o_list == NULL)
    return;

  o_current = o_list;

  while (o_current != NULL) {

    if (o_current->type == OBJ_TEXT) {
      if (o_current->visibility == INVISIBLE) {
	o_current->visibility = VISIBLE;

	if (o_current->text->prim_objs == NULL) {
	  o_text_recreate(w_current, o_current);
	}

	o_text_draw(w_current, o_current);

	w_current->page_current->CHANGED = 1;
      }
    }
    o_current = o_current->next;
  }
  o_undo_savestate(w_current, UNDO_ALL);
}
