/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "file.h"
#include "graph.h"
#include "task.h"
#include "morph.h"
#include "model.h"
#include "module.h"
#include "select.h"
#include "matrix.h"
#include "space.h"
#include "spatial.h"
#include "opengl.h"
#include "surface.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"

#include "logo_left.xpm"
#include "logo_right.xpm"
#include "folder.xpm"
#include "disk.xpm"
#include "arrow.xpm"
#include "tools.xpm"
#include "palette.xpm"
#include "cross.xpm"
#include "geom.xpm"
#include "axes.xpm"
#include "cell.xpm"
#include "camera.xpm"

/* top level data structure */
extern struct sysenv_pak sysenv;

extern GMutex *gdk_threads_mutex;

/* bit messy - put some of these in sysenv? */
/* main window */
GtkWidget *window;
/* backing pixmap */
GdkPixmap *pixmap = NULL;
/* model tree (put in sysenv?) */
GtkWidget *tree;
/* model pane stuff */
GtkWidget *scrolled_window, *clist;
/* gdisrc? */
gint pane_width=65;
/* current viewing mode combo box */
GtkWidget *viewing_mode;

/************************/
/* EVENT - button press */
/************************/
#define DEBUG_BUTTON_PRESS_EVENT 0
gint button_press_event(GtkWidget *w, GdkEventButton *event)
{
gint refresh=0, x, y;
gint shift=FALSE, ctrl=FALSE;
gdouble r[3];
GdkModifierType state;
GSList *list;
struct model_pak *data;
struct bond_pak *bond;
struct core_pak *core;

/* get model */
data = sysenv.active_model;
if (!data)
  return(FALSE);

/* event info */
x = event->x;
y = event->y;
sysenv.moving = FALSE;

/* analyse the current state */
state = (GdkModifierType) event->state;
if ((state & GDK_SHIFT_MASK))
  shift = TRUE;
if ((state & GDK_CONTROL_MASK))
  ctrl = TRUE;

/* only want button 1 (for now) */
if (event->button != 1)
  return(FALSE);

/* NEW - diffraction peak search */
if (data->graph_active)
  {
  diffract_select_peak(x, y, data);
  return(FALSE);
  }

/* can we assoc. with a single atom */
/* TODO - maybe return a list? (could be more than one) */
pixel_coord_map(x, y, r, data);
core = seek_coord3d(r, data);

/* allow shift+click to add/remove single atoms in the selection */
/* TODO - depending on mode add/remove objects in selection eg atom/mols */
if (shift)
  {
  switch(data->mode)
    {
    default:
      if (core)
        {
        switch (sysenv.select_mode)
          {
          case ATOM_LABEL:
            select_all_labels(core, data);
            break;
          case CORE:
/* if atom is already in the selection - remove it */
            if (select_add_core(core, data))
              select_del_core(core, data);
            break;
          case ELEM:
            select_all_elem(core->atom_code, data);
            break;
          case MOL:
            select_add_mol(core->mol, data);
            break;
          case REGION:
            select_add_region(core, data);
            break;

          default:
            g_assert_not_reached();
          }
        refresh++;
        }
      else
        {
/* otherwise start a box selection */
        update_box(x,y,data,START);
        }
      break;
    }
  }
else
  {
/* determine the type of action required */
  switch(data->mode)
    {
    case BOND_INFO:
      info_bond(w,x,y,data);
      break;

    case BOND_DELETE:
/* delete via constituent atoms, */
      if (core)
        {
        make_bond(core, data->mode, data); 
        refresh++;
        }
      else
        {
/* or by bond midpoints */
        list = gl_seek_bond(w,x,y,data);
        if (list)
          {
          bond = (struct bond_pak *) list->data;
          user_bond(bond->atom1, bond->atom2, data->mode, data);
          refresh++;
          }
        }
      break;

    case BOND_SINGLE:
    case BOND_DOUBLE:
    case BOND_TRIPLE:
      if (core)
        {
        make_bond(core, data->mode, data); 
        refresh++;
        }
      break;

    case ATOM_ADD:
      add_atom(r, data);
      refresh++; 
      break;
    case ATOM_MOVE:
      move_atom(r, START);
      refresh++;
      break;
    case ATOM_DELETE:
      delete_atom_at(r, data);
      calc_bonds(data);
      calc_mols(data);
      refresh++;
      break;

    case DELETE_VECTOR:
      delete_vector_at(r, data);
      refresh++;
      break;

    case MOL_DELETE:
      if (core)
        {
        delete_mol(core->mol, data);
        refresh++;
        }
      break;

    case DEFINE_RIBBON:
      if (core)
        construct_backbone(core, data);
      break;

    case DEFINE_VECTOR:
    case DEFINE_PLANE:
      if (core)
        new_spatial_point(data->mode, core, data);
      break;

/* selection stuff */
    default:
      select_clear();
/* NEW - don't select if a core is under the mouse */
/* instead we allow the selection box to be drawn */
      update_box(x,y,data,START);
      refresh++;
      break; 
    }
  }
return(FALSE);
}

/***************************/
/* EVENT - button released */
/***************************/
gint button_release_event(GtkWidget *w, GdkEventButton *event)
{
gint x, y;
struct model_pak *data;
GdkModifierType state;

/* get model */
data = sysenv.active_model;
if (!data)
  return(FALSE);

/* get event info */
x = event->x;
y = event->y;
state = (GdkModifierType) event->state;

/* NEW - motion flag */
sysenv.moving = FALSE;
refresh_view();

/* first mouse button */
if (event->button == 1) 
  {
/* clean up after move operations */
  switch(data->mode)
    {
    case DIST_INFO:
      info_dist(x,y,data);
      break;
    case ANGLE_INFO:
      info_angle(x,y,data);
      break;
    case DIHEDRAL_INFO:
      info_torsion(x,y,data);
      break;
    case ATOM_MOVE:
      move_atom(NULL, STOP);
      break;
    default:
/* commit the box contents to the selection */
/* if shift pressed - append selection, otherwise create new */
      if (data->box_on)
        gl_select_box(w);
/* turn the box off */
      data->box_on = FALSE;
      break;
    }
  }

redraw_canvas(SINGLE);
return(FALSE);
}

/************************/
/* EVENT - mouse motion */
/************************/
#define DEBUG_MOTION 0
gint motion_notify_event(GtkWidget *w, GdkEventMotion *event)
{
gint x, y, dx, dy, fx, fy, ds, da, refresh;
gint shift=FALSE, ctrl=FALSE;
gdouble r[3], mat[9], vec[3], scale;
GSList *glist;
GdkModifierType state;
static gint ox=0, oy=0;
struct model_pak *data;

/* get model */
data = sysenv.active_model;
if (!data)
  return(FALSE);

/* get mouse data */
if (event->is_hint)
  gdk_window_get_pointer(event->window, &x, &y, &state);
else
  {
/* windows apparently reaches this */
  x = event->x;
  y = event->y;
  state = (GdkModifierType) event->state;
  }

/* discard discontinuous jumps (ie ox,oy are invalid on 1st call) */
if (!sysenv.moving)
  {
  ox = x;
  oy = y;
  sysenv.moving = TRUE;
  return(TRUE);
  }

/* convert relative mouse motion to an increment */
dx = x-ox;
dy = oy-y;        /* inverted y */

/* single update */
refresh=0;

/* analyse the current state */
if ((state & GDK_SHIFT_MASK))
  shift = TRUE;
if ((state & GDK_CONTROL_MASK))
  ctrl = TRUE;

pixel_coord_map(x, y, r, data);

/* first mouse button - mostly selection stuff */
if (state & GDK_BUTTON1_MASK)
  {
  switch(data->mode)
    {
    case ATOM_MOVE:
      move_atom(r, UPDATE);
      refresh++;
      break;
    case ATOM_ADD:
      break;
    case FREE:
      update_box(x, y, data, UPDATE);
      refresh++;
      break;
    default:
      break;
    }
/* don't switch to low quality drawing */
  sysenv.moving = FALSE;
  if (refresh)
    redraw_canvas(SINGLE);
  ox=x;
  oy=y;
  return(TRUE);
  }

/* second mouse button */
if (state & GDK_BUTTON2_MASK)
  {
/* shift - zoom */
  if (shift)
    {
/* zoom */
    data->scale += PIX2SCALE * (gdouble) (dx+dy);
    if (data->scale < 0.1)
      data->scale = 0.1;
    data->zoom = data->rmax/data->scale;
    gtksh_relation_update(data);

/* camera z translation */
/*
    data->offset[2] += 0.01 * data->rmax * (gdouble) dy;
*/

    init_objs(REDO_COORDS, data);
    refresh++;
    sysenv.moving = TRUE;
    }
/* no shift - translate */
  else
    {
/* selection only translation */
    if (ctrl)
      {
/* FIXME - how does the 500 fit in ??? */
      scale = data->rmax/500.0;
/* convert pixel translation to real space translation */
      VEC3SET(vec, scale*dx, -scale*dy, 0.0);

      vecmat(data->irotmat, vec);
      vecmat(data->ilatmat, vec);
/* apply to all */
      xlat_select(vec, data);

/* recalc coords */
      calc_coords(REFRESH, data);
      refresh++;
      sysenv.moving = TRUE;
      }
    else
      {
/* translate everything */
      data->offset[0] += x-ox;
      data->offset[1] += y-oy;
      glist = data->ghosts;
      while (glist != NULL)
        {
        data = model_ptr(GPOINTER_TO_INT(glist->data), RECALL);
        data->offset[0] += x-ox;
        data->offset[1] += y-oy;
        glist = g_slist_next(glist);
        }
      refresh++;
      sysenv.moving = TRUE;
      }
    }
  }

/* third mouse button clicked? */
if (state & GDK_BUTTON3_MASK)
  {
/* shift clicked? */
  if (shift)
    {
/* yaw */
    if (dx || dy)
      {
/* rotation amount */
      da = abs(dx) + abs(dy);
/* vector from center to mouse pointer (different for OpenGL window) */
/* FIXME - if ctrl is pressed, the center should be the selection */
/* centroid & not the middle of the window */
      fx = x - data->offset[0] - (sysenv.x + sysenv.width/2);
      fy = data->offset[1] + (sysenv.y + sysenv.height/2 - y);

/* rotation direction via z component of cross product (+=clock, -=anti) */
      if ((fx*dy - dx*fy) > 0)
        ds = -1;
      else
        ds = 1;

#if DEBUG_MOTION
printf("(%d,%d) x (%d,%d) : %d\n",dx,dy,fx,fy, ds);
#endif

/* assign and calculate */
      data->da = (gdouble) (ds*da);
      data->da *= ROTATE_SCALE * PI / 180.0;
/* NEW - selection */
      if (ctrl)
        {
        make_rotmat(mat, data->da, ROLL);
        rotate_select(data, mat);
        data->da = 0.0;
        }
/* update coords */
      calc_coords(ROLL, data);
      data->da = 0.0;
      }
    refresh++;
    sysenv.moving = TRUE;
    }
  else
    {
#if DEBUG_MOTION
printf("(%d,%d)\n",dx,dy);
#endif
/* pitch and roll */
    dy *= -1;
    if (dy)
      {
      data->da = (gdouble) ROTATE_SCALE * dy * PI / 180.0;
/* NEW - selection */
      if (ctrl)
        {
        make_rotmat(mat, data->da, PITCH);
        rotate_select(data, mat);
        data->da = 0.0;
        }
/* update coords */
      calc_coords(PITCH, data);
      data->da = 0.0;
      refresh++;
      sysenv.moving = TRUE;
      }
    if (dx)
      {
      data->da = (gdouble) ROTATE_SCALE * dx * PI / 180.0;
/* NEW - selection */
      if (ctrl)
        {
        make_rotmat(mat, data->da, YAW);
        rotate_select(data, mat);
        data->da = 0.0;
        }
/* update coords */
      calc_coords(YAW, data);
      data->da = 0.0;
      refresh++;
      sysenv.moving = TRUE;
      }
    }
  }

/* save old values */
ox=x;
oy=y;

/* redraw? */
if (refresh)
  redraw_canvas(SINGLE);

return(TRUE);
}

/***************************/
/* expose all hidden atoms */
/***************************/
void unhide_atoms(void)
{
GSList *list;
struct model_pak *data;
struct core_pak *core;

/* deletion for the active model only */
data = sysenv.active_model;

/* unhide */
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  core->status &= ~HIDDEN;
  }

/* update */
redraw_canvas(SINGLE);
}

/****************/
/* change mode */
/***************/
void switch_mode(gint new_mode)
{
GSList *list;
struct model_pak *data;
struct core_pak *core1;

/* get selected model */
data = sysenv.active_model;
if (!data)
  return;

/* special case for morphology */
if (data->id == MORPH)
  {
  switch(new_mode)
    {
    case FREE:
    case RECORD:
      break;

    default:
      show_text(WARNING, "Disallowed mode.\n");
      return;
    }
  }

/* clean up (if necessary) from previous mode */
switch (data->mode)
  {
  case ATOM_MOVE:
    move_atom(NULL, STOP);
    break;
  case DIST_INFO:
  case BOND_INFO:
    for (list=data->cores ; list ; list=g_slist_next(list))
      {
      core1 = (struct core_pak *) list->data;
      core1->status &= ~SELECT;
      }
    break;

  case RECORD:
    if (data->num_frames > 2)
      data->num_frames--;
    else
      {
/* if no extra frames - it's not an animation any more */
      data->num_frames = 1;
      data->animation = FALSE;
      }
    break;
  }

/* initialize for new mode */
switch (new_mode)
  {
  case BOND_INFO:
  case DIST_INFO:
  case ANGLE_INFO:
  case DIHEDRAL_INFO:
    select_clear();
    break;

  case RECORD:
/* NEW - disallow transformation record mode if file is */
/* a standard animation as this will confuse read_frame() */
    if (data->frame_list || data->gulp.trj_file)
      {
      show_text(ERROR, "Disallowed mode change.\n");
      return;
      }
    data->animation = TRUE;
    break;
  }

/* set the new mode */
data->mode = new_mode;
data->state = 0;
redraw_canvas(SINGLE);
}

/* Need this for the menu item factory widget */
void gtk_switch_mode(GtkWidget *w, gint new_mode)
{
switch_mode(new_mode);
}

/***************/
/* change view */
/***************/
void switch_view(gint new_mode)
{
}

/************/
/* GTK hook */
/************/
void gtk_switch_view(GtkWidget *w, gint new_mode)
{
switch_view(new_mode);
}

/**********************************/
/* update the model/canvas layout */
/**********************************/
#define DEBUG_CANVAS_SHUFFLE 0
void canvas_shuffle(void)
{
GSList *list, *mlist;
struct model_pak *model;
struct canvas_pak *canvas;

model = sysenv.active_model;

/* this becomes non NULL when we find the first */
/* available spot for the active model */
mlist = NULL;

for (list=sysenv.canvas_list ; list ; list=g_slist_next(list))
  {
  canvas = (struct canvas_pak *) list->data;
  if (model)
    {
    if (!canvas->model)
      {
/* first time exception - the currently active model */
      if (!mlist)
        {
        canvas->active = TRUE;
        mlist = g_slist_find(sysenv.mal, model);
        }
      }
/* at this stage we've found a candidate spot for the active model */
    if (mlist)
      {
      model = (struct model_pak *) mlist->data;
      canvas->model = model;

#if DEBUG_CANVAS_SHUFFLE 
printf("assigning model: %p to canvas: %p\n", model, canvas);
#endif

/* if no more models in the current sequence - fill out the */
/* rest of the canvas model slots with NULL */
      mlist = g_slist_next(mlist);
      if (!mlist)
        model = NULL;
      }
    }
  else
    canvas->model = NULL;
  }

/* if no NULL spots to put the active model - force it somewhere */
if (model)
  {
/* FIXME - single canvas hack */
  list = sysenv.canvas_list;
  canvas = (struct canvas_pak *) list->data;
  
  canvas->active = TRUE;
  canvas->model = model;

#if DEBUG_CANVAS_SHUFFLE 
printf("assigning model: %p to canvas: %p\n", model, canvas);
#endif
  }
}

/********************/
/* property widgets */
/********************/

GtkListStore *mpb_ls=NULL;
GtkWidget *mpb_tv=NULL;

/******************************************************/
/* update model property hash table for current model */
/******************************************************/
void property_widget_redraw(void)
{
GSList *list;
GtkTreeIter iter;
struct model_pak *model;

/* checks */
g_assert(GTK_IS_LIST_STORE(mpb_ls));
gtk_list_store_clear(mpb_ls);
model = sysenv.active_model;
if (!model)
  return;

/* re-populate */
for (list=model->property_list ; list ; list=g_slist_next(list))
  {
  if (property_rank(list->data))
    {
    gtk_list_store_append(mpb_ls, &iter);
    gtk_list_store_set(mpb_ls, &iter, 0, property_label(list->data), -1);
    gtk_list_store_set(mpb_ls, &iter, 1, property_value(list->data), -1);
    }
  }
}

/*************************************/
/* init model property display panel */
/*************************************/
void property_widget_init(GtkWidget *box)
{
GtkCellRenderer *renderer;
GtkTreeViewColumn *column;

/* new tree list */
mpb_ls = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
mpb_tv = gtk_tree_view_new_with_model(GTK_TREE_MODEL(mpb_ls));
/* TODO - scrolled window instead of box? */
gtk_box_pack_start(GTK_BOX(box), mpb_tv, TRUE, TRUE, 0);

/* setup cell renderers */
renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes(" ", renderer, "text", 0, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(mpb_tv), column);
renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes(" ", renderer, "text", 1, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(mpb_tv), column);
gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(mpb_tv), FALSE);

/* currently, allow nothing to be selected */
gtk_tree_selection_set_mode(gtk_tree_view_get_selection(GTK_TREE_VIEW(mpb_tv)),
                            GTK_SELECTION_NONE);
}

/**************************/
/* model symmetry globals */
/**************************/

GSList *msb_list=NULL;
GtkListStore *msb_ls=NULL;
GtkWidget *msb_tv=NULL;

gchar *msb_label[] = {"space group", "system", "atoms",
                      "a", "b", "c", "alpha", "beta", "gamma",
                      "surface area", "volume", NULL};

/**************************************/
/* symmetry list store free primitive */
/**************************************/
gboolean symmetry_free(GtkTreeModel *treemodel, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
gchar *a, *b;

gtk_tree_model_get(treemodel, iter, 0, &a, 1, &b, -1);
if (a)
  g_free(a);
if (b)
  g_free(b);
return(TRUE);
}

/******************************/
/* redraw the symmetry widget */
/******************************/
void symmetry_widget_redraw(void)
{
gint i;
gchar *value;
GtkTreeIter iter;
GtkTreeModel *tree_model;
struct model_pak *model;

/* checks */
g_assert(GTK_IS_LIST_STORE(msb_ls));

/* free allocated strings in list */
tree_model = gtk_tree_view_get_model(GTK_TREE_VIEW(msb_tv));
g_assert(tree_model != NULL);
gtk_tree_model_foreach(tree_model, (gpointer) symmetry_free, NULL);

/* clear */
gtk_list_store_clear(msb_ls);
free_slist(msb_list);
msb_list = NULL;

/* get model and use to populate */
model = sysenv.active_model;
if (!model)
  return;
if (!model->periodic)
  {
/* populate with symmetry info */
/* TODO - could list symops from pavel's symmetry finder??? */
  gtk_list_store_append(msb_ls, &iter);
  gtk_list_store_set(msb_ls, &iter, 0, "none", -1);
  gtk_list_store_set(msb_ls, &iter, 1, NULL, -1);
  return;
  }

/* populate with space group info only if 3D */
if (model->periodic == 3)
  i=0;
else
  i=3;

while (msb_label[i])
  {
  value = NULL;
  switch (i)
    {
    case 0:
      value = g_strdup_printf("%s", model->sginfo.spacename);
      break;
    case 1:
      value = g_strdup_printf("%s", model->sginfo.latticename);
      break;
    case 2:
      value = g_strdup_printf("%d (%d)", model->num_atoms, model->num_asym);
      break;
    case 3:
      value = g_strdup_printf("%8.4f", model->pbc[0]);
      break;
    case 4:
      if (model->periodic > 1)
        value = g_strdup_printf("%8.4f", model->pbc[1]);
      break;
    case 5:
      if (model->periodic > 2)
        value = g_strdup_printf("%8.4f", model->pbc[2]);
      break;
    case 6:
      if (model->periodic > 2)
        value = g_strdup_printf("%8.2f", R2D*model->pbc[3]);
      break;
    case 7:
      if (model->periodic > 2)
        value = g_strdup_printf("%8.2f", R2D*model->pbc[4]);
      break;
    case 8:
      if (model->periodic > 1)
        value = g_strdup_printf("%8.2f", R2D*model->pbc[5]);
      break;
    case 9:
      if (model->periodic == 2)
        value = g_strdup_printf("%.4f", model->area);
      break;
    case 10:
      if (model->periodic == 3)
        value = g_strdup_printf("%.2f", model->volume);
      break;
    } 
/* only add if a valid value */
  if (value)
    {
    gtk_list_store_append(msb_ls, &iter);
    gtk_list_store_set(msb_ls, &iter, 0, msb_label[i], -1);
    gtk_list_store_set(msb_ls, &iter, 1, value, -1);
    msb_list = g_slist_prepend(msb_list, value);
    }
  i++;
  }
}

/****************************/
/* init the symmetry widget */
/****************************/
void symmetry_widget_init(GtkWidget *box)
{
GtkCellRenderer *renderer;
GtkTreeViewColumn *column;

/* new tree list */
msb_ls = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
msb_tv = gtk_tree_view_new_with_model(GTK_TREE_MODEL(msb_ls));
/* TODO - scrolled window instead of box? */
gtk_box_pack_start(GTK_BOX(box), msb_tv, TRUE, TRUE, 0);

/* setup cell renderers */
renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes(" ", renderer, "text", 0, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(msb_tv), column);
renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes(" ", renderer, "text", 1, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(msb_tv), column);
gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(msb_tv), FALSE);

/* currently, allow nothing to be selected */
gtk_tree_selection_set_mode(gtk_tree_view_get_selection(GTK_TREE_VIEW(msb_tv)),
                            GTK_SELECTION_NONE);
}

/*************************/
/* module widget gloabls */
/*************************/
GtkTreeStore *module_ts=NULL;
GtkWidget *module_tv=NULL;

/******************************/
/* module invocation callback */
/******************************/
void cb_module_activate(GtkTreeView *tree_view,
                        GtkTreePath *tree_path,
                        GtkTreeViewColumn *tree_view_column,
                        gpointer data)
{
gchar *symbol;
gpointer module;
GtkTreeModel *tree_model;
GtkTreeIter iter;
struct model_pak *model;

model = sysenv.active_model;

tree_model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree_view));
if (gtk_tree_model_get_iter(tree_model, &iter, tree_path))
  {
  gtk_tree_model_get(tree_model, &iter, 0, &symbol, 1, &module, -1);

  if (module)
    module_invoke(symbol, model, module);
  else
    show_text(ERROR, "Please select function(s) listed below the module.\n");
  }
}

/***********************/
/* refresh module list */
/***********************/
void module_widget_redraw(void)
{
GSList *list1, *list2;
GtkTreeIter root, branch;

/* checks */
g_assert(GTK_IS_TREE_STORE(module_ts));
g_assert(GTK_IS_TREE_VIEW(module_tv));

/* populate with valid modules */
for (list1=sysenv.module_list ; list1 ; list1=g_slist_next(list1))
  {
  gtk_tree_store_append(module_ts, &root, NULL);
  gtk_tree_store_set(module_ts, &root, 0, module_label(list1->data), -1);
  gtk_tree_store_set(module_ts, &root, 1, NULL, -1);
/* populate with module symbols */
  for (list2=module_symbols(list1->data) ; list2 ; list2=g_slist_next(list2))
    {
    gtk_tree_store_append(module_ts, &branch, &root);
    gtk_tree_store_set(module_ts, &branch, 0, (gchar *) list2->data, -1);
    gtk_tree_store_set(module_ts, &branch, 1, (gpointer) list1->data, -1);
    }
  }
}

/***********************************************/
/* init the available module (plug-in) listing */
/***********************************************/
void module_widget_setup(GtkWidget *box)
{
GtkWidget *label;
GtkCellRenderer *renderer;
GtkTreeViewColumn *column;

label = gtk_label_new("Loaded modules");
gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

/* underlying data storage */
module_ts = gtk_tree_store_new(2, G_TYPE_STRING, G_TYPE_POINTER);

/* actual tree widget */
module_tv = gtk_tree_view_new_with_model(GTK_TREE_MODEL(module_ts));
gtk_box_pack_start(GTK_BOX(box), module_tv, TRUE, TRUE, 0);

/* setup the text rendering colum */
renderer = gtk_cell_renderer_text_new();
column = gtk_tree_view_column_new_with_attributes("a", renderer, "text", 0, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(module_tv), column);
gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(module_tv), FALSE);

/* fill in with valid modules */
module_widget_redraw();

/* setup the selection handler */
g_signal_connect(G_OBJECT(module_tv), "row-activated",
                 G_CALLBACK(cb_module_activate),
                 NULL);
}

/************************************/
/* pick a (loaded) model to display */
/************************************/
#define DEBUG_PICK_MODEL 0
void pick_model(struct model_pak *model)
{
/* checks */
if (!model)
  return;

/* make the model active */
sysenv.active_model = model;

/* updates */
canvas_shuffle();
refresh_view();
update_creator();
gtksh_relation_update(model);

/* CURRENT */
property_widget_redraw();
symmetry_widget_redraw();
space_image_widget_redraw();

/* update atom info box */
if (g_slist_length(model->selection) == 1)
  atom_properties_update((model->selection)->data, model);

redraw_canvas(ALL);
}

/****************************/
/* event hook for the above */
/****************************/
void event_pick_model(GtkWidget *w, struct model_pak *data)
{
g_return_if_fail(data != NULL);
pick_model(data);
}

/*****************************************/
/* force the selection of main tree item */
/*****************************************/
/*
void tree_select(gint model)
{
if (GTK_IS_TREE(tree))
  gtk_tree_select_item(GTK_TREE(tree), model);
}
*/

/*****************************/
/* view panel button handler */
/*****************************/
void reset_view(void)
{
struct model_pak *data;

/* do we have a loaded model? */
data = sysenv.active_model;
if (data == NULL)
  return;

/* make the changes */
VEC3SET(data->angle, 0.0, 0.0, 0.0);
data->da = 0.0;
data->scale = 1.0;
if (sysenv.rab_on)
  {
  VEC3SET(sysenv.angle, 0.0, 0.0, 0.0);
  }


/* update */
calc_coords(INITIAL, data);
cent_coords(1, data);
calc_coords(REFRESH, data);
 /*
init_objs(INIT_COORDS, data);
*/
/* this'll trigger any dialog updates */
pick_model(data);
}

/*******************************/
/* refresh spin buttons values */
/*******************************/
void refresh_view()
{
struct model_pak *model;

model = sysenv.active_model;
if (!model)
  return;

ARR3SET(sysenv.angle, model->angle);
gtksh_relation_update(NULL);
}

/******************************/
/* The omnipresent view panel */
/******************************/
void change_view(GtkWidget *w, gpointer dummy)
{
gint i;
gdouble da[3];
struct model_pak *model;

model = sysenv.active_model;
if (!model)
  {
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), 0.0);
  return;
  }

/* get angle difference */
ARR3SET(da, sysenv.angle);
ARR3SUB(da, model->angle);

/* apply the appropriate rotation matrix */
for (i=3 ; i-- ; )
  {
  if (da[i])
    {
    model->da = D2R*da[i];
    switch (i)
      {
      case 0:
        calc_coords(YAW, model);
        break;
      case 1:
        calc_coords(PITCH, model);
        break;
      case 2:
        calc_coords(ROLL, model);
        break;
      }
    model->da = 0.0;
    model->angle[i] = sysenv.angle[i];
    }
  }
redraw_canvas(SINGLE);
}

/***************************/
/* create/update text tray */
/***************************/
void show_text(gint type, gchar *message)
{
GtkWidget *view;
static GtkTextBuffer *buffer=NULL;
static GtkTextIter iter;

/* checks */
if (!message)
  return;

/* try this for improved stability of tasks */
/*
if (sysenv.running_tasks)
  return;
*/

if (sysenv.canvas)
  {
  if (!buffer)
    {
/* first time init */
    view = gtk_text_view_new();
    gtk_text_view_set_editable(GTK_TEXT_VIEW(view), FALSE);
    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(view));
    gtk_container_add(GTK_CONTAINER(sysenv.tpane), view);
    gtk_widget_show(view);
/* tags */
    gtk_text_buffer_create_tag(buffer, "fg_blue", "foreground", "blue", NULL);  
    gtk_text_buffer_create_tag(buffer, "fg_red", "foreground", "red", NULL);
    gtk_text_buffer_create_tag(buffer, "italic", "style", PANGO_STYLE_ITALIC, NULL);
/* position iterator */
    gtk_text_buffer_get_iter_at_line(buffer, &iter, 0);
    }

/* assign colour via message type */
  switch(type)
    {
    case ERROR:
      gtk_text_buffer_insert_with_tags_by_name
        (buffer, &iter, message, -1, "fg_red", NULL); 
      break;

    case WARNING:
      gtk_text_buffer_insert_with_tags_by_name
       (buffer, &iter, message, -1, "fg_blue", NULL); 
      break;

    case ITALIC:
      gtk_text_buffer_insert_with_tags_by_name
        (buffer, &iter, message, -1, "italic", NULL); 
      break;

    default:
      gtk_text_buffer_insert(buffer, &iter, message, -1);
    }
/* TODO - delete items from buffer when line gets too big */
  }
else
  printf("[%s]\n", message);
}

/***********************************/
/* general panel visibility toggle */
/***********************************/
void pane_toggle(GtkWidget *w, GtkWidget *frame)
{
sysenv.write_gdisrc = TRUE;

if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
  gtk_widget_show(frame);
else
  gtk_widget_hide(frame);
}

/********************************/
/* rotation adjustment spinners */
/********************************/
void rot_adjust_box(GtkWidget *box)
{
GtkWidget *frame, *hbox, *vbox;

/* rotation frame */
frame = gtk_frame_new(NULL);
gtk_container_add(GTK_CONTAINER(box), frame);
hbox = gtk_hbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);

vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox),vbox,TRUE,TRUE,0);

gtksh_direct_spin(NULL, &sysenv.angle[0],
                  0.0, 360.0, 5.0,
                  change_view, GINT_TO_POINTER(0), vbox);

vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox),vbox,TRUE,TRUE,0);

gtksh_direct_spin(NULL, &sysenv.angle[1],
                  0.0, 360.0, 5.0,
                  change_view, GINT_TO_POINTER(1), vbox);

vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox),vbox,TRUE,TRUE,0);

gtksh_direct_spin(NULL, &sysenv.angle[2],
                  0.0, 360.0, 5.0,
                  change_view, GINT_TO_POINTER(2), vbox);
}

/*******************************/
/* write gdisrc before exiting */
/*******************************/
void gdis_exit(void)
{
gint button;
GtkWidget *dialog;

/* FIXME - the window destroy signal has already closed the parent */
/* popup if any background jobs are running */
if (g_thread_pool_get_num_threads(sysenv.thread_pool))
  {
  dialog = gtk_message_dialog_new(GTK_WINDOW(window),
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_WARNING,
                                  GTK_BUTTONS_YES_NO,
                                  "There are running tasks. Really exit?");
  button = gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
  if (button != -8)
    return;
  }

/* if user has changed element colours, but doesn't want them saved */
/* this will do it anyway - but they can just click reset next time */
if (sysenv.write_gdisrc)
  {
  sysenv.x = sysenv.display_box->allocation.x;
  sysenv.y = sysenv.display_box->allocation.y;
  sysenv.width = sysenv.display_box->allocation.width;
  sysenv.height = sysenv.display_box->allocation.height;

  write_gdisrc();
  }

/* cleanup */
sys_free();
/*
unlink("gdis.ps");
*/

task_queue_free();

if (sysenv.canvas)
  {
  gtk_widget_pop_visual();
  gtk_widget_pop_colormap();
  gtk_exit(0);
  }

exit(0);
}

/**************************/
/* multi canvas splitting */
/**************************/
void canvas_split(GtkWidget *w, gpointer *type)
{
struct canvas_pak *canvas;

/*
printf("split: %d\n", GPOINTER_TO_INT(type));
*/

switch(GPOINTER_TO_INT(type))
  {
  case CANVAS_INIT:
    canvas = gl_new_canvas(sysenv.width, sysenv.height);
    gtk_box_pack_start(GTK_BOX(sysenv.display_box), canvas->glarea, TRUE, TRUE, 0);
    gtk_widget_show(canvas->glarea);
  case CANVAS_SINGLE:
    sysenv.canvas_rows=sysenv.canvas_cols=1;
    break;

  case CANVAS_VSPLIT:
/*
    sysenv.canvas_cols++;
*/
    break;
  case CANVAS_HSPLIT:
/*
    sysenv.canvas_rows++;
*/
    break;
  case CANVAS_HVSPLIT:
/*
    sysenv.canvas_rows++;
    sysenv.canvas_cols++;
*/
    break;
  }
}

/* left pane widgets */

GtkWidget *frame_left_pane[7];

/*******************************/
/* left pane toggles callbacks */
/*******************************/
void mtb_widget_toggle(void)
{
sysenv.mtb_on ^= 1;

if (sysenv.mtb_on)
  gtk_widget_show(frame_left_pane[0]);
else
  gtk_widget_hide(frame_left_pane[0]);
}
void mpb_widget_update(void)
{
if (sysenv.mpb_on)
  gtk_widget_show(frame_left_pane[1]);
else
  gtk_widget_hide(frame_left_pane[1]);
}
void msb_widget_update(void)
{
if (sysenv.msb_on)
  gtk_widget_show(frame_left_pane[2]);
else
  gtk_widget_hide(frame_left_pane[2]);
}
void apb_widget_update(void)
{
if (sysenv.apb_on)
  gtk_widget_show(frame_left_pane[3]);
else
  gtk_widget_hide(frame_left_pane[3]);
}

void pib_widget_update(void)
{
if (sysenv.pib_on)
  gtk_widget_show(frame_left_pane[6]);
else
  gtk_widget_hide(frame_left_pane[6]);
}

void rab_widget_toggle(void)
{
sysenv.rab_on ^= 1;

if (sysenv.rab_on)
  gtk_widget_show(frame_left_pane[4]);
else
  gtk_widget_hide(frame_left_pane[4]);
}
void lmb_widget_toggle(void)
{
sysenv.lmb_on ^= 1;

if (sysenv.lmb_on)
  gtk_widget_show(frame_left_pane[5]);
else
  gtk_widget_hide(frame_left_pane[5]);
}

/***************************************/
/* display available modules & symbols */
/***************************************/
void mod_widget(void)
{
GSList *list, *list2;

for (list=sysenv.module_list ; list ; list=g_slist_next(list))
  {
  printf("module: %s\n", module_label(list->data));
  for (list2=module_symbols(list->data) ; list2 ; list2=g_slist_next(list2))
    {
    printf("  - %s\n", (gchar *) list2->data);
    }
  }
}

/******************/
/* MENU structure */
/******************/
static GtkItemFactoryEntry menu_items[] = 
{
  { "/_File",                 NULL, NULL, 0, "<Branch>" },
  { "/File/_New",             NULL, create_new_model, 1, NULL },
  { "/File/sep1",             NULL, NULL, 0, "<Separator>" },
  { "/File/_Open...",         NULL, file_load_dialog, 1, NULL },
  { "/File/_Close",           NULL, tree_select_delete, 1, NULL },
  { "/File/sep1",             NULL, NULL, 0, "<Separator>" },
  { "/File/_Save",            NULL, file_save, 1, NULL },
  { "/File/Save _as...",      NULL, file_save_dialog, 1, NULL },
  { "/File/sep1",             NULL, NULL, 0, "<Separator>" },

  { "/File/Export",  NULL, NULL, 0, "<Branch>" },
  { "/File/Export/Canvas snapshot...",  NULL, image_export_dialog, 1, NULL },
  { "/File/Export/Graph data...",  NULL, analysis_export_dialog, 1, NULL },

  { "/File/sep1",             NULL, NULL, 0, "<Separator>" },
  { "/File/_Quit",            NULL, gdis_exit, 1, NULL },

  { "/_Edit",               NULL, NULL, 0, "<Branch>" },
  { "/Edit/_Copy",          NULL, select_copy, 0, NULL },
  { "/Edit/_Paste",         NULL, select_paste, 0, NULL },
  { "/Edit/sep1",           NULL, NULL, 0, "<Separator>" },
  { "/Edit/Colour...",      NULL, select_colour, 0, NULL },
  { "/Edit/Delete",         NULL, select_delete, 0, NULL },
  { "/Edit/Hide",           NULL, select_hide, 0, NULL },
  { "/Edit/Unhide all",     NULL, unhide_atoms, 0, NULL },
  { "/Edit/sep1",           NULL, NULL, 0, "<Separator>" },
  { "/Edit/Select all",     NULL, select_all, 0, NULL },
  { "/Edit/Invert",         NULL, select_invert, 0, NULL },

  { "/_View",                       NULL, NULL, 0, "<Branch>" },
/*
  { "/View/Loaded models",          NULL, mtb_widget_toggle, 0, "<CheckItem>" },
  { "/View/Properties",             NULL, mpb_widget_toggle, 0, "<CheckItem>" },
  { "/View/Symmetry",               NULL, msb_widget_toggle, 0, "<CheckItem>" },
*/
  { "/View/Display properties...", NULL, render_dialog, 0, NULL },
  { "/View/Periodic table...",      NULL, gperiodic_win, 0, NULL },
  { "/View/sep1",                   NULL, NULL, 0, "<Separator>" },
  { "/View/Animation...",           NULL, animate_setup_widget, 0, NULL },
  { "/View/Measurements...",        NULL, geom_info, 0, NULL },
  { "/View/sep1",                   NULL, NULL, 0, "<Separator>" },
  { "/View/Orientation",            NULL, rab_widget_toggle, 0, "<CheckItem>" },
  { "/View/Modules",                NULL, lmb_widget_toggle, 0, "<CheckItem>" },

  { "/_Tools",                      NULL, NULL, 0, "<Branch>" },
  { "/Tools/Model editing...",      NULL, med_interface, 0, NULL },
  { "/Tools/sep1",                  NULL, NULL, 0, "<Separator>" },
  { "/Tools/Diffraction...",        NULL, diffract_dialog, 0, NULL },
  { "/Tools/Energetics...",         NULL, energetics_dialog, 0, NULL },
  { "/Tools/Symmetry...",           NULL, symmetry_dialog, 0, NULL },
  { "/Tools/Surfaces...",           NULL, surface_dialog, 0, NULL },
  { "/Tools/sep1",                  NULL, NULL, 0, "<Separator>" },
  { "/Tools/MD initializer...",     NULL, mdi_setup_widget, 0, NULL },
  { "/Tools/MD analysis...",        NULL, md_dialog, 0, NULL },
  { "/Tools/sep1",                  NULL, NULL, 0, "<Separator>" },
/*
  { "/Tools/Hirshfeld surfaces...", NULL, hfs_dialog, 0, NULL },
*/
  { "/Tools/Molecular surfaces...", NULL, surf_dialog, 0, NULL },

/*
  { "/Info",                  NULL, NULL, 0, "<LastBranch>" },
*/
  { "/_Info",                  NULL, NULL, 0, "<Branch>" },
  { "/Info/About...",         NULL, about, 0, NULL },
  { "/Info/sep1",             NULL, NULL, 0, "<Separator>" },
  { "/Info/_Task manager...",  NULL, task_dialog, 0, NULL }
};

/********************************/
/* process key presses directly */
/********************************/
gint cb_key_press(GtkWidget *w, GdkEventKey *event, gpointer dummy)
{
switch(event->keyval)
  {
  case GDK_Delete:
    select_delete();
    break;

/* fullscreen */
  case GDK_F1:
    if (!sysenv.stereo)
      {
      sysenv.stereo = TRUE;
      sysenv.stereo_fullscreen = TRUE;
      sysenv.render.perspective = TRUE;
      stereo_open_window();
      }
    break;

/* windowed */
  case GDK_F2:
    if (!sysenv.stereo && sysenv.stereo_windowed)
      {
      sysenv.stereo = TRUE;
      sysenv.stereo_fullscreen = FALSE;
      sysenv.render.perspective = TRUE;
      redraw_canvas(SINGLE);
      }
    break;

/* NEW - switching between windowed/fullscreen stereo while in */
/* stereo mode can screw things up - so just use esc to exit */
/* for both. ie F1/F2 do not act as stereo toggles */
  case GDK_Escape:
    sysenv.stereo = FALSE;
    sysenv.render.perspective = FALSE;
    stereo_close_window();
    redraw_canvas(SINGLE);
    break;
  }
return(FALSE);
}

/*******************************************************/
/* record current model/property pane divider position */
/*******************************************************/
gboolean cb_pane_refresh(GtkPaned *w, gpointer data)
{
sysenv.tree_divider = gtk_paned_get_position(w);
return(FALSE);
}

/*********/
/* SETUP */
/*********/
void connect_events()
{
gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);
gchar *text;
GList *list;
GtkWidget *gdis_wid;
GtkWidget *hpaned, *vpaned;
GtkWidget *vbox, *hbox, *vbox_lower, *menu_bar, *toolbar, *combo;
GtkWidget *frame, *event_box;
GdkBitmap *mask;
GdkPixmap *gdis_pix=NULL;
GtkStyle *style, *logo_style;
GtkItemFactory *item;

/* enforce true colour (fixes SG problems) */
sysenv.visual = gdk_visual_get_best_with_type(GDK_VISUAL_TRUE_COLOR);
if (!sysenv.visual)
  {
  printf("Error: could not get requested visual.\n");
  exit(1);
  }
sysenv.depth = sysenv.visual->depth;
sysenv.colourmap = gdk_colormap_new(sysenv.visual, TRUE);
if (!sysenv.colourmap)
  {
  printf("Error: could not allocate colourmap.\n");
  exit(1);
  }
gtk_widget_push_colormap(sysenv.colourmap);
gtk_widget_push_visual(sysenv.visual);


/* main window */
window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);
gtk_window_set_title(GTK_WINDOW(window),"GTK Display Interface for Structures");
gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);

g_signal_connect(GTK_OBJECT(window), "key_press_event",
                (GtkSignalFunc) cb_key_press, NULL);


/* vbox for the menubar */
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(window), vbox);

/* item factory menu creation */
item = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", NULL);
gtk_item_factory_create_items(item, nmenu_items, menu_items, NULL);
menu_bar = gtk_item_factory_get_widget(item, "<main>");

/* FALSE,FALSE => don't expand to fill (eg on resize) */
gtk_box_pack_start(GTK_BOX(vbox), menu_bar, FALSE, FALSE, 0);

/* hbox for the toolbar */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
toolbar = gtk_toolbar_new();
gtk_box_pack_start(GTK_BOX(hbox), toolbar, FALSE, FALSE, 0);

/* extract some important style info */
gtk_widget_realize(window);
style = gtk_widget_get_style(window);

sysenv.gtk_fontsize = pango_font_description_get_size(style->font_desc) 
                    / PANGO_SCALE;

/*
printf("psize = %d\n", pango_font_description_get_size(style->font_desc));
printf("scale = %d\n", PANGO_SCALE);
printf(" size = %d\n", sysenv.gtk_fontsize);
*/

/* load button */
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, folder_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), NULL,
                        "Open new model", "Private",
                        gdis_wid, GTK_SIGNAL_FUNC(file_load_dialog), NULL);

/* save button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, disk_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), NULL,
                        "Save active model", "Private", gdis_wid,
                        GTK_SIGNAL_FUNC(file_save_dialog), NULL);

/* delete button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, cross_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,                   /* label */
                        "Close active model",   /* tooltip */
                        "Private",              /* private info */
                        gdis_wid,               /* icon widget */
                        GTK_SIGNAL_FUNC(tree_select_delete), /* signal */
                        NULL);                  /* signal data */

/* display properties */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, palette_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Display properties",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(render_dialog),
                        NULL);

/* model editing */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, tools_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Model editing",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(med_interface),
                        NULL);

/* geometry button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, geom_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Measurements",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(geom_info),
                        NULL);

/* gperiodic button */
/*
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, element_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Periodic table",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gperiodic_win),
                        NULL);
*/

/* reset model geometry */
/* TODO - different icon */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, axes_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Reset model geometry",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(reset_view),
                        NULL);

/* reset model geometry */
/* TODO - different icon */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, cell_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Reset model images",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(space_image_widget_reset),
                        NULL);

/* normal viewing button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, arrow_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Normal mode",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_mode),
                        GINT_TO_POINTER(FREE));

/* transformation record button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, camera_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Record mode",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_mode),
                        GINT_TO_POINTER(RECORD));


#if THIS_IS_FIXED
/* make prev model active */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, left_arrow1_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "display previous model",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_view),
                        (gpointer) PREV_MODEL);

gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

/* make next model active */
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, right_arrow1_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "display next model",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_view),
                        (gpointer) NEXT_MODEL);
#endif




#if CANVAS_SPLIT
/* canvas splitting button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, split_none_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Single canvas",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(canvas_split),
                        GINT_TO_POINTER(CANVAS_SINGLE));

/* canvas splitting button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, split_vert_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "split canvas",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(canvas_split),
                        GINT_TO_POINTER(CANVAS_VSPLIT));

/* canvas splitting button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, split_horz_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "split canvas",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(canvas_split),
                        GINT_TO_POINTER(CANVAS_HSPLIT));

/* canvas splitting button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, split_both_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "split canvas",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(canvas_split),
                        GINT_TO_POINTER(CANVAS_HVSPLIT));
#endif


/* TODO - can we get rid of the cursor??? */
/* combo box for the selection mode */
list = NULL;
list = g_list_append(list, "atoms");
list = g_list_append(list, "atom labels");
list = g_list_append(list, "elements");
list = g_list_append(list, "elements in molecule");
list = g_list_append(list, "molecules");
list = g_list_append(list, "molecular fragments");
list = g_list_append(list, "regions");

combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
gtk_box_pack_end(GTK_BOX(hbox), combo, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed", 
                 GTK_SIGNAL_FUNC(set_selection_mode), (gpointer *) combo);

/*
label = gtk_label_new("Select: ");
gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, PANEL_SPACING);
*/

/* MAIN LEFT/RIGHT HBOX PANE */
/* paned window */
hpaned = gtk_hpaned_new();
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox)), hpaned);

/* LEFT PANE */
sysenv.mpane = gtk_vbox_new(FALSE, 0);
gtk_paned_pack1(GTK_PANED(hpaned), sysenv.mpane, TRUE, TRUE);
gtk_widget_set_size_request(sysenv.mpane, sysenv.tree_width, -1);

/* split pane - model tree in upper, everything else in lower */
vpaned = gtk_vpaned_new();
gtk_box_pack_start(GTK_BOX(sysenv.mpane), vpaned, TRUE, TRUE, 0);

/* model tree box */
frame_left_pane[0] = gtk_vbox_new(FALSE, 0);
gtk_paned_pack1(GTK_PANED(vpaned), frame_left_pane[0], TRUE, TRUE);
tree_init(frame_left_pane[0]);

/* lower pane */
vbox_lower = gtk_vbox_new(FALSE, 0);
gtk_paned_pack2(GTK_PANED(vpaned), vbox_lower, TRUE, TRUE);

/* model properties box */
frame_left_pane[1] = gtk_vbox_new(FALSE, 0);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox_lower), hbox, FALSE, FALSE, 0);
sysenv.mpb_on = TRUE;
gtk_box_pack_start(GTK_BOX(vbox_lower), frame_left_pane[1], TRUE, TRUE, 0);
property_widget_init(frame_left_pane[1]);
vbox_lower = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(sysenv.mpane), vbox_lower, FALSE, FALSE, 0);

/* model symmetry box */
frame_left_pane[2] = gtk_vbox_new(FALSE, 0);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox_lower), hbox, FALSE, FALSE, 0);
gtksh_direct_check("Model cell", &sysenv.msb_on, msb_widget_update, frame_left_pane[2], hbox);
gtk_box_pack_start(GTK_BOX(vbox_lower), frame_left_pane[2], TRUE, TRUE, 0);
symmetry_widget_init(frame_left_pane[2]);

/* periodic images */
frame_left_pane[6] = gtk_vbox_new(FALSE, 0);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox_lower), hbox, FALSE, FALSE, 0);
gtksh_direct_check("Model periodic images", &sysenv.pib_on, pib_widget_update, frame_left_pane[6], hbox);
gtk_box_pack_start(GTK_BOX(vbox_lower), frame_left_pane[6], FALSE, FALSE, 0);
space_image_widget_setup(frame_left_pane[6]);

/* current atom properties */
frame_left_pane[3] = gtk_vbox_new(FALSE, 0);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox_lower), hbox, FALSE, FALSE, 0);
gtksh_direct_check("Atom properties", &sysenv.apb_on, apb_widget_update, frame_left_pane[3], hbox);
gtk_box_pack_start(GTK_BOX(vbox_lower), frame_left_pane[3], FALSE, FALSE, 0);
atom_properties_box(frame_left_pane[3]);

/* current orientation values */
frame_left_pane[4] = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox_lower), frame_left_pane[4], FALSE, FALSE, 0);
rot_adjust_box(frame_left_pane[4]);

/* loaded modules */
frame_left_pane[5] = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox_lower), frame_left_pane[5], FALSE, FALSE, 0);
module_widget_setup(frame_left_pane[5]);

/* GTK is a bit dumb when it sets the initial sizes of the two panes */
/* set the pane position and store when changed */
gtk_paned_set_position(GTK_PANED(vpaned), sysenv.tree_divider);
g_signal_connect(GTK_OBJECT(vpaned), "size-request",
                 G_CALLBACK(cb_pane_refresh), NULL);

/* GDIS logo */
frame = gtk_frame_new(NULL);
gtk_box_pack_end(GTK_BOX(sysenv.mpane), frame, FALSE, TRUE, 0);

/* event box (get an x window for setting black background) */
event_box = gtk_event_box_new();
gtk_container_add(GTK_CONTAINER(frame), event_box);

/* force black background & white foreground */
logo_style = gtk_style_copy(style);
logo_style->bg[GTK_STATE_NORMAL].red   = 0;
logo_style->bg[GTK_STATE_NORMAL].green = 0;
logo_style->bg[GTK_STATE_NORMAL].blue  = 0;
gtk_widget_set_style(GTK_WIDGET(event_box), logo_style);

hbox = gtk_hbox_new (FALSE, 10);
gtk_container_add (GTK_CONTAINER(event_box), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);


gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                          &style->bg[GTK_STATE_NORMAL], logo_left_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_box_pack_start(GTK_BOX(hbox), gdis_wid, FALSE, FALSE, 0);


gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                          &style->bg[GTK_STATE_NORMAL], logo_right_81_xpm);
gdis_wid = gtk_image_new_from_pixmap(gdis_pix, mask);
gtk_box_pack_end(GTK_BOX(hbox), gdis_wid, FALSE, FALSE, 0);


/* RIGHT PANE */
vbox = gtk_vbox_new(FALSE, 0);
gtk_paned_pack2(GTK_PANED(hpaned), vbox, TRUE, TRUE);

/* paned window */
vpaned = gtk_vpaned_new();
gtk_container_add(GTK_CONTAINER(vbox), vpaned);

sysenv.display_box = gtk_vbox_new(FALSE, 0);
gtk_paned_pack1(GTK_PANED(vpaned), sysenv.display_box, TRUE, TRUE);

/* create the main drawing area */
if (gl_init_visual())
  exit(1);

canvas_split(NULL, GINT_TO_POINTER(CANVAS_INIT));

/* text status pane */
sysenv.tpane = gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sysenv.tpane),
                               GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_paned_pack2(GTK_PANED(vpaned), sysenv.tpane, TRUE, TRUE);
gtk_widget_set_size_request(sysenv.tpane, -1, sysenv.tray_height);
gtk_widget_show(sysenv.tpane);

text = g_strdup_printf("This is free software, distributed under the terms of the GNU public license (GPL).\nFor more information visit http://www.gnu.org\n");
show_text(WARNING, text);
g_free(text);

text = g_strdup_printf("Welcome to GDIS version %4.2f.%d, Copyright (C) 2004 by Sean Fleming\n",VERSION,PATCH); 
show_text(STANDARD, text);
g_free(text);

/*
text = g_strdup_printf("Current development supported by the Western Australian Premiers Research Fellowship.\n");
show_text(ITALIC, text);
*/

/* window signals */
g_signal_connect(GTK_OBJECT(window), "destroy",
                 GTK_SIGNAL_FUNC(gdis_exit), NULL);

/* show all */
gtk_widget_show_all(window);

/* initial visibility */
if (!sysenv.mtb_on)
  gtk_widget_hide(frame_left_pane[0]);

mpb_widget_update();
msb_widget_update();
pib_widget_update();
apb_widget_update();

if (!sysenv.lmb_on)
  gtk_widget_hide(frame_left_pane[5]);
if (!sysenv.rab_on)
  gtk_widget_hide(frame_left_pane[4]);
}

