/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <GL/gl.h>

#include "gdis.h"
#include "coords.h"
#include "file.h"
#include "matrix.h"
#include "morph.h"
#include "opengl.h"
#include "render.h"
#include "select.h"
#include "spatial.h"
#include "interface.h"

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/* global render parameters */
gdouble acm[9];
gdouble p2a;

/***************************/
/* Setup POVray parameters */
/***************************/
/* TODO - put this stuff in file_povray.c */
void povray_hdr(FILE *fp, struct model_pak *data)
{
gdouble xvec, yvec, amb, cam[3], pos[3], colour[3];
GSList *list;
struct light_pak *light;

fprintf(fp,"#include \"colors.inc\" \n");
fprintf(fp,"#include \"finish.inc\" \n");
fprintf(fp,"#include \"glass.inc\" \n");
fprintf(fp,"#include \"metals.inc\" \n");
fprintf(fp,"#include \"textures.inc\" \n");

/* background colour (except for glass morphologies) */
fprintf(fp,"background { color rgb<%f,%f,%f0> }\n", sysenv.render.bg_colour[0],
                        sysenv.render.bg_colour[1], sysenv.render.bg_colour[2]);  

/* pixel to angstrom conversion, with yet another magic number... */
p2a = 0.565 * (gdouble) sysenv.render.width * data->scale / data->rmax;

/* setup camera position (z is just a value far enough away) */
VEC3SET(cam, data->offset[0]/p2a, data->offset[1]/p2a, 4.0*sysenv.rsize*data->scale);

/* width and height */
xvec = yvec = 2.0*sysenv.rsize/data->scale;

/* NEW - adjust to preserve aspect ratio at the desired image size */
if (sysenv.render.width > sysenv.render.height)
  xvec *= sysenv.render.width/sysenv.render.height;
if (sysenv.render.height > sysenv.render.width)
  yvec *= sysenv.render.height/sysenv.render.width;

/* orientation */
fprintf(fp,"camera { orthographic location <%f,%f,%f>\n", cam[0], cam[1], cam[2]);
fprintf(fp,"    right <%f,0,0> up <0,%f,0>\n", xvec, yvec);
fprintf(fp,"    look_at <%f,%f,0> }\n", data->offset[0]/p2a, data->offset[1]/p2a);

/* attempt to mimic OpenGL lights */
for (list=sysenv.render.light_list ; list ; list=g_slist_next(list))
  {
  light = (struct light_pak *) list->data;
  ARR3SET(pos, light->x);

/* FIXME - I don't fully understand all the coordinate transforms */
/* necessary to make these positions match up with the on screen axes */
  switch (light->type)
    {
    case POSITIONAL:
      vecmat(acm, pos);
/* NB: y & z switch mimics gl_acm */
      fprintf(fp,"light_source\n  {\n <%f,%f,%f>\n", pos[0], -pos[2], pos[1]);
      break;

    case DIRECTIONAL:
      vecmat(acm, pos);
/* move away far enough so the rays are ~ // */
      VEC3MUL(pos, -100.0*data->rmax);
/* NB: y & z switch mimics gl_acm */
      fprintf(fp,"light_source\n  {\n <%f,%f,%f>\n", pos[0], -pos[2], pos[1]);
      break;

    default:
      continue;
    }

  ARR3SET(colour, light->colour);
/* NEW - hack to simulate OpenGL style lights */
  VEC3MUL(colour, light->specular);

  if (sysenv.render.shadowless)
    fprintf(fp,"  color rgb<%f,%f,%f> shadowless }\n", colour[0], colour[1], colour[2]);
  else
    fprintf(fp,"  color rgb<%f,%f,%f> }\n", colour[0], colour[1], colour[2]);
  }

/* morph is too dark with just the above, sky_sphere is *nice* */
/* TODO - leave choice of colour eg white/grey/light blue to the user? */
if (data->id == MORPH)
  {
  if (!sysenv.render.wire_surface && !sysenv.render.shadowless)
    {
    fprintf(fp,"sky_sphere { pigment {gradient y  color_map "
               "{[0, 1 color Gray20 color White]} rotate x*45}}\n");
    }
  }

/* white is a bit too bright (even with 0 ambience) */
/*
if (data->id == MORPH && !sysenv.render.wire_surface)
  fprintf(fp,"sky_sphere { pigment {White}}\n");
*/

/* POVRay is a bit darker than OpenGL */
amb = 20.0*sysenv.render.ambience;

/* FIXME - adjustable gamma? */
fprintf(fp,"global_settings { ambient_light rgb<%f, %f, %f> assumed_gamma 2.2}\n",amb,amb,amb);
}

/****************************/
/* make a POVRAY input file */
/****************************/
/* TODO - eliminate redundancy between this and the opengl code */
/* TODO - put this stuff in file_povray.c */
#define DEBUG_MKPOV 0
gint write_povray(gchar *povfile, struct model_pak *data)
{
gint n, m, i, j;
gint r, g, b, flag;
gdouble rad, scale, len;
gdouble x1, y1, z1, x2, y2, z2;
gdouble xmin, ymin, zmin, xmax, ymax, zmax;
gdouble rf, gf, bf;
gdouble vec[3], vec1[3], vec2[3];
gdouble v1[4], v2[4], v3[3], n1[3], n2[3], n3[3];
gdouble ctrl[8][3], mat4[16], tmp4[16];
GSList *list, *ilist, *plist, *rlist, *list1, *list2, *pipe_list[4];
struct core_pak *core1;
struct plane_pak *plane;
struct object_pak *object;
struct ribbon_pak *ribbon;
struct spatial_pak *spatial;
struct pipe_pak *pipe;
struct image_pak *image;
struct vec_pak *p1, *p2, *p3;
struct vertex_pak *vx1, *vx2;
struct elem_pak elem;
FILE *fp;

/* checks */
g_assert(data != NULL);

/* open file & init */
fp = fopen(povfile, "w");
if (!fp)
  {
  printf("Error, render(): can't open %s!\n", povfile);
  return(1);
  }

/* setup axes conversion matrix */
VEC3SET(&acm[0],-1.0, 0.0, 0.0);
VEC3SET(&acm[3], 0.0,-1.0, 0.0);
VEC3SET(&acm[6], 0.0, 0.0,-1.0);

/* strcpy(povfile,models[model].filename); */
/* file_extension -> .pov */

/* setup */
povray_hdr(fp,data);
/* limits - for intelligent axes placement */
xmin = ymin = zmin = 99999999.9;
xmax = ymax = zmax = -99999999.9;
/* pixel to coord scaling factor */
scale =  data->scale * sysenv.subscale * 0.5 / data->rmax;

/* precalc matrix products */
ARR3SET(&mat4[0], &data->latmat[0]);
ARR3SET(&mat4[4], &data->latmat[3]);
ARR3SET(&mat4[8], &data->latmat[6]);
ARR3SET(v1, data->centroid);
vecmat(data->latmat, v1);
mat4[3] = -v1[0];
mat4[7] = -v1[1];
mat4[11] = -v1[2];
VEC4SET(&mat4[12], 0.0, 0.0, 0.0, 1.0);

ARR3SET(&tmp4[0], &data->rotmat[0]);
ARR3SET(&tmp4[4], &data->rotmat[3]);
ARR3SET(&tmp4[8], &data->rotmat[6]);
tmp4[3] = 0.0;
tmp4[7] = 0.0;
tmp4[11] = 0.0;
VEC4SET(&tmp4[12], 0.0, 0.0, 0.0, 1.0);
mat4mat(tmp4, mat4);

/* do atoms */ 
#if DEBUG_MKPOV
printf("Doing atoms...\n");
#endif
/* enumerate periodic images */
plist=NULL;
do
  {
  if (plist)
    {
/* image */
    image = (struct image_pak *) plist->data;
    ARR3SET(vec, image->rx);
    plist = g_slist_next(plist);
    }
  else
    {
/* original */
    VEC3SET(vec, 0.0, 0.0, 0.0);
    plist = data->images;
    }

/* only if same type */
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core1 = (struct core_pak *) list->data;
  if (core1->status & (DELETED | HIDDEN))
    continue;

  ARR3SET(vec1, core1->rx);
  ARR3ADD(vec1, vec);
  vecmat(acm, vec1);

/* get current colour */
    r = core1->colour[0];
    g = core1->colour[1];
    b = core1->colour[2];
/* convert to povray rgb format */
    rf = (gdouble) (r) / 65535.0;
    gf = (gdouble) (g) / 65535.0;
    bf = (gdouble) (b) / 65535.0;
    switch(core1->render_mode)
      {
/* rounded end are now draw in the bond section */
      case LIQUORICE:
      case STICK:
        if (core1->bonds)
          break;
      case BALL_STICK:
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",vec1[0],vec1[1],vec1[2],
                                             sysenv.render.ball_rad);
/* TODO - can we adjust this to get a wire frame sphere? */
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>",rf,gf,bf);  

        if (core1->ghost)
          fprintf(fp, " transmit 0.6}\n");
        else
          fprintf(fp, "}\n");

        fprintf(fp,"  finish{phong %f phong_size %d }}}\n",
                    sysenv.render.ahl_strength, (gint) sysenv.render.ahl_size); 
/*
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
*/

        break;

      case CPK:
/* FIXME - calling get_elem_data() all the time is inefficient */
        get_elem_data(core1->atom_code, &elem, data);
        rad = elem.vdw;
        rad *= sysenv.render.cpk_scale;
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",vec1[0],vec1[1],vec1[2],rad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>",rf,gf,bf);  

        if (core1->ghost)
          fprintf(fp, " transmit 0.6}\n");
        else
          fprintf(fp, "}\n");

        fprintf(fp,"  finish{phong %f phong_size %d }}}\n",
                    sysenv.render.ahl_strength, (gint) sysenv.render.ahl_size); 
/*
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
*/
        break;
      }
  }
  }
while (plist);

#if DEBUG_MKPOV
printf("Doing bonds...\n");
#endif

/* calculate all bonds */
render_make_pipes(pipe_list, data);

/* enumerate the supplied pipes (half bonds) */
for (i=0 ; i<4 ; i++)
  {
  for (list=pipe_list[i] ; list ; list=g_slist_next(list))
    {
    pipe = (struct pipe_pak *) list->data;

/* original + image iteration */
    ilist = NULL;
    do
      {
/* original */
      ARR3SET(v1, pipe->v1);
      ARR3SET(v2, pipe->v2);
      if (ilist)
        {
        image = (struct image_pak *) ilist->data;
/* image */
        ARR3ADD(v1, image->rx);
        ARR3ADD(v2, image->rx);
        ilist = g_slist_next(ilist);
        }
      else
        ilist = data->images;

      vecmat(acm, v1);
      vecmat(acm, v2);

/* bond type */
      switch(i)
        {
/* normal */
        case 0:
          fprintf(fp,"cylinder { <%f,%f,%f>,\n<%f,%f,%f>, %f\n",
                      v1[0],v1[1],v1[2],v2[0],v2[1],v2[2], sysenv.render.stick_rad);
          fprintf(fp,"open texture{pigment{color rgb<%f,%f,%f>}\n",
                      pipe->colour[0], pipe->colour[1], pipe->colour[2]);
          fprintf(fp,"  finish{phong %f phong_size %d }}}\n",
                      sysenv.render.ahl_strength, (gint) sysenv.render.ahl_size); 
          break;

/* ghosts */
        case 1:
          fprintf(fp,"cylinder { <%f,%f,%f>,\n<%f,%f,%f>, %f\n",
                      v1[0],v1[1],v1[2],v2[0],v2[1],v2[2], sysenv.render.stick_rad);
          fprintf(fp,"open texture{pigment{color rgb<%f,%f,%f> transmit 0.6}",
                      pipe->colour[0], pipe->colour[1], pipe->colour[2]);
          fprintf(fp,"  finish{phong %f phong_size %d }}}\n",
                      sysenv.render.ahl_strength, (gint) sysenv.render.ahl_size); 
          break;

/* line */
        case 3:
/* FIXME - need a better way to set "line" thicknesses */
          fprintf(fp,"cylinder { <%f,%f,%f>,\n<%f,%f,%f>, %f\n",
                      v1[0],v1[1],v1[2],v2[0],v2[1],v2[2], 2.0/p2a);
          fprintf(fp,"open texture{pigment{color rgb<%f,%f,%f>}\n",
                      pipe->colour[0], pipe->colour[1], pipe->colour[2]);
          fprintf(fp,"  finish{phong %f phong_size %d }}}\n",
                      sysenv.render.ahl_strength, (gint) sysenv.render.ahl_size); 
          break;

/* TODO - wire frame bonds??? */
        default:
          break;
        }
      }
    while (ilist);
    }
  }

/* free all pipes */
for (i=4 ; i-- ; )
  free_slist(pipe_list[i]);

#if DEBUG_MKPOV
printf("Doing special objects...\n");
#endif

/********/
/* AXES */
/********/
/* FIXME - can't cope with model translation */
if (data->show_axes)
  {
/* origin */
  VEC3SET(vec1, 1.0, 1.0, 0.0);
  VEC3MUL(vec1, data->rmax/data->scale);
/* cope with scaling */
  vec1[2] -= 4.0*sysenv.rsize*data->scale;
  for (j=0 ; j<3 ; j++)
    {
/* get end point */
    ARR3SET(vec2, data->axes[j].rx);
    VEC3MUL(vec2, 1.0/data->scale);
    vecmat(acm, vec2);
    ARR3ADD(vec2, vec1);
/* cope with scaling */
    vec2[2] -= 4.0*sysenv.rsize*data->scale;
/* draw */
    fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>, %f\n",
                vec1[0],vec1[1],vec1[2],vec2[0],vec2[1],vec2[2], 2.0/p2a);
    fprintf(fp," texture { pigment {White} } }\n");
    }
  }

/********/
/* CELL */
/********/
if (data->show_cell && data->periodic)
  {
  rad = sysenv.render.frame_thickness / p2a;

/* ends */
  for (j=0 ; j<8 ; j++)
    {
    m = 2*(j/2) + 1;
    n = 4*(j/4);

    ARR3SET(vec, data->cell[m].rx);
    vecmat(acm, vec);
    x1 = vec[0];
    y1 = vec[1];
    z1 = vec[2];

    ARR3SET(vec, data->cell[n].rx);
    vecmat(acm, vec);
    x2 = vec[0];
    y2 = vec[1];
    z2 = vec[2];
    fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>, %f\n",
                                 x1,y1,z1,x2,y2,z2,rad);
    fprintf(fp," texture { pigment {White} } }\n");

    ARR3SET(vec, data->cell[n+2].rx);
    vecmat(acm, vec);
    x2 = vec[0];
    y2 = vec[1];
    z2 = vec[2];
    fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>, %f\n",
                                 x1,y1,z1,x2,y2,z2,rad);
    fprintf(fp," texture { pigment {White} } }\n");
    }

/* sides */
/* skip for 2D periodic models */
  if (data->periodic == 3)
    {
    m = 0;
    n = 4;
    for (j=0 ; j<4 ; j++)
      {
      ARR3SET(vec, data->cell[m].rx);
      vecmat(acm, vec);
      x1 = vec[0];
      y1 = vec[1];
      z1 = vec[2];

      ARR3SET(vec, data->cell[n].rx);
      vecmat(acm, vec);
      x2 = vec[0];
      y2 = vec[1];
      z2 = vec[2];

      fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>, %f\n",
                                   x1,y1,z1,x2,y2,z2,rad);
      fprintf(fp," texture { pigment {White} } }\n");
      m++;
      n++;
      }
    }
  }

/***********/
/* ribbons */
/***********/
for (list=data->ribbons ; list ; list=g_slist_next(list))
  {
  object = (struct object_pak *) list->data;
  g_assert(object->type == RIBBON);

  rlist = (GSList *) object->data;
  while (rlist)
    {

    ribbon = (struct ribbon_pak *) rlist->data;

  fprintf(fp, "bicubic_patch {\n  type 1 flatness 0.001\n");
/* NB: POVRay is very slow at rendering ribbons, so halve the quality */
  fprintf(fp, "  u_steps %d  v_steps 2\n", (gint) (sysenv.render.ribbon_quality/2.0));

/* end points */
    ARR3SET(&ctrl[0][0], ribbon->r1);
    ARR3SET(&ctrl[3][0], ribbon->r2);

/* get distance between ribbon points */
    ARR3SET(vec1, ribbon->x1);
    ARR3SUB(vec1, ribbon->x2);
    len = VEC3MAG(vec1);

/* shape control points */
    ARR3SET(&ctrl[1][0], ribbon->r1);
    ARR3SET(&ctrl[2][0], ribbon->r2);

/* segment length based curvature - controls how flat it is at the cyclic group */
    ARR3SET(vec1, ribbon->o1);
    VEC3MUL(vec1, len*sysenv.render.ribbon_curvature);
    ARR3ADD(&ctrl[1][0], vec1);
    ARR3SET(vec2, ribbon->o2);
    VEC3MUL(vec2, len*sysenv.render.ribbon_curvature);
    ARR3ADD(&ctrl[2][0], vec2);

/* compute offsets for ribbon thickness */
    crossprod(vec1, ribbon->n1, ribbon->o1);
    crossprod(vec2, ribbon->n2, ribbon->o2);
    normalize(vec1, 3);
    normalize(vec2, 3);

/* thickness vectors for the two ribbon endpoints */
    VEC3MUL(vec1, 0.5*sysenv.render.ribbon_thickness);
    VEC3MUL(vec2, 0.5*sysenv.render.ribbon_thickness);

/* ensure these are pointing the same way */
    if (via(vec1, vec2, 3) > PI/2.0)
      {
      VEC3MUL(vec2, -1.0);
      }

/* init the bottom edge control points */
    ARR3SET(&ctrl[4][0], &ctrl[0][0]);
    ARR3SET(&ctrl[5][0], &ctrl[1][0]);
    ARR3SET(&ctrl[6][0], &ctrl[2][0]);
    ARR3SET(&ctrl[7][0], &ctrl[3][0]);
/* lift points to make the top edge */
    ARR3ADD(&ctrl[0][0], vec1);
    ARR3ADD(&ctrl[1][0], vec1);
    ARR3ADD(&ctrl[2][0], vec2);
    ARR3ADD(&ctrl[3][0], vec2);
/* lower points to make the bottom edge */
    ARR3SUB(&ctrl[4][0], vec1);
    ARR3SUB(&ctrl[5][0], vec1);
    ARR3SUB(&ctrl[6][0], vec2);
    ARR3SUB(&ctrl[7][0], vec2);

    for (i=0 ; i<8 ; i++)
      {
      vecmat(acm, &ctrl[i][0]);
/*
      VEC3MUL(&ctrl[i][0], data->scale);
*/
      }

    fprintf(fp, "<%f, %f, %f>\n", ctrl[0][0], ctrl[0][1], ctrl[0][2]);
    for (i=1 ; i<4 ; i++)
      fprintf(fp, ", <%f, %f, %f>\n", ctrl[i][0], ctrl[i][1], ctrl[i][2]);
    for (i=0 ; i<4 ; i++)
      fprintf(fp, ", <%f, %f, %f>\n", ctrl[i][0], ctrl[i][1], ctrl[i][2]);
    for (i=4 ; i<8 ; i++)
      fprintf(fp, ", <%f, %f, %f>\n", ctrl[i][0], ctrl[i][1], ctrl[i][2]);
    for (i=4 ; i<8 ; i++)
      fprintf(fp, ", <%f, %f, %f>\n", ctrl[i][0], ctrl[i][1], ctrl[i][2]);

    fprintf(fp," texture { pigment { color rgbt<%f, %f, %f, %f> } } no_shadow }\n",
               ribbon->colour[0], ribbon->colour[1], ribbon->colour[2],
               1.0-sysenv.render.transmit);

    rlist = g_slist_next(rlist);
    }
  }

/******************/
/* spatial planes */
/******************/
for (list=data->spatial ; list ; list=g_slist_next(list))
  {
  spatial = (struct spatial_pak *) list->data;

  switch(spatial->type)
    {
/* special cases */
    case SPATIAL_VECTOR:
      break;
    case SPATIAL_PLANE:
      break;

/* general cases */
    default:
/* enumerate periodic images */
      plist=NULL;
      do
        {
        if (plist)
          {
/* image */
          image = (struct image_pak *) plist->data;
          ARR3SET(vec2, image->rx);
          plist = g_slist_next(plist);
          }
        else
          {
/* original */
          VEC3SET(vec2, 0.0, 0.0, 0.0);
          plist = data->images;
          }
/* enumerate vertices */
        rlist = (GSList *) spatial->data;
        p1 = NULL;

/* TODO - the best way of doing this would be through smooth_triangle() and mesh() */
/* note that these two seem to have to be used together ie can't just use smooth triangle */
/* by itself.  Unfortunately, a mesh is something that can have only one colour which makes */
/* it harder to implement - a rewrite would be needed. */
/* TODO - the normal calcs are redundant at present, but may be used in the future */
        switch(spatial->method)
          {
          case GL_TRIANGLES:
            g_assert(g_slist_length(spatial->data) == 3);

            fprintf(fp, "triangle { \n");
            while (rlist)
              {
              p1 = (struct vec_pak *) rlist->data;
              ARR3SET(vec1, vec2);
              ARR3ADD(vec1, p1->rx);
              vecmat(acm, vec1);
/* supply normal at the vertex as well */
              ARR3SET(vec, p1->n);
              vecmat(acm, vec);

              fprintf(fp, "< %f, %f, %f>  ", vec1[0], vec1[1], vec1[2]);

              rlist = g_slist_next(rlist);
  
              if (rlist)
                fprintf(fp, ",\n");
              else
                fprintf(fp, "\n");
              }
            break;

          case GL_TRIANGLE_STRIP:
            g_assert(g_slist_length(spatial->data) > 2);

            p1 = (struct vec_pak *) rlist->data;
            ARR3SET(v1, vec2);
            ARR3ADD(v1, p1->rx);
            vecmat(acm, v1);
            ARR3SET(n1, p1->n);
            vecmat(acm, n1);

            rlist = g_slist_next(rlist);

            p2 = (struct vec_pak *) rlist->data;
            ARR3SET(v2, vec2);
            ARR3ADD(v2, p2->rx);
            vecmat(acm, v2);
            ARR3SET(n2, p2->n);
            vecmat(acm, n2);

            rlist = g_slist_next(rlist);

            while (rlist)
              {
              p3 = (struct vec_pak *) rlist->data;
              ARR3SET(v3, vec2);
              ARR3ADD(v3, p3->rx);
              vecmat(acm, v3);
              ARR3SET(n3, p3->n);
              vecmat(acm, n3);

              fprintf(fp, "triangle { \n");

              fprintf(fp, "< %f, %f, %f>", v1[0], v1[1], v1[2]);
              fprintf(fp, "< %f, %f, %f>", v2[0], v2[1], v2[2]);
              fprintf(fp, "< %f, %f, %f>\n", v3[0], v3[1], v3[2]);
 
/* FIXME - can we have individual colours at each vertex? */
              fprintf(fp, " texture { \n");
              fprintf(fp, " pigment { color rgbt<%f, %f, %f, %f> } }\n",
                           p3->colour[0], p3->colour[1], p3->colour[2],
                           1.0-sysenv.render.transmit);

/* NB: POVRay highlight size has reverse sense. */
              fprintf(fp, " finish { phong %f phong_size %d } }\n",
                           sysenv.render.shl_strength, (gint) sysenv.render.shl_size);

              ARR3SET(v1, v2);
              ARR3SET(v2, v3);
              ARR3SET(n1, n2);
              ARR3SET(n2, n3);

              rlist = g_slist_next(rlist);
              }
            break;

/* general case */
          default:
            fprintf(fp, "polygon { %d\n", g_slist_length(rlist));
            while (rlist)
              {
              p1 = (struct vec_pak *) rlist->data;
              ARR3SET(vec1, vec2);
              ARR3ADD(vec1, p1->rx);
              vecmat(acm, vec1);
              fprintf(fp, "< %f, %f, %f>", vec1[0], vec1[1], vec1[2]);
  
              rlist = g_slist_next(rlist);
  
              if (rlist)
                fprintf(fp, ",\n");
              else
                fprintf(fp, "\n");
              }
          }

/* FIXME - can we have individual colours at each vertex? */
if (spatial->method != GL_TRIANGLE_STRIP)
  {
        g_assert(p1 != NULL);
        fprintf(fp, " texture { \n");
        fprintf(fp, " pigment { color rgbt<%f, %f, %f, %f> } }\n",
                                               p1->colour[0],
                                               p1->colour[1],
                                               p1->colour[2],
                                      1.0-sysenv.render.transmit);

/* NB: POVRay highlight size has reverse sense. */
        fprintf(fp, " finish { phong %f phong_size %d } }\n",
          sysenv.render.shl_strength, (gint) sysenv.render.shl_size);
  }

/*
if (spatial->method == GL_TRIANGLES)
  fprintf(fp, "}\n");
*/
  
/*
        fprintf(fp, " finish { specular %d roughness %d } }\n",
          sysenv.render.shl_strength, sysenv.render.shl_size);
*/

        }
      while (plist);
      break;
    }
  }

/**************/
/* MORPHOLOGY */
/**************/
if (data->num_vertices)
  {
/* morphology rendering style */
  if (!sysenv.render.wire_surface)
    {
/* render via plane intersection */
    fprintf(fp,"object\n{\nintersection{\n");

    plist = data->planes;
    while (plist != NULL)
      {
      plane = (struct plane_pak *) plist->data;
      if (plane->present)
        {
/* rotate cartesian normals to match viewing angle */
        ARR3SET(vec, plane->norm);
        vecmat(data->rotmat, vec);
/* apply axes conversion matrix */
        vecmat(acm, vec);

/* NEW - distance to facet */
switch(data->morph_type)
  {
  case EQUIL_UN:
    rad = plane->esurf[0];
    break;
  case GROWTH_UN:
    rad = fabs(plane->eatt[0]);
    break;
  case EQUIL_RE:
    rad = plane->esurf[1];
    break;
  case GROWTH_RE:
    rad = fabs(plane->eatt[1]);
    break;
  case DHKL:
  default:
    rad = 1.0/plane->dhkl;
    break;
  }

        fprintf(fp,"plane { <%f, %f, %f>,  %f }\n",vec[0],vec[1],vec[2], rad);
        }
      plist = g_slist_next(plist);
      }
/* crystal composition */
    fprintf(fp,"} interior { ior %f } texture { finish { %s } "
               " pigment { color rgbf <%f,%f,%f,%f> } }\n }\n", 
               sysenv.render.ref_index,
               sysenv.render.morph_finish,
               sysenv.render.morph_colour[0],
               sysenv.render.morph_colour[1],
               sysenv.render.morph_colour[2],
               1.0-sysenv.render.transmit);  
    }
  else
    {
    rad = sysenv.render.frame_thickness / p2a;
    for (list1=data->vertices ; list1 ; list1=g_slist_next(list1))
      {
      vx1 = (struct vertex_pak *) list1->data;

      for (list2=vx1->adj ; list2 ; list2=g_slist_next(list2))
        {
        vx2 = (struct vertex_pak *) list2->data;

        if (!ridge_visible(vx1, vx2, data))
          continue;

        ARR3SET(vec, vx1->rx);
        vecmat(acm, vec);
        x1 = vec[0];
        y1 = vec[1];
        z1 = vec[2];

        ARR3SET(vec, vx2->rx);
        vecmat(acm, vec);
        x2 = vec[0];
        y2 = vec[1];
        z2 = vec[2];

        flag=0;
        if (fabs(x2-x1) < 0.001)
          flag++;
        if (fabs(y2-y1) < 0.001)
          flag++;
        if (fabs(z2-z1) < 0.001)
          flag++;
        if (flag==3)
          continue;

        fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>,%f\n",
                                  x1,y1,z1,x2,y2,z2,rad);
        fprintf(fp," texture { pigment { color rgb<%f %f %f>} } }\n",
                                          sysenv.render.morph_colour[0],
                                          sysenv.render.morph_colour[1],
                                          sysenv.render.morph_colour[2]);
        }
      }
    }
  }

/* finished making the .pov file */
fclose(fp);
return(0);
}

