/*
 *  Brute force symmetry analyzer.
 *  This is actually C++ program, masquerading as a C one!
 *
 *  (C) 1996 S. Pachkovsky, ps@oci.unizh.ch
 *  This code comes with NO WARRANTY, neither explicit nor implied.
 *  Non-profit use of the code is welcome, provided that the
 *  copyright above is retained in all derived works and the
 *  origin of the code is clearly indicated. Any for-profit use
 *  requires prior written permission of the author.
 *
 * $Log: symmetry.c,v $
 * Revision 1.6  2002/01/10 07:28:38  sean
 * Added text/font drawing to OpenGL. Renamed wire_draw() to update_canvas().
 *
 * Revision 1.5  2001/10/30 13:18:04  sean
 * New status flag method (bitwise ops) should now work.
 *
 * Revision 1.4  2001/10/29 14:20:18  sean
 * New bitwise flag implementation for delete/hidden etc.
 *
 * Revision 1.3  2001/09/17 09:39:01  sean
 * Altered the model tree highlighting scheme.
 *
 * Revision 1.2  2001/09/03 05:02:57  sean
 * Marvin's surface generation code added
 *
 * Revision 1.1.1.1  2001/07/06 02:57:07  andrew
 * initial import
 *
 * Revision 1.15  2000/01/25  16:47:17  patchkov
 * *** empty log message ***
 *
 * Revision 1.14  2000/01/25  16:39:08  patchkov
 * *** empty log message ***
 *
 * Revision 1.13  1996/05/24  12:32:08  ps
 * Revision 1.12  1996/05/23  16:10:47  ps
 * First reasonably stable version.
 *
 */

/* Modified by Sean Fleming to use glib, and for inclusion in gdis */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "gdis.h"

/* TODO - probably needs a few more dp... */
#ifndef M_PI
#define M_PI 3.1415926535897932384626433832795028841971694
#endif
#define	DIMENSION 3
#define MAXPARAM  7
#define DEBUG 0

extern struct sysenv_pak sysenv;

typedef struct 
  {
  gint     type;
  gdouble  x[DIMENSION];
  } OBJECT;

/*
 *  All specific structures should have corresponding elements in the
 *  same position generic structure does.
 *
 *  Planes are characterized by the surface normal direction
 *  (taken in the direction *from* the coordinate origin)
 *  and distance from the coordinate origin to the plane
 *  in the direction of the surface normal.
 *
 *  Inversion is characterized by location of the inversion center.
 *
 *  Rotation is characterized by a vector (distance+direction) from the origin
 *  to the rotation axis, axis direction and rotation order. Rotations
 *  are in the clockwise direction looking opposite to the direction
 *  of the axis. Note that this definition of the rotation axis
 *  is *not* unique, since an arbitrary multiple of the axis direction
 *  can be added to the position vector without changing actual operation.
 *
 *  Mirror rotation is defined by the same parameters as normal rotation,
 *  but the origin is now unambiguous since it defines the position of the
 *  plane associated with the axis.
 *
 */

typedef struct _SYMMETRY_ELEMENT_ 
  {
  void (*transform_atom) 
       (struct _SYMMETRY_ELEMENT_ *el, OBJECT *from, OBJECT *to );
  gint *transform;   /* Correspondence table for the transformation  */
  gint order;        /* Applying transformation this many times is identity */
  gint nparam;       /* 4 for inversion and planes, 7 for axes */
  gdouble maxdev;    /* Larges error associated with the element */
  gdouble distance ;
  gdouble normal[DIMENSION];
  gdouble direction[DIMENSION];
  } SYMMETRY_ELEMENT;

typedef struct 
  {
  gchar *group_name;        /* Canonical group name */
  gchar *symmetry_code;     /* Group symmetry code  */
  gint (*check)(void);      /* Additional verification routine, not used */
  } POINT_GROUP;

/****************/
/* globals, yuk */
/****************/
gdouble                 ToleranceSame         = 1e-3 ;
gdouble                 TolerancePrimary      = 5e-2 ;
gdouble                 ToleranceFinal        = 1e-4 ;
gdouble                 MaxOptStep            = 5e-1 ;
gdouble                 MinOptStep            = 1e-7 ;
gdouble                 GradientStep          = 1e-7 ;
gdouble                 OptChangeThreshold    = 1e-10 ;
gdouble                 CenterOfSomething[ DIMENSION ] ;
gdouble *               DistanceFromCenter    = NULL ;
gint                    verbose               = 0 ;
gint                    MaxOptCycles          = 200 ;
gint                    OptChangeHits         = 5 ;
gint                    MaxAxisOrder          = 20 ;
gint                    AtomsCount            = 0 ;
OBJECT *                 Atoms                 = NULL ;
gint                    PlanesCount           = 0 ;
SYMMETRY_ELEMENT **    Planes                = NULL ;
SYMMETRY_ELEMENT *     MolecularPlane        = NULL ;
gint                    InversionCentersCount = 0 ;
SYMMETRY_ELEMENT **    InversionCenters      = NULL ;
gint                    NormalAxesCount       = 0 ;
SYMMETRY_ELEMENT **    NormalAxes            = NULL ;
gint                    ImproperAxesCount     = 0 ;
SYMMETRY_ELEMENT **    ImproperAxes          = NULL ;
gint *                  NormalAxesCounts      = NULL ;
gint *                  ImproperAxesCounts    = NULL ;
gint                    BadOptimization       = 0 ;
gchar *                 SymmetryCode          = NULL ;
/*
 *    Statistics
 */
gint                   StatTotal             = 0 ;
gint                   StatEarly             = 0 ;
gint                   StatPairs             = 0 ;
gint                   StatDups              = 0 ;
gint                   StatOrder             = 0 ;
gint                   StatOpt               = 0 ;
gint                   StatAccept            = 0 ;

/*
 *    Point groups I know about
 */
gint true(void){return 1;}

POINT_GROUP PointGroups[] = 
  {
  {"C1",    "",                                                       true},
  {"Cs",    "(sigma) ",                                               true},
  {"Ci",    "(i) ",                                                   true},
  {"C2",    "(C2) ",                                                  true},
  {"C3",    "(C3) ",                                                  true},
  {"C4",    "(C4) (C2) ",                                             true},
  {"C5",    "(C5) ",                                                  true},
  {"C6",    "(C6) (C3) (C2) ",                                        true},
  {"C7",    "(C7) ",                                                  true},
  {"C8",    "(C8) (C4) (C2) ",                                        true},
  {"D2",    "3*(C2) ",                                                true},
  {"D3",    "(C3) 3*(C2) ",                                           true},
  {"D4",    "(C4) 5*(C2) ",                                           true},
  {"D5",    "(C5) 5*(C2) ",                                           true},
  {"D6",    "(C6) (C3) 7*(C2) ",                                      true},
  {"D7",    "(C7) 7*(C2) ",                                           true},
  {"D8",    "(C8) (C4) 9*(C2) ",                                      true},
  {"C2v",   "(C2) 2*(sigma) ",                                        true},
  {"C3v",   "(C3) 3*(sigma) ",                                        true},
  {"C4v",   "(C4) (C2) 4*(sigma) ",                                   true},
  {"C5v",   "(C5) 5*(sigma) ",                                        true},
  {"C6v",   "(C6) (C3) (C2) 6*(sigma) ",                              true},
  {"C7v",   "(C7) 7*(sigma) ",                                        true},
  {"C8v",   "(C8) (C4) (C2) 8*(sigma) ",                              true},
  {"C2h",   "(i) (C2) (sigma) ",                                      true},
  {"C3h",   "(C3) (S3) (sigma) ",                                     true},
  {"C4h",   "(i) (C4) (C2) (S4) (sigma) ",                            true},
  {"C5h",   "(C5) (S5) (sigma) ",                                     true},
  {"C6h",   "(i) (C6) (C3) (C2) (S6) (S3) (sigma) ",                  true},
  {"C7h",   "(C7) (S7) (sigma) ",                                     true},
  {"C8h",   "(i) (C8) (C4) (C2) (S8) (S4) (sigma) ",                  true},
  {"D2h",   "(i) 3*(C2) 3*(sigma) ",                                  true},
  {"D3h",   "(C3) 3*(C2) (S3) 4*(sigma) ",                            true},
  {"D4h",   "(i) (C4) 5*(C2) (S4) 5*(sigma) ",                        true},
  {"D5h",   "(C5) 5*(C2) (S5) 6*(sigma) ",                            true},
  {"D6h",   "(i) (C6) (C3) 7*(C2) (S6) (S3) 7*(sigma) ",              true},
  {"D7h",   "(C7) 7*(C2) (S7) 8*(sigma) ",                            true},
  {"D8h",   "(i) (C8) (C4) 9*(C2) (S8) (S4) 9*(sigma) ",              true},
  {"D2d",   "3*(C2) (S4) 2*(sigma) ",                                 true},
  {"D3d",   "(i) (C3) 3*(C2) (S6) 3*(sigma) ",                        true},
  {"D4d",   "(C4) 5*(C2) (S8) 4*(sigma) ",                            true},
  {"D5d",   "(i) (C5) 5*(C2) (S10) 5*(sigma) ",                       true},
  {"D6d",   "(C6) (C3) 7*(C2) (S12) (S4) 6*(sigma) ",                 true},
  {"D7d",   "(i) (C7) 7*(C2) (S14) 7*(sigma) ",                       true},
  {"D8d",   "(C8) (C4) 9*(C2) (S16) 8*(sigma) ",                      true},
  {"S4",    "(C2) (S4) ",                                             true},
  {"S6",    "(i) (C3) (S6) ",                                         true},
  {"S8",    "(C4) (C2) (S8) ",                                        true},
  {"T",     "4*(C3) 3*(C2) ",                                         true},
  {"Th",    "(i) 4*(C3) 3*(C2) 4*(S6) 3*(sigma) ",                    true},
  {"Td",    "4*(C3) 3*(C2) 3*(S4) 6*(sigma) ",                        true},
  {"O",     "3*(C4) 4*(C3) 9*(C2) ",                                  true},
  {"Oh",    "(i) 3*(C4) 4*(C3) 9*(C2) 4*(S6) 3*(S4) 9*(sigma) ",      true},
  {"Cinfv", "(Cinf) (sigma) ",                                        true},
  {"Dinfh", "(i) (Cinf) (C2) 2*(sigma) ",                             true},
  {"I",     "6*(C5) 10*(C3) 15*(C2) ",                                true},
  {"Ih",    "(i) 6*(C5) 10*(C3) 15*(C2) 6*(S10) 10*(S6) 15*(sigma) ", true},
  {"Kh",    "(i) (Cinf) (sigma) ",                                    true},
  };


#define PointGroupsCount (sizeof(PointGroups)/sizeof(POINT_GROUP))
gchar *                 PointGroupRejectionReason = NULL ;

/*
 *   Generic functions
 */

gdouble pow2( gdouble x )
{
return x * x ;
}

gint establish_pairs( SYMMETRY_ELEMENT *elem )
{
gint i, j, k, best_j ;
gchar *atom_used = g_malloc0(AtomsCount);
gdouble distance, best_distance;
OBJECT symmetric;

for( i = 0 ; i < AtomsCount ; i++ ){
    if( elem->transform[i] >= AtomsCount ){ /* No symmetric atom yet          */
        if( verbose > 2 ) printf( "        looking for a pair for %d\n", i ) ;
        elem->transform_atom( elem, Atoms+i, &symmetric ) ;
        if( verbose > 2 ) printf( "        new coordinates are: (%g,%g,%g)\n", 
                              symmetric.x[0], symmetric.x[1], symmetric.x[2] ) ;
        best_j        = i ;
        best_distance = 2*TolerancePrimary ;/* Performance value we'll reject */
        for( j = 0 ; j < AtomsCount ; j++ ){
            if( Atoms[j].type != symmetric.type || atom_used[j] )
                continue ;
            for( k = 0, distance = 0 ; k < DIMENSION ; k++ ){
                distance += pow2( symmetric.x[k] - Atoms[j].x[k] ) ;
                }
            distance = sqrt( distance ) ;
            if( verbose > 2 ) printf( "        distance to %d is %g\n", j, distance ) ;
            if( distance < best_distance ){
                best_j        = j ;
                best_distance = distance ;
                }
            }
        if( best_distance > TolerancePrimary ){ /* Too bad, there is no symmetric atom */
            if( verbose > 0 ) 
                printf( "        no pair for atom %d - best was %d with err = %g\n", i, best_j, best_distance ) ;
            g_free( atom_used ) ;
            return -1 ;
            }
        elem->transform[i] = best_j ;
        atom_used[best_j]  = 1 ;
        if( verbose > 1 ) printf( "        atom %d transforms to the atom %d, err = %g\n", i, best_j, best_distance ) ;
        }
    }
g_free( atom_used ) ;
return 0 ;
}

gint check_transform_order( SYMMETRY_ELEMENT *elem )
{
gint             i, j, k ;
void            rotate_reflect_atom( SYMMETRY_ELEMENT *, OBJECT *, OBJECT *) ;

for( i = 0 ; i < AtomsCount ; i++ ){
    if( elem->transform[i] == i )   /* Identity transform is Ok for any order */
        continue ;
    if( elem->transform_atom == rotate_reflect_atom ){
        j = elem->transform[i] ;
        if( elem->transform[j] == i )
            continue ; /* Second-order transform is Ok for improper axis */
        }
    for( j = elem->order - 1, k = elem->transform[i] ; j > 0 ; j--, k = elem->transform[k] ){
        if( k == i ){
            if( verbose > 0 ) printf( "        transform looped %d steps too early from atom %d\n", j, i ) ;
            return -1 ;
            }
        }
    if( k != i && elem->transform_atom == rotate_reflect_atom ){
        /* For improper axes, the complete loop may also take twice the order */
        for( j = elem->order ; j > 0 ; j--, k = elem->transform[k] ){
            if( k == i ){
                if( verbose > 0 ) printf( "        (improper) transform looped %d steps too early from atom %d\n", j, i ) ;
                return -1 ;
                }
            }
        }
    if( k != i ){
        if( verbose > 0 ) printf( "        transform failed to loop after %d steps from atom %d\n", elem->order, i ) ;
        return -1 ;
        }
    }
return 0 ;
}

gint same_transform( SYMMETRY_ELEMENT *a, SYMMETRY_ELEMENT *b )
{
gint               i, j ;
gint               code ;

if( ( a->order != b->order ) || ( a->nparam != b->nparam ) || ( a->transform_atom != b->transform_atom ) )
    return 0 ;
for( i = 0, code = 1 ; i < AtomsCount ; i++ ){
    if( a->transform[i] != b->transform[i] ){
        code = 0 ;
        break ;
        }
    }
if( code == 0 && a->order > 2 ){  /* b can also be a reverse transformation for a */
    for( i = 0 ; i < AtomsCount ; i++ ){
        j = a->transform[i] ;
        if( b->transform[j] != i )
            return 0 ;
        }
    return 1 ;
    }
return code ;
}

SYMMETRY_ELEMENT *alloc_symmetry_element( void )
{
gint                i;
SYMMETRY_ELEMENT *elem;

elem = g_malloc0(sizeof(SYMMETRY_ELEMENT));
elem->transform = g_malloc0(AtomsCount*sizeof(int));

for (i=0 ; i<AtomsCount ; i++)
  elem->transform[i] = AtomsCount + 1 ; /* An impossible value */

return elem ;
}

void destroy_symmetry_element( SYMMETRY_ELEMENT *elem )
{
if (elem != NULL)
  {
  if (elem->transform != NULL)
    g_free(elem->transform);
  g_free(elem);
  }
}

gint check_transform_quality( SYMMETRY_ELEMENT *elem )
{
gint               i, j, k ;
OBJECT              symmetric ;
gdouble            r, max_r ;

for( i = 0, max_r = 0 ; i < AtomsCount ; i++ ){
    j = elem->transform[i] ;
    elem->transform_atom( elem, Atoms + i, &symmetric ) ;
    for( k = 0, r = 0 ; k < DIMENSION ; k++ ){
        r += pow2( symmetric.x[k] - Atoms[j].x[k] ) ;
        }
    r = sqrt( r ) ;
    if( r > ToleranceFinal ){
        if( verbose > 0 ) printf( "        distance to symmetric atom (%g) is too big for %d\n", r, i ) ;
        return -1 ;
        }
    if( r > max_r ) max_r = r ;
    }
elem->maxdev = max_r ;
return 0 ;
}

gdouble eval_optimization_target_function(SYMMETRY_ELEMENT *elem, gint *finish)
{
gint               i, j, k ;
OBJECT              symmetric ;
gdouble            target, r, maxr ;

if( elem->nparam >= 4 ){
    for( k = 0, r = 0 ; k < DIMENSION ; k++ ){
        r += elem->normal[k]*elem->normal[k] ;
        }
    r = sqrt( r ) ;
    if( r < ToleranceSame ){
        fprintf( stderr, "Normal collapsed!\n" ) ;
        exit( EXIT_FAILURE ) ;
        }
    for( k = 0 ; k < DIMENSION ; k++ ){
        elem->normal[k] /= r ;
        }
    if( elem->distance < 0 ){
        elem->distance = -elem->distance ;
        for( k = 0 ; k < DIMENSION ; k++ ){
            elem->normal[k] = -elem->normal[k] ;
            }
        }
    }
if( elem->nparam >= 7 ){
    for( k = 0, r = 0 ; k < DIMENSION ; k++ ){
        r += elem->direction[k]*elem->direction[k] ;
        }
    r = sqrt( r ) ;
    if( r < ToleranceSame ){
        fprintf( stderr, "Direction collapced!\n" ) ;
        exit( EXIT_FAILURE ) ;
        }
    for( k = 0 ; k < DIMENSION ; k++ ){
        elem->direction[k] /= r ;
        }
    }
for( i = 0, target = maxr = 0 ; i < AtomsCount ; i++ ){
    elem->transform_atom( elem, Atoms + i, &symmetric ) ;
    j = elem->transform[i] ;
    for( k = 0, r = 0 ; k < DIMENSION ; k++ ){
        r += pow2( Atoms[j].x[k] - symmetric.x[k] ) ;
        }
    if( r > maxr ) maxr = r ;
    target += r ;
    }
if( finish != NULL ){
    *finish = 0 ;
    if( sqrt( maxr ) < ToleranceFinal )
        *finish = 1 ;
    }
return target ;
}

void get_params( SYMMETRY_ELEMENT *elem, gdouble values[] )
{
memcpy( values, &elem->distance, elem->nparam * sizeof( gdouble ) ) ;
}

void set_params( SYMMETRY_ELEMENT *elem, gdouble values[] )
{
memcpy( &elem->distance, values, elem->nparam * sizeof( gdouble ) ) ;
}

void optimize_transformation_params( SYMMETRY_ELEMENT *elem )
{
gdouble            values[ MAXPARAM ] ;
gdouble            grad  [ MAXPARAM ] ;
gdouble            force [ MAXPARAM ] ;
gdouble            step  [ MAXPARAM ] ;
gdouble            f, fold, fnew, fnew2, fdn, fup, snorm ;
gdouble            a, b, x ;
gint               vars  = elem->nparam ;
gint               cycle = 0 ;
gint               i, finish ;
gint               hits = 0 ;

if( vars > MAXPARAM ){
    fprintf( stderr, "Catastrophe in optimize_transformation_params()!\n" ) ;
    exit( EXIT_FAILURE ) ;
    }
f = 0 ;
do {
    fold = f ;
    f    = eval_optimization_target_function( elem, &finish ) ;
    /* Evaluate function, gradient and diagonal force constants */
    if( verbose > 1 ) printf( "            function value = %g\n", f ) ;
    if( finish ){
        if( verbose > 1 ) printf( "        function value is small enough\n" ) ;
        break ;
        }
    if( cycle > 0 ){
        if( fabs( f-fold ) > OptChangeThreshold )
             hits = 0 ;
        else hits++ ;
        if( hits >= OptChangeHits ){
            if( verbose > 1 ) printf( "        no progress is made, stop optimization\n" ) ;
            break ;
            }
        }
    get_params( elem, values ) ;
    for( i = 0 ; i < vars ; i++ ){
        values[i] -= GradientStep ;
        set_params( elem, values ) ;
        fdn        = eval_optimization_target_function( elem, NULL ) ;
        values[i] += 2*GradientStep ;
        set_params( elem, values ) ;
        fup        = eval_optimization_target_function( elem, NULL ) ;
        values[i] -= GradientStep ;
        grad[i]    = ( fup - fdn ) / ( 2 * GradientStep ) ;
        force[i]   = ( fup + fdn - 2*f ) / ( GradientStep * GradientStep ) ;
        if( verbose > 1 ) printf( "        i = %d, grad = %12.6e, force = %12.6e\n", i, grad[i], force[i] ) ;
        }
    /* Do a quasy-Newton step */
    for( i = 0, snorm = 0 ; i < vars ; i++ ){
        if( force[i] <  0   ) force[i] = -force[i] ;
        if( force[i] < 1e-3 ) force[i] = 1e-3 ;
        if( force[i] > 1e3  ) force[i] = 1e3 ;
        step[i] = - grad[i]/force[i] ;
        snorm += step[i] * step[i] ;
        }
    snorm = sqrt( snorm ) ;
    if( snorm > MaxOptStep ){ /* Renormalize step */
        for( i = 0 ; i < vars ; i++ )
            step[i] *= MaxOptStep/snorm ;
        snorm = MaxOptStep ;
        }
    do {
        for( i = 0 ; i < vars ; i++ ){
            values[i] += step[i] ;
            }
        set_params( elem, values ) ;
        fnew = eval_optimization_target_function( elem, NULL ) ;
        if( fnew < f )
            break ;
        for( i = 0 ; i < vars ; i++ ){
            values[i] -= step[i] ;
            step  [i] /= 2 ;
            }
        set_params( elem, values ) ;
        snorm /= 2 ;
        } while( snorm > MinOptStep ) ;
        if( (snorm > MinOptStep) && (snorm < MaxOptStep / 2) ){  /* try to do quadratic interpolation */
            for( i = 0 ; i < vars ; i++ )
                values[i] += step[i] ;
            set_params( elem, values ) ;
            fnew2 = eval_optimization_target_function( elem, NULL ) ;
            if( verbose > 1 ) printf( "        interpolation base points: %g, %g, %g\n", f, fnew, fnew2 ) ;
            for( i = 0 ; i < vars ; i++ )
                values[i] -= 2*step[i] ;
            a     = ( 4*f - fnew2 - 3*fnew ) / 2 ;
            b     = ( f + fnew2 - 2*fnew ) / 2 ;
            if( verbose > 1 ) printf( "        linear interpolation coefficients %g, %g\n", a, b ) ;
            if( b > 0 ){
                x = -a/(2*b) ;
                if( x > 0.2 && x < 1.8 ){
                    if( verbose > 1 ) printf( "        interpolated: %g\n", x ) ;
                    for( i = 0 ; i < vars ; i++ )
                        values[i] += x*step[i] ;
                    }
                else b = 0 ;
                }
            if( b <= 0 ){
                if( fnew2 < fnew ){
                    for( i = 0 ; i < vars ; i++ )
                        values[i] += 2*step[i] ;
                    }
                else {
                    for( i = 0 ; i < vars ; i++ )
                        values[i] += step[i] ;
                    }
                }
            set_params( elem, values ) ;
            }
    } while( snorm > MinOptStep && ++cycle < MaxOptCycles ) ;
f = eval_optimization_target_function( elem, NULL ) ;
if( cycle >= MaxOptCycles ) BadOptimization = 1 ;
if( verbose > 0 ) {
    if( cycle >= MaxOptCycles )
        printf( "        maximum number of optimization cycles made\n" ) ;
        printf( "        optimization completed after %d cycles with f = %g\n", cycle, f ) ;
    }
}

gint
refine_symmetry_element( SYMMETRY_ELEMENT *elem, gint build_table )
{
        gint               i ;


if( build_table && (establish_pairs( elem ) < 0) ){
    StatPairs++ ;
    if( verbose > 0 ) printf( "        no transformation correspondence table can be constructed\n" ) ;
    return -1 ;
    }
for( i = 0 ; i < PlanesCount ; i++ ){
    if( same_transform( Planes[i], elem ) ){
        StatDups++ ;
        if( verbose > 0 ) printf( "        transformation is identical to plane %d\n", i ) ;
        return -1 ;
        }
    }

for( i = 0 ; i < InversionCentersCount ; i++ ){
    if( same_transform( InversionCenters[i], elem ) ){
        StatDups++ ;
        if( verbose > 0 ) printf( "        transformation is identical to inversion center %d\n", i ) ;
        return -1 ;
        }
    }
for( i = 0 ; i < NormalAxesCount ; i++ ){
    if( same_transform( NormalAxes[i], elem ) ){
        StatDups++ ;
        if( verbose > 0 ) printf( "        transformation is identical to normal axis %d\n", i ) ;
        return -1 ;
        }
    }
for( i = 0 ; i < ImproperAxesCount ; i++ ){
    if( same_transform( ImproperAxes[i], elem ) ){
        StatDups++ ;
        if( verbose > 0 ) printf( "        transformation is identical to improper axis %d\n", i ) ;
        return -1 ;
        }
    }
if( check_transform_order( elem ) < 0 ){
    StatOrder++ ;
    if( verbose > 0 ) printf( "        incorrect transformation order\n" ) ;
    return -1 ;
    }
optimize_transformation_params( elem ) ;
if( check_transform_quality( elem ) < 0 ){
    StatOpt++ ;
    if( verbose > 0 ) printf( "        refined transformation does not pass the numeric threshold\n" ) ;
    return -1 ;
    }
StatAccept++ ;
return 0 ;
}

/*
 *   Plane-specific functions
 */

void mirror_atom( SYMMETRY_ELEMENT *plane, OBJECT *from, OBJECT *to )
{
gint                i ;
gdouble             r ;

for( i = 0, r = plane->distance ; i < DIMENSION ; i++ ){
    r -= from->x[i] * plane->normal[i] ;
    }
to->type = from->type ;
for( i = 0 ; i < DIMENSION ; i++ ){
    to->x[i] = from->x[i] + 2*r*plane->normal[i] ;
    }
}

SYMMETRY_ELEMENT * init_mirror_plane( gint i, gint j )
{
SYMMETRY_ELEMENT * plane = alloc_symmetry_element() ;
gdouble             dx[ DIMENSION ], midpoint[ DIMENSION ], rab, r ;
gint                k ;

if( verbose > 0 ) printf( "Trying mirror plane for atoms %d,%d\n", i, j ) ;
StatTotal++ ;
plane->transform_atom = mirror_atom ;
plane->order          = 2 ;
plane->nparam         = 4 ;
for( k = 0, rab = 0 ; k < DIMENSION ; k++ ){
    dx[k]       = Atoms[i].x[k] - Atoms[j].x[k] ;
    midpoint[k] = ( Atoms[i].x[k] + Atoms[j].x[k] ) / 2.0 ;
    rab        += dx[k]*dx[k] ;
    }
rab = sqrt(rab) ;
if( rab < ToleranceSame ){
    fprintf( stderr, "Atoms %d and %d coincide (r = %g)\n", i, j, rab ) ;
    exit( EXIT_FAILURE ) ;
    }
for( k = 0, r = 0 ; k < DIMENSION ; k++ ){
    plane->normal[k] = dx[k]/rab ;
    r += midpoint[k]*plane->normal[k] ;
    }
if( r < 0 ){  /* Reverce normal direction, distance is always positive! */
    r = -r ;
    for( k = 0 ; k < DIMENSION ; k++ ){
        plane->normal[k] = -plane->normal[k] ;
        }
    }
plane->distance = r ;
if( verbose > 0 ) printf( "    initial plane is at %g from the origin\n", r ) ;
if( refine_symmetry_element( plane, 1 ) < 0 ){
    if( verbose > 0 ) printf( "    refinement failed for the plane\n" ) ;
    destroy_symmetry_element( plane ) ;
    return NULL ;
    }
return plane ;
}

SYMMETRY_ELEMENT *init_ultimate_plane( void )
{
SYMMETRY_ELEMENT * plane = alloc_symmetry_element() ;
gdouble             d0[ DIMENSION ], d1[ DIMENSION ], d2[ DIMENSION ] ;
gdouble             p[ DIMENSION ] ;
gdouble             r, s0, s1, s2 ;
gdouble *           d ;
gint                i, j, k ;

if( verbose > 0 ) printf( "Trying whole-molecule mirror plane\n" ) ;
StatTotal++ ;
plane->transform_atom = mirror_atom ;
plane->order          = 1 ;
plane->nparam         = 4 ;
for( k = 0 ; k < DIMENSION ; k++ )
    d0[k] = d1[k] = d2[k] = 0 ;
d0[0] = 1 ; d1[1] = 1 ; d2[2] = 1 ;
for( i = 1 ; i < AtomsCount ; i++ ){
    for( j = 0 ; j < i ; j++ ){
        for( k = 0, r = 0 ; k < DIMENSION ; k++ ){
            p[k] = Atoms[i].x[k] - Atoms[j].x[k] ;
            r   += p[k]*p[k] ;
            }
        r = sqrt(r) ;
        for( k = 0, s0=s1=s2=0 ; k < DIMENSION ; k++ ){
            p[k] /= r ;
            s0   += p[k]*d0[k] ;
            s1   += p[k]*d1[k] ;
            s2   += p[k]*d2[k] ;
            }
        for( k = 0 ; k < DIMENSION ; k++ ){
            d0[k] -= s0*p[k] ;
            d1[k] -= s1*p[k] ;
            d2[k] -= s2*p[k] ;
            }
        }
    }
for( k = 0, s0=s1=s2=0 ; k < DIMENSION ; k++ ){
    s0 += d0[k] ;
    s1 += d1[k] ;
    s2 += d2[k] ;
    }
d = NULL ;
if( s0 >= s1 && s0 >= s2 ) d = d0 ;
if( s1 >= s0 && s1 >= s2 ) d = d1 ;
if( s2 >= s0 && s2 >= s1 ) d = d2 ;
if( d == NULL ){
    fprintf( stderr, "Catastrophe in init_ultimate_plane(): %g, %g and %g have no ordering!\n", s0, s1, s2 ) ;
    exit( EXIT_FAILURE ) ;
    }
for( k = 0, r = 0 ; k < DIMENSION ; k++ )
    r += d[k]*d[k] ;
r = sqrt(r) ;
if( r > 0 ){
    for( k = 0 ; k < DIMENSION ; k++ )
        plane->normal[k] = d[k]/r ;
    }
else {
    for( k = 1 ; k < DIMENSION ; k++ )
        plane->normal[k] = 0 ;
    plane->normal[0] = 1 ;
    }
for( k = 0, r = 0 ; k < DIMENSION ; k++ )
    r += CenterOfSomething[k]*plane->normal[k] ;
plane->distance = r ;
for( k = 0 ; k < AtomsCount ; k++ )
    plane->transform[k] = k ;
if( refine_symmetry_element( plane, 0 ) < 0 ){
    if( verbose > 0 ) printf( "    refinement failed for the plane\n" ) ;
    destroy_symmetry_element( plane ) ;
    return NULL ;
    }
return plane ;
}

/*
 *   Inversion-center specific functions
 */
void invert_atom( SYMMETRY_ELEMENT *center, OBJECT *from, OBJECT *to )
{
gint                i ;

to->type = from->type ;
for( i = 0 ; i < DIMENSION ; i++ ){
    to->x[i] = 2*center->distance*center->normal[i] - from->x[i] ;
    }
}

SYMMETRY_ELEMENT *init_inversion_center( void )
{
SYMMETRY_ELEMENT * center = alloc_symmetry_element() ;
gint                k ;
gdouble             r ;

if( verbose > 0 ) printf( "Trying inversion center at the center of something\n" ) ;
StatTotal++ ;
center->transform_atom = invert_atom ;
center->order          = 2 ;
center->nparam         = 4 ;
for( k = 0, r = 0 ; k < DIMENSION ; k++ )
    r += CenterOfSomething[k]*CenterOfSomething[k] ;
r = sqrt(r) ;
if( r > 0 ){
    for( k = 0 ; k < DIMENSION ; k++ )
        center->normal[k] = CenterOfSomething[k]/r ;
    }
else {
    center->normal[0] = 1 ;
    for( k = 1 ; k < DIMENSION ; k++ )
        center->normal[k] = 0 ;
    }
center->distance = r ;
if( verbose > 0 ) printf( "    initial inversion center is at %g from the origin\n", r ) ;
if( refine_symmetry_element( center, 1 ) < 0 ){
    if( verbose > 0 ) printf( "    refinement failed for the inversion center\n" ) ;
    destroy_symmetry_element( center ) ;
    return NULL ;
    }
return center ;
}

/*
 *   Normal rotation axis-specific routines.
 */
void rotate_atom( SYMMETRY_ELEMENT *axis, OBJECT *from, OBJECT *to )
{
gdouble             x[3], y[3], a[3], b[3], c[3] ;
gdouble             angle = axis->order ? 2*M_PI/axis->order : 1.0 ;
gdouble             a_sin = sin( angle ) ;
gdouble             a_cos = cos( angle ) ;
gdouble             dot ;
gint                i ;

if( DIMENSION != 3 ){
    fprintf( stderr, "Catastrophe in rotate_atom!\n" ) ;
    exit( EXIT_FAILURE ) ;
    }
for( i = 0 ; i < 3 ; i++ )
    x[i] = from->x[i] - axis->distance * axis->normal[i] ;
for( i = 0, dot = 0 ; i < 3 ; i++ )
    dot += x[i] * axis->direction[i] ;
for( i = 0 ; i < 3 ; i++ )
    a[i] = axis->direction[i] * dot ;
for( i = 0 ; i < 3 ; i++ )
    b[i] = x[i] - a[i] ;
c[0] = b[1]*axis->direction[2] - b[2]*axis->direction[1] ;
c[1] = b[2]*axis->direction[0] - b[0]*axis->direction[2] ;
c[2] = b[0]*axis->direction[1] - b[1]*axis->direction[0] ;
for( i = 0 ; i < 3 ; i++ )
    y[i] = a[i] + b[i]*a_cos + c[i]*a_sin ;
for( i = 0 ; i < 3 ; i++ )
    to->x[i] = y[i] + axis->distance * axis->normal[i] ;
to->type = from->type ;
}

SYMMETRY_ELEMENT *init_ultimate_axis(void)
{
SYMMETRY_ELEMENT * axis = alloc_symmetry_element() ;
gdouble             dir[ DIMENSION ], rel[ DIMENSION ] ;
gdouble             s ;
gint                i, k ;

if( verbose > 0 ) printf( "Trying infinity axis\n" ) ;
StatTotal++ ;
axis->transform_atom = rotate_atom ;
axis->order          = 0 ;
axis->nparam         = 7 ;
for( k = 0 ; k < DIMENSION ; k++ )
    dir[k] = 0 ;
for( i = 0 ; i < AtomsCount ; i++ ){
    for( k = 0, s = 0 ; k < DIMENSION ; k++ ){
        rel[k] = Atoms[i].x[k] - CenterOfSomething[k] ;
        s     += rel[k]*dir[k] ;
        }
    if( s >= 0 )
         for( k = 0 ; k < DIMENSION ; k++ )
             dir[k] += rel[k] ;
    else for( k = 0 ; k < DIMENSION ; k++ )
             dir[k] -= rel[k] ;
    }
for( k = 0, s = 0 ; k < DIMENSION ; k++ )
    s += pow2( dir[k] ) ;
s = sqrt(s) ;
if( s > 0 )
     for( k = 0 ; k < DIMENSION ; k++ )
         dir[k] /= s ;
else dir[0] = 1 ;
for( k = 0 ; k < DIMENSION ; k++ )
    axis->direction[k] = dir[k] ;
for( k = 0, s = 0 ; k < DIMENSION ; k++ )
    s += pow2( CenterOfSomething[k] ) ;
s = sqrt(s) ;
if( s > 0 )
    for( k = 0 ; k < DIMENSION ; k++ )
        axis->normal[k] = CenterOfSomething[k]/s ;
else {
    for( k = 1 ; k < DIMENSION ; k++ )
        axis->normal[k] = 0 ;
    axis->normal[0] = 1 ;
    }
axis->distance = s ;
for( k = 0 ; k < AtomsCount ; k++ )
    axis->transform[k] = k ;
if( refine_symmetry_element( axis, 0 ) < 0 ){
    if( verbose > 0 ) printf( "    refinement failed for the infinity axis\n" ) ;
    destroy_symmetry_element( axis ) ;
    return NULL ;
    }
return axis ;
}


SYMMETRY_ELEMENT *init_c2_axis( gint i, gint j, gdouble support[ DIMENSION ] )
{
SYMMETRY_ELEMENT * axis ;
gint                k ;
gdouble             ris, rjs ;
gdouble             r, center[ DIMENSION ] ;

if( verbose > 0 ) 
    printf( "Trying c2 axis for the pair (%d,%d) with the support (%g,%g,%g)\n", 
             i, j, support[0], support[1], support[2] ) ;
StatTotal++ ;
/* First, do a quick sanity check */
for( k = 0, ris = rjs = 0 ; k < DIMENSION ; k++ ){
    ris += pow2( Atoms[i].x[k] - support[k] ) ;
    rjs += pow2( Atoms[j].x[k] - support[k] ) ;
    }
ris = sqrt( ris ) ;
rjs = sqrt( rjs ) ;
if( fabs( ris - rjs ) > TolerancePrimary ){
    StatEarly++ ;
    if( verbose > 0 ) printf( "    Support can't actually define a rotation axis\n" ) ;
    return NULL ;
    }
axis                 = alloc_symmetry_element() ;
axis->transform_atom = rotate_atom ;
axis->order          = 2 ;
axis->nparam         = 7 ;
for( k = 0, r = 0 ; k < DIMENSION ; k++ )
    r += CenterOfSomething[k]*CenterOfSomething[k] ;
r = sqrt(r) ;
if( r > 0 ){
    for( k = 0 ; k < DIMENSION ; k++ )
        axis->normal[k] = CenterOfSomething[k]/r ;
    }
else {
    axis->normal[0] = 1 ;
    for( k = 1 ; k < DIMENSION ; k++ )
        axis->normal[k] = 0 ;
    }
axis->distance = r ;
for( k = 0, r = 0 ; k < DIMENSION ; k++ ){
    center[k] = ( Atoms[i].x[k] + Atoms[j].x[k] ) / 2 - support[k] ;
    r        += center[k]*center[k] ;
    }
r = sqrt(r) ;
if( r <= TolerancePrimary ){ /* c2 is underdefined, let's do something special */
    if( MolecularPlane != NULL ){
        if( verbose > 0 ) printf( "    c2 is underdefined, but there is a molecular plane\n" ) ;
        for( k = 0 ; k < DIMENSION ; k++ )
            axis->direction[k] = MolecularPlane->normal[k] ;
        }
    else {
        if( verbose > 0 ) printf( "    c2 is underdefined, trying random direction\n" ) ;
        for( k = 0 ; k < DIMENSION ; k++ )
            center[k] = Atoms[i].x[k] - Atoms[j].x[k] ;
        if( fabs( center[2] ) + fabs( center[1] ) > ToleranceSame ){
            axis->direction[0] =  0 ;
            axis->direction[1] =  center[2] ;
            axis->direction[2] = -center[1] ;
            }
        else {
            axis->direction[0] = -center[2] ;
            axis->direction[1] =  0 ;
            axis->direction[2] =  center[0] ;
            }
        for( k = 0, r = 0 ; k < DIMENSION ; k++ )
            r += axis->direction[k] * axis->direction[k] ;
        r = sqrt(r) ;
        for( k = 0 ; k < DIMENSION ; k++ )
            axis->direction[k] /= r ;
        }
    }
else { /* direction is Ok, renormalize it */
    for( k = 0 ; k < DIMENSION ; k++ )
        axis->direction[k] = center[k]/r ;
    }
if( refine_symmetry_element( axis, 1 ) < 0 ){
    if( verbose > 0 ) printf( "    refinement failed for the c2 axis\n" ) ;
    destroy_symmetry_element( axis ) ;
    return NULL ;
    }
return axis ;
}

SYMMETRY_ELEMENT *init_axis_parameters
                  (gdouble a[3], gdouble b[3], gdouble c[3])
{
        SYMMETRY_ELEMENT * axis ;
        gint                i, order, sign ;
        gdouble             ra, rb, rc, rab, rbc, rac, r ;
        gdouble             angle ;

ra = rb = rc = rab = rbc = rac = 0 ;
for( i = 0 ; i < DIMENSION ; i++ ){
    ra  += a[i]*a[i] ;
    rb  += b[i]*b[i] ;
    rc  += c[i]*c[i] ;
    }
ra = sqrt(ra) ; rb  = sqrt(rb) ; rc  = sqrt(rc) ;
if( fabs( ra - rb ) > TolerancePrimary || fabs( ra - rc ) > TolerancePrimary || fabs( rb - rc ) > TolerancePrimary ){
    StatEarly++ ;
    if( verbose > 0 ) printf( "    points are not on a sphere\n" ) ;
    return NULL ;
    }
for( i = 0 ; i < DIMENSION ; i++ ){
    rab += (a[i]-b[i])*(a[i]-b[i]) ;
    rac += (a[i]-c[i])*(a[i]-c[i]) ;
    rbc += (c[i]-b[i])*(c[i]-b[i]) ;
    }
rab = sqrt(rab) ;
rac = sqrt(rac) ;
rbc = sqrt(rbc) ;
if( fabs( rab - rbc ) > TolerancePrimary ){
    StatEarly++ ;
    if( verbose > 0 ) printf( "    points can't be rotation-equivalent\n" ) ;
    return NULL ;
    }
if( rab <= ToleranceSame || rbc <= ToleranceSame || rac <= ToleranceSame ){
    StatEarly++ ;
    if( verbose > 0 ) printf( "    rotation is underdefined by these points\n" ) ;
    return NULL ;
    }
rab   = (rab+rbc)/2 ;
angle = M_PI - 2*asin( rac/(2*rab) ) ;
if( verbose > 1 ) printf( "    rotation angle is %f\n", angle ) ;
if( fabs(angle) <= M_PI/(MaxAxisOrder+1) ){
    StatEarly++ ;
    if( verbose > 0 ) printf( "    atoms are too close to a straight line\n" ) ;
    return NULL ;
    }
order = floor( (2*M_PI)/angle + 0.5 ) ;
if( order <= 2 || order > MaxAxisOrder ){
    StatEarly++ ;
    if( verbose > 0 ) printf( "    rotation axis order (%d) is not from 3 to %d\n", order, MaxAxisOrder ) ;
    return NULL ;
    }
axis = alloc_symmetry_element() ;
axis->order          = order ;
axis->nparam         = 7 ;
for( i = 0, r = 0 ; i < DIMENSION ; i++ )
    r += CenterOfSomething[i]*CenterOfSomething[i] ;
r = sqrt(r) ;
if( r > 0 ){
    for( i = 0 ; i < DIMENSION ; i++ )
        axis->normal[i] = CenterOfSomething[i]/r ;
    }
else {
    axis->normal[0] = 1 ;
    for( i = 1 ; i < DIMENSION ; i++ )
        axis->normal[i] = 0 ;
    }
axis->distance = r ;
axis->direction[0] = (b[1]-a[1])*(c[2]-b[2]) - (b[2]-a[2])*(c[1]-b[1]) ;
axis->direction[1] = (b[2]-a[2])*(c[0]-b[0]) - (b[0]-a[0])*(c[2]-b[2]) ;
axis->direction[2] = (b[0]-a[0])*(c[1]-b[1]) - (b[1]-a[1])*(c[0]-b[0]) ;
/*
 *  Arbitrarily select axis direction so that first non-zero component
 *  or the direction is positive.
 */
sign = 0 ;
if( axis->direction[0] <= 0 )
{
    if( axis->direction[0] < 0 )
         sign = 1 ;
    else if( axis->direction[1] <= 0 )
{
             if( axis->direction[1] < 0 )
                  sign = 1 ;
             else if( axis->direction[2] < 0 )
                      sign = 1 ;
}
}
if( sign )
    for( i = 0 ; i < DIMENSION ; i++ )
        axis->direction[i] = -axis->direction[i] ;
for( i = 0, r = 0 ; i < DIMENSION ; i++ )
    r += axis->direction[i]*axis->direction[i] ;
r = sqrt(r) ;
for( i = 0 ; i < DIMENSION ; i++ )
    axis->direction[i] /= r ;
if( verbose > 1 ){
    printf( "    axis origin is at (%g,%g,%g)\n", 
        axis->normal[0]*axis->distance, axis->normal[1]*axis->distance, axis->normal[2]*axis->distance ) ;
    printf( "    axis is in the direction (%g,%g,%g)\n", axis->direction[0], axis->direction[1], axis->direction[2] ) ;
    }
return axis ;
}

SYMMETRY_ELEMENT *init_higher_axis( gint ia, gint ib, gint ic )
{
SYMMETRY_ELEMENT * axis ;
gdouble             a[ DIMENSION ], b[ DIMENSION ], c[ DIMENSION ] ;
gint                i ;

if( verbose > 0 ) printf( "Trying cn axis for the triplet (%d,%d,%d)\n", ia, ib, ic ) ;
StatTotal++ ;
/* Do a quick check of geometry validity */
for( i = 0 ; i < DIMENSION ; i++ ){
    a[i] = Atoms[ia].x[i] - CenterOfSomething[i] ;
    b[i] = Atoms[ib].x[i] - CenterOfSomething[i] ;
    c[i] = Atoms[ic].x[i] - CenterOfSomething[i] ;
    }
if( ( axis = init_axis_parameters( a, b, c ) ) == NULL ){
    if( verbose > 0 ) printf( "    no coherrent axis is defined by the points\n" ) ;
    return NULL ;
    }
axis->transform_atom = rotate_atom ;
if( refine_symmetry_element( axis, 1 ) < 0 ){
    if( verbose > 0 ) printf( "    refinement failed for the c%d axis\n", axis->order ) ;
    destroy_symmetry_element( axis ) ;
    return NULL ;
    }
return axis ;
}

/*
 *   Improper axes-specific routines.
 *   These are obtained by slight modifications of normal rotation
 *       routines.
 */
void rotate_reflect_atom( SYMMETRY_ELEMENT *axis, OBJECT *from, OBJECT *to )
{
gdouble             x[3], y[3], a[3], b[3], c[3] ;
gdouble             angle = 2*M_PI/axis->order ;
gdouble             a_sin = sin( angle ) ;
gdouble             a_cos = cos( angle ) ;
gdouble             dot ;
gint                i ;

if( DIMENSION != 3 ){
    fprintf( stderr, "Catastrophe in rotate_reflect_atom!\n" ) ;
    exit( EXIT_FAILURE ) ;
    }
for( i = 0 ; i < 3 ; i++ )
    x[i] = from->x[i] - axis->distance * axis->normal[i] ;
for( i = 0, dot = 0 ; i < 3 ; i++ )
    dot += x[i] * axis->direction[i] ;
for( i = 0 ; i < 3 ; i++ )
    a[i] = axis->direction[i] * dot ;
for( i = 0 ; i < 3 ; i++ )
    b[i] = x[i] - a[i] ;
c[0] = b[1]*axis->direction[2] - b[2]*axis->direction[1] ;
c[1] = b[2]*axis->direction[0] - b[0]*axis->direction[2] ;
c[2] = b[0]*axis->direction[1] - b[1]*axis->direction[0] ;
for( i = 0 ; i < 3 ; i++ )
    y[i] = -a[i] + b[i]*a_cos + c[i]*a_sin ;
for( i = 0 ; i < 3 ; i++ )
    to->x[i] = y[i] + axis->distance * axis->normal[i] ;
to->type = from->type ;
}

SYMMETRY_ELEMENT * init_improper_axis( gint ia, gint ib, gint ic )
{
SYMMETRY_ELEMENT * axis ;
gdouble             a[ DIMENSION ], b[ DIMENSION ], c[ DIMENSION ] ;
gdouble             centerpoint[ DIMENSION ] ;
gdouble             r ;
gint                i ;

if( verbose > 0 ) printf( "Trying sn axis for the triplet (%d,%d,%d)\n", ia, ib, ic ) ;
StatTotal++ ;
/* First, reduce the problem to Cn case */
for( i = 0 ; i < DIMENSION ; i++ ){
    a[i] = Atoms[ia].x[i] - CenterOfSomething[i] ;
    b[i] = Atoms[ib].x[i] - CenterOfSomething[i] ;
    c[i] = Atoms[ic].x[i] - CenterOfSomething[i] ;
    }
for( i = 0, r = 0 ; i < DIMENSION ; i++ ){
    centerpoint[i] = a[i] + c[i] + 2*b[i] ;
    r             += centerpoint[i]*centerpoint[i] ;
    }
r = sqrt(r) ;
if( r <= ToleranceSame ){
    StatEarly++ ;
    if( verbose > 0 ) printf( "    atoms can not define improper axis of the order more than 2\n" ) ;
    return NULL ;
    }
for( i = 0 ; i < DIMENSION ; i++ )
    centerpoint[i] /= r ;
for( i = 0, r = 0 ; i < DIMENSION ; i++ )
    r += centerpoint[i] * b[i] ;
for( i = 0 ; i < DIMENSION ; i++ )
    b[i] = 2*r*centerpoint[i] - b[i] ;
/* Do a quick check of geometry validity */
if( ( axis = init_axis_parameters( a, b, c ) ) == NULL ){
    if( verbose > 0 ) printf( "    no coherrent improper axis is defined by the points\n" ) ;
    return NULL ;
    }
axis->transform_atom = rotate_reflect_atom ;
if( refine_symmetry_element( axis, 1 ) < 0 ){
    if( verbose > 0 ) printf( "    refinement failed for the s%d axis\n", axis->order ) ;
    destroy_symmetry_element( axis ) ;
    return NULL ;
    }
return axis ;
}

/*
 *   Control routines
 */

/****************************************/
/* calculate atom to centroid distances */
/****************************************/
void find_distances(void)
{
gint i;
gdouble r[3];

#if DEBUG
P3VEC("Centroid is ", CenterOfSomething);
#endif

DistanceFromCenter = g_malloc(AtomsCount*sizeof(gdouble));
for (i=0 ; i<AtomsCount ; i++)
  {
  ARR3SET(r, &Atoms[i].x[0]); 
  ARR3SUB(r, CenterOfSomething);
  DistanceFromCenter[i] = VEC3MAGSQ(r);
  } 
}

void find_planes(void)
{
gint i, j;
SYMMETRY_ELEMENT * plane;

plane = init_ultimate_plane() ;
if( plane != NULL ){
    MolecularPlane = plane ;
    PlanesCount++ ;
    Planes = g_realloc(Planes, sizeof(SYMMETRY_ELEMENT *)*PlanesCount);
    Planes[ PlanesCount - 1 ] = plane ;
    }
for( i = 1 ; i < AtomsCount ; i++ ){
    for( j = 0 ; j < i ; j++ ){
        if( Atoms[i].type != Atoms[j].type )
            continue ;
        if( ( plane = init_mirror_plane( i, j ) ) != NULL ){
            PlanesCount++ ;
            Planes = g_realloc(Planes, sizeof(SYMMETRY_ELEMENT *)*PlanesCount);
            Planes[ PlanesCount - 1 ] = plane ;
            }
        }
    }
}

void find_inversion_centers(void)
{
SYMMETRY_ELEMENT *center;

if ((center = init_inversion_center()) != NULL)
  {
  InversionCenters = g_malloc0(sizeof(SYMMETRY_ELEMENT *));
  InversionCenters[0] = center;
  InversionCentersCount = 1;
  }
}

void find_infinity_axis(void)
{
SYMMETRY_ELEMENT *axis;

if ((axis = init_ultimate_axis()) != NULL)
  {
  NormalAxesCount++;
  NormalAxes = g_realloc(NormalAxes, 
                         sizeof(SYMMETRY_ELEMENT *)*NormalAxesCount);
  NormalAxes[NormalAxesCount-1] = axis;
  }
}

void find_c2_axes(void)
{
gint i, j, k, l, m ;
gdouble center[ DIMENSION ] ;
gdouble *distances;
gdouble r;
SYMMETRY_ELEMENT * axis ;

distances = g_malloc0(AtomsCount*sizeof(gdouble));

for( i = 1 ; i < AtomsCount ; i++ ){
    for( j = 0 ; j < i ; j++ ){
        if( Atoms[i].type != Atoms[j].type )
            continue ;
        if( fabs( DistanceFromCenter[i] - DistanceFromCenter[j] ) > TolerancePrimary )
            continue ; /* A very cheap, but quite effective check */
        /*
         *   First, let's try to get it cheap and use CenterOfSomething
         */
        for( k = 0, r = 0 ; k < DIMENSION ; k++ ){
            center[k] = ( Atoms[i].x[k] + Atoms[j].x[k] ) / 2 ;
            r        += pow2( center[k] - CenterOfSomething[k] ) ;
            }
        r = sqrt(r) ;
        if( r > 5*TolerancePrimary ){ /* It's Ok to use CenterOfSomething */
            if( ( axis = init_c2_axis( i, j, CenterOfSomething ) ) != NULL ){
                NormalAxesCount++ ;
                NormalAxes = g_realloc(NormalAxes, sizeof(SYMMETRY_ELEMENT *)
                                       * NormalAxesCount);
                NormalAxes[ NormalAxesCount - 1 ] = axis;
                }
            continue;
            }
        /*
         *  Now, C2 axis can either pass through an atom, or through the
         *  middle of the other pair.
         */
        for( k = 0 ; k < AtomsCount ; k++ ){
            if( ( axis = init_c2_axis( i, j, Atoms[k].x ) ) != NULL ){
                NormalAxesCount++ ;
                NormalAxes = g_realloc(NormalAxes, sizeof(SYMMETRY_ELEMENT *)
                                       * NormalAxesCount);
                NormalAxes[ NormalAxesCount - 1 ] = axis;
                }
            }
        /*
         *  Prepare data for an additional pre-screening check
         */
        for( k = 0 ; k < AtomsCount ; k++ ){
            for( l = 0, r = 0 ; l < DIMENSION ; l++ )
                r += pow2( Atoms[k].x[l] - center[l] ) ;
            distances[k] = sqrt(r) ;
            }
        for( k = 0 ; k < AtomsCount ; k++ ){
            for( l = 0 ; l < AtomsCount ; l++ ){
                if( Atoms[k].type != Atoms[l].type )
                    continue ;
                if( fabs( DistanceFromCenter[k] - DistanceFromCenter[l] ) > TolerancePrimary ||
                    fabs( distances[k] - distances[l] ) > TolerancePrimary )
                        continue ; /* We really need this one to run reasonably fast! */
                for( m = 0 ; m < DIMENSION ; m++ )
                    center[m] = ( Atoms[k].x[m] + Atoms[l].x[m] ) / 2 ;
                if( ( axis = init_c2_axis( i, j, center ) ) != NULL )
                  {
                  NormalAxesCount++ ;
                  NormalAxes = g_realloc(NormalAxes, sizeof(SYMMETRY_ELEMENT * )                                         * NormalAxesCount);
                  NormalAxes[ NormalAxesCount - 1 ] = axis ;
                  }
                }
            }
        }
    }
g_free(distances);
}

void find_higher_axes(void)
{
gint i, j, k ;
SYMMETRY_ELEMENT *axis;

for( i = 0 ; i < AtomsCount ; i++ ){
    for( j = i + 1 ; j < AtomsCount ; j++ ){
        if( Atoms[i].type != Atoms[j].type )
            continue ;
        if( fabs( DistanceFromCenter[i] - DistanceFromCenter[j] ) > TolerancePrimary )
            continue ; /* A very cheap, but quite effective check */
        for( k = 0 ; k < AtomsCount ; k++ ){
            if( Atoms[i].type != Atoms[k].type )
                continue ;
            if( ( fabs( DistanceFromCenter[i] - DistanceFromCenter[k] ) > TolerancePrimary ) ||
                ( fabs( DistanceFromCenter[j] - DistanceFromCenter[k] ) > TolerancePrimary ) )
                    continue ;
            if( ( axis = init_higher_axis( i, j, k ) ) != NULL ){
                NormalAxesCount++ ;
                NormalAxes = g_realloc(NormalAxes, sizeof(SYMMETRY_ELEMENT *)
                                       * NormalAxesCount);
                NormalAxes[ NormalAxesCount - 1 ] = axis ;
                }
            }
        }
    }
}

void find_improper_axes(void)
{
gint                i, j, k ;
SYMMETRY_ELEMENT * axis ;

for( i = 0 ; i < AtomsCount ; i++ ){
  for( j = i + 1 ; j < AtomsCount ; j++ ){
    for( k = 0 ; k < AtomsCount ; k++ ){
      if ((axis = init_improper_axis( i, j, k ) ) != NULL)
        {
        ImproperAxesCount++ ;
        ImproperAxes = g_realloc(ImproperAxes, sizeof(SYMMETRY_ELEMENT *)                                        * ImproperAxesCount);
        ImproperAxes[ ImproperAxesCount - 1 ] = axis ;
        }
      }
    }
  }
}

/*****************/
/* main sequence */
/*****************/
void find_symmetry_elements( void )
{
find_distances();
find_inversion_centers();
find_planes();
find_infinity_axis();
find_c2_axes();
find_higher_axes();
find_improper_axes();
}

gint compare_axes( const void *a, const void *b )
{
SYMMETRY_ELEMENT * axis_a = *(SYMMETRY_ELEMENT**) a ;
SYMMETRY_ELEMENT * axis_b = *(SYMMETRY_ELEMENT**) b ;
gint i, order_a, order_b ;

order_a = axis_a->order ; if( order_a == 0 ) order_a = 10000 ;
order_b = axis_b->order ; if( order_b == 0 ) order_b = 10000 ;
if( ( i = order_b - order_a ) != 0 ) return i ;
if( axis_a->maxdev > axis_b->maxdev ) return -1 ;
if( axis_a->maxdev < axis_b->maxdev ) return  1 ;
return 0 ;
}

void sort_symmetry_elements( void )
{
if( PlanesCount > 1 ){
    qsort( Planes, PlanesCount, sizeof( SYMMETRY_ELEMENT * ), compare_axes ) ;
    }
if( NormalAxesCount > 1 ){
    qsort( NormalAxes, NormalAxesCount, sizeof( SYMMETRY_ELEMENT * ), compare_axes ) ;
    }
if( ImproperAxesCount > 1 ){
    qsort( ImproperAxes, ImproperAxesCount, sizeof( SYMMETRY_ELEMENT * ), compare_axes ) ;
    }
}

void summarize_symmetry_elements( void )
{
gint i;

NormalAxesCounts   = g_malloc0((MaxAxisOrder+1)*sizeof(gint));
ImproperAxesCounts = g_malloc0((MaxAxisOrder+1)*sizeof(gint));

for( i = 0 ; i < NormalAxesCount ; i++ )
    NormalAxesCounts[ NormalAxes[i]->order ]++ ;
for( i = 0 ; i < ImproperAxesCount ; i++ )
    ImproperAxesCounts[ ImproperAxes[i]->order ]++ ;
}

/************************************/
/* store symmetry info in model pak */
/************************************/
#define  DEBUG_UPDATE_SYMMETRY 0
void update_symmetry(struct model_pak *data)
{
gint i, j , k, n, m, p;
GString *code;
struct symop_pak *symops;

/* delete all existing entries */
gtk_clist_freeze((GtkCList *) data->symmetry.summary);
gtk_clist_clear((GtkCList *) data->symmetry.summary);
/* free the list items */
g_strfreev(data->symmetry.items);

n = PlanesCount + NormalAxesCount + ImproperAxesCount + InversionCentersCount;

#if DEBUG_UPDATE_SYMMETRY
printf("Model has %d symmetry elements:\n", n);
printf("%d Planes,\n", PlanesCount);
printf("%d Rotation axes,\n", NormalAxesCount);
printf("%d Improper rotation axes,\n", ImproperAxesCount);
printf("%d Inversion centers.\n", InversionCentersCount);
#endif

/* nothing found? */
if (!n)
  {
  data->symmetry.num_items = 1;
  data->symmetry.items = g_malloc(2*sizeof(gchar *));
  *(data->symmetry.items) = g_strdup("none");
  *(data->symmetry.items+1) = NULL;
  return;
  }

/* otherwise continue */
g_free((data->symmetry.symops));
data->symmetry.symops = g_malloc(n * sizeof(struct symop_pak));
/* this probably will overallocate (but ensures space for NULL termination) */
data->symmetry.items = g_malloc((n+1) * sizeof(gchar *));

code = g_string_new(NULL);

/* fill out with operations found */
/* j follows number of symop elements/matrices */
symops = data->symmetry.symops;
j = k = 0;
if (InversionCentersCount) 
  {
  g_string_sprintfa(code, "(i) " ) ;
  *((data->symmetry.items)+k++) = g_strdup("Inversion centre");
/* store the symmetry operation */
  (symops+j)->type = INVERSION;
  VEC3SET(&((symops+j)->matrix[0]),-1.0, 0.0, 0.0);
  VEC3SET(&((symops+j)->matrix[3]), 0.0,-1.0, 0.0);
  VEC3SET(&((symops+j)->matrix[6]), 0.0, 0.0,-1.0);
  j++;
  }

/* infinite axes */
if (NormalAxesCounts[0])
  {
  *((data->symmetry.items)+k++) = g_strdup_printf("%d infinite rotation axes",
                                                   NormalAxesCounts[0]);
  g_string_sprintfa(code, "%d*(Cinf) ", NormalAxesCounts[0] ) ;
/*
  j += NormalAxesCounts[0];
*/
  }

p=0;
for (i=MaxAxisOrder ; i>=2 ; i--)
  {
  switch (NormalAxesCounts[i])
    {
    case 0:
      continue;
    case 1:
      g_string_sprintfa(code, "(C%d) ", i );
      break;
    default:
      g_string_sprintfa(code, "%d*(C%d) ", NormalAxesCounts[i], i );
    }
  *((data->symmetry.items)+k++) = g_strdup_printf("%d order %d rotation axes",
                                                    NormalAxesCounts[i],i);
/* store the symmetry operation */
  for (m=0 ; m<NormalAxesCounts[i] ; m++)
    {
    (symops+j)->type = PAXIS;
    get_matrix((symops+j)->matrix, PAXIS, &(NormalAxes[p++]->direction[0]), i);
    j++;
    }
  }

for (i=MaxAxisOrder ; i>=2 ; i--)
  {
/*
  j += ImproperAxesCounts[i];
*/
  switch (ImproperAxesCounts[i])
    {
    case 0:
      continue;
    case 1:
      g_string_sprintfa(code, "(S%d) ", i);
      break;
    default:
      g_string_sprintfa(code, "%d*(S%d) ", ImproperAxesCounts[i], i);
    }
  *((data->symmetry.items)+k++) = g_strdup_printf("%d order %d improper axes",
                                                    ImproperAxesCounts[i],i);
  }

p=0;
switch (PlanesCount) 
  {
  case 0:
    break;
  case 1:
    g_string_sprintfa(code, "(sigma) ");
    *((data->symmetry.items)+k++) = g_strdup_printf("reflection plane");
    break;
  default:
    g_string_sprintfa(code, "%d*(sigma) ", PlanesCount);
    *((data->symmetry.items)+k++) = g_strdup_printf("%d reflection planes",
                                                             PlanesCount);
/* store the symmetry operation */
  for (m=0 ; m<PlanesCount ; m++)
    {
/* NIY */
/* FIXME - infreq. core dump on line below */
/* symops alloc seemed ok, perhaps NormalAxes is not what we want? */
/*
    get_matrix((symops+j)->matrix, PLANE, &(NormalAxes[p++]->direction[0]), i);
    (symops+j)->type = PLANE;
    j++;
*/
    }
  }
#if DEBUG_UPDATE_SYMMETRY
printf("symmetry code: {%s}\n", code->str) ;
printf("Filled out %d items (allocated for %d)\n",j,n);
for (m=0 ; m<j ; m++)
  {
  P3MAT("symop ",(symops+m)->matrix);
  }
#endif
SymmetryCode = g_strdup(code->str);

/* terminate, so that g_strfreev works */
data->symmetry.num_symops = j;
*((data->symmetry.items)+k++) = NULL;
data->symmetry.num_items = k;

/* update */
gtk_clist_thaw((GtkCList *) data->symmetry.summary);

g_string_free(code, TRUE);
}

void identify_point_group(struct model_pak *data)
{
gint i;
gint last_matching = -1;
gint matching_count = 0;

for (i=0 ; i<PointGroupsCount ; i++ )
  {
  if (g_strcasecmp(SymmetryCode, PointGroups[i].symmetry_code) == 0)
    {
    if (PointGroups[i].check() == 1 )
      {
      last_matching = i ;
      matching_count++ ;
      }
#if DEBUG
    else
      printf("It looks very much like %s, but it is not since %s\n", 
              PointGroups[i].group_name, PointGroupRejectionReason);
#endif
    }
  }

#if DEBUG
switch(matching_count)
  {
  case 0:
    printf("No matching point group.\n");
    break;
  case 1:
    printf("Apparent point group: %s\n",PointGroups[last_matching].group_name);
    break;
  default:
    printf("Error in point group lookup.\n");
    return;
  }
#endif

/* update dialog */
if (PointGroups[last_matching].group_name != NULL)
  {
  g_free(data->symmetry.pg_name);
  data->symmetry.pg_name = g_strdup(PointGroups[last_matching].group_name);
  }
else
  {
  g_free(data->symmetry.pg_name);
  data->symmetry.pg_name = g_strdup("unknown");
  }
}

/*
 *  Input/Output
 */
gint get_coords(struct model_pak *data)
{
gint i;

Atoms = g_malloc0( data->num_atoms* sizeof( OBJECT ) ) ;

/* read 'em in */
AtomsCount=0;
for (i=0 ; i<data->num_atoms ; i++)
  {
  if ((data->atoms+i)->status & (DELETED | HIDDEN))
    continue;
  Atoms[i].type = (data->atoms+i)->atom_code;
  Atoms[i].x[0] = (data->atoms+i)->x;
  Atoms[i].x[1] = (data->atoms+i)->y;
  Atoms[i].x[2] = (data->atoms+i)->z;
  AtomsCount++;
  }

/* transfer centroid */
ARR3SET(CenterOfSomething, data->centroid);

return(0);
}

/*************/
/* main call */
/*************/
void seek_sym(struct model_pak *data)
{
/* checks */
g_return_if_fail(data != NULL);
if (!data->num_atoms)
  return;

/* init pointers, so we don't free something that wasn't found & alloc'd */
DistanceFromCenter = NULL;
InversionCenters = NULL;
SymmetryCode = NULL;

/* convert formats */
if (get_coords(data))
  {
  printf("Error reading in atomic coordinates\n");
  return;
  }

PlanesCount = NormalAxesCount = ImproperAxesCount = InversionCentersCount = 0;

/* FIXME - globals need init, as successive calls overlap */
find_symmetry_elements();
sort_symmetry_elements();
summarize_symmetry_elements();
#if DEBUG
if (BadOptimization)
  {
  printf("Refinement of some symmetry elements was terminated before"
         "convergence was reached.\n Some symmetry elements may remain"
         "unidentified.\n") ;
  }
#endif
update_symmetry(data);
/* if 0 sym elements, can identify wrongly for some reason */
if (data->symmetry.num_symops)
  identify_point_group(data);

/* free if allocated */
if (DistanceFromCenter)
  g_free(DistanceFromCenter);
if (SymmetryCode)
  g_free(SymmetryCode);
if (InversionCentersCount)
  g_free(InversionCenters);

/* update dialog */
update_symlist(data);
}

void update_symlist(struct model_pak *data)
{
gint i;
gchar *info[1];

/* symmetry elements */
gtk_clist_freeze((GtkCList *) data->symmetry.summary);
gtk_clist_clear((GtkCList *) data->symmetry.summary);

for (i=0 ; i<data->symmetry.num_items ; i++)
  {
  info[0] = *((data->symmetry.items)+i);
  gtk_clist_append((GtkCList *) data->symmetry.summary, &info[0]);
  }
gtk_clist_thaw((GtkCList *) data->symmetry.summary);

/* point group */
gtk_label_set_text(GTK_LABEL(data->symmetry.pg_entry),data->symmetry.pg_name);
return;
}
/*******************************/
/* verify a symmetry operation */
/*******************************/
#define DEBUG_DELETE_SYMOP 0
gint delete_symop(struct model_pak *data, gint num)
{
gint i, j, flag, cycle;
gfloat *mat, vec[3], dx, dy, dz, r2;

/* get the matrix */
mat = (data->symmetry.symops+num)->matrix;

#if DEBUG_DELETE_SYMOP
P3MAT("",mat);
#endif

for (i=0 ; i<data->num_atoms ; i++)
  {
/* only operate on original/non-deleted atoms */
/*
  if ((data->atoms+i)->orig == NO || (data->atoms+i)->status & DELETED)
    continue;
*/
  if ((data->atoms+i)->status & HIDDEN)
    continue;

/* no need for latmat? (ie isolated molecules only) */
  vec[0] = (data->atoms+i)->x - data->centroid[0];
  vec[1] = (data->atoms+i)->y - data->centroid[1];
  vec[2] = (data->atoms+i)->z - data->centroid[2];
/* do rotation */
/* multiple applications (until 1st arrive at start again) */
  cycle=0;
  while(!cycle)
    {
    vecmat(mat, vec);

#if DEBUG_DELETE_SYMOP
printf("[%d] [%f %f %f]\n",i,vec[0],vec[1],vec[2]);
#endif

/* which atom of the same type (if any) does it generate? */
    flag=0;
    for (j=0 ; j<data->num_atoms ; j++)
      {
/* only if same type */
      if ((data->atoms+i)->atom_code != (data->atoms+j)->atom_code)
        continue;
#if DEBUG_DELETE_SYMOP
printf("(%d) (%f %f %f)\n", j, (data->atoms+j)->x,
           (data->atoms+j)->y, (data->atoms+j)->z);
#endif
/* attempt to match rotated i with some j */
      dx = vec[0] - (data->atoms+j)->x + data->centroid[0];
      dy = vec[1] - (data->atoms+j)->y + data->centroid[1];
      dz = vec[2] - (data->atoms+j)->z + data->centroid[2];
      r2 = dx*dx + dy*dy + dz*dz;
      if (r2 < POSITION_TOLERANCE)
        {
#if DEBUG_DELETE_SYMOP
printf("[%d : %d]\n",i,j);
#endif
/* we've found at least 1 atom this sym op produces */
        flag++;
/* special case where i = j, don't delete j (as i is primary) */
        if (i != j)
          {
/* mark as non unique */
/*
          (data->atoms+j)->primary = NO;
*/
          (data->atoms+j)->status |= HIDDEN;
          }
        else
          cycle=1;
/* found a match, so we can stop searching */
        break;
        }
      }
/* atom generates nothing -> sym op candidate failed */
    if (!flag)
      return(1);
    }
  }

return(0);
}

/**********************************/
/* reduce to asymetric components */
/**********************************/
gint unique_toggle(struct model_pak *data)
{
gint i;

/* toggle */
data->asym_on ^= 1;
if (data->asym_on)
  {
/* permanently remove deleted atoms */
  mem_shrink(data);
/* cycle through */
  for (i=data->symmetry.num_symops ; i-- ; )
    delete_symop(data, i);
  }
else
  {
  for (i=data->num_atoms ; i-- ; )
    (data->atoms+i)->status &= ~HIDDEN;
  }

/* update - (only if atoms were hidden/deleted) */
init_objs(REDO_COORDS, data);
calc_bonds(data);
calc_mols(data);

redraw_canvas(SINGLE);
return(FALSE);
}

/*******************/
/* symmetry dialog */
/*******************/
/* TODO - periodicity counterpart, ie sym elements/pt group/asym etc. */
/* => when model is loaded, if not periodic -> run the above code */
/* might have problems with editing/minimization resulting in diff */
/* symmetry than prev calc'd (especially when removing sym elements) */
void sym_info(GtkWidget *w, struct model_pak *data)
{
gint id;
GtkWidget *frame, *mainvbox, *vbox, *hbox, *label, *button;
struct dialog_pak *sym;

g_return_if_fail(data != NULL);

/* retrieve a suitable dialog */
if ((id = request_dialog(data->number, SYMMETRY)) < 0)
  return;
sym = &sysenv.dialog[id];

sym->win = gtk_dialog_new();
gtk_window_set_title (GTK_WINDOW (sym->win), "Symmetry analyzer");
gtk_signal_connect(GTK_OBJECT(sym->win), "destroy",
                   GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);
/* FIXME - why is the width value being ignored??? */
gtk_window_set_default_size(GTK_WINDOW(sym->win), 250, 250);

/* main dialog vbox */
mainvbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(GTK_BOX(GTK_DIALOG(sym->win)->vbox)),mainvbox);
gtk_container_set_border_width (GTK_CONTAINER(mainvbox), 3);

/* model name */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(mainvbox),frame,FALSE,FALSE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
hbox = gtk_hbox_new(FALSE,5);
gtk_container_add(GTK_CONTAINER(frame), hbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(hbox)), 3);

label = gtk_label_new(g_strdup_printf("Model: %s",data->basename));
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

/* main list */
frame = gtk_frame_new("Symmetry elements");
gtk_box_pack_start(GTK_BOX(mainvbox),frame,TRUE,TRUE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
/* create a vbox in the frame */
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width (GTK_CONTAINER(GTK_BOX(vbox)), 3);
/* scrolled model pane */
/*
scr_win = gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scr_win),
                               GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(vbox), scr_win, TRUE, TRUE, 0);
*/

/* create the CList */
data->symmetry.summary = gtk_clist_new(1);
/* events */
/*
gtk_signal_connect(GTK_OBJECT(label_list), "select_row",
                   GTK_SIGNAL_FUNC(select_label), NULL);
gtk_signal_connect(GTK_OBJECT(label_list), "unselect_row",
                   GTK_SIGNAL_FUNC(unselect_label), NULL);
gtk_container_add(GTK_CONTAINER(scr_win), data->symmetry.summary);
*/
gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(data->symmetry.summary), TRUE, TRUE, 0);

/* guessed point group */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(mainvbox),frame,FALSE,FALSE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
hbox = gtk_hbox_new(FALSE,5);
gtk_container_add(GTK_CONTAINER(frame), hbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(hbox)), 3);

label = gtk_label_new("Apparent point group: ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

/* make new, if none exists */
data->symmetry.pg_entry = gtk_label_new(data->symmetry.pg_name);
gtk_box_pack_start(GTK_BOX(hbox), data->symmetry.pg_entry, FALSE, FALSE, 0);

/* terminating buttons */
hbox = gtk_hbox_new(TRUE, 5);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(sym->win)->action_area),
                   hbox, TRUE, TRUE, 0);

button = gtk_button_new_with_label (" Analyze ");
gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                          GTK_SIGNAL_FUNC (seek_sym),
                         (gpointer) data);

button = gtk_button_new_with_label (" Condense ");
gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                          GTK_SIGNAL_FUNC (unique_toggle),
                         (gpointer) data);

button = gtk_button_new_with_label (" Close ");
gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (close_dialog),
                           (gpointer) id);

/* done, draw and exit */
update_symlist(data);
gtk_widget_show_all(sym->win);
pick_model(data->number);
}

