/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <sys/wait.h>

#include "gdis.h"

#define DEBUG 0

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/* global render parameters */
struct povray_pak povray_data;

/************************************/
/* get an unused name for rendering */
/************************************/
#define DEBUG_GUN 0
gchar *gun(void)
{
gint i;
gchar *basename;
GString *filename;

/* seek a file name that doesn't exist (avoid background overwriting) */
filename = g_string_new(NULL);
i=0;
do
  {
  g_string_sprintf(filename,"dummy_%d.pov",i);
#if DEBUG_GUN
printf("testing: %s\n",filename->str);
#endif
  i++;
  }
while(!access(filename->str, F_OK));

/* basename for .pov & .tga */
g_string_sprintf(filename,"dummy_%d",i-1);
basename = g_strdup(filename->str);
g_string_free(filename, TRUE);

#if DEBUG_GUN
printf("using base: %s\n",basename);
#endif
return(basename);
}

/****************/
/* render setup */
/****************/
void render_setup(GtkWidget *w)
{
gint i, last_flag;
gchar *file, *basename;
GString *cmd;
struct model_pak *data;

if (!sysenv.have_povray)
  {
  show_text("Sorry, povray executable was not found.");
  return;
  }

cmd = g_string_new(NULL);

/* TODO - progress widget? */
if (povray_data.animate)
  {
/* remove any extension */
  file = strdup_no_extension(povray_data.filename);
  if (!strlen(file))
    {
    show_text("Please enter a filename");
    return;
    }

/* render active model */
  data = model_ptr(sysenv.active, RECALL);
  g_return_if_fail(data != NULL);

/* animation loop */
  last_flag = get_frame(data, 1);
  i=0;
  while(!last_flag)
    {
    if (data->periodic)
      {
/* refresh the cell (in case it changed) */
      make_cell(data);
/* TODO - PBC applied only if desired ie if periodic AND constrain on */
      pbc_constrain_mols(data);
      init_objs(REDO_COORDS, data);
      calc_bonds(data);
      calc_mols(data);
      }
/* go */
    basename = gun();
    make_pov(sysenv.active, basename);
    exec_pov(basename);
/* make into sequence (pre pad with 0's to keep list sequential) */
/* FIXME - determine max width & construct format using num_frames */
    g_string_sprintf(cmd,"mv -f %s.tga %s_%03d.tga",basename,file,i);
    system(cmd->str);

/* get next frame */
    if (get_frame(data,0))
      last_flag = 1;
    i++;
    }

/* TODO - checks for required packages? */
/* setup for conversion to anim GIF (or whatever) */
  switch(povray_data.atype)
    {
    case ANIM_GIF:
      g_string_sprintf(cmd,"%s -delay %d %s_*.tga %s.gif",
                       sysenv.convert_exe, povray_data.delay, file, file);
      break;

    case ANIM_MPEG:
/* TODO - need mpeg2encode to make this work */
/* adding a '-quality xx' option doesn't seem to help */
      g_string_sprintf(cmd,"%s -delay %d %s_*.tga %s.mpg",
                       sysenv.convert_exe, povray_data.delay, file, file);
      break;

    default:
      printf("Error in render_setup(): this shouldn't happen\n");
      return;
    }
/* execute the conversion command */
  system(cmd->str);
/* clean up */
  g_free(file);
  g_string_free(cmd, TRUE);
  }
else
  {
/* single model */
  show_text("Rendering in the background, please wait.");
/*
  basename = gun();
  make_pov(sysenv.active, basename);
  bg_pov(basename);
*/
  run_pov();
  }
/*
  pthread_create(&thread1, NULL, (void *) &bg_pov, (void*) NULL);
*/
}

/****************/
/* unified hook */
/****************/
gint event_render_modify(GtkWidget *w, gpointer *obj)
{
gint id, len;
gchar *entry;
gpointer frame, button;
GtkStyle *style;
struct model_pak *data;

/* checks */
g_return_val_if_fail(obj != NULL, FALSE);

/* ascertain type of modification required */
id = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(obj), "id"));

data = model_ptr(sysenv.active, RECALL);

switch(id)
  {
/* spinners */
  case BALL_RADIUS:
    povray_data.arad = SPIN_FVAL(GTK_SPIN_BUTTON(obj));
    sysenv.render.ball_rad = SPIN_FVAL(GTK_SPIN_BUTTON(obj));
    break;
  case STICK_RADIUS:
    povray_data.brad = SPIN_FVAL(GTK_SPIN_BUTTON(obj));
    sysenv.render.stick_rad = SPIN_FVAL(GTK_SPIN_BUTTON(obj));
    break;
  case CPK_SCALE:
    povray_data.cpk_scale = SPIN_FVAL(GTK_SPIN_BUTTON(obj));
    sysenv.render.cpk_scale = SPIN_FVAL(GTK_SPIN_BUTTON(obj));
    break;
  case FRAME_RADIUS:
    povray_data.frad = SPIN_FVAL(GTK_SPIN_BUTTON(obj));
    sysenv.render.frame_thickness = SPIN_FVAL(GTK_SPIN_BUTTON(obj))/PIX2ANG;
    break;
  case WIDTH:
    povray_data.width = SPIN_IVAL(GTK_SPIN_BUTTON(obj));
    sysenv.render.width = SPIN_FVAL(GTK_SPIN_BUTTON(obj));
    break;
  case HEIGHT:
    povray_data.height = SPIN_IVAL(GTK_SPIN_BUTTON(obj));
    sysenv.render.height = SPIN_FVAL(GTK_SPIN_BUTTON(obj));
    break;
  case DELAY:
    povray_data.delay = SPIN_IVAL(GTK_SPIN_BUTTON(obj));
    break;
  case VP_DIST:
    sysenv.render.vp_dist = SPIN_IVAL(GTK_SPIN_BUTTON(obj));
    break;

/* buttons */
  case STICK:
  case BALL_STICK:
  case CPK:
    povray_data.render_type = id;
    sysenv.render.type = id;
    break;
/* check boxes */
  case ANTIALIAS:
    povray_data.antialias ^= 1;
    sysenv.render.antialias ^= 1;
    break;
  case SHADOWLESS:
    povray_data.shadowless ^= 1;
    sysenv.render.shadowless ^= 1;
    break;
  case BACKGROUND:
    povray_data.background ^= 1;
    break;
  case DRAW_AXES:
    povray_data.axes ^= 1;
    break;
  case ANIMATE:
    povray_data.animate ^= 1;
/* hide gory details, until animate-to-file selected */
    frame = gtk_object_get_data(GTK_OBJECT(obj), "ptr");
    if (povray_data.animate)
      gtk_widget_show(GTK_WIDGET(frame));
    else
      gtk_widget_hide(GTK_WIDGET(frame));
    break;
  case ANIM_GIF:
    povray_data.atype = ANIM_GIF;
    break;
  case ANIM_MPEG:
    povray_data.atype = ANIM_MPEG;
    break;
  case ANIM_NAME:
/* retrieve current contents of the entry */
    entry = (gchar *) gtk_entry_get_text(GTK_ENTRY(obj));
/* entry has been confined to be < FILELEN, so we should be safe */
    len = strlen(entry);
/* plus one for the termination character */
    g_snprintf(povray_data.filename,len+1,"%s",entry);
    break;
/*
  case LIGHT_LEVEL:
    povray_data.ambience = AMBIENCE_MAX * GTK_ADJUSTMENT(obj)->value;
    break;
*/
  case LIGHT_AMBIENT:
    sysenv.render.ambience = GTK_ADJUSTMENT(obj)->value;
    break;
  case LIGHT_DIFFUSE:
    sysenv.render.diffuse = GTK_ADJUSTMENT(obj)->value;
    break;
  case LIGHT_SPECULAR:
    sysenv.render.specular = GTK_ADJUSTMENT(obj)->value;
    break;

/* morphology */
  case MORPH_FINISH:
/* retrieve current contents of the entry */
    entry = (gchar *) gtk_entry_get_text(GTK_ENTRY(obj));
/* entry has been confined to be < FILELEN, so we should be safe */
    len = strlen(entry);
/* plus one for the termination character */
    g_snprintf(povray_data.morph_finish,len+1,"%s",entry);
    break;

  case MORPH_STYLE:
    povray_data.wire_frame ^= 1;
    frame = gtk_object_get_data(GTK_OBJECT(obj), "ptr");
    if (povray_data.wire_frame)
      gtk_widget_hide(GTK_WIDGET(frame));
    else
      gtk_widget_show(GTK_WIDGET(frame));
    break;

  case MORPH_COLOUR:
    gtk_color_selection_get_color((GtkColorSelection *) obj, 
                                  povray_data.morph_colour);
    button = gtk_object_get_data(GTK_OBJECT(obj), "ptr");
    style = gtk_style_copy(gtk_widget_get_style(w));
    style->bg[0].red   = povray_data.morph_colour[0] * 65535;
    style->bg[0].green = povray_data.morph_colour[1] * 65535; 
    style->bg[0].blue  = povray_data.morph_colour[2] * 65535; 
/* only update if widget exists */
    if (GTK_IS_WIDGET(button))
      gtk_widget_set_style(GTK_WIDGET(button), style);
/* hack to change opengl morph colour */
    ARR3SET(sysenv.render.morph_colour, povray_data.morph_colour);
    break;
  case REF_INDEX:
    povray_data.ref_index = SPIN_FVAL(GTK_SPIN_BUTTON(obj));
    break;
  case TRANSMIT:
    povray_data.transmit = SPIN_FVAL(GTK_SPIN_BUTTON(obj));
    break;
  }

/* NEW - update drawing area (also triggers an OpenGL update) */
redraw_canvas(SINGLE);

return(FALSE);
}

void scale_set_default_values( GtkScale *scale )
{
gtk_range_set_update_policy(GTK_RANGE(scale), GTK_UPDATE_CONTINUOUS);
gtk_scale_set_digits (scale, 1);
gtk_scale_set_value_pos (scale, GTK_POS_TOP);
gtk_scale_set_draw_value (scale, TRUE);
}

/***********************/
/* render setup widget */
/***********************/
void render_setup_widget()
{
gint i, id;
GtkWidget *vbox1, *vbox2, *vbox, *hbox, *frame, *a_frame=NULL;
GtkWidget *label, *button, *spin, *file_entry, *entry;
GtkAdjustment *adj;
GtkStyle *style;
GSList *group;
struct model_pak *data;
struct dialog_pak *rsd;
struct callback_pak dimensions[] = {
{"Ball radius", BALL_RADIUS},
{"Stick radius", STICK_RADIUS},
{"CPK scaling", CPK_SCALE},
{"Frame radius", FRAME_RADIUS},
{"Vanishing point", VP_DIST},
{NULL,0}
};

/* do we have to do anything? */
if (!sysenv.num_models)
  return;

/* request dialog slot */
if ((id = request_dialog(sysenv.active, POVRAY)) < 0)
  return;
rsd = &sysenv.dialog[id];
rsd->win = gtk_dialog_new();
gtk_signal_connect(GTK_OBJECT (rsd->win), "destroy",
                   GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);
gtk_window_set_title(GTK_WINDOW(rsd->win), "POVRay/OpenGL setup");

data = model_ptr(sysenv.active, RECALL);

/* defaults */
povray_data.width = 400;
povray_data.height = 400;
povray_data.delay = 20;
povray_data.arad = 0.22;
povray_data.brad = 0.05;
povray_data.frad = 0.05;
povray_data.ambience = 0.0125;
povray_data.cpk_scale = 1.0;
povray_data.render_type = STICK;
povray_data.antialias = 0;
povray_data.shadowless = 0;
povray_data.background = 0;
povray_data.axes = 0;
povray_data.animate = 0;
povray_data.atype = ANIM_GIF;
g_snprintf(povray_data.filename,6,"movie");
povray_data.ref_index = 2.5;
g_snprintf(povray_data.morph_finish,9,"F_Glass4");
povray_data.morph_colour[0] = 1.00;
povray_data.morph_colour[1] = 0.00;
povray_data.morph_colour[2] = 0.00;
povray_data.transmit = 0.9;
povray_data.wire_frame = 0;

/* main hbox */
hbox = gtk_hbox_new(FALSE,5);
gtk_container_add(GTK_CONTAINER(GTK_DIALOG(rsd->win)->vbox),hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox),3);

/* LEFT & RIGHT PANES */
vbox1 = gtk_vbox_new(FALSE,3);
gtk_box_pack_start (GTK_BOX (hbox), vbox1, TRUE, TRUE, 0);
vbox2 = gtk_vbox_new(FALSE,3);
gtk_box_pack_start (GTK_BOX (hbox), vbox2, TRUE, TRUE, 0);

/* LEFT PANE */
/* Frame 1 */
frame = gtk_frame_new("Display mode");
gtk_box_pack_start(GTK_BOX (vbox1), frame, TRUE, TRUE, 0);

/* create a vbox in the frame */
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add (GTK_CONTAINER(frame),vbox);
/* do the first radio button in the vbox */
button = gtk_radio_button_new_with_label (NULL, "Stick");
gtk_signal_connect_object(GTK_OBJECT (button), "pressed",
                          GTK_SIGNAL_FUNC (event_render_modify),
                         (gpointer) button);
gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) STICK);
gtk_widget_show(button);
if (sysenv.render.type == STICK)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
 

/* make a radio group */
group = gtk_radio_button_group(GTK_RADIO_BUTTON(button));
/* do the rest of the buttons */
button = gtk_radio_button_new_with_label(group, "Ball & stick");
gtk_signal_connect_object(GTK_OBJECT(button), "pressed",
                          GTK_SIGNAL_FUNC(event_render_modify),
                          (gpointer) button);
gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) BALL_STICK);
gtk_widget_show(button);
if (sysenv.render.type == BALL_STICK)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* last button */
button = gtk_radio_button_new_with_label(
                      gtk_radio_button_group(GTK_RADIO_BUTTON(button)),
                      "CPK");
gtk_signal_connect_object(GTK_OBJECT(button), "pressed",
                          GTK_SIGNAL_FUNC (event_render_modify),
                          (gpointer) button);
gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) CPK);
gtk_widget_show(button);
if (sysenv.render.type == CPK)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* Frame 2 */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX (vbox1), frame, TRUE, TRUE, 0);

/* create a vbox in the frame */
vbox = gtk_vbox_new (FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);

/* Antialias toggle */
button = gtk_check_button_new_with_label ("Antialias");
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (event_render_modify), NULL);
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) ANTIALIAS);

/* Shadowless toggle */
button = gtk_check_button_new_with_label ("Shadowless");
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (event_render_modify), NULL);
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) SHADOWLESS);

/* White background toggle */
button = gtk_check_button_new_with_label ("White background");
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (event_render_modify), NULL);
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) BACKGROUND);

/* FIXME - axes display toggle */
/*
button = gtk_check_button_new_with_label ("Draw axes");
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (event_render_modify), NULL);
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) DRAW_AXES);
*/

/* FRAME image size */
frame = gtk_frame_new("Image size");
gtk_box_pack_start(GTK_BOX(vbox1), frame, TRUE, TRUE, 0);

/* create a vbox in the frame */
vbox = gtk_vbox_new (FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)),1);

/* width spinner */
hbox = gtk_hbox_new (FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
/* label */
label = gtk_label_new(" width ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);

adj = (GtkAdjustment *) gtk_adjustment_new
      (povray_data.width, 100, 2000, 100, 200, 0);
spin = gtk_spin_button_new (adj, 0, 0);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
gtk_box_pack_end(GTK_BOX (hbox), spin, FALSE, FALSE, 0);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(event_render_modify), (gpointer) spin);
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) WIDTH);

/* height spinner */
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(vbox), hbox);
/* label */
label = gtk_label_new(" height ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);

adj = (GtkAdjustment *) gtk_adjustment_new
      (povray_data.height, 100, 2000, 100, 200, 0);
spin = gtk_spin_button_new (adj, 0, 0);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON (spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON (spin), GTK_SHADOW_OUT);
gtk_box_pack_end(GTK_BOX (hbox), spin, FALSE, FALSE, 0);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(event_render_modify), (gpointer) spin);
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) HEIGHT);

/* RIGHT PANE */
/* FRAME - dimensions */
frame = gtk_frame_new("Dimensions");
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame),vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)),1);

/* NEW - for loop to cut down on repeated code */
i=0;
while(dimensions[i].label)
  {
  hbox = gtk_hbox_new(FALSE, 10);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
  label = gtk_label_new(dimensions[i].label);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

  switch(dimensions[i].id)
    {
    case BALL_RADIUS:
      adj = (GtkAdjustment *) gtk_adjustment_new(povray_data.arad, 0.1, 0.5, 0.02, 0.04, 0);
      break;
    case STICK_RADIUS:
      adj = (GtkAdjustment *) gtk_adjustment_new(povray_data.brad, 0.02, 0.2, 0.01, 0.02, 0);
      break;
    case CPK_SCALE:
      adj = (GtkAdjustment *) gtk_adjustment_new(povray_data.cpk_scale, 0.1, 3.0, 0.1, 0.1, 0);
      break;
    case FRAME_RADIUS:
      adj = (GtkAdjustment *) gtk_adjustment_new(povray_data.frad, 0.01, 0.3, 0.01, 0.01, 0);
      break;
    case VP_DIST:
      adj = (GtkAdjustment *) gtk_adjustment_new(sysenv.render.vp_dist, 1.0, 10.0, 1.0, 1.0, 0);
      break;
    }
  spin = gtk_spin_button_new(adj, 0.01, 2);

  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
  gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin),GTK_SHADOW_OUT);
  gtk_box_pack_end(GTK_BOX(hbox), spin, FALSE, TRUE, 0);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                     GTK_SIGNAL_FUNC(event_render_modify), (gpointer) spin);
  gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) dimensions[i].id);
  hbox = gtk_hbox_new (FALSE, 0);

  i++;
  }

/* FRAME - dimensions */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)),1);

/* create a hbox for ambient light */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
label = gtk_label_new("Ambient light");
gtk_box_pack_start(GTK_BOX (hbox), label, FALSE, FALSE, 0);

adj = (GtkAdjustment *) gtk_adjustment_new
      (sysenv.render.ambience, 0.0, 1.0, 0.1, 0.1, 0.0);

spin = gtk_spin_button_new (adj, 0.1, 1);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin),GTK_SHADOW_OUT);

gtk_box_pack_end(GTK_BOX(hbox), spin, FALSE, TRUE, 0);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(event_render_modify), (gpointer) adj);
gtk_object_set_data(GTK_OBJECT(adj), "id", (gpointer) LIGHT_AMBIENT);

/* diffuse component */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
label = gtk_label_new("Diffuse light");
gtk_box_pack_start(GTK_BOX (hbox), label, FALSE, FALSE, 0);

adj = (GtkAdjustment *) gtk_adjustment_new
      (sysenv.render.diffuse, 0.0, 1.0, 0.1, 0.1, 0.0);

spin = gtk_spin_button_new (adj, 0.1, 1);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin),GTK_SHADOW_OUT);

gtk_box_pack_end(GTK_BOX(hbox), spin, FALSE, TRUE, 0);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(event_render_modify), (gpointer) adj);
gtk_object_set_data(GTK_OBJECT(adj), "id", (gpointer) LIGHT_DIFFUSE);

/* specular component */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
label = gtk_label_new("Specular light");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

adj = (GtkAdjustment *) gtk_adjustment_new
      (sysenv.render.specular, 0.0, 1.0, 0.1, 0.1, 0.0);

spin = gtk_spin_button_new (adj, 0.1, 1);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin),GTK_SHADOW_OUT);

gtk_box_pack_end(GTK_BOX(hbox), spin, FALSE, TRUE, 0);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(event_render_modify), (gpointer) adj);
gtk_object_set_data(GTK_OBJECT(adj), "id", (gpointer) LIGHT_SPECULAR);


/* Animation */
if (data->num_frames > 1)
  {
/* Frame 5 - filename/animation rendering */
  frame = gtk_frame_new("File only");
  gtk_box_pack_start(GTK_BOX(vbox2), frame, TRUE, TRUE, 0);

/* check block - save to file only */
  vbox = gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(frame), vbox);

/* type of file */
  button = gtk_check_button_new_with_label ("Rendered animation");
  gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, TRUE, 0);

/* Details of the animation */
  a_frame = gtk_frame_new("Animation details");
  gtk_box_pack_start(GTK_BOX(vbox2), a_frame, TRUE, TRUE, 0);
  vbox = gtk_vbox_new(TRUE, 0);
  gtk_container_add(GTK_CONTAINER(a_frame), vbox);
  gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)),1);

/* attach frame to check button for show/hide toggling */
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                            GTK_SIGNAL_FUNC(event_render_modify),
                           (gpointer) button);
  gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) ANIMATE);
  gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) a_frame);

/* start off with a button */
  button = gtk_radio_button_new_with_label (NULL, "Animated GIF");
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                            GTK_SIGNAL_FUNC(event_render_modify),
                            (gpointer) button);
  gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, TRUE, 0);
  gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) ANIM_GIF);

/* make a radio group */
  group = gtk_radio_button_group(GTK_RADIO_BUTTON(button));
/* do the rest of the buttons */
  button = gtk_radio_button_new_with_label(group, "MPEG");
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                            GTK_SIGNAL_FUNC(event_render_modify),
                             (gpointer) button);
  gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, TRUE, 0);
  gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) ANIM_MPEG);

/* delay */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,TRUE,0);
  label = gtk_label_new("Delay (ms)");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

  adj = (GtkAdjustment *) gtk_adjustment_new
                            (povray_data.delay, 0, 100, 5, 10, 0);
  spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
  gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
  gtk_box_pack_end(GTK_BOX(hbox), spin, FALSE, TRUE, 5);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                     GTK_SIGNAL_FUNC(event_render_modify), (gpointer) spin);
  gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) DELAY);

/* file entry */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,TRUE,0);
  label = gtk_label_new("Filename ");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);
  file_entry = gtk_entry_new_with_max_length(FILELEN-1);
  gtk_box_pack_end(GTK_BOX (hbox), file_entry, FALSE, TRUE, 0);
  gtk_entry_set_text(GTK_ENTRY(file_entry), povray_data.filename);

/* update hook */
  gtk_signal_connect(GTK_OBJECT(file_entry), "changed",
      GTK_SIGNAL_FUNC(event_render_modify), (gpointer) file_entry);
  gtk_object_set_data(GTK_OBJECT(file_entry), "id", (gpointer) ANIM_NAME);
  }

if (data->id == MORPH)
  {
/* wire frame or solid morphology rendering */
  frame = gtk_frame_new(NULL);
  gtk_box_pack_start(GTK_BOX(vbox2), frame, TRUE, TRUE, 0);
  vbox = gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  button = gtk_check_button_new_with_label ("Wire frame morphology");
  gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, TRUE, 0);

/* rendered crystal properties */
  frame = gtk_frame_new("Solid morphology");
  gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
  vbox = gtk_vbox_new(TRUE, 5);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)),1);

/* hide/show callback */
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC (event_render_modify),
                     (gpointer) button);
  gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) MORPH_STYLE);
  gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) frame);

/* glass transmission */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,TRUE,0);
  label = gtk_label_new("Light transmission");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

  adj = (GtkAdjustment *) gtk_adjustment_new
                            (povray_data.transmit, 0.0, 1.0, 0.1, 0.1, 0);
  spin = gtk_spin_button_new(adj, 0.1, 2);
  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
  gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
  gtk_box_pack_end(GTK_BOX(hbox), spin, FALSE, TRUE, 5);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                     GTK_SIGNAL_FUNC(event_render_modify), (gpointer) spin);
  gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) TRANSMIT);

/* refractive index */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,TRUE,0);
  label = gtk_label_new("Refractive index");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

  adj = (GtkAdjustment *) gtk_adjustment_new
                            (povray_data.ref_index, 1.0, 3.0, 0.1, 0.1, 0);
  spin = gtk_spin_button_new(adj, 0.1, 2);
  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
  gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
  gtk_box_pack_end(GTK_BOX(hbox), spin, FALSE, TRUE, 5);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                     GTK_SIGNAL_FUNC(event_render_modify), (gpointer) spin);
  gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) REF_INDEX);

/* colour */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,TRUE,0);

  label = gtk_label_new("Crystal colour");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

  button = gtk_button_new_with_label ("   Edit   ");
  gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, TRUE, 0);

/* morph drawing colour */
  style = gtk_style_copy(gtk_widget_get_style(rsd->win));
  style->bg[0].red   = povray_data.morph_colour[0] * 65535;
  style->bg[0].green = povray_data.morph_colour[1] * 65535; 
  style->bg[0].blue  = povray_data.morph_colour[2] * 65535; 
  gtk_widget_set_style(GTK_WIDGET(button), style);

/* TODO - csd dialog hook? rgb spinners? */
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                             GTK_SIGNAL_FUNC (colsel_dialog),
                             GTK_OBJECT (button));
  gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) MORPH_COLOUR);
  gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) button);

/* texture - glassy, metallic, etc. */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,TRUE,0);
  label = gtk_label_new("Surface finish ");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);
  entry = gtk_entry_new_with_max_length(LINELEN-1);
  gtk_box_pack_end(GTK_BOX (hbox), entry, FALSE, TRUE, 0);
  gtk_entry_set_text(GTK_ENTRY(entry), povray_data.morph_finish);

/* update hook */
  gtk_signal_connect(GTK_OBJECT(entry), "changed",
      GTK_SIGNAL_FUNC(event_render_modify), (gpointer) entry);
  gtk_object_set_data(GTK_OBJECT(entry), "id", (gpointer) MORPH_FINISH);
  }

/* terminating buttons */
button = gtk_button_new_with_label ("POVRay");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(rsd->win)->action_area),
                                           button,FALSE,TRUE,0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (render_setup),
                           GTK_OBJECT (rsd->win));

button = gtk_button_new_with_label ("Close");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(rsd->win)->action_area),
                                           button,FALSE,TRUE,0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (close_dialog),
                          (gpointer) id);

/* display the dialog */
gtk_widget_show_all(rsd->win);

/* default - animate menu off */
if (data->num_frames > 1 && a_frame != NULL)
  gtk_widget_hide(a_frame);
}

/***************************/
/* Setup POVray parameters */
/***************************/
void povray_hdr(FILE *fp, struct model_pak *data)
{
gfloat xvec, yvec, amb;

fprintf(fp,"#include \"colors.inc\" \n");
fprintf(fp,"#include \"finish.inc\" \n");
fprintf(fp,"#include \"glass.inc\" \n");
fprintf(fp,"#include \"metals.inc\" \n");
fprintf(fp,"#include \"textures.inc\" \n");

if (povray_data.background)
  fprintf(fp,"background { color rgb<1.0,1.0,1.0> }\n");
else
  fprintf(fp,"background { color rgb<0.0,0.0,0.0> }\n");

/* scaling */
xvec = yvec = 2.0*data->rmax/data->scale;

/* BIOSYM orientation */
fprintf(fp,"camera { orthographic location <0,0,%d>\n", FRAMEZ);
fprintf(fp,"    right <%f,0,0> up <0,%f,0>\n", xvec, yvec);
fprintf(fp,"    look_at <0,0,0> }\n");

/* normal lights */
fprintf(fp,"light_source\n  {\n <%d,%d,%d>\n",FRAMEZ/2,FRAMEZ/2,FRAMEZ);

/* The trouble with shadowless is that the nice shiny highlights are lost */
if (povray_data.shadowless)
  fprintf(fp,"  color rgb<1,1,1> shadowless }\n");
else
  fprintf(fp,"  color rgb<1,1,1> }\n");

/* morph is too dark with just the above, sky_sphere is *nice* */
/* TODO - leave choice of colour eg white/grey/light blue to the user? */
/* white can be a bit too bright (even with 0 ambience) */
if (data->id == MORPH && !povray_data.wire_frame)
  fprintf(fp,"sky_sphere { pigment {gradient y  color_map "
             "{[0, 1  color Gray20 color Gray80]} rotate x*45}}\n");
/*
  fprintf(fp,"sky_sphere { pigment { Gray80 }}\n");
*/

amb = povray_data.ambience * 255.0;

/* TODO - put the gamma in an extended features dialog? */
fprintf(fp,"global_settings { ambient_light rgb<%f, %f, %f> assumed_gamma 2.2}\n",amb,amb,amb);
}

/*****************************/
/* task orientated rendering */
/*****************************/
void exec_pov_task(gpointer *ptr)
{
GString *cmd;

g_return_if_fail(ptr != NULL);
cmd = g_string_new(NULL);

/* build the command line */
g_string_sprintf(cmd, "%s +I%s.pov -Ga -P +W%d +H%d +FT", 
                       sysenv.povray_exe, (gchar *) ptr, 
                       povray_data.width, povray_data.height); 

if (povray_data.antialias)
  g_string_sprintfa(cmd, " +A +AM2");

exec(cmd->str);
_exit;
}

/***************************/
/* task orientated viewing */
/***************************/
void exec_img_task(gpointer *ptr)
{
gint pid1, pid2, stat;
GString *cmd;

/* post povray command */
g_return_if_fail(ptr != NULL);

cmd = g_string_new(NULL);

/* build the viewing command , */
g_string_sprintf(cmd, "rm -rf %s.pov ; %s %s.tga ; rm -rf %s.tga",
        (gchar *) ptr, sysenv.viewer_exe, (gchar *) ptr, (gchar *) ptr);

/* double fork to completely free child */
pid1 = fork();
switch(pid1)
  {
/* failure */
  case -1:
    printf("end_task() error: fork failed.\n");
    break;
/* child */
  case 0:
    pid2 = fork();
    if (pid2 == -1)
      {
      printf("end_task() error: fork failed.\n");
      return;
      }
    if (pid2 == 0)
      {
      exec(cmd->str);
      _exit(0);
      }
    _exit(0);
/* parent */
  default:
    while(wait(&stat) != pid1); 
  }

/* free the string holding the filename */
g_free(ptr);
g_string_free(cmd, TRUE);
return;
}

/************************/
/* background rendering */
/************************/
/* TODO - pthreads (SG's ???) */
void run_pov()
{
struct task_pak task;
gchar *basename;

/* make an input file */
basename = gun();
make_pov(sysenv.active, basename);

/* execute */
task.label = g_strdup("POVRay");
task.primary = &exec_pov_task;
task.ptr1 = (gpointer *) basename;
task.cleanup = &exec_img_task;
task.ptr2 = (gpointer *) basename;

start_task(&task);

printf("\n");

/*
pthread_exit(0);
*/
}

/*****************************************/
/* foreground rendering (for animation!) */
/*****************************************/
void exec_pov(gchar *basename)
{
GString *cmd;

cmd = g_string_new(NULL);

/* build the command line */
g_string_sprintf(cmd,"%s +I%s.pov -GA -P +W%d +H%d +FT ",sysenv.povray_exe
                                                        ,basename
                                                        ,povray_data.width
                                                        ,povray_data.height);
if (povray_data.antialias)
  g_string_sprintfa(cmd,"+A +AM2 ");
/* after rendering delete input file, */
g_string_sprintfa(cmd,"; rm -rf %s.pov ",basename);

/* execute */
system(cmd->str);
printf("\n");
g_string_free(cmd, TRUE);
}

/****************************/
/* make a POVRAY input file */
/****************************/
#define DEBUG_MKPOV 1
/* FIXME - morph via plane cuts (ie solid) has problems with non 90 angles */
gint make_pov(gint model, gchar *basename)
{
gint ai,bi,ci;
gint n,m,i,j,k;
gint r,g,b, flag;
gfloat rad, sep, scale;
gfloat x1,y1,z1,x2,y2,z2,mx,my,mz;
gfloat xmin,ymin,zmin,xmax,ymax,zmax;
gfloat dx,dy,ox,oy,a;
gfloat rf,gf,bf,rf2,gf2,bf2;
gfloat vec[3], acm[9], tmp[3];
GString *povfile;
GSList *bond, *plist;
struct model_pak *data;
struct plane_pak *plane;
FILE *fp;

/* FIXME - scan for existing & rename with _1, _2 etc. */
povfile = g_string_new(basename);
g_string_sprintfa(povfile,".pov");

/*
printf("Creating [%s]\n",povfile->str);
*/

/* open file & init */
fp = fopen(povfile->str,"w");
if (!fp)
  {
  printf("Error, render(): can't open %s!\n",povfile->str);
  return(1);
  }

data = model_ptr(model, RECALL);

/* setup axes conversion matrix */
switch(data->id)
  {
  default:
    VEC3SET(&acm[0],-1.0, 0.0, 0.0);
    VEC3SET(&acm[3], 0.0,-1.0, 0.0);
    VEC3SET(&acm[6], 0.0, 0.0,-1.0);
  }

/* strcpy(povfile,models[model].filename); */
/* file_extension -> .pov */

/* setup */
povray_hdr(fp,data);
/* overlay user defined bonds onto precalculated ones */
bond_overlay(data);
/* limits - for intelligent axes placement */
xmin = ymin = zmin = 99999999.9;
xmax = ymax = zmax = -99999999.9;
/* pixel to coord scaling factor */
scale =  data->scale * sysenv.subscale * 0.5 / data->rmax;

/* do atoms */ 
#if DEBUG_MKPOV
printf("Doing atoms...\n");
#endif
for (n=0 ; n<data->num_atoms ; n++)
  {
  if ((data->atoms+n)->status & (DELETED | HIDDEN))
    continue;

/* POVRAY data (atoms) */
  VEC3SET(vec, (data->atoms+n)->rx, (data->atoms+n)->ry, (data->atoms+n)->rz);
  vecmat(acm, vec);
  x1 = vec[0];
  y1 = vec[1];
  z1 = vec[2];
/* limits */
  if (x1 < xmin)
    xmin = x1;
  if (y1 < ymin)
    ymin = y1;
  if (z1 < zmin)
    zmin = z1;
  if (x1 > xmax)
    xmax = x1;
  if (y1 > ymax)
    ymax = y1;
  if (z1 > zmax)
    zmax = z1;
/* if ($ageom{$atype[$n]} eq "stick") */
/* get current colour */
    r = (data->atoms+n)->colour[0];
    g = (data->atoms+n)->colour[1];
    b = (data->atoms+n)->colour[2];
/* convert to povray rgb format */
    rf = (gfloat) (r) / 65535.0;
    gf = (gfloat) (g) / 65535.0;
    bf = (gfloat) (b) / 65535.0;
/* MARVIN region highlighting */
/* NEW - only fade regions if in REGION_LITE mode */
    if (data->id == MARVIN && data->mode == REGION_LITE)
      if (!data->region[(data->atoms+n)->region])
        {
        rf *= 0.3;
        gf *= 0.3;
        bf *= 0.3;
        }
    switch(povray_data.render_type)
      {
/* rounded end are now draw in the bond section */
      case STICK:
/* even if stick, draw a sphere if isolated */
        if ((data->atoms+n)->atom_bonds)
          break;
      case BALL_STICK:
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",x1,y1,z1,povray_data.arad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);  
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
        break;
      case CPK:
        rad = elements[(data->atoms+n)->atom_code].vdw;
        rad *= povray_data.cpk_scale;
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",x1,y1,z1,rad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);  
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
        break;
      }
  }

/* do bonds */ 
#if DEBUG_MKPOV
printf("Doing bonds...\n");
#endif
if (povray_data.render_type != CPK)
  {
bond = data->bonds;
while(bond != NULL)
  {
  i = ((struct bond_pak *) bond->data)->atom1;
  j = ((struct bond_pak *) bond->data)->atom2;
/* coordinates of bonded atoms */
  VEC3SET(vec, (data->atoms+i)->rx, (data->atoms+i)->ry, (data->atoms+i)->rz);
  vecmat(acm, vec);
  x1 = vec[0];
  y1 = vec[1];
  z1 = vec[2];
  VEC3SET(vec, (data->atoms+j)->rx, (data->atoms+j)->ry, (data->atoms+j)->rz);
  vecmat(acm, vec);
  x2 = vec[0];
  y2 = vec[1];
  z2 = vec[2];
/* limits */
  if (x1 < xmin)
    xmin = x1;
  if (y1 < ymin)
    ymin = y1;
  if (z1 < zmin)
    zmin = z1;
  if (x1 > xmax)
    xmax = x1;
  if (y1 > ymax)
    ymax = y1;
  if (z1 > zmax)
    zmax = z1;

/* bond midpoint TODO - account for different radii */
  mx = (x1 + x2) / 2.0;
  my = (y1 + y2) / 2.0;
  mz = (z1 + z2) / 2.0;
/* colour of atom i */
  r = (data->atoms+i)->colour[0];
  g = (data->atoms+i)->colour[1];
  b = (data->atoms+i)->colour[2];
/* convert to povray rgb format */
  rf = (gfloat) (r) / 65535.0;
  gf = (gfloat) (g) / 65535.0;
  bf = (gfloat) (b) / 65535.0;
/* colour of atom j */
  r = (data->atoms+j)->colour[0];
  g = (data->atoms+j)->colour[1];
  b = (data->atoms+j)->colour[2];
/* convert to povray rgb format */
  rf2 = (gfloat) (r) / 65535.0;
  gf2 = (gfloat) (g) / 65535.0;
  bf2 = (gfloat) (b) / 65535.0;
/* MARVIN region highlighting */
  flag=0;
  if (data->id == MARVIN)
    if (!data->region[(data->atoms+i)->region] ||
        !data->region[(data->atoms+j)->region])
      {
      rf *= 0.6;
      gf *= 0.6;
      bf *= 0.6;
      rf2 *= 0.6;
      gf2 *= 0.6;
      bf2 *= 0.6;
      }

/* bond type dependence */
/* separation constant for // cylinders (double/triple bonds) */
    sep = 1.5;
    rad = povray_data.brad;
    switch(((struct bond_pak *) bond->data)->type)
      {
      case TRIPLE:
/* print half the bond */
        fprintf(fp,"cylinder { <%f,%f,%f>,\n<%f,%f,%f>, %f\n"
                             ,x1,y1,z1, mx,my,mz, TRIPLE_SCALE*rad);
        fprintf(fp,"open texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
/* print half the bond */
        fprintf(fp,"cylinder { <%f,%f,%f>,\n<%f,%f,%f>, %f\n"
                              ,x2,y2,z2, mx,my,mz, TRIPLE_SCALE*rad);
        fprintf(fp,"open texture{pigment{color rgb<%f,%f,%f>}\n",rf2,gf2,bf2);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
/* round off the ends */
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",x1,y1,z1,TRIPLE_SCALE*rad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",x2,y2,z2,TRIPLE_SCALE*rad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf2,gf2,bf2);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");

        sep += 1.0;
        rad *= TRIPLE_SCALE / DOUBLE_SCALE;
      case DOUBLE:
        rad *= DOUBLE_SCALE;
        dy = my - y1;
        dx = mx - x1;
        if (dx)
          a = PI/2.0 - atan(dy/dx);
        else
          a = 0.0;
/* normalize */
        ox = sep*rad*cos(a);
        oy = -1.0*sep*rad*sin(a);   /* inv y */
/* print half the bond */
        fprintf(fp,"cylinder { <%f,%f,%f>,\n<%f,%f,%f>, %f\n"
                   ,x1+ox,y1+oy,z1, mx+ox,my+oy,mz, rad);
        fprintf(fp,"open texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
/* print half the bond */
        fprintf(fp,"cylinder { <%f,%f,%f>,\n<%f,%f,%f>, %f\n"
                   ,x1-ox,y1-oy,z1, mx-ox,my-oy,mz, rad);
        fprintf(fp,"open texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
/* round off the ends */
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",x1+ox,y1+oy,z1,rad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",x1-ox,y1-oy,z1,rad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
/* print half the bond */
        fprintf(fp,"cylinder { <%f,%f,%f>,\n<%f,%f,%f>, %f\n"
                    ,x2+ox,y2+oy,z2, mx+ox,my+oy,mz, rad);
        fprintf(fp,"open texture{pigment{color rgb<%f,%f,%f>}\n",rf2,gf2,bf2);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
/* print half the bond */
        fprintf(fp,"cylinder { <%f,%f,%f>,\n<%f,%f,%f>, %f\n"
                    ,x2-ox,y2-oy,z2, mx-ox,my-oy,mz, rad);
        fprintf(fp,"open texture{pigment{color rgb<%f,%f,%f>}\n",rf2,gf2,bf2);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
/* round off the ends */
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",x2+ox,y2+oy,z2,rad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf2,gf2,bf2);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",x2-ox,y2-oy,z2,rad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf2,gf2,bf2);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
        break;
      case SINGLE:
      default:
/* print half the bond */
        fprintf(fp,"cylinder { <%f,%f,%f>,\n<%f,%f,%f>, %f\n"
                             ,x1,y1,z1, mx,my,mz, rad);
        fprintf(fp,"open texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
/* print half the bond */
        fprintf(fp,"cylinder { <%f,%f,%f>,\n<%f,%f,%f>, %f\n"
                                ,x2,y2,z2, mx,my,mz, rad);
        fprintf(fp,"open texture{pigment{color rgb<%f,%f,%f>}\n",rf2,gf2,bf2);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
/* round off the ends */
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",x1,y1,z1,rad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
        fprintf(fp,"sphere { <%f, %f, %f>, %f ",x2,y2,z2,rad);
        fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf2,gf2,bf2);
        fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
        break;
    }
  bond = g_slist_next(bond);
  }
  }

#if DEBUG_MKPOV
printf("Doing special objects...\n");
#endif

/********/
/* AXES */
/********/
/* FIXME - doesn't work yet, placement problem */
povray_data.axes = FALSE;
if (povray_data.axes)
  {
printf("Doing axes.\n");
/*
xvec = yvec = 2.0*data->rmax/data->scale;
*/
  x1 = data->axes[0].rx[0] + data->axes[1].rx[0] + data->axes[2].rx[0];
  y1 = data->axes[0].rx[1] + data->axes[1].rx[1] + data->axes[2].rx[1];
  z1 = data->axes[0].rx[2] + data->axes[1].rx[2] + data->axes[2].rx[2];
  rad = sqrt(x1*x1 + y1*y1 + z1*z1);
/* origin */
  x1 = x2 = (1.05*xmin - 0.05*xmax);
  y1 = y2 = ymin;
  z1 = z2 = (zmin + zmax)/2.0;
  for (j=0 ; j<3 ; j++)
    {
/* end points */
    x2 += data->axes[j].rx[0];
    y2 += data->axes[j].rx[1];
    z2 += data->axes[j].rx[2];
/* draw the axis */
    fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>, 0.01\n",x1,y1,z1,x2,y2,z2);
    fprintf(fp," texture { pigment {White} } }\n");
    }
  }

/********/
/* CELL */
/********/
if (data->cell_on)
  {
/* ends */
  for (j=0 ; j<8 ; j++)
    {
    m = 2*(j/2) + 1;
    n = 4*(j/4);

    ARR3SET(vec, data->cell[m].rx);
    vecmat(acm, vec);
    x1 = vec[0];
    y1 = vec[1];
    z1 = vec[2];

    ARR3SET(vec, data->cell[n].rx);
    vecmat(acm, vec);
    x2 = vec[0];
    y2 = vec[1];
    z2 = vec[2];
    fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>, %f\n",
                            x1,y1,z1,x2,y2,z2,povray_data.frad);
    fprintf(fp," texture { pigment {White} } }\n");

    ARR3SET(vec, data->cell[n+2].rx);
    vecmat(acm, vec);
    x2 = vec[0];
    y2 = vec[1];
    z2 = vec[2];
    fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>, %f\n",
                            x1,y1,z1,x2,y2,z2,povray_data.frad);
    fprintf(fp," texture { pigment {White} } }\n");
    }

/* sides */
/* skip for 2D periodic models */
  if (data->periodic == 3)
    {
    m = 0;
    n = 4;
    for (j=0 ; j<4 ; j++)
      {
      ARR3SET(vec, data->cell[m].rx);
      vecmat(acm, vec);
      x1 = vec[0];
      y1 = vec[1];
      z1 = vec[2];

      ARR3SET(vec, data->cell[n].rx);
      vecmat(acm, vec);
      x2 = vec[0];
      y2 = vec[1];
      z2 = vec[2];

      fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>, %f\n",
                              x1,y1,z1,x2,y2,z2,povray_data.frad);
      fprintf(fp," texture { pigment {White} } }\n");
      m++;
      n++;
      }
    }
  }

/********************/
/* Connolly surface */
/********************/
if (data->csurf_on)
  {
/* CURRENT - nothing like a bit of black magic... */
/*
rad = 38400.0 / (scale * povray_data.width * povray_data.height);
rad = (scale/10 * povray_data.width/100 * povray_data.height/100);
rad *= rad;
rad /= 100;
rad = 1/rad;
*/
sep = (scale/10 * povray_data.width/100 * povray_data.height/100);
rad = exp(-0.001 * sep*sep);
/*
printf("scale = %f, rad = %f\n",scale,rad);
*/
  for(n=0 ; n<data->csurf.grid*data->csurf.grid ; n++)
    {
/* get atom touched (if any) */
    j = *(data->csurf.touch+n); 
    if (j == -2)
      continue;

/* periodic images? */
    for (ai=-data->image_limit[0] ; ai<data->image_limit[1] ; ai++)
      {
    for (bi=-data->image_limit[2] ; bi<data->image_limit[3] ; bi++)
      {
    for (ci=-data->image_limit[4] ; ci<data->image_limit[5] ; ci++)
      {
      if (!data->periodic)
        if (ai || bi || ci)
          continue;
/* get real coordinates */
      x1 = *(data->csurf.rx+n);
      y1 = *(data->csurf.ry+n);
      z1 = *(data->csurf.rz+n);
/* create periodic images */
      x1 += ai*data->piv_real[0];
      x1 += bi*data->piv_real[1];
      x1 += ci*data->piv_real[2];
      y1 += ai*data->piv_real[3];
      y1 += bi*data->piv_real[4];
      y1 += ci*data->piv_real[5];
      z1 += ai*data->piv_real[6];
      z1 += bi*data->piv_real[7];
      z1 += ci*data->piv_real[8];
/* axes conversion */
      VEC3SET(vec, x1, y1, z1);
      vecmat(acm, vec);
      x1 = vec[0];
      y1 = vec[1];
      z1 = vec[2];
/* colour of atom j */
      if (j == -1)
        {
/* re-entrant surface */
        r = 64000;
        g = 30000;
        b = 64000;
        }
      else
        {
/* current atom colour */
r = elements[*(data->csurf.code+n)].colour[0];
g = elements[*(data->csurf.code+n)].colour[1];
b = elements[*(data->csurf.code+n)].colour[2];
        }
/* convert to povray rgb format */
      rf = (gfloat) (r) / 65535.0;
      gf = (gfloat) (g) / 65535.0;
      bf = (gfloat) (b) / 65535.0;

x1 -= (gfloat) data->offset[0] / scale;
y1 -= (gfloat) data->offset[1] / scale;

/* print small sphere */
      fprintf(fp,"sphere { <%f, %f, %f>, %f ",x1,y1,z1,0.03);
      fprintf(fp,"texture{pigment{color rgb<%f,%f,%f>}\n",rf,gf,bf);  
      fprintf(fp,"  finish{specular 0.6 diffuse 0.8 }}}\n");
      }
      }
      }
    } 
  }

/**************/
/* MORPHOLOGY */
/**************/
if (data->num_vertices)
  {
/* morphology rendering style */
  if (!povray_data.wire_frame)
    {
/* render via plane intersection */
    fprintf(fp,"object\n{\nintersection{\n");

    plist = data->planes;
    while (plist != NULL)
      {
      plane = (struct plane_pak *) plist->data;
      if (plane->present)
        {
/* rotate cartesian normals to match viewing angle */
        ARR3SET(vec, plane->norm);
        vecmat(data->rotmat, vec);
/* apply axes conversion matrix */
        vecmat(acm, vec);
/* now need distance to facet (this is not the length of norm) */
/* use the position of the facet labels */
        tmp[0] = plane->rx;
        tmp[1] = plane->ry;
        tmp[2] = plane->rz;
        fprintf(fp,"plane { <%f, %f, %f>,  %f }\n",vec[0],vec[1],vec[2],
                                                          VEC3MAG(tmp));
        }
      plist = g_slist_next(plist);
      }
/* crystal composition */
    fprintf(fp,"} interior { ior %f } texture { finish { %s } "
               " pigment { color rgbf <%f,%f,%f,%f> } }\n }\n", 
               povray_data.ref_index,
               povray_data.morph_finish,
               povray_data.morph_colour[0],
               povray_data.morph_colour[1],
               povray_data.morph_colour[2],
               povray_data.transmit);  
    }
  else
    {
/* draw by ridges */
    for (i=0 ; i<data->num_vertices ; i++)
      {
      for (j=0 ; j<(data->vertices+i)->num_adj ; j++)
        {
/* retrieve vertex i and neighbour k */
        k = *((data->vertices+i)->adj+j);
/* hidden line removal */
        if (!ridge_visible(data,i,k) && data->hpr)
          continue;

        VEC3SET(vec, (data->vertices+i)->rx, (data->vertices+i)->ry,
                                             (data->vertices+i)->rz);
        vecmat(acm, vec);
        x1 = vec[0];
        y1 = vec[1];
        z1 = vec[2];

        VEC3SET(vec, (data->vertices+k)->rx, (data->vertices+k)->ry,
                                             (data->vertices+k)->rz);
        vecmat(acm, vec);
        x2 = vec[0];
        y2 = vec[1];
        z2 = vec[2];

/* end point degeneracy check */
        flag=0;
        if (fabs(x2-x1) < 0.001)
          flag++;
        if (fabs(y2-y1) < 0.001)
          flag++;
        if (fabs(z2-z1) < 0.001)
          flag++;
        if (flag==3)
          continue;
/* auto scale the frame radius */
        rad = povray_data.frad * data->rmax / 12.0;
        fprintf(fp,"cylinder { <%f,%f,%f>,<%f,%f,%f>,%f\n",
                                  x1,y1,z1,x2,y2,z2,rad);
/* TODO - choose black/white depending on background colour */
        if (povray_data.background)
          fprintf(fp," texture { pigment {Black} } }\n");
        else
          fprintf(fp," texture { pigment {White} } }\n");
        }
      }
    }
  }

/* finished making the .pov file */
g_string_free(povfile, TRUE);
fclose(fp);
return(0);
}

