/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <fcntl.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>

#include "gdis.h"

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/***************************************************/
/* get single point energy from a GULP output file */
/***************************************************/
#define DEBUG_READ_GULP 0
gint read_single_gulp(gchar *file, struct model_pak *data)
{
gchar line[LINELEN], **buff;
FILE *fp;

/* open & scan gulp output file */
fp = fopen(file,"rt");
if (!fp)
  {
  printf("No output file found!\n");
  return(1);
  }

/* look for last (eV) energy output */
while(!fgetline(fp,line))
  { 
  buff = get_tokens(line, 10);

/* search for energy */
  if (g_strncasecmp("  Total lattice energy",line,22) == 0)
    {
/* if nothing on the rest of this line, then should be on the next */
    if (!strlen(*(buff+5)))
      {
      while(!fgetline(fp,line))
        if (g_strncasecmp("    Primitive unit cell",line,20) == 0)
          break;
      g_strfreev(buff);
      buff = get_tokens(line, 6);
      }

    if (g_strncasecmp("eV",*(buff+5),2) == 0)
      sscanf(*(buff+4),"%f",&data->gulp.energy);
    }

/* search for free energy */
  if (g_strncasecmp("  Total free energy",line,19) == 0)
    {
    if (g_strncasecmp("eV",*(buff+5),2) == 0)
      sscanf(*(buff+4),"%f",&data->gulp.energy);
    }

/* search for surface energy */
  if (g_strncasecmp("  Surface energy ",line,17) == 0)
    {
    sscanf(*(buff+5),"%f",&data->gulp.esurf);
    g_free(data->gulp.esurf_units);
    data->gulp.esurf_units = g_strdup(*(buff+6));
    }

/* search for attachment energy */
  if (g_strncasecmp("  Attachment energy",line,19) == 0)
    {
    data->gulp.eatt = str_to_float(*(buff+3));
    g_free(data->gulp.eatt_units);
    data->gulp.eatt_units = g_strdup(*(buff+4));
    }

  if (g_strncasecmp("  Attachment energy/molecule",line,28) == 0)
    {
    data->gulp.eatt = str_to_float(*(buff+3));
    g_free(data->gulp.eatt_units);
    data->gulp.eatt_units = g_strdup("eV/mol");
    }

/* dipole moment */
  if (g_strncasecmp("  Initial surface dipole ",line,25) == 0)
    sscanf(*(buff+4),"%f",&data->gulp.sdipole);

  g_strfreev(buff);
  }
/* done */
g_snprintf(line,35,"Single point Energy: %10.4f eV\n", data->gulp.energy);
show_text(line);
return(0);
}

/**********************************************/
/* get updated coords from a GULP output file */
/**********************************************/
gint read_minimize_gulp(gchar *file, struct model_pak *data)
{
gint code, model;
gchar *tmp;
FILE *fp;

/* open & scan gulp output file */
fp = fopen(file,"rt");
if (!fp)
  {
  printf("No output file found!\n");
  return(-1);
  }

/* create new file instead of overwriting */
model = sysenv.num_models;
tmp = g_strdup_printf("%s_opti", g_strstrip(data->basename));
if ((code = load_gulp_output(fp, model, tmp)))
  printf("Output file load failed, error code %d.\n",code);
else
  {
/* this will trigger a pick_model() call */
  tree_select(model);
  }

/* done */
g_free(tmp);
fclose(fp);
return(model);
}

/*******************/
/* run a gulp file */
/*******************/
#define DEBUG_EXEC_GULP 0
gint exec_gulp(gchar *input, gchar *output)
{
GString *cmd;

/* stop gulp's renaming (if it exists already) */
unlink(output);

/* TODO - scan output to print updates */
/* build & execute command line */
cmd = g_string_new(NULL);

/* The SG's don't like the ampersand after the '>' */
/*
g_string_sprintf(cmd,"%s < %s >& %s",sysenv.gulp_exe,input,output);
*/
g_string_sprintf(cmd,"%s < %s > %s",sysenv.gulp_exe,input,output);

#if DEBUG_EXEC_GULP
printf("executing: [%s]\n",cmd->str);
#endif

my_system(cmd->str);

/*
unlink(input);
*/

/* done */
g_string_free(cmd, TRUE);
return(0);
}

/*****************************/
/* execute a gulp run (task) */
/*****************************/
#define DEBUG_EXEC_GULP_TASK 0
void exec_gulp_task(gpointer *ptr)
{
gint fi, fo;
gchar *inpfile, *outfile, *argv[2];
gchar **buff;
struct model_pak *data;

g_return_if_fail(ptr != NULL);
data = (struct model_pak *) ptr;

/* get input and output filenames */
/* TODO - can we put =, instead of malloc'ing? */
inpfile = g_strdup(data->gulp.temp_file);

/* TODO - put this in a routine */
buff = g_strsplit(data->gulp.temp_file, ".gin", 0);
/* copy first token as the base name */
outfile = g_malloc((5+strlen(*buff))*sizeof(gchar));
strcpy(outfile, *buff);
strcat(outfile, ".got");
g_strfreev(buff);

#if DEBUG_EXEC_GULP_TASK
printf("output file: %s\n", outfile);
#endif

/* create the input file */
write_gulp(inpfile, data);

/* redirect stdin */
fi = open(inpfile, O_RDONLY);
if (fi < 0)
  {
  printf("failed to open: %s\n", inpfile);
  _exit;
  }
close(0);
dup2(fi, 0);
close(fi);
/* redirect stdout */
/* NB: output file must have correct permissions if created */
fo = open(outfile, O_RDWR | O_CREAT, S_IREAD | S_IWRITE);
if (fo < 0)
  {
  printf("failed to open: %s\n", outfile);
  _exit;
  }
close(1);
dup2(fo, 1);
close(fo);
/* NEW - some machines (mac's) don't like empty argument lists */
argv[0] = "";
argv[1] = NULL;
/* run */
execvp(sysenv.gulp_exe, argv);
_exit;
}

/*****************************/
/* process a gulp run (task) */
/*****************************/
#define DEBUG_PROC_GULP 0
void proc_gulp_task(gpointer *ptr)
{
gint model;
GString *line;
struct model_pak *dest, *data;

/* TODO - model locking (moldel_ptr RO/RW etc) to prevent screw ups */
g_return_if_fail(ptr != NULL);
data = (struct model_pak *) ptr;

/* TODO - make it possile to get dialog data by request */
/* so that we can check if a dialog exsits to be updated */

/* get new coords */
if (data->gulp.run == E_OPTIMIZE)
  {
/* NEW - if minimize, we deal with the newly created model */
  model = read_minimize_gulp(data->gulp.out_file, data);
  dest = model_ptr(model, RECALL);
  if (dest)
    {
/* transfer data to new model */
    copy_gulp_data(data, dest);
    copy_gulp_extra(data, dest);
    copy_elem_data(data, dest);
/* update coords etc - INIT's */
    init_objs(REDO_COORDS, dest);
    calc_bonds(dest);
    calc_mols(dest);
    redraw_canvas(ALL);
    }
  }
else
  {
/* same model (ie with current energetics dialog) so update */
  read_single_gulp(data->gulp.out_file, data);
/* update energy (TODO - only if successful) */
  line = g_string_new(NULL);
  if (data->gulp.free)
    g_string_sprintf(line,"%f (free energy)",data->gulp.energy);
  else
    g_string_sprintf(line,"%f",data->gulp.energy);

/* is there a dialog entry to be updated? */
  if (GTK_IS_ENTRY(data->gulp.energy_entry))
    gtk_entry_set_text(GTK_ENTRY(data->gulp.energy_entry), line->str);
  if (data->periodic == 2)
    {
/* update surface energy dialog entry */
    g_string_sprintf(line,"%f    %s",data->gulp.esurf, data->gulp.esurf_units);
    if (GTK_IS_ENTRY(data->gulp.esurf_entry))
      gtk_entry_set_text(GTK_ENTRY(data->gulp.esurf_entry), line->str);

/* update attachment energy dialog entry */
    g_string_sprintf(line,"%f    %s",data->gulp.eatt, data->gulp.eatt_units);
    if (GTK_IS_ENTRY(data->gulp.eatt_entry))
      gtk_entry_set_text(GTK_ENTRY(data->gulp.eatt_entry), line->str);
    }
  g_string_free(line, TRUE);
  }

/*
unlink(data->gulp.out_file);
*/

return;
}

/***********************************************/
/* controls the use of GULP to optimise coords */
/***********************************************/
#define DEBUG_RUN_GULP 0
void run_gulp(gint id)
{
gint model;
gchar **buff;
struct model_pak *data;
struct task_pak task;

if (!sysenv.have_gulp)
  {
  show_text("Sorry, gulp executable was not found.");
  return;
  }

/* get model attached to dialog id */
model = sysenv.dialog[id].model;
data = model_ptr(model, RECALL);
g_return_if_fail(data != NULL);

/* create input file to execute */
write_gulp(data->gulp.temp_file, data);

/* set up output file to use */
g_free(data->gulp.out_file);

/* TODO - put this in a routine */
buff = g_strsplit(data->gulp.temp_file, ".gin", 0);
/* copy first token as the base name */
data->gulp.out_file = g_malloc((5+strlen(*buff))*sizeof(gchar));
strcpy(data->gulp.out_file, *buff);
strcat(data->gulp.out_file, ".got");
g_strfreev(buff);

#if DEBUG_RUN_GULP
printf("output file: %s\n", data->gulp.out_file);
#endif

show_text("Running GULP in the background.");

/* run as a background process */
task.label = g_strdup("Gulp");
task.primary = &exec_gulp_task;
task.ptr1 = (gpointer *) data;
task.cleanup = &proc_gulp_task;
task.ptr2 = (gpointer *) data;

start_task(&task);
}

/***********************/
/* GULP keyword toggle */
/***********************/
gint gulp_keyword(GtkWidget *w, gpointer *obj)
{
gint keyword;
struct model_pak *data;

keyword = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(obj), "key"));

data = (struct model_pak *) gtk_object_get_data(GTK_OBJECT(obj), "ptr");
g_return_val_if_fail(data != NULL, FALSE);

switch(keyword)
  {
  case E_SINGLE:
  case E_OPTIMIZE:
  case MD:
    data->gulp.run = keyword;
    break;
  case NO_EATT:
    data->gulp.no_eatt ^= 1;
    break;
  case NO_ESURF:
    data->gulp.no_esurf ^= 1;
    break;
  case NVE:
  case NVT:
  case NPT:
    if (data->gulp.run != MD)
      {
      show_text("I'm ignoring this as it's not a dynamics run type.");
      return(TRUE);
      }
    else
      show_text("");
    data->gulp.ensemble = keyword;
    break;
  case CONP:
  case CONV:
    data->gulp.method = keyword;
    break;
  case BFGS_OPT:
  case CONJ_OPT:
  case RFO_OPT:
    data->gulp.optimiser = keyword;
    break;
  case MOLE:
  case MOLMEC:
  case MOLQ:
  case NOBUILD:
    data->gulp.coulomb = keyword;
    break;
  case TEMPERATURE:
    data->gulp.temp = (gfloat) GTK_ADJUSTMENT(obj)->value;
    break;
  case MAXCYC:
    data->gulp.maxcyc = (gint) GTK_ADJUSTMENT(obj)->value;
    break;
  case TEMP_FILE:
    g_free(data->gulp.temp_file);
    data->gulp.temp_file = g_strdup(gtk_entry_get_text(GTK_ENTRY(obj)));
    break; 
  case DUMP_FILE:
    g_free(data->gulp.dump_file);
    data->gulp.dump_file = g_strdup(gtk_entry_get_text(GTK_ENTRY(obj)));
    break;
  case LIB_FILE:
    g_free(data->gulp.lib_file);
    data->gulp.lib_file = g_strdup(gtk_entry_get_text(GTK_ENTRY(obj)));
    break;
  }

return(FALSE);
}

/***********************************/
/* register structure name changes */
/***********************************/
gint name_entry_changed(gpointer *entry)
{
struct model_pak *data;

/* retrieve the appropriate model */
data = (struct model_pak *) gtk_object_get_data(GTK_OBJECT(entry), "ptr");
g_return_val_if_fail(data != NULL, FALSE);

/* set the new basename */
g_free(data->basename);
data->basename = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));

/* update the model tree */
new_tree_item(data->number, REPLACE);
return(FALSE);
}

/*************************/
/* The GULP setup widget */
/*************************/
void gulp_widget(GtkWidget *w, struct model_pak *data)
{
gint id, model;
struct dialog_pak *gulp_dialog;
GtkWidget *main_hbox, *hbox, *vbox, *vbox1, *vbox2, *page;
GtkWidget *frame, *button, *label, *entry, *notebook;
GtkAdjustment *adj;
GSList *keyword[5];
GString *line;

/* do we have to do anything? */
g_return_if_fail(data != NULL);

/* NEW - default run setup is done by the template routine */
model = data->number;
if ((id = request_dialog(model, GULP)) < 0)
  return;
gulp_dialog = &sysenv.dialog[id];

/* string manipulation scratchpad */
line = g_string_new(NULL);

/* NEW - setup some gulp file name defaults */
if (g_strncasecmp(data->gulp.temp_file, "none", 4) == 0)
  {
/* FIXME - what if temp_file = filename? (may not want to overwrite) */
  g_string_sprintf(line,"%s.gin",g_strstrip(data->basename));
  g_free(data->gulp.temp_file);
  data->gulp.temp_file = g_strdup(line->str); 
  }
if (g_strncasecmp(data->gulp.dump_file, "none", 4) == 0)
  {
  g_string_sprintf(line,"%s.res",g_strstrip(data->basename));
  g_free(data->gulp.dump_file);
  data->gulp.dump_file = g_strdup(line->str); 
  }

/* TODO - show model's name */
/* create a new dialogue */
gulp_dialog->win = gtk_dialog_new();
gtk_window_set_title (GTK_WINDOW(gulp_dialog->win), "GULP interface");
gtk_signal_connect(GTK_OBJECT(gulp_dialog->win), "destroy",
                   GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

main_hbox = gtk_hbox_new(TRUE,3);
gtk_container_add(GTK_CONTAINER(GTK_DIALOG(gulp_dialog->win)->vbox),main_hbox);

/* left vbox */
vbox1 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start (GTK_BOX(main_hbox), vbox1, TRUE, TRUE, 0);

/* frame 1 */
frame = gtk_frame_new("run type");
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox1)),frame);
gtk_container_set_border_width(GTK_CONTAINER(frame),3);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);

/* do the first (DEFAULT MODE) radio button */
button = gtk_radio_button_new_with_label(NULL, "single point");
gtk_box_pack_start(GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) E_SINGLE);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.run == E_SINGLE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* make a radio group */
keyword[0] = gtk_radio_button_group(GTK_RADIO_BUTTON(button));
/* do the next button */
button = gtk_radio_button_new_with_label (keyword[0], "minimize");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) E_OPTIMIZE);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.run == E_OPTIMIZE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(gtk_radio_button_group
       (GTK_RADIO_BUTTON(button)), "dynamics");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) MD);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.run == MD)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);






/* NEW - notebook display for extended options */
frame = gtk_frame_new("Run options");
gtk_box_pack_start(GTK_BOX (vbox1), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame),3);

vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);
/*
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)),1);
*/

/* create notebook */
notebook = gtk_notebook_new();
gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
/*
gtk_container_add(GTK_CONTAINER(frame),notebook);
*/
gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);

/* create minimize page */
page = gtk_vbox_new(FALSE,0);
gtk_container_set_border_width (GTK_CONTAINER(GTK_BOX(vbox)), 10);

/* minimize constraint */
frame = gtk_frame_new("method");
gtk_box_pack_start (GTK_BOX (page), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame),3);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);

/* do the first (DEFAULT MODE) radio button */
button = gtk_radio_button_new_with_label (NULL, "constant pressure");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) CONP);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.method == CONP)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* make a radio group */
keyword[1] = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
/* do the next button */
button = gtk_radio_button_new_with_label (keyword[1], "constant volume");
gtk_box_pack_start(GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) CONV);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.method == CONV)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);


/* optimiser to use */
frame = gtk_frame_new("optimiser");
gtk_box_pack_start (GTK_BOX (page), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame),3);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);

/* do the first (DEFAULT MODE) radio button */
button = gtk_radio_button_new_with_label (NULL, "bfgs (default)");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) BFGS_OPT);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.optimiser == BFGS_OPT || data->gulp.optimiser == -1)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* make a radio group */
keyword[1] = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
/* do the next button */
button = gtk_radio_button_new_with_label (keyword[1], "conj");
gtk_box_pack_start(GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) CONJ_OPT);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.optimiser == CONJ_OPT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(gtk_radio_button_group
       (GTK_RADIO_BUTTON (button)), "rfo");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) RFO_OPT);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.optimiser == RFO_OPT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* add the page & title */
label = gtk_label_new ("minimize");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);








/* create dynamics page */
page = gtk_vbox_new(FALSE,0);
gtk_container_set_border_width (GTK_CONTAINER(GTK_BOX(vbox)), 10);

/* dynamics ensemble */
frame = gtk_frame_new("ensemble");
gtk_box_pack_start(GTK_BOX (page), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame),3);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);

/* do the first (DEFAULT MODE) radio button */
button = gtk_radio_button_new_with_label (NULL, "NVE");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) NVE);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.ensemble == NVE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* make a radio group */
keyword[1] = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
/* do the next button */
button = gtk_radio_button_new_with_label (keyword[1], "NVT");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) NVT);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.ensemble == NVT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(gtk_radio_button_group
       (GTK_RADIO_BUTTON (button)), "NPT");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) NPT);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.ensemble == NPT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* TODO */
/* step size/run length, trajectory file etc. */

/* add the page & title */
label = gtk_label_new ("dynamics");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);


/* create misc page */
page = gtk_vbox_new(FALSE,0);
gtk_container_set_border_width (GTK_CONTAINER(GTK_BOX(vbox)), 10);

/* dynamics ensemble */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX (page), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame),3);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);

if (data->periodic == 2)
  {
  button = gtk_check_button_new_with_label("No attachment energy calculation");
  gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
  gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                            GTK_SIGNAL_FUNC (gulp_keyword),
                           (gpointer) button);
  gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) NO_EATT);
  gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
  }

/* add the page & title */
label = gtk_label_new ("misc");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);


/* starting page */
gtk_widget_show_all(notebook); /* needed in order to make this work */
if (data->gulp.run == MD)
  gtk_notebook_set_page(GTK_NOTEBOOK(notebook), 1);
else
  gtk_notebook_set_page(GTK_NOTEBOOK(notebook), 0);


/* right vbox */
vbox2 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start (GTK_BOX(main_hbox), vbox2, TRUE, TRUE, 0);

/* frame 3 */
frame = gtk_frame_new("Molecule options");
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox2)),frame);
gtk_container_set_border_width(GTK_CONTAINER(frame),3);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);
/* do the first radio button */
button = gtk_radio_button_new_with_label (NULL, "1-2 coulomb subtraction");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) MOLE);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.coulomb == MOLE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* make a radio group */
keyword[2] = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
/* do the next button */
button = gtk_radio_button_new_with_label (keyword[2], 
                  "1-2 and 1-3 coulomb subtraction");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) MOLMEC);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.coulomb == MOLMEC)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(gtk_radio_button_group
       (GTK_RADIO_BUTTON (button)), "build, but retain coulomb interactions");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) MOLQ);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.coulomb == MOLQ)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(gtk_radio_button_group 
          (GTK_RADIO_BUTTON (button)), "molecule building off");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "key", (gpointer) NOBUILD);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.coulomb == NOBUILD)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);



/* frame */
frame = gtk_frame_new(NULL);
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox2)),frame);
gtk_container_set_border_width(GTK_CONTAINER(frame),3);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame),hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox),3);

/* TODO - ability to edit (via another dialog) */
label = gtk_label_new(" Temperature ");
gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, FALSE, 0);

adj = (GtkAdjustment *) gtk_adjustment_new
                        (data->gulp.temp, 0, 500, 10, 10, 0);
button = gtk_spin_button_new(adj, 0, 0);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(button), FALSE);
gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object(GTK_OBJECT (adj), "value_changed",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) adj);
gtk_object_set_data(GTK_OBJECT(adj), "key", (gpointer) TEMPERATURE);
gtk_object_set_data(GTK_OBJECT(adj), "ptr", (gpointer) data);


/* frame */
frame = gtk_frame_new(NULL);
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox2)),frame);
gtk_container_set_border_width(GTK_CONTAINER(frame),3);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame),hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox),3);

/* TODO - ability to edit (via another dialog) */
label = gtk_label_new(" Maximum cycles ");
gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, FALSE, 0);

adj = (GtkAdjustment *) gtk_adjustment_new
                        (data->gulp.maxcyc, 0, 500, 10, 10, 0);
button = gtk_spin_button_new(adj, 0, 0);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(button), FALSE);
gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object(GTK_OBJECT (adj), "value_changed",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) adj);
gtk_object_set_data(GTK_OBJECT(adj), "key", (gpointer) MAXCYC);
gtk_object_set_data(GTK_OBJECT(adj), "ptr", (gpointer) data);

/* setup the filename entry part */
frame = gtk_frame_new("Supplied data");


gtk_box_pack_start(GTK_BOX(vbox2),frame,FALSE,FALSE,0);
/*
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(gulp_dialog->win)->vbox),frame,FALSE,FALSE,0);
*/
gtk_container_set_border_width(GTK_CONTAINER(frame),3);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame),hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox),3);

/* left vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

/* GULP template data */
/* TODO - ability to edit (via another dialog) */
label = gtk_label_new("Potentials");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.0f);
label = gtk_label_new("Species data");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.0f);
label = gtk_label_new("Element data");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.0f);

/* right vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

g_string_sprintf(line,"%d",data->gulp.num_potentials);
label = gtk_label_new(line->str);
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
g_string_sprintf(line,"%d",data->gulp.num_species);
label = gtk_label_new(line->str);
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
g_string_sprintf(line,"%d",data->gulp.num_elem);
label = gtk_label_new(line->str);
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);

/* setup the filename entry */
frame = gtk_frame_new("Files");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(gulp_dialog->win)->vbox),frame,FALSE,FALSE,0);
gtk_container_set_border_width(GTK_CONTAINER(frame),3);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame),hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox),3);

/* left vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

/* temporary GULP input filename */
label = gtk_label_new(" Job input file ");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
label = gtk_label_new(" Dump file ");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
label = gtk_label_new(" Potential library ");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);

/* right vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

/* temporary input */
entry = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(entry), data->gulp.temp_file);
gtk_box_pack_start (GTK_BOX (vbox), entry, TRUE, TRUE, 0);
/* event */
gtk_signal_connect_object(GTK_OBJECT (entry), "changed",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) entry);
gtk_object_set_data(GTK_OBJECT(entry), "key", (gpointer) TEMP_FILE);
gtk_object_set_data(GTK_OBJECT(entry), "ptr", (gpointer) data);

/* dump file */
entry = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(entry), data->gulp.dump_file);
gtk_box_pack_start (GTK_BOX (vbox), entry, TRUE, TRUE, 0);
/* event */
gtk_signal_connect_object(GTK_OBJECT (entry), "changed",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) entry);
gtk_object_set_data(GTK_OBJECT(entry), "key", (gpointer) DUMP_FILE);
gtk_object_set_data(GTK_OBJECT(entry), "ptr", (gpointer) data);

/* potential library */
entry = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(entry),data->gulp.lib_file);
gtk_box_pack_start (GTK_BOX (vbox), entry, TRUE, TRUE, 0);
/* event */
gtk_signal_connect_object(GTK_OBJECT (entry), "changed",
                          GTK_SIGNAL_FUNC (gulp_keyword),
                         (gpointer) entry);
gtk_object_set_data(GTK_OBJECT(entry), "key", (gpointer) LIB_FILE);
gtk_object_set_data(GTK_OBJECT(entry), "ptr", (gpointer) data);


/* setup the filename entry */
frame = gtk_frame_new("Details");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(gulp_dialog->win)->vbox),frame,FALSE,FALSE,0);
gtk_container_set_border_width(GTK_CONTAINER(frame),3);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame),hbox);

/* left vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

/* temporary GULP input filename */
label = gtk_label_new("Structure name");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
label = gtk_label_new("Lattice energy (eV)");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
if (data->periodic == 2)
  {
  label = gtk_label_new("Surface bulk energy (eV)");
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
  label = gtk_label_new("Surface energy");
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
  label = gtk_label_new("Attachment energy");
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
  }

/* right vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

entry = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(entry),data->basename);
gtk_box_pack_start (GTK_BOX (vbox), entry, TRUE, TRUE, 0);
/* CURRENT */
gtk_signal_connect_object(GTK_OBJECT (entry), "changed",
                          GTK_SIGNAL_FUNC (name_entry_changed),
                         (gpointer) entry);
gtk_object_set_data(GTK_OBJECT(entry), "ptr", (gpointer) data);


data->gulp.energy_entry = gtk_entry_new_with_max_length(LINELEN);
if (data->gulp.free)
  g_string_sprintf(line,"%f (free energy)",data->gulp.energy);
else
  g_string_sprintf(line,"%f",data->gulp.energy);

gtk_entry_set_text(GTK_ENTRY(data->gulp.energy_entry),line->str);
gtk_box_pack_start (GTK_BOX (vbox), data->gulp.energy_entry, TRUE, TRUE, 0);
gtk_entry_set_editable(GTK_ENTRY(data->gulp.energy_entry), FALSE);

if (data->periodic == 2)
  {
  data->gulp.sbe_entry = gtk_entry_new_with_max_length(LINELEN);
  g_string_sprintf(line,"%f",data->gulp.sbulkenergy);
  gtk_entry_set_text(GTK_ENTRY(data->gulp.sbe_entry),line->str);
  gtk_box_pack_start (GTK_BOX (vbox), data->gulp.sbe_entry, TRUE, TRUE, 0);
  gtk_entry_set_editable(GTK_ENTRY(data->gulp.sbe_entry), FALSE);

  data->gulp.esurf_entry = gtk_entry_new_with_max_length(LINELEN);
  g_string_sprintf(line,"%f    %s",data->gulp.esurf, data->gulp.esurf_units);
  gtk_entry_set_text(GTK_ENTRY(data->gulp.esurf_entry),line->str);
  gtk_box_pack_start (GTK_BOX (vbox), data->gulp.esurf_entry, TRUE, TRUE, 0);
  gtk_entry_set_editable(GTK_ENTRY(data->gulp.esurf_entry), FALSE);

  data->gulp.eatt_entry = gtk_entry_new_with_max_length(LINELEN);
  g_string_sprintf(line,"%f    %s",data->gulp.eatt, data->gulp.eatt_units);
  gtk_entry_set_text(GTK_ENTRY(data->gulp.eatt_entry),line->str);
  gtk_box_pack_start (GTK_BOX (vbox), data->gulp.eatt_entry, TRUE, TRUE, 0);
  gtk_entry_set_editable(GTK_ENTRY(data->gulp.eatt_entry), FALSE);
  }

/* TODO - choose foreground or background run ?*/
/* terminating buttons */
button = gtk_button_new_with_label ("Run");
gtk_box_pack_start (GTK_BOX (GTK_DIALOG (gulp_dialog->win)->action_area),
                                         button, FALSE, TRUE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (run_gulp), (gpointer) id);

button = gtk_button_new_with_label ("Close");
gtk_box_pack_start (GTK_BOX (GTK_DIALOG (gulp_dialog->win)->action_area),
                                         button, FALSE, TRUE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (close_dialog),
                          (gpointer) id);
/* done */
gtk_widget_show_all(gulp_dialog->win);

g_string_free(line, TRUE);
pick_model(data->number);
}

