/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/param.h>
#ifdef __sun
#include <sys/dirent.h>
#else
#include <sys/dir.h>
#endif

#include "gdis.h"

#define DEBUG_MORE 0
#define MAX_KEYS 15

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/*********************/
/* write a GULP file */
/*********************/
#define DEBUG_WRITE_GULP 0
gint write_gulp(gchar *file, struct model_pak *data)
{
gint i, j, r1size;
gchar *cmd, *tail, *libfile, *name, fake_tail[] = "";
gfloat vec[3];
GString *line;
FILE *fp;

g_return_val_if_fail(data != NULL, 1);

if (data->gulp.num_potentials)
  name = g_strdup("none");
else
  {
/* test for library existance */
  libfile = data->gulp.libfile;
  if (!access(libfile, R_OK))
    {
#if DEBUG_WRITE_GULP
  printf("Found: %s ",libfile);
#endif
  name = g_strdup(g_basename(libfile));
/* was it in cwd or somewhere else? */
    if (!access(name, R_OK))
      {
#if DEBUG_WRITE_GULP
printf("in cwd.\n");
#endif
      }
    else
      {
/* elsewhere - copy it to cwd */
#if DEBUG_WRITE_GULP
printf("elsewhere.\n");
#endif
      cmd = g_strdup_printf("cp %s %s",libfile,name);
      system(cmd);
      g_free(cmd);
      }
    }
  else
    {
/* missing library - copy default to cwd */
    printf("Warning: %s not found, using default library.\n", libfile);
    name = g_strdup(LIBRARY);
    cmd = g_strdup_printf("cp %s/%s .",INSTALL, LIBRARY);
    system(cmd);
    g_free(cmd);
    }
  }
#if DEBUG_WRITE_GULP
printf("Library: %s\n",name);
#endif

/* off we go */
fp = fopen(file,"wt");

/* run type */
line = g_string_new(NULL);
switch(data->gulp.run)
  {
  case E_SINGLE:
    g_string_assign(line,"single");
    break;
  case E_OPTIMIZE:
    g_string_assign(line,"opti");
    break;
  case MD:
    g_string_assign(line,"md");
    break;
/* no run -> single pt (valid default?) */
  default:
    g_string_assign(line,"single");
    break;
  }

switch(data->gulp.method)
  {
  case CONP:
    g_string_append(line," conp");
    break;
  case CONV:
    g_string_append(line," conv");
    break;
  default:
    break;
  }

switch(data->gulp.optimiser)
  {
  case CONJ_OPT:
    g_string_append(line," conjugate");
    break;
  case RFO_OPT:
    g_string_append(line," rfo");
    break;
  }

switch(data->gulp.coulomb)
  {
  case MOLE:
    g_string_append(line," mole");
    break;
  case MOLMEC:
    g_string_append(line," molmec");
    break;
  case MOLQ:
    g_string_append(line," molq");
    break;
  default:
    break;
  }

/* additional keywords */
if (data->gulp.free)
  g_string_append(line," free");
if (data->gulp.zsisa)
  g_string_append(line," zsisa");
if (data->gulp.compare)
  g_string_append(line," comp");

/* print out the keywords line */
fprintf(fp,"%s\n\n",line->str);

/* optional data */
if (data->gulp.maxcyc > 0)
  fprintf(fp,"maxcyc %d\n\n",data->gulp.maxcyc);
if (data->gulp.minimiser)
  fprintf(fp,"%s\n\n",data->gulp.minimiser);
/* if no potentials then use library */
if (!data->gulp.num_potentials)
  fprintf(fp,"library %s\n\n",name);
fprintf(fp,"name %s\n\n",data->basename);

/* md stuff */
if (data->gulp.temp > FRACTION_TOLERANCE)
  fprintf(fp,"temperature %f\n",data->gulp.temp);

switch(data->gulp.ensemble)
  {
  case NVE:
    fprintf(fp,"ensemble nve\n");
    break;
  case NVT:
    fprintf(fp,"ensemble nvt\n");
    break;
  case NPT:
    fprintf(fp,"ensemble npt\n");
    break;
  }

/* NB: gulp matrices are transposed wrt gdis */
switch(data->periodic)
  {
  case 3:
    if (data->construct_pbc)
      {
      fprintf(fp,"\nvectors\n");
      fprintf(fp,"%15.10f %15.10f %15.10f\n",
                  data->latmat[0], data->latmat[3], data->latmat[6]);
      fprintf(fp,"%15.10f %15.10f %15.10f\n",
                  data->latmat[1], data->latmat[4], data->latmat[7]);
      fprintf(fp,"%15.10f %15.10f %15.10f\n",
                  data->latmat[2], data->latmat[5], data->latmat[8]);
      }
    else
      {
      fprintf(fp,"\ncell\n");
      fprintf(fp,"%f %f %f  %f %f %f\n",data->pbc[0],data->pbc[1],data->pbc[2],
                data->pbc[3]*180/PI, data->pbc[4]*180/PI, data->pbc[5]*180/PI);
      }
  break;
  case 2:
    if (data->construct_pbc)
      {
      fprintf(fp,"\nsvectors\n");
      fprintf(fp,"%f  %f\n%f  %f\n",data->latmat[0], data->latmat[3],
                                    data->latmat[1], data->latmat[4]);
      }
    else
      {
      fprintf(fp,"\nscell\n");
      fprintf(fp,"%f %f %f\n",data->pbc[0], data->pbc[1],
                              180.0*data->pbc[5]/PI);
      }
  break;
  }

/* new - some models may be 2D periodic & fractional (eg 2D xtl) */
/* which will cause problems with z & 0 pbc -> save as cartesian */
switch(data->periodic)
  {
  case 2:
    if (data->fractional)
      fprintf(fp,"sfractional region 1\n");
    else
      fprintf(fp,"cartesian region 1\n");
    break;
/* 3 or none */
  default:
    if (data->fractional)
      fprintf(fp,"fractional\n");
    else
      fprintf(fp,"cartesian\n");
    break;
  }

/* write coords */
r1size=0;
for (i=0 ; i<data->num_atoms ; i++)
  {
  if ((data->atoms+i)->status & DELETED)
    continue;

/* only print tail data if GULP style, not (eg) BIOSYM */
  if (data->id != GULP)
    tail = fake_tail;
  else
    tail = (data->atoms+i)->tail;

/* periodic/isolated */
  if (data->fractional)
    {
    if (!(data->atoms+i)->primary)
      continue;
/* NEW - print to 6 dp, as this appears to be needed for things like 1/3 */
/* handle the saving of 2D fractional model */
    switch (data->periodic)
      {
      case 3:
        fprintf(fp,"%-4s core %10.6f %10.6f %10.6f %s\n",
                                      (data->atoms+i)->element,
                                      (data->atoms+i)->x,
                                      (data->atoms+i)->y,
                                      (data->atoms+i)->z,
                                      tail);
        break;
      case 2:
/* get coordinates */
        VEC3SET(vec, (data->atoms+i)->x,(data->atoms+i)->y,(data->atoms+i)->z);
/* apply lattice vectors (convert ie frac -> cart for z only) */
        vecmat(data->latmat,vec); 
        fprintf(fp,"%-4s core %10.6f %10.6f %10.6f %s\n",
                                      (data->atoms+i)->element,
                                      (data->atoms+i)->x,
                                      (data->atoms+i)->y,
                                      vec[2],
                                      tail);
        break;
      default:
        printf("save_gulp() error, bad fractional model.\n");
      }

    if ((data->atoms+i)->has_shell)
      {
      j = (data->atoms+i)->idx_shell;
      if (data->id != GULP)
        tail = fake_tail;
      else
        tail = (data->shells+j)->tail;
      fprintf(fp,"%-4s shel %10.6f %10.6f %10.6f %s\n",
                                        (data->shells+j)->element,
                                        (data->shells+j)->x,
                                        (data->shells+j)->y,
                                        (data->shells+j)->z,
                                        tail);
      } 
    }
  else
    {
    switch(data->periodic)
      {
/* only write region 1a on this iteration */
      case 2:
        if ((data->atoms+i)->region != REGION1A)
          break;
        r1size++;
      default:
/* get coordinates */
        VEC3SET(vec, (data->atoms+i)->x,(data->atoms+i)->y,(data->atoms+i)->z);
/* apply lattice vectors (converting frac -> cart) */
        vecmat(data->latmat,vec); 
        fprintf(fp,"%-4s core %10.5f %10.5f %10.5f %s\n",
                                      (data->atoms+i)->element,
                                      vec[0], vec[1], vec[2],
                                      tail);
        if ((data->atoms+i)->has_shell)
          {
/* NB: don't inc r1size - GULP only counts cores in ascertaining r2 position */
          j = (data->atoms+i)->idx_shell;
          if (data->id != GULP)
            tail = fake_tail;
          else
            tail = (data->shells+j)->tail;

/* get coordinates */
VEC3SET(vec, (data->shells+j)->x,(data->shells+j)->y,(data->shells+j)->z);
/* apply lattice vectors (converting frac -> cart) */
        vecmat(data->latmat,vec); 
        fprintf(fp,"%-4s shel %10.5f %10.5f %10.5f %s\n",
                                      (data->shells+j)->element,
                                      vec[0], vec[1], vec[2],
                                      tail);
          }
      } 
    }
  }       /* for i - all atoms */

/* 2D periodic, now write all region 2 atoms (if any) */
if (data->periodic == 2)
  {
  if (r1size < data->num_atoms)
    {
/* header */
    if (data->fractional)
      fprintf(fp,"sfractional region 2\n");
    else
      fprintf(fp,"cartesian region 2\n");
/* coord write */
    for (i=0 ; i<data->num_atoms ; i++)
      {
/* only write region 2a on this iteration */
     if ((data->atoms+i)->region != REGION2A)
       continue;
/* setup stuff after coords */
     if (data->id != GULP)
       tail = fake_tail;
     else
       tail = (data->atoms+i)->tail;
/* get coordinates */
      VEC3SET(vec, (data->atoms+i)->x,(data->atoms+i)->y,(data->atoms+i)->z);
/* apply lattice vectors (converting frac -> cart) */
      vecmat(data->latmat,vec); 
      fprintf(fp,"%-4s core %10.5f %10.5f %10.5f %s\n",
             (data->atoms+i)->element, vec[0], vec[1], vec[2], tail);
/* assoc shell */
      if ((data->atoms+i)->has_shell)
        {
        j = (data->atoms+i)->idx_shell;
        if (data->id != GULP)
          tail = fake_tail;
        else
          tail = (data->shells+j)->tail;
/* get coordinates */
VEC3SET(vec, (data->shells+j)->x,(data->shells+j)->y,(data->shells+j)->z);
/* apply lattice vectors (converting frac -> cart) */
        vecmat(data->latmat,vec); 
        fprintf(fp,"%-4s shel %10.5f %10.5f %10.5f %s\n", 
               (data->shells+j)->element, vec[0], vec[1], vec[2], tail);
        }
      }
    }
/* write dhkl for Eatt calcs */
  if (data->surface.dspacing > 0.1 && !data->gulp.no_eatt)
    fprintf(fp, "\ndhkl %f\n",data->surface.dspacing);

/* write surface bulk energy for Esurf calcs */
  if (!data->gulp.no_esurf)
    fprintf(fp, "\nsbulkenergy %f\n",data->gulp.sbulkenergy);
  }

/* makes it easier for GDIS to read back in */
fprintf(fp,"\n");

/* SG info */
if (data->periodic == 3)
  fprintf(fp,"space\n%s\n\n",g_strstrip(data->sginfo.spacename));

/* extra data */
if (data->gulp.num_elem)
  {
  fprintf(fp,"elements\n");
  for (i=0 ; i<data->gulp.num_elem ; i++)
    fprintf(fp,"%s\n",*(data->gulp.elem+i));
  fprintf(fp,"end\n\n");
  }
if (data->gulp.num_species)
  {
  fprintf(fp,"species\n");
  for (i=0 ; i<data->gulp.num_species ; i++)
    fprintf(fp,"%s\n",*(data->gulp.species+i));
  fprintf(fp,"end\n\n");
  }
for (i=0 ; i<data->gulp.num_potentials ; i++)
  fprintf(fp,"%s\n",*(data->gulp.pot+i));
if (g_strncasecmp(data->gulp.dump_file, "none", 4))
  fprintf(fp,"\ndump %s\n",data->gulp.dump_file);

if (data->gulp.extra)
  {
  fprintf(fp, "\n#--- Additional options/data\n");
  fprintf(fp, "%s\n", data->gulp.extra);
  }

/* done */
fclose(fp);
g_free(name);
g_string_free(line, TRUE);
return(0);
}

/*****************/
/* debugging aid */
/*****************/
void print_gulp_flags(gint *list, gint size)
{
gint i;

printf("Flags:");
for (i=0 ; i<size ; i++)
  {
/* active? */
  if (*(list+i))
    {
    switch(i)
      {
      case NAME:
        printf(" name");
        break;
      case SURFACE_CELL:
        printf(" scell");
        break;
      case CELL:
        printf(" cell");
        break;
      case FRAC:
        printf(" frac");
        break;
      case S_FRAC:
        printf(" sfrac");
        break;
      case CART:
        printf(" cart");
        break;
      case DUMP_FILE:
        printf(" dump");
        break;
      case SURFACE_VECTORS:
        printf(" svec");
        break;
      case LATTICE_VECTORS:
        printf(" lvec");
        break;
      }
    }
  }
printf("\n");
}

/****************************/
/* new gulp parsing routine */
/****************************/
#define DEBUG_LOAD_GULP 0
gint load_gulp(FILE *fp, gint model, gchar *inp_file)
{
gint i, j, nc=0, ns=0;
gint current, type, region, code;
gint gen_flag, new_flag, flag_count, old_count, *flag;
gint run, method, optimiser, coulomb, maxcyc, free, zsisa, compare;
gint num_potentials, num_species, num_elem, num_keyword;
gint *keyword;
gfloat vec1[3], vec2[3], vec3[3];
gchar **buff, *tmp, line[LINELEN];
gchar *species[MAX_POTENTIALS], *elem[MAX_POTENTIALS];
gchar *pot[MAX_POTENTIALS];
gchar *minimiser;
GString *pad, *extra;
struct model_pak *data;
struct elem_pak elem_data;

pad = g_string_new(NULL);
/* append lines we don't recognize to this string */
extra = g_string_new("\n");

/* make sure this is assigned b4 parse as we don't */
/* want to encounter data that goes into a model ptr b4 */
/* one has been assigned - which may not happen if keyword */
/* order is changed around */
data = model_ptr(model, ASSIGN);
current = model-1;

/* setup */
num_keyword=num_keys();
flag = g_malloc(num_keyword * sizeof(gint));
for (i=0 ; i<num_keyword ; i++)
  *(flag+i) = 0;
flag_count = 0;
old_count=0;

/* no default gulp run type */
/* so we don't force anything that wasn't in the file */
run = -1;
method = -1;
optimiser = -1;
coulomb = NOBUILD;
maxcyc = 0;
num_potentials=num_species=num_elem=0;
minimiser=NULL;
region = REGION1A;
free = zsisa = compare = FALSE;

/* main option search */
keyword = get_keyword(line, MAX_KEYS);
for(;;)
  {
/* search for keyword(s) */
  while (!*keyword)
    {
    if (fgetline(fp, line))
      goto done;
    g_free(keyword);
    keyword = get_keyword(line, MAX_KEYS);

/* save lines that have no keywords (for later printing) */
    if (!*keyword)
      {
/* TODO - ignore lines with only one char (ie '\n') */
      g_string_sprintfa(extra,"%s",line);
      }
    }
/* search for starting keywords */
  if (*keyword)
    {
/* PROCESS FOR FLAG SETTING */
    for (i=1 ; i<=*keyword ; i++)
      {
      switch(*(keyword+i))
        {
/* these are to be applied to all loaded models */
        case E_SINGLE:
        case E_OPTIMIZE:
        case MD:
          run = *(keyword+i);
          break;
        case FREE_ENERGY:
          free = TRUE;
          break;
        case ZSISA:
          zsisa = TRUE;
          break;
        case COMPARE:
          compare = TRUE;
          break;
        case CONP:
        case CONV:
          method = *(keyword+i);
          break;
        case CONJ_OPT:
        case RFO_OPT:
/* avoid switch option triggering this */
          if (i>1)
            if (*(keyword+i-1) != SWITCH_MINIMISER)
              optimiser = *(keyword+i);
          break;
        case MOLE:
        case MOLMEC:
        case MOLQ:
          coulomb = *(keyword+i);
          break;
        case MAXCYC:
          sscanf(get_token_pos(line,1),"%d",&maxcyc);
          break;
        case SWITCH_MINIMISER:
          if (!minimiser)
            minimiser = g_strdup(line);
          break;
        case S_FRAC:
        case FRAC:
          buff = get_tokens(line, 4);
          if (g_strncasecmp("region",*(buff+1),6) == 0)
            {
            if (g_strncasecmp("1",*(buff+2),1) == 0)
              region = REGION1A;
            else if (g_strncasecmp("2",*(buff+2),1) == 0)
              region = REGION2A;
            else
              printf("Unrecognized region in [%s]\n",g_strstrip(line));
            }
          g_strfreev(buff);
          data->region_empty[region] = FALSE; 
/* only change flags if it's region 1 */
/* that way region 2 -> keep on reading as if nothing happened */
/* (apart from the fact that we now have a different region number) */
          if (region == REGION1A)
            flag[*(keyword+i)]++;
          break;
        case CART:
          buff = get_tokens(line, 4);
          if (g_strncasecmp("region",*(buff+1),6) == 0)
            {
            if (g_strncasecmp("1",*(buff+2),1) == 0)
              region = REGION1A;
            else if (g_strncasecmp("2",*(buff+2),1) == 0)
              region = REGION2A;
            else
              printf("Unrecognized region in [%s]\n",g_strstrip(line));
            }
          g_strfreev(buff);
          data->region_empty[region] = FALSE; 
/* begin new model only if it's region 1 */
          if (region == REGION1A)
            flag[CART]++;
          break;

/* other keywords that can initiate a new model */
        case CELL:
        case NAME:
        case SURFACE_CELL:
        case SURFACE_VECTORS:
        case LATTICE_VECTORS:
          flag[*(keyword+i)]++;
          break;
        }
      old_count = flag_count;
      }

#if DEBUG_LOAD_GULP
print_gulp_flags(flag, num_keyword);
#endif

/* reset found flag */
    *keyword=0;
/* PROCESS FOR MODEL CREATION */
    new_flag=flag_count=0;
    for (i=0 ; i<num_keyword ; i++)
      {
/* repeated keyword -> next model in file */
      if (flag[i] == 2)
        {
/* check we have minimum data */
        if (!flag[CART])
          if (!flag[FRAC] && !flag[CELL])
            {
            printf("***ERROR: incomplete data.\n");
            return(1);
            }
        new_flag++;
        }
/* sum the number of starting keywords currently found */
      if (flag[i])
        flag_count++;
      }
/* NEW - new model only on 1st occurrence of any data start keyword */
    if (flag_count == 1 && old_count == 0)
      new_flag++;

/* are we required to make a new model? */
    if (new_flag)
      {
      current++;
#if DEBUG_LOAD_GULP
printf("Creating new model: %d\n",current);
#endif
/* special case for 1st time around (ie ptr is already assigned) */
      if (current > model)
        data = model_ptr(current, ASSIGN);
      if (data == NULL)
        return(2);
      data->id = GULP;
/* init indices for num cores & shells read into this model */
      nc=ns=0;
/* setup name */
      g_free(data->basename);
      g_string_sprintf(pad,"%s_%-2d",strdup_no_extension(g_basename(inp_file)),
                                                              current-model+1);
      data->basename = g_strdup(pad->str);
      data->gulp.srcfile = strdup(inp_file);
/* reset flags unless first model (special case) */
      if (new_flag && current > model)
        {
/* new models beyond the first may be triggered by a repeat */
/* option (eg the second occurrance of name) in which case */
/* we only want to dec the flag (ie from 2 to 1 - NB: all other */
/* flags should be set to zero) so the parser won't make a new */
/* model if it bumps into another main data option (eg cell, cart...) */
        for (i=0 ; i<num_keyword ; i++)
          {
/* NEW - safer way of initializing flags */
          if (flag[i] >= 2)
            flag[i] = 1;
          else
            flag[i] = 0;
          }
        }
      }
    }

/* PROCESS FOR DATA - options only (ie 1st keyword) */
    switch(*(keyword+1))
      {
/* avoid keyword parsing the crap in a title/end block */
      case TITLE:
        g_string_sprintfa(extra, "%s", line);
        while (*(keyword+1) != END)
          {
          if (fgetline(fp,line))
            goto done;
          g_string_sprintfa(extra, "%s", line);
          keyword = get_keyword(line, MAX_KEYS);
          }
        break;
/* hack for getting an associated .trg file */
      case OUTPUT:
        buff = get_tokens(line, 3);
        if (g_strncasecmp(*(buff+1),"traj",4) == 0)
          {
          g_free(data->gulp.trj_file);
          data->gulp.trj_file = g_strdup(*(buff+2));
          data->animation = TRUE;
          }
        g_strfreev(buff);
        break;
/* avoid keyword parsing the crap in an observables/end block */
      case GULP_OBSERVABLES:
        g_string_sprintfa(extra, "%s", line);
        while (*(keyword+1) != END)
          {
          if (fgetline(fp,line))
            goto done;
          g_string_sprintfa(extra, "%s", line);
          keyword = get_keyword(line, MAX_KEYS);
          }
        break;
/* NEW stuff */
        case ENSEMBLE:
          buff = get_tokens(line, 3);
          if (g_strncasecmp(*(buff+1),"nve",3) == 0)
            data->gulp.ensemble = NVE;
          if (g_strncasecmp(*(buff+1),"nvt",3) == 0)
            data->gulp.ensemble = NVT;
          if (g_strncasecmp(*(buff+1),"npt",3) == 0)
            data->gulp.ensemble = NPT;
          g_strfreev(buff);
          break;
        case TEMPERATURE:
          buff = get_tokens(line, 3);
          sscanf(*(buff+1),"%f",&(data->gulp.temp));
          g_strfreev(buff);
          break;
        case SUPER_CELL:
          buff = get_tokens(line, 5);
          sscanf(*(buff+1),"%d",&(data->gulp.super[0]));
          sscanf(*(buff+2),"%d",&(data->gulp.super[1]));
          sscanf(*(buff+3),"%d",&(data->gulp.super[2]));
          g_strfreev(buff);
          break;
/* model specific data - the *data ptr MUST be allocated */
        case NAME:
          g_free(data->basename);
          data->basename = g_strstrip(g_strdup(get_token_pos(line,1)));
          strcpy(data->filename, data->basename);
          break;
        case SPACE:
/* anything on the rest of the line? */
          buff = get_tokens(line, 2);
/* position of actual space data */
          i=1;
/* no, use the next line instead */
          if (!strlen(*(buff+1)))
            {
            fgetline(fp,line);
            i--;
            }
          tmp = g_strdup(g_strstrip(get_token_pos(line, i)));
          gen_flag=0;
          for (i=0 ; i<strlen(tmp) ; i++)
            {
/* any alphabetic chars => space *name* */
            if (isalpha((int) *(tmp+i)))
              {
              g_free(data->sginfo.spacename);
              data->sginfo.spacename = g_strdup_printf("%s",tmp);
/* indicate that name should used in lookup */
              data->sginfo.spacenum = -1;
              gen_flag++;
              break;
              }
            }
/* no alphabetic chars, assume space group number */
          if (!gen_flag)
            sscanf(tmp,"%3d",&data->sginfo.spacenum);

          g_strfreev(buff);
          g_free(tmp);
          break;
        case ORIGIN:
          buff = get_tokens(line, 5);
/* FIXME - could be coords & not just the cell choice */
          if (strlen(*(buff+3)))
            printf("Warning: unsupported origin specification.\n");
          else
            sscanf(*(buff+1),"%d",&(data->sginfo.cellchoice));
          g_strfreev(buff);
          break;
        case CELL:
/* found it - get next line (if possible) */
          if (fgetline(fp,line))
            return(3);
          buff = get_tokens(line, 6);
          sscanf(*(buff+0),"%f",&(data->pbc[0]));
          sscanf(*(buff+1),"%f",&(data->pbc[1]));
          sscanf(*(buff+2),"%f",&(data->pbc[2]));
          sscanf(*(buff+3),"%f",&(data->pbc[3]));
          sscanf(*(buff+4),"%f",&(data->pbc[4]));
          sscanf(*(buff+5),"%f",&(data->pbc[5]));
          g_strfreev(buff);
/* hack to determine if 2D or 3D */
/* FIXME - possible for pbc[3] or pbc[4] to be 0 instead */
          if (fabs(data->pbc[2]) < FRACTION_TOLERANCE)
            data->periodic = 2;
          else
            data->periodic = 3;
/* degrees -> radians */
          data->pbc[3] *= PI/180.0;
          data->pbc[4] *= PI/180.0;
          data->pbc[5] *= PI/180.0;
          break;

/* read in gulp surface cell */
        case SURFACE_CELL:
          if (fgetline(fp,line))
            return(4);
          buff = get_tokens(line, 5);
          sscanf(*(buff+0),"%f",&data->pbc[0]);
          sscanf(*(buff+1),"%f",&data->pbc[1]);
          sscanf(*(buff+2),"%f",&data->pbc[5]);
          data->pbc[5] *= PI/180.0;
          data->pbc[2] = 0.0;
          data->periodic = 2;
          break;

/* read in gulp surface vectors */
/* NB: gdis wants transposed gulp matrix */
        case SURFACE_VECTORS:
          if (fgetline(fp,line))
            return(4);
          buff = get_tokens(line, 3);
/* is this one always colinear to x? (FIXME - no) */
          sscanf(*(buff+0),"%f",&vec1[0]);
          sscanf(*(buff+1),"%f",&vec2[0]);
          g_strfreev(buff);
          if (fgetline(fp,line))
            return(5);
          buff = get_tokens(line, 3);
          sscanf(*(buff+0),"%f",&vec1[1]);
          sscanf(*(buff+1),"%f",&vec2[1]);
          g_strfreev(buff);
          VEC3SET(&(data->latmat[0]), vec1[0], vec1[1], 0.0);
          VEC3SET(&(data->latmat[3]), vec2[0], vec2[1], 0.0);
          VEC3SET(&(data->latmat[6]), 0.0, 0.0, 1.0);
          data->construct_pbc = TRUE;
          data->periodic = 2;
          make_xlat(data);
          break;

/* read in gulp lattice vectors */
/* NB: gdis wants transposed gulp matrix */
        case LATTICE_VECTORS:
          if (fgetline(fp,line))
            return(4);
          buff = get_tokens(line, 4);
          sscanf(*(buff+0),"%f",&vec1[0]);
          sscanf(*(buff+1),"%f",&vec2[0]);
          sscanf(*(buff+2),"%f",&vec3[0]);
          g_strfreev(buff);
          if (fgetline(fp,line))
            return(4);
          buff = get_tokens(line, 4);
          sscanf(*(buff+0),"%f",&vec1[1]);
          sscanf(*(buff+1),"%f",&vec2[1]);
          sscanf(*(buff+2),"%f",&vec3[1]);
          g_strfreev(buff);
          if (fgetline(fp,line))
            return(4);
          buff = get_tokens(line, 4);
          sscanf(*(buff+0),"%f",&vec1[2]);
          sscanf(*(buff+1),"%f",&vec2[2]);
          sscanf(*(buff+2),"%f",&vec3[2]);
          g_strfreev(buff);
/* NEW - use the supplied lattice matrix (it will be normalized, however) */
          ARR3SET(&(data->latmat[0]), vec1);
          ARR3SET(&(data->latmat[3]), vec2);
          ARR3SET(&(data->latmat[6]), vec3);
/* NB: data->pbc's are constructed in make_xlat() */
          data->construct_pbc = TRUE;
          data->periodic = 3;
          make_xlat(data);
          break;

/* main coord read in */
        case S_FRAC:
        case FRAC:
          data->fractional = TRUE;
        case CART:
/* au check */
          buff = get_tokens(line,7);
          gen_flag = 0;
          if (g_strncasecmp("au",*(buff+1),2) == 0)
            gen_flag++;
/* get new line */
          if (fgetline(fp, line))
            return(4);
/* insufficient items => stupid gulp frac/cart \n number \n coords format */
          buff = get_tokens(line,5);
          if (!strlen(*(buff+3)))
            if (fgetline(fp, line))
              return(5);
          g_strfreev(buff);
          g_free(keyword);
          keyword = get_keyword(line, MAX_KEYS);
/* core,shell indices */
          i=j=0;
          while (!*keyword)
            {
            buff = get_tokens(line,7);
/* terminate on a blank/insufficient line */
            if (!strlen(*(buff+3)))
              goto last;
/* if an atom type (ie core/shell) is specified - adjust data positions */
            type = 0;
            if (g_strncasecmp(*(buff+1),"core",4) == 0)
              type=1;
            if (g_strncasecmp(*(buff+1),"shel",4) == 0)
              type=2;
/* read appropriately */
            switch(type)
              {
              case 0:
              case 1:
/* new core */
                if (nc >= data->atom_limit)
                  mem_grow(data, ATOM, 10);
/* setup */
                (data->atoms+nc)->status = NORMAL;
                (data->atoms+nc)->primary = TRUE;
                (data->atoms+nc)->orig = TRUE;
                (data->atoms+nc)->region = region;
                (data->atoms+nc)->offx = 0;
                (data->atoms+nc)->offy = 0;
/* read the atom type & corresponding coords */
/* NB: assumes if there is core/shell l_pos = 1 always */
                sscanf(*buff,"%s",(data->atoms+nc)->element);
                sscanf(*buff,"%s",(data->atoms+nc)->label);
/* cope with gulp's fractional representation */
                (data->atoms+nc)->x = str_to_float(*(buff+1+type));
                (data->atoms+nc)->y = str_to_float(*(buff+2+type));
                (data->atoms+nc)->z = str_to_float(*(buff+3+type));
/* colourize & special bond calc if partial occ. (shells?) */
                if (strlen(*(buff+5+type)))
                  {
                  data->has_sof = TRUE;
                  (data->atoms+nc)->sof = str_to_float(*(buff+5+type));
                  }
                else
                  (data->atoms+nc)->sof = 1.0;

/* copy_items(..ALL) replacement */
                tmp = g_strdup(get_token_pos(line,4+type));
                strcpy((data->atoms+nc)->tail, g_strstrip(tmp));
                g_free(tmp);
/* atomic units used? */
                if (gen_flag)
                  {
                  (data->atoms+nc)->x *= AU2ANG;
                  (data->atoms+nc)->y *= AU2ANG;
                  (data->atoms+nc)->z *= AU2ANG;
                  }
/* setup (colour, code etc.) */
                elem_seek(nc, ATOM, data);
#if DEBUG_LOAD_GULP
/*
printf("[%d] %s %f %f %f [%d]\n", nc,
             (data->atoms+nc)->element,
             (data->atoms+nc)->x,
             (data->atoms+nc)->y,
             (data->atoms+nc)->z,
             (data->atoms+nc)->region);
*/
#endif
                nc++;
                break;
              case 2:
/* new shell */
                if (ns >= data->shell_limit)
                  mem_grow(data, SHELL, 10);
/* setup the shell */
                (data->shells+ns)->status = NORMAL;
                (data->shells+ns)->primary = TRUE;
                (data->shells+ns)->orig = TRUE;
                (data->shells+ns)->region = region;
                (data->shells+ns)->offx = 0;
                (data->shells+ns)->offy = 0;
/* read the atom type & corresponding coords */
/* NB: assumes its always [atom] [core/shell] x y z */
                sscanf(*buff,"%s",(data->shells+ns)->label);
                sscanf(*buff,"%s",(data->shells+ns)->element);
                (data->shells+ns)->x = str_to_float(*(buff+2));
                (data->shells+ns)->y = str_to_float(*(buff+3));
                (data->shells+ns)->z = str_to_float(*(buff+4));
/* copy_items(..ALL) replacement */
                tmp = g_strdup(get_token_pos(line,5));
                strcpy((data->shells+ns)->tail, g_strstrip(tmp));
                g_free(tmp);
/* GULP atomic units hack */
                if (gen_flag)
                  {
                  (data->shells+ns)->x *= AU2ANG;
                  (data->shells+ns)->y *= AU2ANG;
                  (data->shells+ns)->z *= AU2ANG;
                  }
                elem_seek(ns, SHELL, data);
                ns++;
                break;
              }
            g_strfreev(buff);
/* get new line & check for keywords */
            if (fgetline(fp, line))
              break;
            g_free(keyword);
            keyword = get_keyword(line, MAX_KEYS);
            }
last:;
/* finished coord read */
          data->num_asym = data->num_orig = data->num_atoms = nc;
          data->num_shells = ns;
          break;
/* one liners... */
        case D_HKL:
          buff = get_tokens(line, 2);
          sscanf(*(buff+1),"%f",&data->surface.dspacing);
          g_strfreev(buff);
        case SBULK_ENERGY:
          buff = get_tokens(line, 2);
          sscanf(*(buff+1),"%f",&data->gulp.sbulkenergy);
          g_strfreev(buff);
          break;
        case TOTAL_ENERGY:
          buff = get_tokens(line, 2);
          sscanf(*(buff+1),"%f",&data->gulp.energy);
          g_strfreev(buff);
          break;
        case SPECIES:
          while (!fgetline(fp,line))
            {
/* end on a keyword */
            g_free(keyword);
            keyword = get_keyword(line, MAX_KEYS);
            if (*keyword)
              break;
            buff = get_tokens(line, 1);
/* is the 1st item an element? (no - end) */
            i = elem_type(*buff);
            g_strfreev(buff);
            if (i && num_species < MAX_POTENTIALS)
              {
              *(species+num_species) = g_strdup(g_strstrip(line));
              num_species++;
              }
            else
              break;
            }
          break;
        case ELEMENT:
          while (!fgetline(fp,line))
            {
/* end on a keyword */
            g_free(keyword);
            keyword = get_keyword(line, MAX_KEYS);
            if (*keyword)
              break;
            buff = get_tokens(line, 3);
/* end if insufficient data */
            if (!strlen(*(buff+2)))
              {
              g_strfreev(buff);
              break;
              }
/* is the 2nd item an element? (symbol check) */
            if (!elem_type(*(buff+1)))
              {
/* FIXME - a better way to cope with ugly fortran ** possibility? */
              if (*(*(buff+1)) == '*')
                {
                g_strfreev(buff);
/* ignore - skip to next line */
                continue;
                }
/* no - atomic number check */
              if (atoi(*(buff+1)) < 0 || atoi(*(buff+1)) > sysenv.num_elements)
                {
/* not a recognized element - exit element loop */
                g_strfreev(buff);
                break;
                }
              }
/* otherwise save if we have the room */
            if (num_elem < MAX_POTENTIALS)
              {
              *(elem+num_elem) = g_strdup(g_strstrip(line));
              num_elem++;
              }
            g_strfreev(buff);
            }
          break;
        case DUMP_FILE:
/* ignore, for the moment */
          break;
        case HARMONIC:
        case MORSE:
        case BUCKINGHAM:
        case LENNARD:
        case THREE:
        case FOUR:
        case TORSION:
        case OUTOFPLANE:
        case SPRING:
          if (num_potentials < MAX_POTENTIALS)
            {
/* TODO - use only one gchar ptr for potential storage */
/* store potential type line */
            g_string_sprintf(pad,"%s",line);
/* NEW - while not a keyword, keep tacking the lines on */
            while (!fgetline(fp,line))
              {
/* end on a keyword */
              g_free(keyword);
              keyword = get_keyword(line, MAX_KEYS);
              if (*keyword)
                break;
              g_string_sprintfa(pad,"%s",line);
              }
            *(pot+num_potentials) = g_strdup(pad->str);
            num_potentials++;

/* if missing next line - don't include */
/*
            if (fgetline(fp,line))
              g_free(*(pot1+num_potentials));
            else
              {
              *(pot2+num_potentials) = g_strdup(g_strstrip(line));
              num_potentials++;
              }
*/
            }
          break;
        }
  }
/* finished with file reading */
done:;

/* summary */
#if DEBUG_LOAD_GULP
printf("-------------------------------------\n");
printf("         Structures found: %d [%d;%d]\n",current-model+1,model,current);
printf("                run type : %d\n",run);
printf("             method type : %d\n",method);
printf("          optimiser type : %d\n",optimiser);
printf("coulomb subtraction type : %d\n",coulomb);
printf("         iteration limit : %d\n",maxcyc);
printf("        potentials found : %d\n",num_potentials);
printf("      species data found : %d\n",num_species);
printf("      element data found : %d\n",num_elem);
printf("-------------------------------------\n");
#endif

/* init models & exit */
for (i=model ; i<=current ; i++)
  {
  data = model_ptr(i, RECALL);
  data->mode = FREE;
  if (data == NULL)
    return(7);
  strcpy(data->filename, inp_file);

/* transfer universal gulp data */
  data->gulp.run = run;
  data->gulp.method = method;
  data->gulp.optimiser = optimiser;
  data->gulp.coulomb = coulomb;
  data->gulp.maxcyc = maxcyc;
  data->gulp.free = free;
  data->gulp.zsisa = zsisa;
  data->gulp.compare = compare;
  if (minimiser)
    data->gulp.minimiser = g_strdup(g_strstrip(minimiser));
  if (num_potentials)
    {
    data->gulp.num_potentials = num_potentials;
    data->gulp.pot = g_malloc(num_potentials*sizeof(gchar *));
    for (j=0 ; j<num_potentials ; j++)
      *(data->gulp.pot+j) = g_strdup(*(pot+j));
    }
  if (num_species)
    {
    data->gulp.num_species = num_species;
    data->gulp.species = g_malloc(num_species*sizeof(gchar *));
    for (j=0 ; j<num_species ; j++)
      *(data->gulp.species+j) = g_strdup(*(species+j));
    }
  if (num_elem)
    {
    data->gulp.num_elem = num_elem;
    data->gulp.elem = g_malloc(num_elem*sizeof(gchar *));
    for (j=0 ; j<num_elem ; j++)
      {
      *(data->gulp.elem+j) = g_strdup(*(elem+j));
/* process any cova data */
      if (g_strncasecmp(*(elem+j), "cova", 4) == 0)
        {
        buff = get_tokens(*(elem+j), 4);
        code = elem_type(*(buff+1));
/* no match - perhaps atomic number was specified */
        if (!code)
          code = atoi(*(buff+1));
        if (!get_elem_data(code, &elem_data, data))
          {
          elem_data.cova = str_to_float(*(buff+2));
#if DEBUG_LOAD_GULP
printf("New cova for %s: %f\n",elem_data.element,elem_data.cova);
#endif
          put_elem_data(&elem_data, data);
          }

        g_strfreev(buff);
        }
      }
    }
/* extra (unrecognized lines) */
  if (strlen(extra->str))
    data->gulp.extra = g_strdup(extra->str);

/* display init */
  data->axes_on = TRUE;
  if (data->periodic)
    {
    data->cell_on = TRUE;
    data->axes_type = OTHER;
/* space group init here */
    update_shells(data);
    if (!data->sginfo.spacenum)
      {
#if DEBUG_LOAD_GULP
printf("No Space Group found: assuming P1.\n");
#endif
      data->sginfo.spacenum=1;
      }
    if (genpos(data))
      printf("Error in Space Group lookup.\n");
    }

/* main coord setup */
  init_objs(INIT_COORDS, data);
/* preserve orig cartesian coords */
  save_cart(data);
/* if surface - special case as gdis saves as mixed frac/cart */
  if (data->periodic && !data->fractional)
    {
    latmat_fudge(data);
    init_objs(INIT_COORDS, data);
    }
/* setup for display */
  calc_bonds(data);
  calc_mols(data);
  new_tree_item(i, APPEND);
  }

/* clean up & exit */
/* attached data to all model, now we can free it */
for (i=0 ; i<num_potentials ; i++)
  {
/*
printf("%s\n",*(pot+i));
*/
  g_free(*(pot+i));
  }
for (i=0 ; i<num_elem ; i++)
  {
/*
printf("%s\n",*(elem+i));
*/
  g_free(*(elem+i));
  }
for (i=0 ; i<num_species ; i++)
  {
/*
printf("%s\n",*(species+i));
*/
  g_free(*(species+i));
  }

/*
printf("super = %d %d %d\n",
  data->gulp.super[0],data->gulp.super[1],data->gulp.super[2]);
printf("ensemble = %d\n",data->gulp.ensemble);
printf("temp = %f\n",data->gulp.temp);
*/

g_string_free(pad, TRUE);
g_string_free(extra, TRUE);
g_free(keyword);
return(0);
}

/*****************************/
/* generic gulp output parse */
/*****************************/
#define DEBUG_READ_GULP_OUTPUT 0
gint load_gulp_output(FILE *fp, gint model, gchar *inp_file)
{
gint i, j, m, n, flag;
gint sflag=0, dflag, count, region;
gchar line[LINELEN], **buff;
struct model_pak *data;

/* TODO - how to sensibly handle multiple structures */
/* alloc, setup & read */
data = model_ptr(model, ASSIGN);
if (data == NULL)
  return(1);

data->id = GULPOUT;
data->mode = FREE;
g_free(data->basename);
data->basename = strdup_no_extension(g_basename(inp_file));
strcpy(data->filename, g_basename(inp_file));
data->region_empty[REGION1A] = FALSE;

/* look for interesting output */
flag=0;
while(!fgetline(fp,line))
  {
/* space group search */
  if (g_strncasecmp("  Space group ",line,14) == 0)
    {
    g_free(data->sginfo.spacename);
    data->sginfo.spacename = g_strstrip(g_strdup(get_token_pos(line,4)));
#if DEBUG_READ_GULP_OUTPUT
printf("space group: %s\n", data->sginfo.spacename);
#endif
    sflag++;
    }

/* minimized energy search */
  if (g_strncasecmp("  Final energy =",line,16) == 0)
    {
    buff = get_tokens(line,4);
    sscanf(*(buff+3),"%f",&data->gulp.energy);
    flag++;
    g_strfreev(buff);
    }

/* surface energy search */
  if (g_strncasecmp("  Surface energy ",line,17) == 0)
    {
    buff = get_tokens(line,8);
    sscanf(*(buff+5),"%f",&data->gulp.esurf);
    g_free(data->gulp.esurf_units);
    data->gulp.esurf_units = g_strdup(*(buff+6));
    g_strfreev(buff);
    }

/* attachment energy search 1 */
  if (g_strncasecmp("  Attachment energy ",line,20) == 0)
    {
    buff = get_tokens(line,6);
    data->gulp.eatt = str_to_float(*(buff+3));
    g_free(data->gulp.eatt_units);
    data->gulp.eatt_units = g_strdup(*(buff+4));
    g_strfreev(buff);
    }

/* attachment energy search 2 */
  if (g_strncasecmp("  Attachment energy/mol",line,23) == 0)
    {
    buff = get_tokens(line,6);
    data->gulp.eatt = str_to_float(*(buff+3));
    g_free(data->gulp.eatt_units);
    data->gulp.eatt_units = g_strdup("eV/mol");
    g_strfreev(buff);
    }

/* search for gnorm */
  if (g_strncasecmp("  Final Gnorm  =",line,16) == 0)
    {
    buff = get_tokens(line,4);
    sscanf(*(buff+3),"%f",&data->gulp.gnorm);
    flag++;
    g_strfreev(buff);
    }

/* single point energy search */
  if (g_strncasecmp("  Total lattice energy",line,22) == 0)
    {
    buff = get_tokens(line,6);
/* if nothing on the rest of this line, then should be on the next */
    if (!strlen(*(buff+5)))
      {
      while(!fgetline(fp,line))
        if (g_strncasecmp("    Primitive unit cell",line,20) == 0)
          break;
      g_strfreev(buff);
      buff = get_tokens(line, 6);
      }
    if (g_strncasecmp("eV",*(buff+5),2) == 0)
      sscanf(*(buff+4),"%f",&data->gulp.energy);
    g_strfreev(buff);
    }

/* single point free energy search */
  if (g_strncasecmp("  Total free energy",line,19) == 0)
    {
    buff = get_tokens(line,6);
    if (g_strncasecmp("eV",*(buff+5),2) == 0)
      sscanf(*(buff+4),"%f",&data->gulp.energy);
    g_strfreev(buff);
    }

/* minimized coords search */
/* FIXME - unminimized coords? */
  dflag=0;
  if (g_strncasecmp("  Final fractional/Cartesian coordinates",line,40) == 0
   || g_strncasecmp("  Mixed fractional/Cartesian coordinates",line,40) == 0)
    {
    data->periodic = 2;
    data->fractional = TRUE;
    dflag++;
    }
  if (g_strncasecmp("  Final asymmetric unit coordinates",line,35) == 0
   || g_strncasecmp("  Final fractional coordinates",line,30) == 0
   || g_strncasecmp("  Fractional coordinates of asymmetric",line,38) == 0)
    {
    data->periodic = 3;
    data->fractional = TRUE;
    dflag++;
    }
/* read in? */
  if (dflag)
    {
/* skip the two (minimum number) ----- dividers */
    region = REGION1A;
    count=0;
    while (count < 2)
      {
      if (fgetline(fp,line))
        return(2);
      if (g_strncasecmp("-------", line, 7) == 0)
        count++;
      }

/* loop until we run out of atomic data lines */
    i=j=n=0;
    flag=0;
/* read as many atoms as we can */
    while(!fgetline(fp,line))
      {
/* TODO - dont exit if ----- (could be region divider) */
      if (g_strncasecmp("-------", line, 7) == 0)
        continue;
/* CURRENT - region */
      if (g_strncasecmp("  Region ", line, 9) == 0)
        {
        buff = get_tokens(line, 3);
        switch(*(*(buff+1)))
          {
          case '1':
#if DEBUG_READ_GULP_OUTPUT
printf("Labelling region 1...\n");
#endif
            region = REGION1A;
            break;
          case '2':
#if DEBUG_READ_GULP_OUTPUT
printf("Labelling region 2...\n");
#endif
            region = REGION2A;
            break;
          default:
            printf("WARNING: unknown region specification/\n");
            region = REGION1A;
          }
        g_strfreev(buff);
        data->region_empty[region] = FALSE;
        continue;
        }

/* exit if atom number is not consecutive */ 
      buff = get_tokens(line,6);
      sscanf(*buff,"%d",&m);
      if (m != ++n)
        break;
      else
        {
/* read the data in - assume the same order */
/* core */
        if (g_strncasecmp("c", *(buff+2), 1) == 0)
          {
          if (i >= data->atom_limit)
            mem_grow(data, ATOM, 10);

          (data->atoms+i)->status = NORMAL;
          (data->atoms+i)->primary = TRUE;
          (data->atoms+i)->orig = TRUE;
          (data->atoms+i)->region = region;
          (data->atoms+i)->offx = 0;
          (data->atoms+i)->offy = 0;
/* TODO - check if a value is in the file */
          (data->atoms+i)->sof = 1.0;

          sscanf(*(buff+1),"%s",(data->atoms+i)->element);
          sscanf(*(buff+1),"%s",(data->atoms+i)->label);
          sscanf(*(buff+3),"%f",&(data->atoms+i)->x);
          sscanf(*(buff+4),"%f",&(data->atoms+i)->y);
          sscanf(*(buff+5),"%f",&(data->atoms+i)->z);
          strcpy((data->atoms+i)->tail, "");

#if DEBUG_READ_GULP_OUTPUT
printf("coords: %f %f %f\n", 
      (data->atoms+i)->x, (data->atoms+i)->y, (data->atoms+i)->z);
#endif

/* elem setup (colour, code etc.) */
          elem_seek(i, ATOM, data);
          i++;
          }

/* shell */
        if (g_strncasecmp("s", *(buff+2), 1) == 0)
          {
          if (j >= data->shell_limit)
            mem_grow(data, SHELL, 10);

          (data->shells+j)->status = NORMAL;
          (data->shells+j)->primary = TRUE;
          (data->shells+j)->orig = TRUE;
          (data->shells+j)->region = region;
          (data->shells+j)->offx = 0;
          (data->shells+j)->offy = 0;

          sscanf(*(buff+1),"%s",(data->shells+j)->element);
          sscanf(*(buff+1),"%s",(data->shells+j)->label);
          sscanf(*(buff+3),"%f",&(data->shells+j)->x);
          sscanf(*(buff+4),"%f",&(data->shells+j)->y);
          sscanf(*(buff+5),"%f",&(data->shells+j)->z);
          strcpy((data->shells+j)->tail, "");

#if DEBUG_READ_GULP_OUTPUT
printf("coords: %f %f %f\n", 
      (data->shells+j)->x, (data->shells+j)->y, (data->shells+j)->z);
#endif

/* elem setup (colour, code etc.) */
          elem_seek(j, SHELL, data);
          j++;
          }
        }
      g_strfreev(buff);
      }
    data->num_asym = data->num_orig = data->num_atoms = i;
    data->num_shells = j;
#if DEBUG_READ_GULP_OUTPUT
printf("Retrieved %d atoms & %d shells (periodic).\n",i,j);
#endif
    }

/* cell parameters - search 1 */
  if (g_strncasecmp("  Final cell parameters",line,22) == 0)
    {
/* skip to data */
    fgetline(fp,line);
    fgetline(fp,line);
/* get cell lengths */
    for (i=0 ; i<6 ; i++)
      {
      if (fgetline(fp,line))
        break;
      buff = get_tokens(line,3);
      sscanf(*(buff+1),"%f",&data->pbc[i]);
      g_strfreev(buff);
      }
/* convert to radians */
    data->pbc[3] *= PI/180.0;
    data->pbc[4] *= PI/180.0;
    data->pbc[5] *= PI/180.0;
    }

/* cell parameters - search 2 */
  if (g_strncasecmp("  Non-primitive lattice parameters",line,34) == 0)
    {
/* skip to data */
    if (fgetline(fp,line))
      break;
    if (fgetline(fp,line))
      break;
/* get cell lengths */
    buff = get_tokens(line,9);
    sscanf(*(buff+2),"%f",&data->pbc[0]);
    sscanf(*(buff+5),"%f",&data->pbc[1]);
    sscanf(*(buff+8),"%f",&data->pbc[2]);
    if (fgetline(fp,line))
      break;
/* get cell angles */
    g_strfreev(buff);
    buff = get_tokens(line,6);
    sscanf(*(buff+1),"%f",&data->pbc[3]);
    sscanf(*(buff+3),"%f",&data->pbc[4]);
    sscanf(*(buff+5),"%f",&data->pbc[5]);
/* convert to radians */
    data->pbc[3] *= PI/180.0;
    data->pbc[4] *= PI/180.0;
    data->pbc[5] *= PI/180.0;
    g_strfreev(buff);
    }

/* cell parameters - search 3 */
  if (g_strncasecmp("  Final surface cell parameters ",line,32) == 0)
    {
    data->periodic = 2;
    data->fractional = TRUE;
    data->pbc[2] = 0.0;
    data->pbc[3] = PI/2.0;
    data->pbc[4] = PI/2.0;
/* skip to 1st line of data */
    fgetline(fp,line);
    fgetline(fp,line);
    if (fgetline(fp,line))
      break;
    buff = get_tokens(line,4);
    sscanf(*(buff+1),"%f",&data->pbc[0]);
    g_strfreev(buff);

    if (fgetline(fp,line))
      break;
    buff = get_tokens(line,4);
    sscanf(*(buff+1),"%f",&data->pbc[1]);
    g_strfreev(buff);

    if (fgetline(fp,line))
      break;
    buff = get_tokens(line,4);
    sscanf(*(buff+1),"%f",&data->pbc[5]);
/* convert to radians */
    data->pbc[5] *= PI/180.0;
    g_strfreev(buff);
    }

/* cell parameters - search 4 (eg a CONV surface calc) */
  if (g_strncasecmp("  Surface cell parameters ",line, 26) == 0)
    {
    data->periodic = 2;
    data->fractional = TRUE;
    data->pbc[2] = 0.0;
    data->pbc[3] = PI/2.0;
    data->pbc[4] = PI/2.0;
/* skip to 1st line of data */
    fgetline(fp,line);
    if (fgetline(fp,line))
      break;
    buff = get_tokens(line,6);
    sscanf(*(buff+2),"%f",&data->pbc[0]);
    sscanf(*(buff+5),"%f",&data->pbc[5]);
/* convert to radians */
    data->pbc[5] *= PI/180.0;
    g_strfreev(buff);
    if (fgetline(fp,line))
      break;
    buff = get_tokens(line,10);
    sscanf(*(buff+2),"%f",&data->pbc[1]);
    g_strfreev(buff);
    }

/* cell parameters - search 5 (unrelaxed) */
  if (g_strncasecmp("  Cell parameters (Angstroms/Degrees):",line,38) == 0)
    {
/* skip blank line */
    fgetline(fp,line);
/* get cell lengths */
    for (i=0 ; i<3 ; i++)
      {
      if (fgetline(fp,line))
        break;
      buff = get_tokens(line,7);
/* length */
      sscanf(*(buff+2),"%f",&data->pbc[i]);
/* angle */
      sscanf(*(buff+5),"%f",&data->pbc[i+3]);
      g_strfreev(buff);
      }
/* convert to radians */
    data->pbc[3] *= PI/180.0;
    data->pbc[4] *= PI/180.0;
    data->pbc[5] *= PI/180.0;
    }

/* cell parameters - search 6 */
  if (g_strncasecmp("  Surface Cartesian vectors (Angstroms) :",line,41) == 0)
    {
    data->periodic = 2;
    data->construct_pbc = TRUE;
/* skip blank line */
    fgetline(fp,line);
/* get vec 1 */
    fgetline(fp,line);
    buff = get_tokens(line,4);
    data->latmat[0] = str_to_float(*buff);
    data->latmat[3] = str_to_float(*(buff+1));
    data->latmat[6] = 0.0;
    g_strfreev(buff);
/* get vec 2 */
    fgetline(fp,line);
    buff = get_tokens(line,4);
    data->latmat[1] = str_to_float(*buff);
    data->latmat[4] = str_to_float(*(buff+1));
    data->latmat[7] = 0.0;
/* set vec 3 */
    data->latmat[2] = 0.0;
    data->latmat[5] = 0.0;
    data->latmat[8] = 1.0;
    g_strfreev(buff);
    }

/* cell parameters - search 7 */
#if GOKY
  if (g_strncasecmp("  Cartesian lattice vectors (Angstroms) :",line,40) == 0)
    {
    data->periodic = 3;
    data->construct_pbc = TRUE;
/* skip blank line */
    fgetline(fp,line);
/* get vec 1 */
    fgetline(fp,line);
    buff = get_tokens(line,4);
    data->latmat[0] = str_to_float(*buff);
    data->latmat[3] = str_to_float(*(buff+1));
    data->latmat[6] = str_to_float(*(buff+2));
    g_strfreev(buff);
/* get vec 2 */
    fgetline(fp,line);
    buff = get_tokens(line,4);
    data->latmat[1] = str_to_float(*buff);
    data->latmat[4] = str_to_float(*(buff+1));
    data->latmat[7] = str_to_float(*(buff+2));
    g_strfreev(buff);
/* get vec 3 */
    fgetline(fp,line);
    buff = get_tokens(line,4);
    data->latmat[2] = str_to_float(*(buff));
    data->latmat[5] = str_to_float(*(buff+1));
    data->latmat[8] = str_to_float(*(buff+2));
    g_strfreev(buff);
    }
#endif

/* cell parameters - search 8 */
  if (g_strncasecmp("  Primitive cell parameters :",line,29) == 0)
    {
    data->periodic = 3;
/* skip blank line */
    fgetline(fp,line);
/* get line 1 */
    fgetline(fp,line);
    buff = get_tokens(line,12);
    data->pbc[0] = str_to_float(*(buff+8));
    data->pbc[3] = str_to_float(*(buff+11));
    g_strfreev(buff);
/* get line 2 */
    fgetline(fp,line);
    buff = get_tokens(line,12);
    data->pbc[1] = str_to_float(*(buff+8));
    data->pbc[4] = str_to_float(*(buff+11));
    g_strfreev(buff);
/* get line 2 */
    fgetline(fp,line);
    buff = get_tokens(line,12);
    data->pbc[2] = str_to_float(*(buff+8));
    data->pbc[5] = str_to_float(*(buff+11));
    g_strfreev(buff);
/* convert to radians */
    data->pbc[3] *= PI/180.0;
    data->pbc[4] *= PI/180.0;
    data->pbc[5] *= PI/180.0;
    }

/* isolated coords search */
  if (g_strncasecmp("  Final cartesian coordinates",line,29) == 0)
    {
    data->periodic = 0;
    data->fractional = FALSE;

    for (i=0 ; i<5 ; i++)
      fgetline(fp,line);
/* loop until we run out of atomic data lines */
    i=j=n=0;
    flag=0;
/* don't read any more than num_atoms -> memory allocation problems! */
    while(!fgetline(fp,line))
      {
      buff = get_tokens(line,6);
/* exit if atom number is not consecutive */ 
      sscanf(*buff,"%d",&m);
      if (m != ++n)
        flag=1;
      else
        {
/* read the data in - assume the same order */
/* core */
        if (g_strncasecmp("c",*(buff+2),1) == 0)
          {
          if (i >= data->atom_limit)
            mem_grow(data, ATOM, 10);

          (data->atoms+i)->status = NORMAL;
          (data->atoms+i)->primary = TRUE;
          (data->atoms+i)->orig = TRUE;
          (data->atoms+i)->region = REGION1A;
          (data->atoms+i)->offx = 0;
          (data->atoms+i)->offy = 0;

          sscanf(*(buff+1),"%s",(data->atoms+i)->element);
          sscanf(*(buff+1),"%s",(data->atoms+i)->label);
          sscanf(*(buff+3),"%f",&(data->atoms+i)->x);
          sscanf(*(buff+4),"%f",&(data->atoms+i)->y);
          sscanf(*(buff+5),"%f",&(data->atoms+i)->z);
          strcpy((data->atoms+i)->tail, "");

#if DEBUG_READ_GULP_OUTPUT
printf("coords: %f %f %f\n", 
      (data->atoms+i)->x, (data->atoms+i)->y, (data->atoms+i)->z);
#endif
          elem_seek(i, ATOM, data);
          i++;
          }

/* shell */
        if (g_strncasecmp("s",*(buff+2),1) == 0)
          {
          if (j >= data->shell_limit)
            mem_grow(data, SHELL, 10);

          (data->shells+j)->status = NORMAL;
          (data->shells+j)->primary = TRUE;
          (data->shells+j)->orig = TRUE;
          (data->shells+j)->region = REGION1A;
          (data->shells+j)->offx = 0;
          (data->shells+j)->offy = 0;

          sscanf(*(buff+1),"%s",(data->shells+j)->element);
          sscanf(*(buff+1),"%s",(data->shells+j)->label);
          sscanf(*(buff+3),"%f",&(data->shells+j)->x);
          sscanf(*(buff+4),"%f",&(data->shells+j)->y);
          sscanf(*(buff+5),"%f",&(data->shells+j)->z);
          strcpy((data->shells+j)->tail, "");

#if DEBUG_READ_GULP_OUTPUT
printf("coords: %f %f %f\n", 
      (data->shells+j)->x, (data->shells+j)->y, (data->shells+j)->z);
#endif
          elem_seek(j, SHELL, data);
          j++;
          }
        }
      g_strfreev(buff);
      }
    data->num_asym = data->num_orig = data->num_atoms = i;
    data->num_shells = j;
#if DEBUG_READ_GULP_OUTPUT
printf("Retrieved %d atoms & %d shells (cluster).\n",i,j);
#endif
    }
  }      /* while data in file */

#if DEBUG_READ_GULP_OUTPUT
P3VEC("cell lengths ",&data->pbc[0]);
P3VEC("cell angles ",&data->pbc[3]);
#endif

/* surfaces are always const vol */
if (data->periodic == 2)
  data->gulp.method = CONV;

/* init for display */
data->axes_on = TRUE;
if (data->periodic)
  {
  data->cell_on = TRUE;
  data->axes_type = OTHER;
/* space group init here */
  update_shells(data);
  if (!sflag)
    {
#if DEBUG_READ_GULP_OUTPUT
printf("No Space Group found: assuming P1.\n");
#endif
    data->sginfo.spacenum=1;
    }
  if (genpos(data))
    printf("Error in Space Group lookup.\n");
  }
init_objs(INIT_COORDS, data);
if (data->periodic && !data->fractional)
  {
  latmat_fudge(data);
  init_objs(INIT_COORDS, data);
  }

/* preserve cartesian coords */
save_cart(data);

calc_bonds(data);
calc_mols(data);

new_tree_item(data->number, APPEND);

return(0);
}

/*********************************/
/* GULP dynamics trajectory file */
/*********************************/
/* NB: this loads data that is not to become a model */
/* but rather is associated with an existing model */
/* this routine will become a get_frame type call */
#define DEBUG_LOAD_TRJ 0
gint load_gulp_trj(FILE *fp, struct model_pak *data, gint read_header)
{
gint i, j;
int pad, num_atoms, periodic;
gint swap=FALSE;
double version, time, ke, pe, temp, *x[6];
double cell, velc;

/* macro for fortran start/end record padding */
#define RECORD fread(&pad, sizeof(int), 1, fp)

/* record 1 - version */
if (read_header)
  {
  RECORD;
  fread(&version, sizeof(version), 1, fp);
#if DEBUG_LOAD_TRJ
printf("version: %f\n", version);
#endif
  if (fabs(version) > 100.0)
    {
    swap_bytes(&version, sizeof(version));
    if (fabs(version) > 100.0)
      {
      printf("Error: file seems garbled.\n");
      return(1);
      }
    swap = TRUE;
    }
  RECORD;
/* record 2 */
  RECORD;
/* # of atoms + shells */
  fread(&num_atoms, sizeof(num_atoms), 1, fp);
  if (swap)
    swap_bytes(&num_atoms, sizeof(num_atoms));
/* dimension */
  fread(&periodic, sizeof(periodic), 1, fp);
  if (swap)
    swap_bytes(&periodic, sizeof(periodic));
  RECORD;
  }
else
  {
  num_atoms = data->num_atoms + data->num_shells;
  periodic = data->periodic;
  }

/* alloc for x,y,z & vx, vy, vz */
for (j=0 ; j<6 ; j++)
  x[j] = g_malloc(num_atoms * sizeof(double));

/* read one frame */
RECORD;
if (!fread(&time, sizeof(time), 1, fp))
  return(1);
if (swap)
  swap_bytes(&time, sizeof(time));
if (!fread(&ke, sizeof(ke), 1, fp))
  return(1);
if (swap)
  swap_bytes(&ke, sizeof(ke));
if (!fread(&pe, sizeof(pe), 1, fp))
  return(1);
if (swap)
  swap_bytes(&pe, sizeof(pe));
if (!fread(&temp, sizeof(temp), 1, fp))
  return(1);
if (swap)
  swap_bytes(&temp, sizeof(temp));
RECORD;

#if DEBUG_LOAD_TRJ
printf("time : %f\n",time);
printf("  ke : %f\n",ke);
printf("  pe : %f\n",pe);
printf("temp : %f\n",temp);
#endif

/* loop over x,y,z & assoc velocity components */
for (j=0 ; j<6 ; j++)
  {
/* NB: cores first, then shells */
  RECORD;
  for (i=0 ; i<num_atoms ; i++)
    if (!fread(x[j]+i, sizeof(double), 1, fp))
      return(1);
  RECORD;
  }
/*
for (i=0 ; i<num_atoms ; i++)
  printf("%f %f %f\n", *(x[0]+i),*(x[1]+i),*(x[2]+i));
*/

if (data->gulp.ensemble == NPT)
  {
/* get cell vectors */
  RECORD;
  for (i=0 ; i<9 ; i++)
    fread(&cell, sizeof(double), 1, fp);
  RECORD;
/* in help.txt the 9 should be nstrains, but wtf is nstrains??? */
  RECORD;
  for (i=0 ; i<9 ; i++)
    fread(&velc, sizeof(double), 1, fp);
  RECORD;
  }

/* copy data */
/* TODO - parsing checks */
j=0;
for (i=0 ; i<data->num_atoms ; i++)
  {
  (data->atoms+i)->x = *(x[0]+j);
  (data->atoms+i)->y = *(x[1]+j);
  (data->atoms+i)->z = *(x[2]+j);
  j++;
  }
for (i=0 ; i<data->num_shells ; i++)
  {
  (data->shells+i)->x = *(x[0]+j);
  (data->shells+i)->y = *(x[1]+j);
  (data->shells+i)->z = *(x[2]+j);
  j++;
  }
/* TODO - cell changes */
/*
if (data->gulp.ensemble == NPT)
*/

/* FIXME - gulp has the super keyword, which will change the number of atoms */
if (j != num_atoms)
  printf("Warning: mismatch in cores/shells\n");

#if DEBUG_LOAD_TRJ
printf("coords read in: %d (%d+%d)\n",num_atoms,data->num_atoms,data->num_shells);
printf("periodic in %d dimensions.\n",periodic);
#endif

/* convert cartesian to fractional */
latmat_fudge(data);

/* clean up */
for (j=0 ; j<6 ; j++)
  g_free(x[j]);

return(0);
}

