/*
** Copyright (C) 08 Jun 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/**
 * cmd_v.elements is used for execv(), this file tries to keep
 * the implementation details separated from the rest (for
 * simplicity and in case something needs to be changed)
 **/

#include <gtk/gtk.h>
#include <stdlib.h>
#include <string.h>

#include "linebuffer.h"
#include "vector_commands.h"
#include "globals.h"

#define INITIAL_CMD_SPACE 16
#define INCREASE_CMD_SPACE 16

/* this function adds commands to the cdr_v structure */
void add_option_to_cmd(cmd_v *cmd, const char *option) {
	g_assert(cmd->elements_used <= cmd->elements_allocated);

	if (cmd->elements_used == cmd->elements_allocated) {
		cmd->elements = realloc(cmd->elements, (cmd->elements_allocated
				+ INCREASE_CMD_SPACE)*sizeof(char *));
		cmd->elements_allocated += INCREASE_CMD_SPACE;
	}
	if (NULL == option)
		cmd->elements[cmd->elements_used] = NULL;
	else
		cmd->elements[cmd->elements_used] = strdup(option);
	cmd->elements_used++;
}
/* create a new cmd_v */
cmd_v *get_new_cmd(void) {
	cmd_v *cmd;

	cmd = malloc(sizeof(cmd_v));
	cmd->elements = malloc(sizeof(char *)*INITIAL_CMD_SPACE);
	cmd->elements_used = 0;
	cmd->elements_allocated = INITIAL_CMD_SPACE;

	return cmd;
}
/* destroy cmd_v */
void destroy_cmd(cmd_v *cmd) {
	int i;
	g_assert(NULL != cmd);
	for (i = 0; i < cmd->elements_used; i++)
		free(cmd->elements[i]);
	free(cmd->elements);
	free(cmd);
	cmd = NULL; /* ? */
}

void print_cmd(cmd_v *cmd) {
	int i;
	g_assert(NULL != cmd);
	printf("\n");
	for (i = 0; i < cmd->elements_used; i++)
		printf("%s\n", cmd->elements[i]);
	printf("\n");
}

char *cmd_to_string(cmd_v *cmd) {
	int i;
	char *s;
	g_assert(NULL != cmd);
	s = string_append("", NULL);
	for (i = 0; i < cmd->elements_used; i++) {
		s = string_append(s, cmd->elements[i]);
		s = string_append(s, "\n");
	}
	return s;
}

/* Takes the argument string (path) and escapes every occurence of '\\' and '='
 * with '\\', and returns the escaped string.
 * FIXME: this routine isn't very efficiently implemented
 */
char *escape_graft_chars(const char *path) {
	char *result;

	result = string_append("", NULL);
	for (;*path != '\0'; path++) {
		char tmp[2] = {'\0', '\0'};
		switch (*path) {
			case '=':
				result = string_append(result, "\\\\");
				break;
			case '\\':
				if (('\0' != *(path+1)) && ('\\' == *(path+1))) {
					result = string_append(result, "\\\\");
				}
				break;
			default:
				break;
		}
		tmp[0] = *path;
		result = string_append(result, tmp);
	}

	return result;
}

/* #define GCOMBUST_TEST_CASE */
#ifdef GCOMBUST_TEST_CASE 
/*  gcc -DHAVE_CONFIG_H -I. -I. -I.. -DLOCALEDIR=\"/usr/local/share/locale\" -I../intl    -I/usr/lib/glib/include -I/usr/X11R6/include -Wall -g -Wmissing-prototypes -Wmissing-declarations vector_commands.c -o testcase linebuffer.o -lgtk -lgdk */
#include <assert.h>
int main(int argc, char *argv[]) {
	char *before, *after;

	/* tests for strings to be left untouched */
	before = "/foo/bar";
	after = escape_graft_chars(before);
	assert(!strcmp(before, after));
	free(after);

	before = "foobar.c";
	after = escape_graft_chars(before);
	assert(!strcmp(before, after));
	free(after);

	before = "";
	after = escape_graft_chars(before);
	assert(!strcmp(before, after));
	free(after);

	before = "foo\\bar";
	after = escape_graft_chars(before);
	assert(!strcmp(before, after));
	free(after);

	/* tests for strings to be escaped */
	before = "=";
	after = escape_graft_chars(before);
	assert(!strcmp("\\\\=", after));
	free(after);

	before = "\\\\";
	after = escape_graft_chars(before);
	assert(!strcmp("\\\\\\\\", after));
	free(after);

	before = "foo\\=bar";
	after = escape_graft_chars(before);
	assert(!strcmp("foo\\\\\\=bar", after));
	free(after);

	before = "foo\\\\=bar";
	after = escape_graft_chars(before);
	assert(!strcmp("foo\\\\\\\\\\\\=bar", after));
	free(after);

	return 0;
}

#endif
