{ String handling routines (lower level)

  Copyright (C) 1997-2002 Free Software Foundation, Inc.

  Authors: Frank Heckenbach <frank@pascal.gnu.de>
           Jukka Virtanen <jtv@hut.fi>

  This file is part of GNU Pascal.

  GNU Pascal is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published
  by the Free Software Foundation; either version 2, or (at your
  option) any later version.

  GNU Pascal is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with GNU Pascal; see the file COPYING. If not, write to the
  Free Software Foundation, 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  As a special exception, if you link this file with files compiled
  with a GNU compiler to produce an executable, this does not cause
  the resulting executable to be covered by the GNU General Public
  License. This exception does not however invalidate any other
  reasons why the executable file might be covered by the GNU
  General Public License. }

{$gnu-pascal,I-,X+}

unit String; asmname 'GPC';

interface

uses RTSC;

{ TString is a string type that is used for function results and
  local variables, as long as undiscriminated strings are not
  allowed there. The default size of 2048 characters should be
  enough for file names on any system, but can be changed when
  necessary. It should be at least as big as MAXPATHLEN. }

const
  TStringSize = 2048;
  SpaceCharacters = [' ', #9];
  NewLine = "\n";  { the separator of lines within a string }
  LineBreak = {$if defined (__OS_DOS__) and not defined (__CYGWIN__)}
              "\r\n"
              {$else}
              "\n"
              {$endif};  { the separator of lines within a file }

type
  TString    = String (TStringSize);
  TStringBuf = packed array [0 .. TStringSize] of Char;
  PString    = ^String;
  CharSet    = set of Char;

var
  CParamCount: Integer = 1; asmname '_p_argc';
  {@internal}
  CParametersDummy: array [0 .. 1] of CString = ('GPC program', nil);
  {@endinternal}
  CParameters: PCStrings = PCStrings (@CParametersDummy); asmname '_p_argv';

function  MemCmp      (const s1, s2; Size: SizeType): Integer; asmname 'memcmp';
function  MemComp     (const s1, s2; Size: SizeType): Integer; asmname 'memcmp';
function  MemCompCase (const s1, s2; Size: SizeType): Boolean; asmname '_p_memcmpcase';

procedure UpCaseString    (var s: String);                                        asmname '_p_upcase_string';
procedure LoCaseString    (var s: String);                                        asmname '_p_locase_string';
function  UpCaseStr       (const s: String): TString;                             asmname '_p_upcase_str';
function  LoCaseStr       (const s: String): TString;                             asmname '_p_locase_str';

function  StrEqualCase    (const s1, s2: String): Boolean;                        asmname '_p_strequalcase';

function  Pos             (const SubString, s: String): Integer;                  asmname '_p_pos';
function  LastPos         (const SubString, s: String): Integer;                  asmname '_p_lastpos';
function  PosCase         (const SubString, s: String): Integer;                  asmname '_p_poscase';
function  LastPosCase     (const SubString, s: String): Integer;                  asmname '_p_lastposcase';
function  CharPos         (const Chars: CharSet; const s: String): Integer;       asmname '_p_charpos';
function  LastCharPos     (const Chars: CharSet; const s: String): Integer;       asmname '_p_lastcharpos';

function  PosFrom         (const SubString, s: String; From: Integer): Integer;            asmname '_p_posfrom';
function  LastPosTill     (const SubString, s: String; Till: Integer): Integer;            asmname '_p_lastpostill';
function  PosFromCase     (const SubString, s: String; From: Integer): Integer;            asmname '_p_posfromcase';
function  LastPosTillCase (const SubString, s: String; Till: Integer): Integer;            asmname '_p_lastpostillcase';
function  CharPosFrom     (const Chars: CharSet; const s: String; From: Integer): Integer; asmname '_p_charposfrom';
function  LastCharPosTill (const Chars: CharSet; const s: String; Till: Integer): Integer; asmname '_p_lastcharpostill';

function  IsPrefix        (const Prefix, s: String): Boolean;                    asmname '_p_isprefix';
function  IsSuffix        (const Suffix, s: String): Boolean;                    asmname '_p_issuffix';
function  IsPrefixCase    (const Prefix, s: String): Boolean;                    asmname '_p_isprefixcase';
function  IsSuffixCase    (const Suffix, s: String): Boolean;                    asmname '_p_issuffixcase';

function  CStringLength      (Src: CString): SizeType;                           asmname '_p_strlen';
function  CStringEnd         (Src: CString): CString;                            asmname '_p_strend';
function  CStringNew         (Src: CString): CString;                            asmname '_p_strdup';
function  CStringComp        (s1, s2: CString): Integer;                         asmname '_p_strcmp';
function  CStringCaseComp    (s1, s2: CString): Integer;                         asmname '_p_strcasecmp';
function  CStringLComp       (s1, s2: CString; MaxLen: SizeType): Integer;       asmname '_p_strlcmp';
function  CStringLCaseComp   (s1, s2: CString; MaxLen: SizeType): Integer;       asmname '_p_strlcasecmp';
function  CStringCopy        (Dest, Source: CString): CString;                   asmname '_p_strcpy';
function  CStringCopyEnd     (Dest, Source: CString): CString;                   asmname '_p_strecpy';
function  CStringLCopy       (Dest, Source: CString; MaxLen: SizeType): CString; asmname '_p_strlcpy';
function  CStringMove        (Dest, Source: CString; Count: SizeType): CString;  asmname '_p_strmove';
function  CStringCat         (Dest, Source: CString): CString;                   asmname '_p_strcat';
function  CStringLCat        (Dest, Source: CString; MaxLen: SizeType): CString; asmname '_p_strlcat';
function  CStringChPos       (Src: CString; Ch: Char): CString;                  asmname '_p_strscan';
function  CStringLastChPos   (Src: CString; Ch: Char): CString;                  asmname '_p_strrscan';
function  CStringPos         (s, SubString: CString): CString;                   asmname '_p_strpos';
function  CStringLastPos     (s, SubString: CString): CString;                   asmname '_p_strrpos';
function  CStringCasePos     (s, SubString: CString): CString;                   asmname '_p_strcasepos';
function  CStringLastCasePos (s, SubString: CString): CString;                   asmname '_p_strrcasepos';
function  CStringUpCase      (s: CString): CString;                              asmname '_p_strupper';
function  CStringLoCase      (s: CString): CString;                              asmname '_p_strlower';
function  CStringIsEmpty     (s: CString): Boolean;                              asmname '_p_strempty';
function  NewCString         (const Source: String): CString;                    asmname '_p_newcstring';
function  CStringCopyString  (Dest: CString; const Source: String): CString;     asmname '_p_cstringcopystring';
procedure CopyCString        (Source: CString; var Dest: String);                asmname '_p_copycstring';

function  NewString       (const s: String): PString;                            asmname '_p_newstring';
procedure DisposeString   (p: PString);                                          asmname '_p_dispose';

procedure SetString       (var s: String; Buffer: PChar; Count: Integer);        asmname '_p_set_string';
function  StringOfChar    (Ch: Char; Count: Integer) = s: TString;               asmname '_p_string_of_char';

procedure TrimLeft        (var s: String);                                       asmname '_p_trimleft';
procedure TrimRight       (var s: String);                                       asmname '_p_trimright';
procedure TrimBoth        (var s: String);                                       asmname '_p_trimboth';
function  LTrim           (const s: String): TString;                            asmname '_p_trimleft_str';
function  TrimLeftStr     (const s: String): TString;                            asmname '_p_trimleft_str';
function  TrimRightStr    (const s: String): TString;                            asmname '_p_trimright_str';
function  TrimBothStr     (const s: String): TString;                            asmname '_p_trimboth_str';

function  GetStringCapacity (const s: String): Integer;                          asmname '_p_get_string_capacity';

{ A shortcut for a common use of WriteStr as a function }
function  Integer2String (i: Integer): TString;                                  asmname '_p_Integer2String';

{@internal}
function  BP_UpCase       (ch: Char): Char;                                      attribute (const); asmname '_p_BP_UpCase';

{ Compare strings for equality without padding }
function  StrEQ (const s1, s2: String): Boolean; asmname '_p_x_eq';

{ Compare strings for `less-than' without padding }
function  StrLT (const s1, s2: String): Boolean; asmname '_p_x_lt';

{ Compare strings for equality, padding the shorter string with spaces }
function  StrEQPad (const s1, s2: String): Boolean; asmname '_p_x_str_eq';

{ Compare strings for `less-than', padding the shorter string with spaces }
function  StrLTPad (const s1, s2: String): Boolean; asmname '_p_x_str_lt';

{ @@@@@@ from error.pas }
procedure SetReturnAddress (Address: Pointer); asmname '_p_SetReturnAddress';
procedure RestoreReturnAddress; asmname '_p_RestoreReturnAddress';
{@endinternal}

implementation

inline function BP_UpCase (ch: Char): Char;
begin
  if ch in ['a' .. 'z']
    then BP_UpCase := Pred (ch, Ord ('a') - Ord ('A'))
    else BP_UpCase := ch
end;

procedure UpCaseString (var s: String);
var i: Integer;
begin
  for i := 1 to Length (s) do s[i] := UpCase (s[i])
end;

procedure LoCaseString (var s: String);
var i: Integer;
begin
  for i := 1 to Length (s) do s[i] := LoCase (s[i])
end;

function UpCaseStr (const s: String) = Result: TString;
begin
  Result := s;
  UpCaseString (Result)
end;

function LoCaseStr (const s: String) = Result: TString;
begin
  Result := s;
  LoCaseString (Result)
end;

function MemCompCase (const s1, s2; Size: SizeType): Boolean;
var
  i: Integer;
  a1: array [1 .. Size] of Char absolute s1;
  a2: array [1 .. Size] of Char absolute s2;
begin
  for i := 1 to Size do
    if (a1[i] <> a2[i]) and (LoCase (a1[i]) <> LoCase (a2[i])) then Return False;
  MemCompCase := True
end;

function StrEqualCase (const s1, s2: String): Boolean;
begin
  if Length (s1) <> Length (s2)
    then StrEqualCase := False
    else StrEqualCase := MemCompCase (s1[1], s2[1], Length (s1))
end;

function Pos (const SubString, s: String): Integer;
begin
  Pos := PosFrom (SubString, s, 1)
end;

function LastPos (const SubString, s: String): Integer;
begin
  LastPos := LastPosTill (SubString, s, Length (s))
end;

function PosCase (const SubString, s: String): Integer;
begin
  PosCase := PosFromCase (SubString, s, 1)
end;

function LastPosCase (const SubString, s: String): Integer;
begin
  LastPosCase := LastPosTillCase (SubString, s, Length (s))
end;

function CharPos (const Chars: CharSet; const s: String): Integer;
var i: Integer;
begin
  i := 1;
  while (i <= Length (s)) and not (s[i] in Chars) do Inc (i);
  if i > Length (s) then CharPos := 0 else CharPos := i
end;

function LastCharPos (const Chars: CharSet; const s: String): Integer;
var i: Integer;
begin
  i := Length (s);
  while (i > 0) and not (s[i] in Chars) do Dec (i);
  LastCharPos := i
end;

function PosFrom (const SubString, s: String; From: Integer): Integer;
var m, i, n: Integer;
begin
  m := Max (1, From);
  case Length (SubString) of
    0: PosFrom := From;
    1: begin
         i := m;
         while (i <= Length (s)) and (s[i] <> SubString[1]) do Inc (i);
         if i > Length (s) then PosFrom := 0 else PosFrom := i
       end;
    else
      n := Length (s) - Length (SubString) + 1;
      i := m;
      while (i <= n) and (MemComp (s[i], SubString[1], Length (SubString)) <> 0) do Inc (i);
      if i > n then PosFrom := 0 else PosFrom := i
  end
end;

function LastPosTill (const SubString, s: String; Till: Integer): Integer;
var m, i: Integer;
begin
  m := Max (0, Min (Length (s), Till));
  case Length (SubString) of
    0: LastPosTill := m + 1;
    1: begin
         i := m;
         while (i > 0) and (s[i] <> SubString[1]) do Dec (i);
         LastPosTill := i
       end;
    else
      i := m - Length (SubString) + 1;
      while (i > 0) and (MemComp (s[i], SubString[1], Length (SubString)) <> 0) do Dec (i);
      if i < 0 then LastPosTill := 0 else LastPosTill := i
  end
end;

function PosFromCase (const SubString, s: String; From: Integer): Integer;
var m, i, n: Integer;
begin
  m := Max (1, From);
  case Length (SubString) of
    0: PosFromCase := From;
    1: begin
         i := m;
         while (i <= Length (s)) and (s[i] <> SubString[1]) and (LoCase (s[i]) <> LoCase (SubString[1])) do Inc (i);
         if i > Length (s) then PosFromCase := 0 else PosFromCase := i
       end;
    else
      n := Length (s) - Length (SubString) + 1;
      i := m;
      while (i <= n) and not MemCompCase (s[i], SubString[1], Length (SubString)) do Inc (i);
      if i > n then PosFromCase := 0 else PosFromCase := i
  end
end;

function LastPosTillCase (const SubString, s: String; Till: Integer): Integer;
var m, i: Integer;
begin
  m := Max (0, Min (Length (s), Till));
  case Length (SubString) of
    0: LastPosTillCase := m + 1;
    1: begin
         i := m;
         while (i > 0) and (s[i] <> SubString[1]) and (LoCase (s[i]) <> LoCase (SubString[1])) do Dec (i);
         LastPosTillCase := i
       end;
    else
      i := m - Length (SubString) + 1;
      while (i > 0) and not MemCompCase (s[i], SubString[1], Length (SubString)) do Dec (i);
      if i < 0 then LastPosTillCase := 0 else LastPosTillCase := i
  end
end;

function CharPosFrom (const Chars: CharSet; const s: String; From: Integer): Integer;
var i: Integer;
begin
  i := Max (1, From);
  while (i <= Length (s)) and not (s[i] in Chars) do Inc (i);
  if i > Length (s) then CharPosFrom := 0 else CharPosFrom := i
end;

function LastCharPosTill (const Chars: CharSet; const s: String; Till: Integer): Integer;
var i: Integer;
begin
  i := Max (0, Min (Length (s), Till));
  while (i > 0) and not (s[i] in Chars) do Dec (i);
  LastCharPosTill := i
end;

function IsPrefix (const Prefix, s: String): Boolean;
begin
  { @@ fjf226 } if not (Length (s) >= Length (Prefix)) then IsPrefix := False
    else IsPrefix := EQ (s[1 .. Length (Prefix)], Prefix)
end;

function IsSuffix (const Suffix, s: String): Boolean;
begin
  { @@ fjf226 } if not (Length (s) >= Length (Suffix)) then IsSuffix := False
    else IsSuffix := EQ (s[Length (s) - Length (Suffix) + 1 .. Length (s)], Suffix)
end;

function IsPrefixCase (const Prefix, s: String): Boolean;
begin
  { @@ fjf226 } if not (Length (s) >= Length (Prefix)) then IsPrefixCase := False
    else IsPrefixCase := StrEqualCase (s[1 .. Length (Prefix)], Prefix)
end;

function IsSuffixCase (const Suffix, s: String): Boolean;
begin
  { @@ fjf226 } if not (Length (s) >= Length (Suffix)) then IsSuffixCase := False
    else IsSuffixCase := StrEqualCase (s[Length (s) - Length (Suffix) + 1 .. Length (s)], Suffix)
end;

inline function CStringLength (Src: CString): SizeType;
var Temp: CString;
begin
  if Src = nil then Return 0;
  Temp := Src;
  while Temp^ <> #0 do Inc (Temp);
  CStringLength := Temp - Src
end;

inline function CStringEnd (Src: CString): CString;
var Temp: CString;
begin
  if Src = nil then Return nil;
  Temp := Src;
  while Temp^ <> #0 do Inc (Temp);
  CStringEnd := Temp
end;

function CStringNew (Src: CString): CString;
var
  Size: SizeType;
  Dest: CString;
begin
  if Src = nil then Return nil;
  Size := CStringLength (Src) + 1;
  SetReturnAddress (ReturnAddress (0));
  GetMem (Dest, Size);
  RestoreReturnAddress;
  Move (Src^, Dest^, Size);
  CStringNew := Dest
end;

function CStringLComp (s1, s2: CString; MaxLen: SizeType): Integer;
var c1, c2: Char;
begin
  if s1 = nil then
    if (s2 = nil) or (s2^ = #0)
      then CStringLComp := 0
      else CStringLComp := -1
  else if s2 = nil then
    if s1^ = #0
      then CStringLComp := 0
      else CStringLComp := 1
  else
    begin
      if MaxLen > 0 then
        repeat
          c1 := s1^;
          c2 := s2^;
          Inc (s1);
          Inc (s2);
          if c1 <> c2 then Return Ord (c1) - Ord (c2);
          Dec (MaxLen)
        until (c1 = #0) or (MaxLen = 0);
      CStringLComp := 0
    end
end;

function CStringComp (s1, s2: CString): Integer;
begin
  CStringComp := CStringLComp (s1, s2, MaxInt)
end;

function CStringLCaseComp (s1, s2: CString; MaxLen: SizeType): Integer;
var c1, c2: Char;
begin
  if s1 = nil then
    if (s2 = nil) or (s2^ = #0)
      then CStringLCaseComp := 0
      else CStringLCaseComp := -1
  else if s2 = nil then
    if s1^ = #0
      then CStringLCaseComp := 0
      else CStringLCaseComp := 1
  else
    begin
      if MaxLen > 0 then
        repeat
          c1 := LoCase (s1^);
          c2 := LoCase (s2^);
          Inc (s1);
          Inc (s2);
          if c1 <> c2 then Return Ord (c1) - Ord (c2);
          Dec (MaxLen)
        until (c1 = #0) or (MaxLen = 0);
      CStringLCaseComp := 0
    end
end;

function CStringCaseComp (s1, s2: CString): Integer;
begin
  CStringCaseComp := CStringLCaseComp (s1, s2, MaxInt)
end;

function CStringCopy (Dest, Source: CString): CString;
var Size: SizeType;
begin
  if Source = nil then
    Size := 0
  else
    begin
      Size := CStringLength (Source);
      Move (Source^, Dest^, Size)
    end;
  Dest[Size] := #0;
  CStringCopy := Dest
end;

function CStringCopyEnd (Dest, Source: CString): CString;
var Size: SizeType;
begin
  if Source = nil then
    Size := 0
  else
    begin
      Size := CStringLength (Source);
      Move (Source^, Dest^, Size)
    end;
  Dest[Size] := #0;
  CStringCopyEnd := Dest + Size
end;

function CStringLCopy (Dest, Source: CString; MaxLen: SizeType): CString;
var Size: SizeType;
begin
  if Source = nil then
    Size := 0
  else
    begin
      Size := Min (CStringLength (Source), MaxLen);
      Move (Source^, Dest^, Size)
    end;
  Dest[Size] := #0;
  CStringLCopy := Dest
end;

function CStringMove (Dest, Source: CString; Count: SizeType): CString;
begin
  if Source = nil then
    FillChar (Dest^, Count, 0)
  else
    Move (Source^, Dest^, Count);
  CStringMove := Dest
end;

function CStringCat (Dest, Source: CString): CString;
begin
  CStringCopy (CStringEnd (Dest), Source);
  CStringCat := Dest
end;

function CStringLCat (Dest, Source: CString; MaxLen: SizeType): CString;
var s: SizeType;
begin
  s := CStringLength (Dest);
  CStringLCopy (Dest + s, Source, Max (MaxLen, s) - s);
  CStringLCat := Dest
end;

inline function CStringChPos (Src: CString; Ch: Char): CString;
var Temp: CString;
begin
  if Src = nil then Return nil;
  Temp := Src;
  while (Temp^ <> #0) and (Temp^ <> Ch) do Inc (Temp);
  if Temp^ = Ch then CStringChPos := Temp else CStringChPos := nil
end;

inline function CStringLastChPos (Src: CString; Ch: Char): CString;
var Temp: CString;
begin
  if Src = nil then Return nil;
  Temp := CStringEnd (Src);
  while (Temp <> Src) and (Temp^ <> Ch) do Dec (Temp);
  if Temp^ = Ch then CStringLastChPos := Temp else CStringLastChPos := nil
end;

function CStringPos (s, SubString: CString): CString;
var
  Temp: CString;
  l: SizeType;
begin
  if (s = nil) or (SubString = nil) then Return s;
  l := CStringLength (SubString);
  Temp := s;
  while Temp^ <> #0 do
    begin
      if CStringLComp (Temp, SubString, l) = 0 then Return Temp;
      Inc (Temp)
    end;
  CStringPos := nil
end;

function CStringLastPos (s, SubString: CString): CString;
var
  Temp: CString;
  l: SizeType;
begin
  if (s = nil) or (SubString = nil) then Return s;
  l := CStringLength (SubString);
  Temp := CStringEnd (s);
  while Temp >= s do
    begin
      if CStringLComp (Temp, SubString, l) = 0 then Return Temp;
      Dec (Temp)
    end;
  CStringLastPos := nil
end;

function CStringCasePos (s, SubString: CString): CString;
var
  Temp: CString;
  l: SizeType;
begin
  if (s = nil) or (SubString = nil) then Return s;
  l := CStringLength (SubString);
  Temp := s;
  while Temp^ <> #0 do
    begin
      if CStringLCaseComp (Temp, SubString, l) = 0 then Return Temp;
      Inc (Temp)
    end;
  CStringCasePos := nil
end;

function CStringLastCasePos (s, SubString: CString): CString;
var
  Temp: CString;
  l: SizeType;
begin
  if (s = nil) or (SubString = nil) then Return s;
  l := CStringLength (SubString);
  Temp := CStringEnd (s);
  while Temp >= s do
    begin
      if CStringLCaseComp (Temp, SubString, l) = 0 then Return Temp;
      Dec (Temp)
    end;
  CStringLastCasePos := nil
end;

function CStringUpCase (s: CString): CString;
var Temp: CString;
begin
  Temp := s;
  if Temp <> nil then
    while Temp^ <> #0 do
      begin
        Temp^ := UpCase (Temp^);
        Inc (Temp)
      end;
  CStringUpCase := s
end;

function CStringLoCase (s: CString): CString;
var Temp: CString;
begin
  Temp := s;
  if Temp <> nil then
    while Temp^ <> #0 do
      begin
        Temp^ := LoCase (Temp^);
        Inc (Temp)
      end;
  CStringLoCase := s
end;

function CStringIsEmpty (s: CString): Boolean;
begin
  CStringIsEmpty := (s = nil) or (s^ = #0)
end;

function NewCString (const Source: String): CString;
var Dest: CString;
begin
  SetReturnAddress (ReturnAddress (0));
  GetMem (Dest, Length (Source) + 1);
  RestoreReturnAddress;
  MoveLeft (Source[1], Dest[0], Length (Source));
  Dest[Length (Source)] := #0;
  NewCString := Dest
end;

function CStringCopyString (Dest: CString; const Source: String): CString;
begin
  MoveLeft (Source[1], Dest[0], Length (Source));
  Dest[Length (Source)] := #0;
  CStringCopyString := Dest
end;

procedure CopyCString (Source: CString; var Dest: String);
var Source_Length: SizeType;
begin
  if Source = nil then
    SetLength (Dest, 0)
  else
    begin
      Source_Length := Min (CStringLength (Source), Dest.Capacity);
      SetLength (Dest, Source_Length);
      MoveLeft (Source[0], Dest[1], Source_Length)
    end
end;

function NewString (const s: String) = Result: PString;
begin
  SetReturnAddress (ReturnAddress (0));
  New (Result, Length (s));
  RestoreReturnAddress;
  Result^ := s
end;

procedure SetString (var s: String; Buffer: PChar; Count: Integer);
var i: Integer;
begin
  SetLength (s, Min (GetStringCapacity (s), Max (0, Min (Count, CStringLength (Buffer)))));
  if Buffer <> nil then
    for i := 1 to { @@ return value of SetLength } Length (s) do s[i] := Buffer[i - 1]
end;

function StringOfChar (Ch: Char; Count: Integer) = s: TString;
var i: Integer;
begin
  SetLength (s, Min (GetStringCapacity (s), Max (0, Count)));
  for i := 1 to { @@ return value of SetLength } Length (s) do s[i] := Ch
end;

procedure TrimLeft (var s: String);
var i: Integer;
begin
  i := 1;
  while (i <= Length (s)) and (s[i] in SpaceCharacters) do Inc (i);
  Delete (s, 1, i - 1)
end;

procedure TrimRight (var s: String);
var i: Integer;
begin
  i := Length (s);
  while (i > 0) and (s[i] in SpaceCharacters) do Dec (i);
  Delete (s, i + 1)
end;

procedure TrimBoth (var s: String);
begin
  TrimLeft (s);
  TrimRight (s)
end;

function TrimLeftStr (const s: String) = Result: TString;
begin
  Result := s;
  TrimLeft (Result)
end;

function TrimRightStr (const s: String) = Result: TString;
begin
  Result := s;
  TrimRight (Result)
end;

function TrimBothStr (const s: String) = Result: TString;
begin
  Result := s;
  TrimBoth (Result)
end;

function GetStringCapacity (const s: String): Integer;
begin
  GetStringCapacity := s.Capacity
end;

function Integer2String (i: Integer) = s: TString;
begin
  WriteStr (s, i)
end;

function StrEQ (const s1, s2: String): Boolean;
begin
  StrEq := (Length (s1) = Length (s2)) and (MemComp (s1[1], s2[1], Length (s1)) = 0)
end;

function StrLT (const s1, s2: String): Boolean;
begin
  if Length (s1) < Length (s2)
    then StrLT := MemComp (s1[1], s2[1], Length (s1)) <= 0
    else StrLT := MemComp (s1[1], s2[1], Length (s2)) < 0
end;

function StrEQPad (const s1, s2: String) = Result: Boolean;
var
  c: Integer;
  pLong, pShort: ^const String;
begin
  if Length (s1) > Length (s2) then
    begin
      pLong  := @s1;
      pShort := @s2
    end
  else
    begin
      pLong  := @s2;
      pShort := @s1
    end;
  Result := MemComp (s1[1], s2[1], Length (pShort^)) = 0;
  if Result and (Length (s1) <> Length (s2)) then
    for c := Length (pShort^) + 1 to Length (pLong^) do
      if pLong^[c] <> ' ' then Return False
end;

function StrLTPad (const s1, s2: String): Boolean;
var
  s1IsLonger: Boolean;
  c, r: Integer;
  pLong, pShort: ^const String;
begin
  s1IsLonger := Length (s1) > Length (s2);
  if s1IsLonger then
    begin
      pLong  := @s1;
      pShort := @s2
    end
  else
    begin
      pLong  := @s2;
      pShort := @s1
    end;
  r := MemComp (s1[1], s2[1], Length (pShort^));
  if (r <> 0) or (Length (s1) = Length (s2)) then Return r < 0;
  for c := Length (pShort^) + 1 to Length (pLong^) do
    if pLong^[c] <> ' ' then Return (pLong^[c] > ' ') xor s1IsLonger;
  StrLTPad := False
end;

{$if True}  { @@ Kludge }
{ cf. ../rts.c }
type
  TCharsKludge = array[1 .. MaxVarSize] of Char;
  PCharsKludge = ^TCharsKludge;

function KludgeStrEQ (s1: PCharsKludge; l1: Integer; s2: PCharsKludge; l2: Integer): Boolean; asmname '_p_eq';
function KludgeStrEQ (s1: PCharsKludge; l1: Integer; s2: PCharsKludge; l2: Integer): Boolean;
var c1, c2: Char;
begin
  { @@ kluuuuuudge } if l1 < 0 then begin l1 := 1; c1 := Chr (PtrInt (s1)); s1 := PCharsKludge (@c1) end;
                     if l2 < 0 then begin l2 := 1; c2 := Chr (PtrInt (s2)); s2 := PCharsKludge (@c2) end;
  KludgeStrEQ := StrEQ (s1^[1 .. l1], s2^[1 .. l2])
end;

function KludgeStrLT (s1: PCharsKludge; l1: Integer; s2: PCharsKludge; l2: Integer): Boolean; asmname '_p_lt';
function KludgeStrLT (s1: PCharsKludge; l1: Integer; s2: PCharsKludge; l2: Integer): Boolean;
var c1, c2: Char;
begin
  { @@ kluuuuuudge } if l1 < 0 then begin l1 := 1; c1 := Chr (PtrInt (s1)); s1 := PCharsKludge (@c1) end;
                     if l2 < 0 then begin l2 := 1; c2 := Chr (PtrInt (s2)); s2 := PCharsKludge (@c2) end;
  KludgeStrLT := StrLT (s1^[1 .. l1], s2^[1 .. l2])
end;

function KludgeStrEQPad (s1: PCharsKludge; l1: Integer; s2: PCharsKludge; l2: Integer): Boolean; asmname '_p_str_eq';
function KludgeStrEQPad (s1: PCharsKludge; l1: Integer; s2: PCharsKludge; l2: Integer): Boolean;
var c1, c2: Char;
begin
  { @@ kluuuuuudge } if l1 < 0 then begin l1 := 1; c1 := Chr (PtrInt (s1)); s1 := PCharsKludge (@c1) end;
                     if l2 < 0 then begin l2 := 1; c2 := Chr (PtrInt (s2)); s2 := PCharsKludge (@c2) end;
  KludgeStrEQPad := StrEQPad (s1^[1 .. l1], s2^[1 .. l2])
end;

function KludgeStrLTPad (s1: PCharsKludge; l1: Integer; s2: PCharsKludge; l2: Integer): Boolean; asmname '_p_str_lt';
function KludgeStrLTPad (s1: PCharsKludge; l1: Integer; s2: PCharsKludge; l2: Integer): Boolean;
var c1, c2: Char;
begin
  { @@ kluuuuuudge } if l1 < 0 then begin l1 := 1; c1 := Chr (PtrInt (s1)); s1 := PCharsKludge (@c1) end;
                     if l2 < 0 then begin l2 := 1; c2 := Chr (PtrInt (s2)); s2 := PCharsKludge (@c2) end;
  KludgeStrLTPad := StrLTPad (s1^[1 .. l1], s2^[1 .. l2])
end;
{$endif}

end.
