/* Gnome BibTeX bibentry.C
 *    Alejandro Aguilar Sierra <asierra@servidor.unam.mx>
 *    Felipe Bergo <bergo@seul.org>
 * 
 *    This program is free software; you can redistribute it and'or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2, or (at your option)
 *    any later version.
 */
#include <iostream.h>
#include <map.h>
#include <vector.h>
#include <std/bastring.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "gbib.h"

stack<char *> Unconst::TheStack;

char * Unconst::copy(const char *s) {
  char *t;
  t=(char *)malloc(strlen(s)*4);
  TheStack.push(t);
  strcpy(t,s);
  return t;
}


void Unconst::pop(int n=1) {
  free( TheStack.top() );
  TheStack.pop();
  if (n>1)
    pop(n-1);
}

extern int flag_allow_warnings;


BibEntry::BibEntry()
{
    def = 0;
    req_field = 0;
    opt_field = 0;
}

BibEntry::BibEntry(char *n, const char *k)
{
    def = getBibEntryDef(n);  
    
    if (k) {
      key = k;
    } else 
      key = " ";
    
    if (def) {   
	special = false;
	req_field = new string[def->nreq];
	opt_field = new string[def->nopt];
    } else {
	key = n;
	string::iterator it = key.begin();
	while (it != key.end()) {
	  *it = toupper(*it);
	  it++;
	}
	if (flag_allow_warnings)
	  cerr << "Warning " << key << ": " << "Undefined entry type " << key << "\n";
	req_field = new string[1];
	opt_field = 0;
	special = true;
    }
    nextra = 0;
}


BibEntry::BibEntry(BibEntry& bib)
{
    def = bib.def;
    key = bib.key;
    
    special =  bib.special;
    
    if (def) {
	req_field = new string[def->nreq];
	opt_field = new string[def->nopt];
    } else if (special) {
	req_field = new string[1];
	opt_field = 0;
    }
    nextra = 0;   
    setFields(bib);
}


BibEntry& BibEntry::operator=(BibEntry &bib)
{
    if (def) {
	delete[] req_field;
	delete[] opt_field;
	extra_field.clear();
    }
    
    def = bib.def;
    key = bib.key;
    
    special =  bib.special;
    
    if (def) {
	req_field = new string[def->nreq];
	opt_field = new string[def->nopt];
    }
    
    nextra = 0;    
    setFields(bib);
    return(*this);
}


BibEntry::~BibEntry()
{
    if (def) {
	if (req_field)
	  delete[] req_field;
	if (opt_field)
	  delete[] opt_field;
    } 
}


char *BibEntry::getEntryType()
{
    if (def)
      return ( Unconst::copy(def->name.c_str()) );
    else 
      return ( Unconst::copy(key.c_str()) );
}

void BibEntry::setEntryType(char *et)
{
    BibEntry bib(et, key.c_str());
    
    if (def) 
      bib.setFields(*this);
    else {
	def = getBibEntryDef(et);
	if (def) {
	    req_field = new string[def->nreq];
	    opt_field = new string[def->nopt];
	} else {
	    req_field = new string[1];
	    opt_field = 0;
	}
	nextra = 0;    
    }
    
    *this = bib;
}

char *BibEntry::getKey()
{
    return (Unconst::copy(key.c_str()));
}

const char *BibEntry::getConstKey() {
    return (key.c_str());
}

char *BibEntry::getFieldName(int i)
{
    if (!def)
      return "";
    
    if (i<0 || i >= getNoFields())
      return 0;

    if (i < def->nreq + def->nopt)    
      return (i < def->nreq) ? def->getRequired(i): 
                               def->getOptional(i - def->nreq);
    
    i -= def->nreq + def->nopt;
    
    slist_pair_string::const_iterator it = extra_field.begin();
    
    for (; i > 0 && it!=extra_field.end(); i--) { 
	it++;
    }
    
    return get_globalfield_name(it->first);
}

const char * BibEntry::getConstField(char *fn)
{
    if (strcmp(fn, "key")==0)
	return key.c_str();	
    for (int i=0; i< getNoFields(); i++) {
	if (strcasecmp(getFieldName(i), fn)==0)
	    return getConstField(i);
    }
    return 0;
}

const char * BibEntry::getConstField(const char *fn) {
    if (strcmp(fn, "key")==0)
	return key.c_str();	
    for (int i=0; i< getNoFields(); i++) {
	if (strcasecmp(getFieldName(i), fn)==0)
	    return getConstField(i);
    }
    return 0;
}


const char * BibEntry::getConstField(int i) {
    if (!def) 
      return (i==0) ? req_field[i].c_str() : 0;
    
    if (i<0 || i >= getNoFields())
      return 0;    
    if (i < def->nreq + def->nopt)
      return (i < def->nreq) ? req_field[i].c_str() : 
                               opt_field[i - def->nreq].c_str();
    i -= def->nreq + def->nopt;    
    slist_pair_string::const_iterator it = extra_field.begin();    
    for (; i > 0 && it!=extra_field.end(); i--)
	it++;
    return it->second.c_str();
}

char *BibEntry::getField(int i)
{
    if (!def) 
      return (i==0) ? Unconst::copy(req_field[i].c_str()) : 0;
    
    if (i<0 || i >= getNoFields())
      return 0;    
    if (i < def->nreq + def->nopt)
      return (i < def->nreq) ? Unconst::copy(req_field[i].c_str()) : 
                               Unconst::copy(opt_field[i - def->nreq].c_str());
    i -= def->nreq + def->nopt;    
    slist_pair_string::const_iterator it = extra_field.begin();    
    for (; i > 0 && it!=extra_field.end(); i--)
	it++;
    return Unconst::copy(it->second.c_str());
}


char *BibEntry::getField(char* fn)
{
    if (strcmp(fn, "key")==0)
	return Unconst::copy(key.c_str());
	
    for (int i=0; i< getNoFields(); i++) {
	if (strcmp(getFieldName(i), fn)==0) { 
	    
//	    cerr << "FIeld " << key << ":" << i << ": " << getField(i) << "\n"; 
	    return getField(i);
	}
    }

    return 0;
}


char *BibEntry::getReqField(int i)
{
    return (i < def->nreq) ? Unconst::copy(req_field[i].c_str()) : 0; 
}


char *BibEntry::getOptField(int i)
{
    return (i < def->nopt) ? Unconst::copy(opt_field[i].c_str()) : 0;
}

char *BibEntry::getExtField(int i)
{
    slist_pair_string::const_iterator it = extra_field.begin();
    
    for (; i > 0 && it!=extra_field.end(); i--) { 
	it++;
    }
    
    if (it!=extra_field.end()) 
      return Unconst::copy(it->second.c_str());
    
    return "Kakas";
}

void BibEntry::setField(int i, char *field)
{
    if (i < 0 || i >= getNoFields())
      return;

    if (!def && req_field) {
	req_field[0] = field;
	return;
    }
    
//    cerr << "sField " << key << ":" << i << ": " << field << "\n"; 
    if (i < def->nreq) {
	req_field[i] = field;
    } else
      if (i < def->nreq + def->nopt) {	  
	  opt_field[i - def->nreq] = field;
      } else {
	  i -= def->nreq + def->nopt;
    
	  slist_pair_string::iterator it = extra_field.begin();
    
	  for (; i > 0 && it!=extra_field.end(); i--) { 
	      it++;
	  }
    
	  it->second = field;
      }    
}


void BibEntry::setField(char *fieldname, char *field)
{
    if (!def && req_field) {
	req_field[0] = field;
	return;
    }
    
    int i = def->getFieldIdx(fieldname);
    
    if (i>=0) {
      setField(i, field);
//	cerr << key << ":Field " << fieldname << "|" << field << "\n";
    }
    else {
	if (flag_allow_warnings)
	  cerr << "Warning " << key << ": Ignored field \"" << fieldname 
	  << "\" in the entry type <" << getEntryType() << ">.\n";

	// does exist already this name?
	int id = set_globalfield_name(fieldname);
	
	nextra++;
	extra_field.push_front(pair_string(id, field));
    }
}



void BibEntry::setFields(BibEntry &entry)
{
    for (int i=0; i<entry.getNoFields(); i++) {
	string s = entry.getField(i);
	if (!s.empty()) {
	  setField(entry.getFieldName(i), Unconst::copy(s.c_str()));
	}
    }
}
