
/*
 * Gnome Basic ListBox implementation.
 *
 * Author:
 *	Frank Chiulli	(fc-linux@home.com)
 *
 * Copyright 2000, Helix Code, Inc.
 */
#include "gbrun-form-item.h"

#define ITEM_NAME "gb-vb.listbox"

#undef DEBUG_LB

/*
 * ListBox Properties:
 * COLUMNS		Specifies the number of columns
 * ITEMDATA		Specifies a specific number for each item in the ListBox
 *			window's left edge.
 * LIST			The items contained in the control's list portion.
 * SORTED
 * WIDTH
 */
enum {
	ARG_FIRST = 0,
	COLUMNS,
	ITEMDATA,
	LIST,
	SORTED,
	WIDTH
};

static const char* p_name[] = {
	"Arg_First",
	"Columns",
	"Itemdata",
	"List",
	"Sorted",
	"Width"
};

#define GBRUN_LISTBOX(obj) (GTK_CHECK_CAST ((obj), gbrun_listbox_get_type (), GBRunListBox))

/*
 *  Function Prototypes.
 */
static GBValue *listbox_getarg (GBRunEvalContext *ec,
				GBRunObject      *object,
				int		  property);

static gboolean listbox_setarg (GBRunEvalContext *ec,
				GBRunObject      *object,
				int		  property,
				GBValue          *val);


typedef struct  {
	GBRunFormItem    item;

	GtkWidget	*sw;

	GtkCList	*clist;

	int		 columns;
	GBBoolean	 sorted;

} GBRunListBox;


static void
gbrun_listbox_destroy (GtkObject *object)
{
	g_warning ("Unimplemented %s destroy", ITEM_NAME);
}


/**
 * gbrun_listbox_construct:
 *   @ec
 *   @item
 **/
static void
gbrun_listbox_construct (GBRunEvalContext *ec,
			 GBRunFormItem *item)
{
	GBRunListBox *dest = GBRUN_LISTBOX (item);

	/* Create a scrolled window to pack the CList widget into */
	dest->sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (dest->sw),
				    	GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gbrun_form_item_set_widget (GBRUN_FORM_ITEM (dest), dest->sw);
/*
	dest->clist = GTK_CLIST (gtk_clist_new (1));

	gtk_container_add (GTK_CONTAINER (dest->sw), GTK_WIDGET (dest->clist));
*/
	dest->columns = 1;
}


/**
 * gbrun_listbox_class_init
 *   @klass
 **/
static void
gbrun_listbox_class_init (GBRunObjectClass *klass)
{
	GtkObjectClass		 *gtk_class = (GtkObjectClass *) klass;
	GBRunFormItemClass	*form_class = (GBRunFormItemClass *) klass;

#ifdef DEBUG_LB
	printf ("gbrun_listbox_class_init entered.\n");	
#endif

	klass->set_arg = listbox_setarg;
	klass->get_arg = listbox_getarg;

	gbrun_object_add_property (klass, "columns",  gb_type_int, COLUMNS);

	gbrun_object_add_property (klass, "itemdata", gb_type_list, ITEMDATA);

	gbrun_object_add_property (klass, "list",     gb_type_list, LIST);

	gbrun_object_add_property (klass, "sorted",   gb_type_int, SORTED);

	gbrun_object_add_property (klass, "width",    gb_type_int, WIDTH);

	form_class->construct = gbrun_listbox_construct;

	gtk_class->destroy    = gbrun_listbox_destroy;
}


/**
 * gbrun_listbox_get_type
 **/
GtkType
gbrun_listbox_get_type (void)
{
	static GtkType object_type = 0;

	if (!object_type) {
		static const GtkTypeInfo object_info = {
			ITEM_NAME,
			sizeof (GBRunListBox),
			sizeof (GBRunFormItemClass),
			(GtkClassInitFunc)  gbrun_listbox_class_init,
			(GtkObjectInitFunc) NULL,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		object_type = gtk_type_unique (GBRUN_TYPE_FORM_ITEM, &object_info);
		gtk_type_class (object_type);
	}

	return object_type;	
}


/**
 * listbox_setarg:
 *   @ec
 *   @object
 *   @property
 *   @val
 **/
static gboolean
listbox_setarg (GBRunEvalContext *ec,
		GBRunObject      *object,
		int		  property,
		GBValue          *val)
{
	gchar		*cptr;

	gint		 col;
	gint		 col_width;
	gint		 i;
	gint		 row;
	gint		 rows_per_col;
	
	guint		 no_ptrs;

	GBRunListBox 	*listbox = GBRUN_LISTBOX (object);;

	g_return_val_if_fail (listbox != NULL, FALSE);
/*	g_return_if_fail (listbox->clist != NULL); */

	switch (property) {
	case COLUMNS:
	{
#ifdef DEBUG_LB
		printf ("listbox_setarg: COLUMNS\n");
#endif
		listbox->columns = val->v.i;
		if (val->v.i == 0)
			listbox->columns = 1;
/*
		printf ("                gtk_clist_clear.\n");
		gtk_clist_clear (listbox->clist);
		printf ("                gtk_container_remove.\n");
		gtk_container_remove (GTK_CONTAINER (listbox->sw), 
				      GTK_WIDGET (listbox->clist));
		printf ("                gtk_clist_new.\n");
		listbox->clist = GTK_CLIST (gtk_clist_new (listbox->columns));
		if (listbox->clist == NULL)
			printf ("                NULL clist.\n");
		printf ("                gtk_container_add.\n");
		gtk_container_add (GTK_CONTAINER (listbox->sw), 
				   GTK_WIDGET (listbox->clist));
		printf ("                done.\n");
*/
#ifdef DEBUG_LB
		printf ("listbox_setarg: COLUMNS=%d\n", listbox->columns);
#endif
		return TRUE;
	}
	
	case ITEMDATA:
	{
		g_warning ("listbox: Set of unhandled property '%s'", p_name[property]);
		return FALSE;
	}

	case LIST:
	{
#ifdef DEBUG_LB
		printf ("listbox_setarg: LIST\n");
		printf ("                columns = %d.\n", listbox->columns);
#endif
		listbox->clist = GTK_CLIST (gtk_clist_new (listbox->columns));
		gtk_container_add (GTK_CONTAINER (listbox->sw), GTK_WIDGET (listbox->clist));
		gtk_widget_show (GTK_WIDGET (listbox->clist));

		no_ptrs       = val->v.list->len;
		rows_per_col  = (no_ptrs + 1) / listbox->columns;

		i = listbox->sw->allocation.width;
#ifdef DEBUG_LB
		printf ("                widget width = %d\n", i);
#endif
		
		/*
		 * Initialize the rows with empty cells for now.
		 */
		 {
			gchar *blank[listbox->columns];
			for (i = 0; i < listbox->columns; i++) {
				blank[i] = " ";
			}

			for (i = 0; i < rows_per_col; i++) {
				gtk_clist_insert (listbox->clist, i, blank);
			}
		}

		/*
		 * Now fill the individual cells with data.
		 */
		for (i = 0; i < no_ptrs; i++) {
			cptr = (char *)g_ptr_array_index (val->v.list, i);
#ifdef DEBUG_LB
			printf ("                data = %s\n", cptr);
#endif
			if (listbox->columns == 1) {
				row = gtk_clist_append (listbox->clist, &cptr);
			} else {
				col = i / rows_per_col;
				row = i % rows_per_col;
				gtk_clist_set_text (listbox->clist, row, col, cptr);
			}
		}

		return TRUE;
	}

	case SORTED:
	{
		listbox->sorted = (val->v.i == VB_FALSE);
		return TRUE;
	}
	
	case WIDTH:
	{
#ifdef DEBUG_LB
		printf ("listbox_setarg: WIDTH\n");
#endif
		listbox->sw->allocation.width = GBRUN_FORM_TWIPS_TO_X (val->v.i);
		gtk_widget_set_usize (listbox->sw, listbox->sw->allocation.width, 
		                      listbox->sw->allocation.height);

		for (i = 0; i < listbox->columns; i++) {
			col_width = listbox->sw->allocation.width / listbox->columns;
			gtk_clist_set_column_width (listbox->clist, i, col_width);
		}
		return TRUE;
	}
	
	default:
		g_warning ("listbox: Set of unhandled property '%s'", p_name[property]);
		return FALSE;
	}

}


/**
 * listbox_getarg:
 *   @ec
 *   @object
 *   @property
 **/
static GBValue *
listbox_getarg (GBRunEvalContext *ec,
		GBRunObject      *object,
		int		  property)
{
	GBRunListBox *listbox = GBRUN_LISTBOX (object);
/*	GtkList      *l       = GTK_LIST (gbrun_form_item_get_widget (object));  */

	g_return_val_if_fail (listbox != NULL, NULL);

	switch (property) {
	case COLUMNS:
		g_warning ("listbox: Get of unhandled property '%s'", p_name[property]);
	
	case ITEMDATA:
		g_warning ("listbox: Get of unhandled property '%s'", p_name[property]);
	
	case LIST:
		g_warning ("listbox: Get of unhandled property '%s'", p_name[property]);
	
	case SORTED:
		return gb_value_new_boolean (listbox->sorted);

	default:
		g_warning ("listbox: Get of unhandled property '%s'", p_name[property]);
		break;
	}

	return NULL;
}


/**
 * gbrun_listbox_register
 **/
void
gbrun_listbox_register ()
{
	gbrun_listbox_get_type ();
}


/**
 * gbrun_listbox_shutdown
 **/
void
gbrun_listbox_shutdown ()
{
}
