# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gtk
import gobject

import os

import urlparse
import urllib

from gazpacho.loader.widgettree import GladeParseError
from gazpacho import palette, editor, project, catalog
from gazpacho.placeholder import Placeholder
from gazpacho.path import pixmaps_dir
from gazpacho.command import CommandManager, CommandStackView
from gazpacho import clipboard
from gazpacho.gaction import GActionsView, GAction, GActionGroup
from gazpacho.gaction import GActionDialog, GActionGroupDialog
from gazpacho.widget import Widget, get_widget_from_gtk_widget
from gazpacho import util

class Application(object):

    # DND information
    TARGET_TYPE_URI = 100
    targets = [('text/uri-list', 0, TARGET_TYPE_URI)]
    
    def __init__(self):
        # see catalog.py
        self._catalogs = catalog.load_all()

        # The WidgetClass that we are about to add to a container. None if no
        # class is to be added. This also has to be in sync with the depressed
        # button in the Palette
        self._add_class = None

        # This is our current project
        self._project = None

        # debugging windows
        self._command_stack_window = None
        self._clipboard_window = None

        self._clipboard = clipboard.Clipboard()
        
        # here we put views that should update when changing the project
        self._project_views = []

        self._window = self._application_window_create()

        self._active_view = None

        self._projects = []
        
        self._command_manager = CommandManager(self)

    def _widget_tree_view_create(self):
        from gazpacho.widgettreeview import WidgetTreeView
        view = WidgetTreeView(self)
        self._project_views.insert(0, view)
        view.set_project(self._project)
        view.set_size_request(150, 160)
        view.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        return view
 
    def _command_stack_view_create(self):
        view = CommandStackView()
        self._project_views.insert(0, view)
        view.set_project(self._project)
        view.set_size_request(300, 200)
        view.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        return view

    def _gactions_view_create(self):
        view = GActionsView()
        view.set_project(self._project)
        self._project_views.insert(0, view)        
        view.set_size_request(150, 160)
        view.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        return view
    
    def _application_window_create(self):
        application_window = gtk.Window(gtk.WINDOW_TOPLEVEL)
        application_window.move(0, 0)
        application_window.set_default_size(700, -1)
        iconfilename = os.path.join(pixmaps_dir, 'gazpacho-icon.png')
        gtk.window_set_default_icon_from_file(iconfilename)
                                                
        application_window.connect('delete-event', self._delete_event)

        # Create the different widgets
        menubar, toolbar = self._construct_menu_and_toolbar(application_window)

        self._palette = palette.Palette(self._catalogs)
        self._palette.connect('toggled', self._palette_button_clicked)

        self._editor = editor.Editor(self)

        widget_view = self._widget_tree_view_create()

        self.gactions_view = self._gactions_view_create()
        
        self._statusbar = self._construct_statusbar()

        # Layout them on the window
        main_vbox = gtk.VBox()
        application_window.add(main_vbox)

        main_vbox.pack_start(menubar, False)
        main_vbox.pack_start(toolbar, False)

        hbox = gtk.HBox(spacing=6)
        hbox.pack_start(self._palette, False, False)

        vpaned = gtk.VPaned()
        hbox.pack_start(vpaned, True, True)

        notebook = gtk.Notebook()
        notebook.append_page(widget_view, gtk.Label(_('Widgets')))
        notebook.append_page(self.gactions_view, gtk.Label(_('Actions')))

        vpaned.set_position(150)

        vpaned.add1(notebook)
        vpaned.add2(self._editor)

        main_vbox.pack_start(hbox)
        
        main_vbox.pack_end(self._statusbar, False)

        # dnd doesn't seem to work with Konqueror, at least not when
        # gtk.DEST_DEFAULT_ALL or gtk.DEST_DEFAULT_MOTION is used. If
        # handling the drag-motion event it will work though, but it's
        # a bit tricky.
        application_window.drag_dest_set(gtk.DEST_DEFAULT_ALL,
                                         Application.targets,
                                         gtk.gdk.ACTION_COPY)
        
        application_window.connect('drag_data_received',
                                   self._dnd_data_received_cb)
        
        self.refresh_undo_and_redo()

        return application_window

    def _expander_activated(self, expander, other_expander):
        # Make the widget tree and property editor use all available
        # space when the other one is collapsed.
        box = expander.parent

        # When both are collapsed, neither should expand to get them
        # neatly packed at the top.
        if expander.get_expanded() and not other_expander.get_expanded():
            expand = False
            other_expand = False
        else:
            expand = not expander.get_expanded()
            other_expand = not expand
        
        box.set_child_packing(expander,
                              expand,
                              True,
                              0,
                              gtk.PACK_START)
        box.set_child_packing(other_expander,
                              other_expand,
                              True,
                              0,
                              gtk.PACK_START)
            
    def _create_expander(self, label, widget, expanded=True):
        expander = gtk.Expander('<span size="large" weight="bold">%s</span>' % label)
        expander.set_use_markup(True)
        expander.set_expanded(expanded)
        expander.add(widget)
        return expander
    
    def _create_frame(self, label, widget):
        frame = gtk.Frame()
        label = gtk.Label('<span size="large" weight="bold">%s</span>' % label)
        label.set_use_markup(True)
        frame.set_label_widget(label)
        frame.set_shadow_type(gtk.SHADOW_NONE)
        frame.add(widget)
        return frame
    
    def _construct_menu_and_toolbar(self, application_window):
        actions =(
            ('FileMenu', None, _('_File')),
            ('New', gtk.STOCK_NEW, _('_New'), '<control>N',
             _('New Project'), self._new_cb),
            ('Open', gtk.STOCK_OPEN, _('_Open'), '<control>O',
             _('Open Project'), self._open_cb),
            ('Save', gtk.STOCK_SAVE, _('_Save'), '<control>S',
             _('Save Project'), self._save_cb),
            ('SaveAs', gtk.STOCK_SAVE_AS, _('_Save As...'),
             '<shift><control>S', _('Save project with different name'),
             self._save_as_cb),
            ('Close', gtk.STOCK_CLOSE, _('_Close'), '<control>W',
             _('Close Project'), self._close_cb),
            ('Quit', gtk.STOCK_QUIT, _('_Quit'), '<control>Q', _('Quit'),
             self._quit_cb),
            ('EditMenu', None, _('_Edit')),
            ('Cut', gtk.STOCK_CUT, _('C_ut'), '<control>X', _('Cut'),
             self._cut_cb),
            ('Copy', gtk.STOCK_COPY, _('_Copy'), '<control>C', _('Copy'),
             self._copy_cb),
            ('Paste', gtk.STOCK_PASTE, _('_Paste'), '<control>V', _('Paste'),
             self._paste_cb),
            ('Delete', gtk.STOCK_DELETE, _('_Delete'), '<control>D',
             _('Delete'), self._delete_cb),
            ('ActionMenu', None, _('_Actions')),
            ('AddAction', gtk.STOCK_ADD, _('_Add action'), '<control>A',
             _('Add an action'), self._add_action_cb),
            ('RemoveAction', gtk.STOCK_REMOVE, _('_Remove action'), None,
             _('Remove action'), self._remove_action_cb),
            ('EditAction', None, _('_Edit action'), None, _('Edit Action'),
             self._edit_action_cb),
            ('ProjectMenu', None, _('_Project')),
            ('DebugMenu', None, _('_Debug')),
            ('HelpMenu', None, _('_Help')),
            ('About', None, _('_About'), None, _('About'), self._about_cb),
            ('DumpData', None, _('Dump data'), '<control>M',
              _('Dump debug data'), self._dump_data_cb)
            )

        toggle_actions = (
            ('ShowCommandStack', None, _('Show _command stack'), 'F3',
             _('Show the command stack'), self._show_command_stack_cb, False),
            ('ShowClipboard', None, _('Show _clipboard'), 'F4',
             _('Show the clipboard'), self._show_clipboard_cb, False),
            )
        
        undo_action = (
            ('Undo', gtk.STOCK_UNDO, _('_Undo'), '<control>Z',
             _('Undo last action'), self._undo_cb),
            )

        redo_action = (
            ('Redo', gtk.STOCK_REDO, _('_Redo'), '<control>R',
             _('Redo last action'), self._redo_cb),
            )
        
        ui_description = """
<ui>
  <menubar name="MainMenu">
    <menu action="FileMenu">
      <menuitem action="New"/>
      <menuitem action="Open"/>
      <separator name="FM1"/>
      <menuitem action="Save"/>
      <menuitem action="SaveAs"/>
      <separator name="FM2"/>
      <menuitem action="Close"/>
      <menuitem action="Quit"/>
    </menu>
    <menu action="EditMenu">
      <menuitem action="Undo"/>
      <menuitem action="Redo"/>
      <separator name="EM1"/>
      <menuitem action="Cut"/>
      <menuitem action="Copy"/>
      <menuitem action="Paste"/>
      <menuitem action="Delete"/>
    </menu>
    <menu action="ActionMenu">
      <menuitem action="AddAction"/>
      <menuitem action="RemoveAction"/>
      <menuitem action="EditAction"/>
    </menu>
    <menu action="ProjectMenu">
    </menu>
    <menu action="DebugMenu">
      <menuitem action="ShowCommandStack"/>
      <menuitem action="ShowClipboard"/>
      <separator/>
      <menuitem action="DumpData"/>
    </menu>
    <menu action="HelpMenu">
      <menuitem action="About"/>
    </menu>
  </menubar>
  <toolbar name="MainToolbar">
    <toolitem action="Open"/>
    <toolitem action="Save"/>
    <separator name="MT1"/>
    <toolitem action="Undo"/>
    <toolitem action="Redo"/>    
    <separator name="MT2"/>
    <toolitem action="Cut"/>
    <toolitem action="Copy"/>
    <toolitem action="Paste"/>
    <toolitem action="Delete"/>
  </toolbar>
</ui>
"""
        self._ui_manager = gtk.UIManager()

        action_group = gtk.ActionGroup('MenuActions')
        action_group.add_actions(actions)
        action_group.add_toggle_actions(toggle_actions)
        self._ui_manager.insert_action_group(action_group, 0)

        action_group = gtk.ActionGroup('UndoAction')
        action_group.add_actions(undo_action)
        self._ui_manager.insert_action_group(action_group, 0)
        
        action_group = gtk.ActionGroup('RedoAction')
        action_group.add_actions(redo_action)
        self._ui_manager.insert_action_group(action_group, 0)
        
        self._ui_manager.add_ui_from_string(ui_description)
        
        application_window.add_accel_group(self._ui_manager.get_accel_group())

        menu = self._ui_manager.get_widget('/MainMenu')
        toolbar = self._ui_manager.get_widget('/MainToolbar')

        # Make sure that the project menu isn't removed if empty
        project_action = self._ui_manager.get_action('/MainMenu/ProjectMenu/')
        project_action.set_property('hide-if-empty', False)

        return (menu, toolbar)
    
    def _construct_statusbar(self):
        statusbar = gtk.Statusbar()
        self._statusbar_menu_context_id = statusbar.get_context_id("menu")
        self._statusbar_actions_context_id = statusbar.get_context_id("actions")
        return statusbar

    # some properties
    def get_window(self): return self._window
    window = property(get_window)
    
    def _push_statusbar_hint(self, msg):
        self._statusbar.push(self._statusbar_menu_context_id, tip)

    def _pop_statusbar_hint(self):
        self._statusbar.pop(self._statusbar_menu_context_id)

    def show_message(self, msg, howlong=5000):
        self._statusbar.push(self._statusbar_menu_context_id, msg)
        def remove_msg():
            self._statusbar.pop(self._statusbar_menu_context_id)
            return False
        
        gobject.timeout_add(howlong, remove_msg)
        
    def refresh_undo_and_redo(self):
        undo_item = redo_item = None
        if self._project is not None:
            pri = self._project.prev_redo_item
            if pri != -1:
                undo_item = self._project.undo_stack[pri]
            if pri + 1 < len(self._project.undo_stack):
                redo_item = self._project.undo_stack[pri + 1]

        undo_action = self._ui_manager.get_action('/MainToolbar/Undo')
        undo_group = undo_action.get_property('action-group')
        undo_group.set_sensitive(undo_item is not None)
        undo_widget = self._ui_manager.get_widget('/MainMenu/EditMenu/Undo')
        label = undo_widget.get_child()
        if undo_item is not None:
            label.set_text_with_mnemonic(_('_Undo: %s') % \
                                         undo_item.description)
        else:
            label.set_text_with_mnemonic(_('_Undo: Nothing'))
            
        redo_action = self._ui_manager.get_action('/MainToolbar/Redo')
        redo_group = redo_action.get_property('action-group')
        redo_group.set_sensitive(redo_item is not None)
        redo_widget = self._ui_manager.get_widget('/MainMenu/EditMenu/Redo')
        label = redo_widget.get_child()
        if redo_item is not None:
            label.set_text_with_mnemonic(_('_Redo: %s') % \
                                         redo_item.description)
        else:
            label.set_text_with_mnemonic(_('_Redo: Nothing'))

        if self._command_stack_window is not None:
            command_stack_view = self._command_stack_window.get_child()
            command_stack_view.update()

    def show_all(self):
        self._window.show_all()
        
    # projects
    def new_project(self):
        prj = project.Project(True, self)
        self._add_project(prj)


    def _confirm_open_project(self, old_project, path):
        """Ask the user whether to reopen a project that has already
        been loaded (and thus reverting all changes) or just switch to
        the loaded version.
        """
        # If the project hasn't been modified, just switch to it.
        if not old_project.changed:
            self._set_project(old_project)
            return
        
        submsg1 = _('The project "%s" is already loaded.') % \
                  old_project.name
        submsg2 = _('Do you wish to revert your changes and reload the original glade file?') + \
                  ' ' + _('Your changes will be permanently lost if you choose to revert.')
        msg = '<span weight="bold" size="larger">%s</span>\n\n%s\n' % \
              (submsg1, submsg2)
        dialog = gtk.MessageDialog(self._window, gtk.DIALOG_MODAL,
                                    gtk.MESSAGE_WARNING, gtk.BUTTONS_NONE,
                                    msg)
        dialog.set_title('')
        dialog.label.set_use_markup(True)
        dialog.add_buttons(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                           gtk.STOCK_REVERT_TO_SAVED, gtk.RESPONSE_YES)
        dialog.set_default_response(gtk.RESPONSE_CANCEL)

        ret = dialog.run()
        if ret == gtk.RESPONSE_YES:
            # Close the old project
            old_project.selection_clear(False)
            
            for widget in old_project.widgets:
                widget.destroy()

            self._ui_manager.remove_ui(old_project.uim_id)
            self._projects.remove(old_project)

            # Open the project again
            prj = project.Project.open(path, self)
            if prj is not None:
                self._add_project(prj)
        else:
            # No revert - just switch to the old project.
            self._set_project(old_project)
            
        dialog.destroy()

    def open_project(self, path):
        # Check if the project is loaded and ask the user what to do.
        for prj in self._projects:
            if prj.path and prj.path == path:
                self._confirm_open_project(prj, path)
                return
        
        try:
            prj = project.Project.open(path, self)

            if prj is not None:
                self._add_project(prj)
        except GladeParseError, e:
            submsg1 = _('The project could not be loaded')
            submsg2 = _('An error occurred while parsing the file "%s".') % \
                      os.path.abspath(path)
            msg = '<span weight="bold" size="larger">%s</span>\n\n%s\n' % \
                      (submsg1, submsg2)
            dialog = gtk.MessageDialog(self._window,
                                    gtk.DIALOG_MODAL \
                                    | gtk.DIALOG_DESTROY_WITH_PARENT,
                                    gtk.MESSAGE_ERROR,
                                    gtk.BUTTONS_OK,
                                    msg)
            dialog.set_title('')
            dialog.label.set_use_markup(True)
            dialog.run()
            dialog.destroy()
            self._update_sensitivity()
        
    def _add_project(self, project):
        # if the project was previously added, don't reload
        for prj in self._projects:
            if prj.path and prj.path == project.path:
                self._set_project(prj)
                return

        self._projects.insert(0, project)

        # add the project in the /Project menu
        project_action= gtk.Action(project.name, project.name, project.name,
                                   '')

        project_action.connect('activate', self._set_project_cb, project)
        project_ui = """
        <ui>
          <menubar name="MainMenu">
            <menu action="ProjectMenu">
              <menuitem action="%s"/>
            </menu>
          </menubar>
        </ui>
        """ % project.name
        for action_group in self._ui_manager.get_action_groups():
            if action_group.get_name() == 'MenuActions':
                action_group.add_action(project_action)
                break
        
        project.uim_id = self._ui_manager.add_ui_from_string(project_ui)

        # connect to the project signals so that the editor can be updated
        project.connect('selection_changed',
                         self._project_selection_changed_cb)
        project.connect('project_changed', self._project_changed_cb)

        self._set_project(project)

        # make sure the palette and the right actions are sensitive
        self._update_sensitivity()

    def _project_changed_cb(self, project):
        self._refresh_title()
        self._update_save_action()
        
    def _set_project_cb(self, action, project):
        self._set_project(project)
        
    def _set_project(self, project):
        if project is self._project:
            return
        
        if not project in self._projects:
            print _('Could not set project because it could not be found in the list')
            return

        self._project = project
        self._refresh_title()

        for view in self._project_views:
            view.set_project(self._project)

        self.refresh_undo_and_redo()
        self._update_save_action()

        # trigger the selection changed signal to update the editor
        self._project.selection_changed()
        
    def _refresh_title(self):
        if self._project:
            title = '%s - Gazpacho' % self._project.name
            if self._project.changed:
                title = '*' + title
        else:
            title = 'Gazpacho'
        self._window.set_title(title)

    def _refresh_project_entry(self, project):
        item = self._ui_manager.get_widget(project.entry.path)
        label = item.get_child()
        # change the menu item's label
        label.set(project.name)
        project.entry.path = "/Project/%s" % project.name
        
    # callbacks
    def _delete_event(self, window, event):
        self._quit_cb()

        # return TRUE to stop other handlers
        return True
    
    def _new_cb(self, action):
        self.new_project()

    def _open_cb(self, action):
        filechooser = gtk.FileChooserDialog(_('Open ...'), self._window,
                                            gtk.FILE_CHOOSER_ACTION_OPEN,
                                            (gtk.STOCK_CANCEL,
                                             gtk.RESPONSE_CANCEL,
                                             gtk.STOCK_OPEN,
                                             gtk.RESPONSE_OK))
        file_filter = gtk.FileFilter()
        file_filter.add_pattern("*.glade")
        filechooser.set_filter(file_filter)
        if filechooser.run() == gtk.RESPONSE_OK:
            path = filechooser.get_filename()
            filechooser.destroy()
            
            if path and os.access(path, os.R_OK):
                self.open_project(path)
            else:
                submsg1 = _('Could not open file "%s"') % os.path.abspath(path)
                submsg2 = _('The file "%s" could not be opened. Permission denied.') % \
                          os.path.abspath(path)
                msg = '<span weight="bold" size="larger">%s</span>\n\n%s\n' % \
                      (submsg1, submsg2)
                dialog = gtk.MessageDialog(self._window,
                                           gtk.DIALOG_MODAL | \
                                           gtk.DIALOG_DESTROY_WITH_PARENT,
                                           gtk.MESSAGE_ERROR,
                                           (gtk.BUTTONS_OK),
                                           msg)
                dialog.set_title('')
                dialog.label.set_use_markup(True)
                dialog.run()
                dialog.destroy()
        else:
            filechooser.destroy()

    def _save(self, project, path):
        """ Internal save """
        if not os.path.exists(path) or os.access(path, os.W_OK):
            project.save(path)
            #self._refresh_project_entry(project)
            self._refresh_title()
            success = True
        else:
            text = _('Could not save to file "%s"\nPermission denied.' % path)
            msg = gtk.MessageDialog(parent=self._window,
                                    flags=gtk.DIALOG_MODAL \
                                    | gtk.DIALOG_DESTROY_WITH_PARENT,
                                    type=gtk.MESSAGE_ERROR,
                                    buttons=(gtk.BUTTONS_OK),
                                    message_format=text)
            msg.run()
            msg.destroy()
            success = False

        return success
    
    def _save_cb(self, action):
        project = self._project
        if project.path is not None:
            self._save(project, project.path)
            return

        # If instead we don't have a path yet, fire up a file chooser
        self._save_as_cb(None)

    def _save_as_cb(self, action):
        project = self._project
        if action is None:
            # we were called from the _save_cb callback
            title = _('Save')
        else:
            title = _('Save as')
        self._project_save_as(title, project)

    def _project_save_as(self, title, project):
        saved = False
        filechooser = gtk.FileChooserDialog(title, self._window,
                                            gtk.FILE_CHOOSER_ACTION_SAVE,
                                            (gtk.STOCK_CANCEL,
                                             gtk.RESPONSE_CANCEL,
                                             gtk.STOCK_SAVE,
                                             gtk.RESPONSE_OK))
        current_name = project.name
        if not current_name.endswith(".glade"):
            current_name = "%s.glade" % current_name
        filechooser.set_current_name(current_name)        
        filechooser.set_default_response(gtk.RESPONSE_OK)
        while True:
            if filechooser.run() == gtk.RESPONSE_OK:
                path = filechooser.get_filename()
                if os.path.exists(path):
                    text = _('There is a file with that name already.\nWould you like to overwrite it?')
                    msg = gtk.MessageDialog(parent=self._window,
                                            flags=gtk.DIALOG_MODAL \
                                            | gtk.DIALOG_DESTROY_WITH_PARENT,
                                            type=gtk.MESSAGE_WARNING,
                                            buttons=(gtk.BUTTONS_YES_NO),
                                            message_format=text)
                    result = msg.run()
                    msg.destroy()
                    # the user want to overwrite the file
                    if result == gtk.RESPONSE_YES:
                        saved = self._save(project, path)
                        break

                # the file does not exists. we can save it without worries
                else:
                    saved = self._save(project, path)
                    break
            # maybe the user changed his/her opinion and don't want to
            # save (by clicking cancel on the file chooser)
            else:
                break
                
        filechooser.destroy()
        return saved

    def _confirm_close_project(self, project):
        """Show a dialog asking the user whether or not to save the
        project before closing them.

        Return False if the user has chosen not to close the project.
        """
        return self._confirm_close_projects([project])

    def _confirm_close_projects(self, projects):
        """Show a dialog listing the projects and ask whether the user
        wants to save them before closing them.

        Return False if the user has chosen not to close the projects.
        """
        if not projects:
            return True
        
        if len(projects) == 1:
            dialog = SingleCloseConfirmationDialog(projects[0], self._window)
        else:
            dialog = MultipleCloseConfirmationDialog(projects, self._window)

        ret = dialog.run()
        if ret == gtk.RESPONSE_YES:
            # Go through the chosen projects and save them. Ask for a
            # file name if necessary.
            close = True
            projects_to_save = dialog.get_projects()
            for project in projects_to_save:
                if project.path:
                    project.save(project.path)
                else:
                    title = _('Save as')
                    saved = self._project_save_as(title, project)
                    # If the user has pressed cancel we abort everything.
                    if not saved:
                        close = False
                        break
        elif ret == gtk.RESPONSE_NO:
            # The user has chosen not to save any projects.
            close = True
        else:
            # The user has cancel the close request.
            close = False

        dialog.destroy()
        return close

    def close_current_project(self):
        self._project.selection_clear(False)
        self._project.selection_changed()
        
        for widget in self._project.widgets:
            widget.destroy()

        self._ui_manager.remove_ui(self._project.uim_id)
        self._projects.remove(self._project)

        # If no more projects
        if not self._projects:
            for view in self._project_views:
                view.set_project(None)
            self._project = None
            self._refresh_title()
            self.refresh_undo_and_redo()
            self._update_sensitivity()
            return

        self._set_project(self._projects[0])
        
    def _close_cb(self, action):
        if not self._project:
            return

        if self._project.changed:
            close = self._confirm_close_project(self._project)
            if not close:
                return

        self.close_current_project()


    def _quit_cb(self, action=None):
        unsaved_projects = [p for p in self._projects if p.changed]
        close = self._confirm_close_projects(unsaved_projects)
        if not close:
            return

        gtk.main_quit()

    def _undo_cb(self, action):
        self._command_manager.undo(self._project)
        self.refresh_undo_and_redo()
        self._editor.refresh()

    def _redo_cb(self, action):
        self._command_manager.redo(self._project)
        self.refresh_undo_and_redo()
        self._editor.refresh()

    def _real_cut(self):
        selection = self._project.selection
        if selection:
            gwidget = get_widget_from_gtk_widget(selection[0])
            self._command_manager.cut(gwidget)
            
    def _cut_cb(self, action):
        self._real_cut()

    def _real_copy(self):
        selection = self._project.selection
        if selection:
            gwidget = get_widget_from_gtk_widget(selection[0])
            self._command_manager.copy(gwidget)
            # Make sure the sensitivity of the edit actions (cut,
            # copy,paste,delete) are updated.
            self._update_edit_actions()
    
    def _copy_cb(self, action):
        self._real_copy()

    def _real_paste(self):
        item = self._clipboard.get_selected_item()
        if item is None:
            print _('There is nothing in the clipboard to paste')
            return

        if self._project is None:
            print _("No project has been specified. Cannot paste the widget")
            return
        
        selection = None
        if self._project.selection:
            selection = self._project.selection[0]

        if isinstance(selection, Placeholder) or item.is_toplevel:
            self._command_manager.paste(selection, self._project)
        else:
            print _('Please, select a placeholder to paste the widget on')
        
    def _paste_cb(self, action):
        self._real_paste()
        
    def _delete_cb(self, action):
        if self._project is not None:
            self._project.delete_selection()

    def _about_cb(self, action):
        dialog = gtk.Dialog(_('About Gazpacho'), self._window,
                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                            (gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
        dialog.set_resizable(False)
        dialog.set_border_width(12)
        dialog.vbox.set_spacing(6)
        image = gtk.Image()
        image.set_from_file(os.path.join(pixmaps_dir, 'gazpacho.png'))

        version = gtk.Label('<span size="xx-large"><b>Gazpacho 0.5.3</b></span>')
        version.set_use_markup(True)
        description = gtk.Label(_('Gazpacho is a User Interface Builder for GTK+.\n\n')+'\n\n<small>(C) 2004,2005 SICEm S.L.</small>')
        description.set_use_markup(True)
        description.set_justify(gtk.JUSTIFY_FILL)
        description.set_line_wrap(True)
        
        dialog.vbox.pack_start(image)
        dialog.vbox.pack_start(version)
        dialog.vbox.pack_start(description)
        
        dialog.vbox.show_all()
        dialog.run()
        dialog.destroy()

    def _palette_button_clicked(self, palette):
        klass = palette.current

        # klass may be None if the selector was pressed
        self._add_class = klass
        if klass and klass.is_toplevel():
            self._command_manager.create(klass, None, self._project)
            self._palette.unselect_widget()
            self._add_class = None

    def _project_selection_changed_cb(self, project):
        if self._project != project:
            self._set_project(project)
            return

        if self._editor:
            children = self._project.selection
            if len(children) == 1 and not isinstance(children[0], Placeholder):
                self._editor.load_widget(
                    get_widget_from_gtk_widget(children[0])
                    )
            else:
                self._editor.load_widget(None)

        self._update_edit_actions()
        
    # debugging windows
    def _delete_event_for_debugging_window(self, window, event, action):
        # this will hide the window
        action.activate()
        # we don't want the window to be destroyed
        return True
    
    def _create_debugging_window(self, view, title, action):
        win = gtk.Window()
        win.set_title(title)
        win.set_transient_for(self.window)
        win.add(view)
        view.show_all()
        win.connect('delete-event', self._delete_event_for_debugging_window, action)
        return win
    
    def _show_command_stack_cb(self, action):
        if self._command_stack_window is None:
            view = self._command_stack_view_create()
            title = _('Command Stack')
            action = self._ui_manager.get_action('/MainMenu/DebugMenu/ShowCommandStack')
            self._command_stack_window = self._create_debugging_window(view,
                                                                       title,
                                                                       action)
            self._command_stack_window.show()
        else:
            if self._command_stack_window.get_property('visible'):
                self._command_stack_window.hide()
            else:
                self._command_stack_window.show_all()
                
    def _show_clipboard_cb(self, action):
        """Show/hide the clipboard window."""
        if self._clipboard_window is None:
            action = self._ui_manager.get_action('/MainMenu/DebugMenu/ShowClipboard')
            self._clipboard_window = clipboard.ClipboardWindow(self.window,
                                                               self._clipboard)
            self._clipboard_window.connect('delete-event',
                                           self._delete_event_for_debugging_window,
                                           action)
            self._clipboard_window.show_window()
        else:
            if self._clipboard_window.get_property('visible'):
                self._clipboard_window.hide_window()
            else:
                self._clipboard_window.show_window()

    # useful properties
    def get_add_class(self): return self._add_class
    add_class = property(get_add_class)

    def get_palette(self): return self._palette
    palette = property(get_palette)

    def get_command_manager(self): return self._command_manager
    command_manager = property(get_command_manager)
    
    # some accesors needed by the unit tests
    def get_action(self, action_name):
        action = self._ui_manager.get_action('ui/' + action_name)
        return action

    def get_current_project(self):
        return self._project

    def get_accel_groups(self):
        return (self._ui_manager.get_accel_group(),)
    
    def get_clipboard(self):
        return self._clipboard

    # actions and action groups
    def _real_add_action(self):
        """Can add a simple action or an action group depending on
        the current selection in the ActionsView"""
        gaction = self.gactions_view.get_selected_action()
        if gaction is not None:
            if isinstance(gaction, GActionGroup):
                # create an action with the selected action group
                # as the parent
                parent = gaction
            else:
                # create a brother action of the selected action
                parent = gaction.parent
            
            dialog = GActionDialog(self.window, None)
            if dialog.run() == gtk.RESPONSE_OK:
                values = dialog.get_values()
                self._command_manager.add_action(values, parent, self._project)
            dialog.destroy()
        else:
            # nothing is selected, we create an action group
            dialog = GActionGroupDialog(self.window, None)
            if dialog.run() == gtk.RESPONSE_OK:
                name = dialog.get_action_group_name()
                self._command_manager.add_action_group(name, self._project)
            dialog.destroy()


    def _add_action_cb(self, action):
        self._real_add_action()

    def _real_remove_action(self, gaction):
        if isinstance(gaction, GAction):
            self._command_manager.remove_action(gaction, self._project)
        else:
            self._command_manager.remove_action_group(gaction, self._project)

    def _remove_action_cb(self, action):
        gaction = self.gactions_view.get_selected_action()
        if gaction is not None:
            self._real_remove_action(gaction)

    def _real_edit_action(self, gaction):
        if isinstance(gaction, GAction):
            dialog = GActionDialog(self.window, gaction)
            if dialog.run() == gtk.RESPONSE_OK:
                new_values = dialog.get_values()
                self._command_manager.edit_action(gaction, new_values,
                                                  self._project)
            dialog.destroy()
        else:
            dialog = GActionGroupDialog(self.window, gaction)
            if dialog.run() == gtk.RESPONSE_OK:
                new_name = dialog.get_action_group_name()
                self._command_manager.edit_action_group(gaction, new_name,
                                                        self._project)
            dialog.destroy()
        
    def _edit_action_cb(self, action):
        gaction = self.gactions_view.get_selected_action()
        if gaction is not None:
            self._real_edit_action(gaction)

    def _update_sensitivity(self):
        """Update sensitivity of actions and UI elements (Palette) depending on
        the project list. If the list is empty we should unsensitive
        everything. Otherwise we should set sensitive to True.
        """
        sensitive = len(self._projects) > 0
        
        self._palette.set_sensitive(sensitive)
        
        action_paths = ['/MainMenu/FileMenu/Save',
                       '/MainMenu/FileMenu/SaveAs',
                       '/MainMenu/FileMenu/Close',
                       '/MainMenu/EditMenu/Copy',
                       '/MainMenu/EditMenu/Cut',
                       '/MainMenu/EditMenu/Paste',
                       '/MainMenu/EditMenu/Delete',
                       '/MainMenu/ActionMenu/AddAction',
                       '/MainMenu/ActionMenu/RemoveAction',
                       '/MainMenu/ActionMenu/EditAction']
        u_list = []
        s_list = []
        if sensitive:
            s_list = action_paths
        else:
            u_list = action_paths

        self._change_action_state(sensitive=s_list, unsensitive=u_list)
        self._update_edit_actions()
        self._update_save_action()

    def _update_save_action(self):
        """Update the state of the Save action."""
        if self._project is None:
            return # _update_sensitivity() takes care of this situation
        
        u_list = []
        s_list = []        
        if self._project.changed:
            s_list += ['/MainMenu/FileMenu/Save',]
        else:
            u_list += ['/MainMenu/FileMenu/Save',]            
        self._change_action_state(sensitive=s_list, unsensitive=u_list)

    def _update_edit_actions(self):
        """Update the state of the Cut, Copy, Paste and Delete actions."""
        if self._project is None:
            return # _update_sensitivity() takes care of this situation

        u_list = []
        s_list = []
        selection = self._project.selection
        if selection:
            # No need to Cut, Copy or Delete a placeholder
            if isinstance(selection[0], Placeholder):
                u_list += ['/MainMenu/EditMenu/Copy',
                           '/MainMenu/EditMenu/Cut']
                # Unless it's in a Box
                parent = util.get_parent(selection[0])
                # XXX Not sure if we should hardcode this here
                if parent and isinstance(parent.gtk_widget, gtk.Box):
                    s_list.append('/MainMenu/EditMenu/Delete')
                else:
                    u_list.append('/MainMenu/EditMenu/Delete')
            else:
                s_list += ['/MainMenu/EditMenu/Copy',
                           '/MainMenu/EditMenu/Cut',
                           '/MainMenu/EditMenu/Delete']
        else:
            u_list += ['/MainMenu/EditMenu/Copy',
                       '/MainMenu/EditMenu/Cut',
                       '/MainMenu/EditMenu/Delete']

        # Unless the widget is toplevel it can only be pasted on a placeholder
        item = self._clipboard.get_selected_item()
        if item and (item.is_toplevel
                     or (selection and isinstance(selection[0],
                                                  Placeholder))):
            s_list.append('/MainMenu/EditMenu/Paste')
        else:
            u_list.append('/MainMenu/EditMenu/Paste')
            
        self._change_action_state(sensitive=s_list, unsensitive=u_list)

    def _change_action_state(self, sensitive=[], unsensitive=[]):
        """Set sensitive True for all the actions whose path is on the
        sensitive list. Set sensitive False for all the actions whose path
        is on the unsensitive list.
        """
        for action_path in sensitive:
            action = self._ui_manager.get_action(action_path)
            action.set_property('sensitive', True)
        for action_path in unsensitive:
            action = self._ui_manager.get_action(action_path)
            action.set_property('sensitive', False)            

    def get_current_context(self):
        """Return the context associated with the current project or None if
        there is not such a project.
        """
        if self._project is None:
            return None
        return self._project.context

    def refresh_editor(self):
        self._editor.refresh()

    def _dnd_data_received_cb(self, widget, context, x, y, data, info, time):
        """Callback that handles drag 'n' drop of glade files."""
        if info == Application.TARGET_TYPE_URI:    
            for uri in data.data.split('\r\n'):
                uri_parts = urlparse.urlparse(uri)
                if uri_parts[0] == 'file':                
                    path = urllib.url2pathname(uri_parts[2])
                    self.open_project(path)


    def _dump_data_cb(self, action):
        """This method is only useful for debugging.

        Any developer can print whatever he/she wants here
        the only rule is: clean everything before you commit it.

        This will be called upon CONTROL+M or by using the menu
        item Debug/Dump Data
        """

class CloseConfirmationDialog(gtk.Dialog):
    """A base class for the close confirmation dialogs. It lets the
    user choose to Save the projects, Close without saving them and to
    abort the close operation.
    """
    
    def __init__(self, projects, toplevel=None):
        gtk.Dialog.__init__(self,
                            title='',
                            parent=toplevel,
                            flags=gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT)
        
        self.add_buttons(_("Close _without Saving"), gtk.RESPONSE_NO,
                         gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                         gtk.STOCK_SAVE, gtk.RESPONSE_YES)
        
        self.set_default_response(gtk.RESPONSE_YES)
        
        self.set_border_width(6)
        self.set_resizable(False)
        self.set_has_separator(False)
        self.vbox.set_spacing(12)

        # Add the warning image to the dialog
        warning_image = gtk.image_new_from_stock(gtk.STOCK_DIALOG_WARNING,
                                                 gtk.ICON_SIZE_DIALOG)
        warning_image.set_alignment(0.5, 0)

        self._hbox = gtk.HBox(False, 12)        
        self._hbox.set_border_width(6)
        self._hbox.pack_start(warning_image, False, False, 0)
        
        self.vbox.pack_start(self._hbox)


class SingleCloseConfirmationDialog(CloseConfirmationDialog):
    """A close confirmation dialog for a single project."""
    
    def __init__(self, project, toplevel=None):
        CloseConfirmationDialog.__init__(self, toplevel)
        self._project = project

        submsg1 = _('Save changes to project "%s" before closing?') % \
                  project.name
        submsg2 = _("Your changes will be lost if you don't save them.")
        msg = '<span weight="bold" size="larger">%s</span>\n\n%s\n' % \
              (submsg1, submsg2)
        label = gtk.Label(msg)
        label.set_use_markup(True)
        label.set_line_wrap(True)
        label.set_alignment(0.0, 0.5)        
        self._hbox.pack_start(label)

        self.vbox.show_all()

    def get_projects(self):
        """Get a list of the projects that should be saved."""
        return [self._project]


class MultipleCloseConfirmationDialog(CloseConfirmationDialog):
    """A close confirmation dialog for a multiple project. It presents
    a list of the projects and let the user choose which to save."""
    
    def __init__(self, projects, toplevel=None):
        CloseConfirmationDialog.__init__(self, toplevel)        
        self._projects = projects
        self._projects.sort(lambda x, y: cmp(x.name, y.name))

        # ListStore (boolean, Project) storing information about which
        # projects should be saved.
        self._model = None

        submsg1 = ('There are %d projects with unsaved changes. Do you want to save those changes before closing?') % \
                 len(projects)        
        submsg2 = _("Your changes will be lost if you don't save them.")
        msg = '<span weight="bold" size="larger">%s</span>\n\n%s\n' % \
              (submsg1, submsg2)
        label = gtk.Label(msg)
        label.set_use_markup(True)
        label.set_line_wrap(True)
        label.set_alignment(0.0, 0.5)   

        list_label = gtk.Label()
        list_label.set_markup_with_mnemonic(_("S_elect the projects you want to save:"))
        list_label.set_line_wrap(True)
        list_label.set_alignment(0.0, 0.5)
        list_label.set_padding(0, 6)
        
        view = self._create_project_list()
        list_label.set_mnemonic_widget(view)
        
        scrolled_window = gtk.ScrolledWindow()
        scrolled_window.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scrolled_window.add_with_viewport(view)        
        scrolled_window.set_size_request(200, 100)
        
        project_vbox = gtk.VBox(False)
        project_vbox.pack_start(label)
        project_vbox.pack_start(list_label)
        project_vbox.pack_start(scrolled_window)
        self._hbox.pack_start(project_vbox)
        
        self.vbox.show_all()

    def _create_project_list(self):
        """Create, populate and return a TreeView containing the
        unsaved projects."""
        # Create a ListStore model
        self._model = gtk.ListStore(gobject.TYPE_BOOLEAN, object)
        for project in self._projects:
            self._model.append([True, project])

        # Create the TreeView
        view = gtk.TreeView(self._model)
        view.set_headers_visible(False)

        # Create the check-box column
        toggle_renderer = gtk.CellRendererToggle()
        toggle_renderer.set_property('activatable', True)
        toggle_renderer.connect("toggled", self._toggled_cb, (self._model, 0))
        toggle_column = gtk.TreeViewColumn('Save', toggle_renderer)
        toggle_column.add_attribute(toggle_renderer, 'active', 0)
        view.append_column(toggle_column)

        # Create the project column
        def render_func(treeviewcolumn, renderer, model, iter):
            project = model.get_value(iter, 1)
            renderer.set_property('text', project.name)
            return
        text_renderer = gtk.CellRendererText()        
        text_column = gtk.TreeViewColumn('Project', text_renderer)
        text_column.set_cell_data_func(text_renderer, render_func)
        view.append_column(text_column)

        return view

    def _toggled_cb(self, renderer, path, user_data):
        """Callback to change the state of the check-box."""
        model, column = user_data
        model[path][column] = not model[path][column]
        
    def get_projects(self):
        """Get a list of the projects that should be saved."""        
        return [p[1] for p in self._model if p[0]]
