''' *- python -*-

    FILE: "/home/life/projects/garchiver/garchiver-0.5/src/tar.py"
    LAST MODIFICATION: "Sat, 22 Sep 2001 16:27:07 +0200 (life)"

    (C) 2000 by Danie Roux <droux@tuks.co.za>

    $Id: tar.py,v 1.31 2001/07/04 12:55:57 uid43216 Exp $

    The Tape ARchiver. Doesn't do any compressing by itself.
'''

from archive import *

from settings import *

import os
import string
import shutil

import gettext
_ = gettext.gettext

TRUE = 1
FALSE = 0

class Tar (Archive):
    ''' Class tar

    What makes this class different than the other is that tar, tar.gz and
    tar.bz2 gets parsed by it. This is possible because every format sets
    it's own flags for the different operations below

    '''

    def __init__ (self, name, cb1, cb2, cb3, format_switch=''):
        ''' Sets up the tar object. All the commands and attributes of
        this class.

        The way this works, is that Gzip gets instantiated for example.
        It sets it's format flag ('z') and then call this method to
        initialize the archive object. From then on all the tar archive
        functions work. Except for add and remove. Tar can't update
        compressed archives. So for them I use special compress and
        uncompress commands to temporarily make them into the usual tar
        archive without compression.
        '''

        # By default there is no format flag. All the compressing programs
        # just put their own flag here. gzip's flag is z for example
        self.format_switch = format_switch

        # Let Archive do all the basic setup
        Archive.__init__ (self, name, cb1, cb2, cb3)

        # Sets up these variables
        self.files = []
        self.folders = []

        self.extract_cmd = PS('tar_path') + ' ' + self.format_switch + ' -xvf' + ' 2>&1'
        self.list_cmd = PS('tar_path') + ' ' + self.format_switch + ' -tvf' + ' 2>&1'
        self.view_cmd = PS('tar_path') + ' ' + self.format_switch + ' -xvOf' + ' 2>&1'
        self.create_cmd = PS('tar_path') + ' ' + self.format_switch + ' -cvf' + ' 2>&1'
        # The compressed ones can't be added to. See add.
        self.add_cmd = PS('tar_path') + ' rvf 2>&1'
        # And can not be deleted from. Format flags doesn't matter then
        self.remove_cmd = PS('tar_path') + ' --delete -vf 2>&1'

        # This format needs 'saving'. This is because of a limitation in
        # tar that literally only adds a file to the archive when
        # updating, and it doesn't remove the old one. Which is a terrible
        # waste. But that is what you get with tape technology!
        self.saveable = 1

        # Logs dictionary
        self.logs = {}
        # Current number of logs
        self.log_count = 0
        # Current index in logs
        self.log_index = 0

        self.__errors = []

        # This should be the standalone command that can be called to compress
        # a tar archive. This command should write to standard output. It's
        # used in add. And it's set in the different subsets.
        self.compress_cmd = ''
        # This is the command to uncompress a compressed tar archive back to a
        # normal tar one.
        self.uncompress_cmd = ''

        # Implements a means of caching. When this is TRUE, you don't need
        # to re-create the list
        self.listed = FALSE

        # Whether this archive actually exist, or if it's just an empty
        # file
        self.new_archive = FALSE

    def available (self):
        return os.access (PS ('tar_path'), os.X_OK)

    def _add_to_error_log (self, line):
        ''' This function insures that we don't get the same message all
        the time. It also ignores timestamp warnings.'''

        try: self.__errors.index (line)
        except:
            if string.find (line, 'timestamp') == -1:
                self.__errors.append (line)
                self.error_log = self.error_log + line + '\n'

    # This function gets called by _execute as a 'callback' function that
    # parses a string. All the functions of tar has the same function,
    # more or less.
    def _generic_parse (self, line):
        tar_path = PS ('tar_path')

        if line [:len (tar_path)] == tar_path:
            self._add_to_error_log (line)

        if (line [:-2] != '/'):
            # This is a file, increment the progress
            new_progress = self.progress + self.progress_inc
            # Account for rounding errors
            if new_progress > 1:
                self.progress = 1
            else: self.progress = new_progress

    _save_parse = _generic_parse
    _add_parse = _generic_parse
    _view_parse = _generic_parse
    _remove_parse = _generic_parse
    _extract_parse = _generic_parse
    _create_parse = _generic_parse

    def _compress_and_uncompress_parse (self, line):
        ''' If this gets called, there was output that I didn't want. So
        it is an error. '''

        self._add_to_error_log (line)

    _compress_parse = _compress_and_uncompress_parse
    _uncompress_parse = _compress_and_uncompress_parse

    def _list_parse (self, line):
        # Ignore the blank lines
        if line == '':
            return
        else:
            try:
                (perms, group, size, date, time, pathname) = \
                    string.split (line, None, 5)
            except:
                # If it's an error message it won't fit in the fields
                # always
                self._add_to_error_log (line)
                return

            # That ugly ./ tar puts in front after a saving. Would be nice
            # to find a better fix.
            if pathname [:2] == './':
                pathname = pathname [2:]
                ugly = './'
            else: ugly = ''

            if len (perms) != 10:
                # it is not a file, but an error message
                self._add_to_error_log (line)
            elif perms[0] == 'd': # A directory
                self.folders.append (pathname)
            else:
                pos = string.rfind (pathname, '/')
                if pos != -1:
                    path = pathname [:pos]
                    name = pathname [pos+1:]
                else:
                    path = ''
                    name = pathname

                if path != '':
                    try:
                        if path != self.folders [-1]:
                            ''' I found an archive that has no directory entries,
                            only files with a structure. Instead of running
                            throught the list each time (which would take forever)
                            we just see if the last path is the current path. Tar
                            is sequential, so this should work
                            '''
                            self.folders.append (path + '/')
                    except: 
                        self.folders.append (path + '/') # First path

                self.files.append \
                    ([name, size, ugly+path, date, time, perms, group])

    def _execute (self, cmd, callback=None, files=[], operate_on=None, \
                    show_errors=TRUE):
        ''' All purpose execute function.

        It takes a parameter to a callback function that parses the string
        accordingly.

        files: Is the files that gets used for the action.
        operate_on: Is the name of the archive to operate on. This is
        usually self.name but in the case of add not. If we are working
        with compressed archives we temporarily work with another tar
        archive.

        It returns nothing. The functions edits the data structures itself
        '''

        if files != []:
            # Create a temp file
            tmp = self.make_tempfile ()
            file = open (tmp, 'w')
            # Add a new line to each element
            files = map (lambda x: x + '\n', files)
            file.writelines (files)
            file.close ()

            # Now tell tar to get input from this file
            cmd = cmd + ' -T ' + tmp

        # Just so they all know where the log comes from
        log = cmd + '\n'
        self.error_log = ''
        # The individual errors of this session
        self.__errors == []

        if files != []:
            log = log + _("\nIn temporary file:\n")  + \
                reduce (lambda x, y: x + y, files)

        (child_in, child_out, child_err) = popen2.popen3 (cmd)

        # Don't need child_out in this case
        child_out.close ()

        while 1:
            line = child_in.readline ()
            if line == '': break

            line = string.rstrip (line)

            # Add the line to the log
            log = log + '\n' + line

            # Do some magic on the line
            callback (line)

        child_in.close ()
        child_err.close ()

        # Remove the tempfile if it exists
        try:
            os.remove (tmp)
        except:
            pass

        self.logs [self.log_count] = log
        self.log_index = self.log_count
        self.log_count = self.log_count +1

        if self.error_log != '' and show_errors:
            self.error (self.error_log, log)
            self.error_log = ''
            self.__errors = []

    def extract (self, to, with_path, files=[]):
        if self.new_archive:
            self.error (_("New archive, no files to extract!"))
            return

        self.handle_dot_slash (files)

        # The increment by which the progress indicator must be ... well,
        # incremented!
        if files != []: self.progress_inc = 1.0 / len (files)
        else:
            try: self.progress_inc = 1.0 / len (self.files)
            except ZeroDivisionError: self.progress_inc = 0

        cmd = self.extract_cmd + ' "'+self.name+'" -C ' + '"' + to + '"'

        self._execute (cmd, self._extract_parse, files)

        if not with_path and files == []:
            # TODO
            raise 'Currently you have to select files if you want to extract them without a path.'
        if not with_path:
            for file in files:
                old_path = os.path.join (to, file)
                new_path = os.path.join (to, os.path.basename (file))

                if old_path != new_path:
                    shutil.copy (old_path, new_path)
                    shutil.rmtree (os.path.dirname (old_path))

    def uncompress (self):
        ''' Uncompresses a compressed tar file so that the tar can update
        it. compress must be called before the changes will be reflected
        in the compressed archive

        returns: A temporary file with a tar extension
        '''
        temp_tar = self.make_tempfile (self.name)
        temp_tar = temp_tar + '.tar'

        # Uncompress it to our temp tar file
        self._execute (self.uncompress_cmd + ' "' + self.name + \
                '" 2>&1 > ' + temp_tar, self._uncompress_parse)

        return temp_tar

    def compress (self, temp_tar):
        ''' Compresses a previously uncompressed archive

        temp_tar: Name of an existing tar file to be compressed back
        '''

        # This compresses the tar file to the name it was already. And
        # overwrites the old one.
        self._execute (self.compress_cmd + ' '+temp_tar+' 2>&1 > "' +
            self.name + '" ', self._compress_parse)

        try: os.remove (temp_tar)
        except: pass


    def handle_dot_slash (self, files):
        ''' We filter out ./ at the start of files in a tar file. ./ is as
        good as the current directory. So when you try to delete and view
        the files, you have to add the ./ to be able to find the file in
        the archive.
        '''
        # We filter out ./ so it has to added again
        # Optimize this in some way TODO

        for file in files:
            if file [0] == '/':
                files [files.index (file)] = file [1:]
                file = file [1:]
            try:
                for self_file in self.files:
                    if os.path.join (self_file [2], self_file [0]) == file:
                        break
                else: raise
            except: files [files.index (file)] = './' + file

    def remove (self, files):
        self.listed = FALSE

        self.progress_inc = 1.0 / len (files)

        self.handle_dot_slash (files)

        if self.format_switch != '': # This is a compressed archive
            temp_tar = self.uncompress ()

            self._execute (self.remove_cmd + ' "' + temp_tar + '"', \
                self._remove_parse, files)

            self.compress (temp_tar)

        else: self._execute (self.remove_cmd + ' "' + self.name + '"', \
                self._remove_parse, files)

    def new (self):
        # Create the empty file
        os.system ('touch ' + '"' + self.name + '"')
        self.new_archive = TRUE

    def add (self, files, root_directory='', in_archive='/', flags=''):
        ''' Add files to the archive.

        If it is a compressed archive it gets uncompressed to a temporary
        tar archive, and compressed back later.
        '''

        self.listed = FALSE

        try: self.progress_inc = 1.0 / len (files)
        except ZeroDivisionError:
            # If someone calls add with zero files, they deserve to be
            # shot.
            self.error (_("A zero division should not be possible here!"))

        if files != []:
            self.modified = 1

            # Save all three the executes so that we can display the
            # whole log to the user
            log = ''
            error_log = ''

            if not self.new_archive:
                # decompressing it to a tar, add the files, and
                # compressing it again.
                if self.format_switch != '': # This is a compressed archive

                    temp_tar = self.uncompress ()
                    base_cmd = temp_tar

                    if self.error_log != '':
                        # Error occured, it was reported already in _execute
                        # so all we do now is abort
                        return

                    log = self.log
                    error_log = self.error_log

                else:
                    base_cmd = ' "' + self.name + '" '

                if root_directory != None:
                    base_cmd = base_cmd + ' -C ' + root_directory

                remove_cmd = self.remove_cmd + ' ' + base_cmd

                # Try to remove the files from the archive if they already
                # exist in the archive

                for file in files:
                    for archive_file in self.files:
                        if (os.path.join (archive_file [2], archive_file [0])) == file:
                            # The file exist in the archive. So call tar to remove
                            # it.
                            self._execute (remove_cmd + ' "' + file + '"', \
                                self._remove_parse)
                            break

                # Now add the new files.
                cmd = self.add_cmd + ' ' + base_cmd
                self._execute (cmd, self._add_parse, files)

                log = log + self.log
                error_log = error_log + self.error_log

                if self.format_switch != '':
                    self.compress (temp_tar)

                    log = log + self.log
                    error_log = error_log + self.error_log

                # Full log of events. Too bad the log the user sees is the
                # list log. That's why we have that nifty back button
                self.log = log
                self.error_log = error_log

            else: # It's a new archive, so the create instead of the add
                  # command has to be used

                cmd = self.create_cmd + ' ' + '"' + self.name + '"'

                if root_directory != None:
                    cmd = cmd + ' -C ' + root_directory

                self._execute (cmd, self._create_parse, files)

            self.new_archive = FALSE

    def list_headers (self):
        ''' List the extra headers we want in the clist. '''
        return [_("Permissions"), _("User/Group")]

    def list (self):
        ''' Returns a list of files in this archive. '''

        if self.listed: return self.files
        if self.new_archive: return []

        # else
        self.files = []

        self._execute (self.list_cmd+ ' "'+self.name+'" ', \
            self._list_parse)

        ret = []

        for file in self.files:
            # Make a copy, very important
            work = file
            ret.append (file[:])

            # The ./ again. This is why we can't just return self.files
            if file [2][:2] == './':
                ret [-1:][0][2] = file [2][2:]

        self.listed = TRUE
        return ret

    def list_dir (self):
        ''' Returns a list of all directories in the archive. Very useful
        for the tree interface.
        '''

        if self.new_archive: return []

        self.folders = []

        self._execute (self.list_cmd+' "'+self.name+'" ', \
            self._list_parse, show_errors=FALSE)

        return self.folders

    def view (self, file):
        ''' Extracts the file to a temporary file so that it can be
        viewed.
        '''

        file_list = map (lambda row: os.path.join (row [2], row [0]), \
            self.files)

        # TODO Document
        try:
            for self_file in file_list:
                if self_file == file: break
            else: raise
        except: file = './' + file

        self.progress_inc = 0

        tmp = self.make_tempfile (file)
        cmd = self.view_cmd + ' "' + self.name + '" "' + file + '" > ' + tmp
        self._execute (cmd, self._view_parse)
        return tmp

    def get_log (self):
        return self.logs [self.log_index]

    def has_next_log (self):
        try:
            log = self.logs [self.log_index +1]
            return TRUE
        except: return FALSE

    def has_previous_log (self):
        try:
            log = self.logs [self.log_index -1]
            return TRUE
        except: return FALSE

    def get_previous_log (self):
        try:
            log = self.logs [self.log_index-1]
            self.log_index = self.log_index - 1
            return log
        except: return ''

    def get_next_log (self):
        try:
            log = self.logs [self.log_index +1]
            self.log_index = self.log_index +1
            return log
        except: return ''

    def save (self):
        ''' Lets tar extract to a different directory. Then tar that
        directory again. Note that using -P (absolute pathnames) makes
        this method impossible TODO So user can not use -P as a setting
        for tar
        '''

        self.listed = FALSE

        try: self.progress_inc = 1.0 / len (self.files)
        except ZeroDivisionError: self.progress_inc = 0

        tmpdir = self.make_tempfile ()

        os.mkdir (tmpdir)

        # TODO Error check whether it actually extracted
        self.extract (tmpdir, 1)

        cmd = self.create_cmd + ' "' + self.name + '"' + ' -C ' + \
            tmpdir + ' ./'

        self._execute (cmd, self._save_parse)

        # Remove the temp stuff
        os.system ('rm -rf ' + tmpdir)
