/**
 * File:          $RCSfile: pixel.h,v $
 * Module:        Image pixel definitions and functions
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.37 $
 * Last edited:   $Date: 2003/09/08 17:30:45 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef _GAN_PIXEL_H
#define _GAN_PIXEL_H

#include <gandalf/common/misc_defs.h>
#include <gandalf/common/bit_array.h>
#include <gandalf/linalg/2vector.h>
#include <gandalf/linalg/2vectorf.h>
#include <gandalf/linalg/3vector.h>
#include <gandalf/linalg/3vectorf.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \addtogroup ImageAccessPixel
 * \{
 */

/**
 * \brief Different formats of image pixel.
 */
typedef enum { GAN_GREY_LEVEL_IMAGE,       /**< grey-level images */
               GAN_GREY_LEVEL_ALPHA_IMAGE, /**< grey-level images with alpha
                                                channel */
               GAN_RGB_COLOUR_IMAGE,       /**< RGB colour images */
               GAN_RGB_COLOUR_ALPHA_IMAGE, /**< RGB colour images with alpha
                                                channel */
               GAN_VECTOR_FIELD_2D,        /**< 2D image of 2D vectors */
               GAN_VECTOR_FIELD_3D }       /**< 2D image of 3D vectors */
  Gan_ImageFormat;

/**
 * \brief Structure defining grey-level unsigned character pixel with alpha channel.
 */
typedef struct Gan_GLAPixel_uc
{
   unsigned char I, /**< Intensity channel */
                 A; /**< Alpha channel */
} Gan_GLAPixel_uc;

/**
 * \brief Structure defining grey-level short integer pixel with alpha channel.
 */
typedef struct Gan_GLAPixel_s
{
   short I, /**< Intensity channel */
         A; /**< Alpha channel */
} Gan_GLAPixel_s;

/**
 * \brief Structure defining grey-level unsigned short pixel with alpha channel.
 */
typedef struct Gan_GLAPixel_us
{
   unsigned short I, /**< Intensity channel */
                  A; /**< Alpha channel */
} Gan_GLAPixel_us;

/**
 * \brief Structure defining grey-level signed integer pixel with alpha channel.
 */
typedef struct Gan_GLAPixel_i
{
   int I, /**< Intensity channel */
       A; /**< Alpha channel */
} Gan_GLAPixel_i;

/**
 * \brief Structure defining grey-level unsigned integer pixel with alpha channel.
 */
typedef struct Gan_GLAPixel_ui
{
   unsigned int I, /**< Intensity channel */
                A; /**< Alpha channel */
} Gan_GLAPixel_ui;

/**
 * \brief Structure defining grey-level single precision floating point pixel with alpha channel.
 */
typedef struct Gan_GLAPixel_f
{
   float I, /**< Intensity channel */
         A; /**< Alpha channel */
} Gan_GLAPixel_f;

/**
 * \brief Structure defining grey-level double precision floating point pixel with alpha channel.
 */
typedef struct Gan_GLAPixel_d
{
   double I, /**< Intensity channel */
          A; /**< Alpha channel */
} Gan_GLAPixel_d;

/**
 * \brief Structure defining RGB colour unsigned character pixel.
 */
typedef struct Gan_RGBPixel_uc
{
   unsigned char R, /**< Red channel */
                 G, /**< Green channel */
                 B; /**< Blue channel */
} Gan_RGBPixel_uc;

/**
 * \brief Structure defining RGB colour signed short integer pixel.
 */
typedef struct Gan_RGBPixel_s
{
   short R, /**< Red channel */
         G, /**< Green channel */
         B; /**< Blue channel */
} Gan_RGBPixel_s;

/**
 * \brief Structure defining RGB colour unsigned short integer pixel.
 */
typedef struct Gan_RGBPixel_us
{
   unsigned short R, /**< Red channel */
                  G, /**< Green channel */
                  B; /**< Blue channel */
} Gan_RGBPixel_us;

/**
 * \brief Structure defining RGB colour signed integer pixel.
 */
typedef struct Gan_RGBPixel_i
{
   int R, /**< Red channel */
       G, /**< Green channel */
       B; /**< Blue channel */
} Gan_RGBPixel_i;

/**
 * \brief Structure defining RGB colour unsigned integer pixel.
 */
typedef struct Gan_RGBPixel_ui
{
   unsigned int R, /**< Red channel */
                G, /**< Green channel */
                B; /**< Blue channel */
} Gan_RGBPixel_ui;

/**
 * \brief Structure defining RGB colour singe precision floating point pixel.
 */
typedef struct Gan_RGBPixel_f
{
   float R, /**< Red channel */
         G, /**< Green channel */
         B; /**< Blue channel */
} Gan_RGBPixel_f;

/**
 * \brief Structure defining RGB colour double precision floating point pixel.
 */
typedef struct Gan_RGBPixel_d
{
   double R, /**< Red channel */
          G, /**< Green channel */
          B; /**< Blue channel */
} Gan_RGBPixel_d;

/**
 * \brief Structure defining RGB unsigned character pixel with alpha channel.
 */
typedef struct Gan_RGBAPixel_uc
{
   unsigned char R, /**< Red channel */
                 G, /**< Green channel */
                 B, /**< Blue channel */
                 A; /**< Alpha channel */
} Gan_RGBAPixel_uc;

/**
 * \brief Structure defining RGB signed short integer pixel with alpha channel.
 */
typedef struct Gan_RGBAPixel_s
{
   short R, /**< Red channel */
         G, /**< Green channel */
         B, /**< Blue channel */
         A; /**< Alpha channel */
} Gan_RGBAPixel_s;

/**
 * \brief Structure defining RGB unsigned short integer pixel with alpha channel.
 */
typedef struct Gan_RGBAPixel_us
{
   unsigned short R, /**< Red channel */
                  G, /**< Green channel */
                  B, /**< Blue channel */
                  A; /**< Alpha channel */
} Gan_RGBAPixel_us;

/**
 * \brief Structure defining RGB signed integer pixel with alpha channel.
 */
typedef struct Gan_RGBAPixel_i
{
   int R, /**< Red channel */
       G, /**< Green channel */
       B, /**< Blue channel */
       A; /**< Alpha channel */
} Gan_RGBAPixel_i;

/**
 * \brief Structure defining RGB unsigned integer pixel with alpha channel.
 */
typedef struct Gan_RGBAPixel_ui
{
   unsigned int R, /**< Red channel */
                G, /**< Green channel */
                B, /**< Blue channel */
                A; /**< Alpha channel */
} Gan_RGBAPixel_ui;

/**
 * \brief Structure defining RGB single precision floating point pixel with alpha channel.
 */
typedef struct Gan_RGBAPixel_f
{
   float R, /**< Red channel */
         G, /**< Green channel */
         B, /**< Blue channel */
         A; /**< Alpha channel */
} Gan_RGBAPixel_f;

/**
 * \brief Structure defining RGB double precision floating point pixel with alpha channel.
 */
typedef struct Gan_RGBAPixel_d
{
   double R, /**< Red channel */
          G, /**< Green channel */
          B, /**< Blue channel */
          A; /**< Alpha channel */
} Gan_RGBAPixel_d;

#ifdef GAN_UINT8
#if (SIZEOF_CHAR == 1)
/**
 * \brief Structure defining grey-level 8-bit unsigned integer pixel with alpha channel.
 */
typedef struct Gan_GLAPixel_uc Gan_GLAPixel_ui8;

/**
 * \brief Structure defining RGB colour 8-bit unsigned integer pixel.
 */
typedef struct Gan_RGBPixel_uc Gan_RGBPixel_ui8;

/**
 * \brief Structure defining RGB colour 8-bit unsigned integer pixel with
 *        alpha channel.
 */
typedef struct Gan_RGBAPixel_uc Gan_RGBAPixel_ui8;
#else
#error "8-bit type conflict"
#endif /* #if (SIZEOF_CHAR == 1) */
#endif /* #ifdef GAN_UINT8 */

#ifdef GAN_UINT16
#if (SIZEOF_SHORT == 2)
/**
 * \brief Structure defining grey-level 16-bit unsigned integer pixel with
 *        alpha channel.
 */
typedef struct Gan_GLAPixel_us Gan_GLAPixel_ui16;

/**
 * \brief Structure defining RGB colour 16-bit unsigned integer pixel.
 */
typedef struct Gan_RGBPixel_us Gan_RGBPixel_ui16;

/**
 * \brief Structure defining RGB colour 16-bit unsigned integer pixel with
 *        alpha channel.
 */
typedef struct Gan_RGBAPixel_us Gan_RGBAPixel_ui16;
#else
#error "16-bit type conflict"
#endif /* #if (SIZEOF_SHORT == 2) */
#endif /* #ifdef GAN_UINT16 */

#ifdef GAN_UINT32
#if (SIZEOF_INT == 4)
/**
 * \brief Structure defining grey-level 32-bit unsigned integer pixel with
 *        alpha channel.
 */
typedef struct Gan_GLAPixel_ui Gan_GLAPixel_ui32;

/**
 * \brief Structure defining RGB colour 32-bit unsigned integer pixel.
 */
typedef struct Gan_RGBPixel_ui Gan_RGBPixel_ui32;

/**
 * \brief Structure defining RGB colour 32-bit unsigned integer pixel with
 *        alpha channel.
 */
typedef struct Gan_RGBAPixel_ui Gan_RGBAPixel_ui32;
#else
#error "32-bit type conflict"
#endif /* #if (SIZEOF_INT == 4) */
#endif /* #ifdef GAN_UINT32 */

/**
 * \brief Structure definition for image pixel of any format or type.
 */
typedef struct Gan_Pixel
{
   /// format of image: grey-level, RGB colour etc.
   Gan_ImageFormat format;

   /// type of pixel values: unsigned char, float etc.
   Gan_Type type;

   /// nested union defining pixel types
   union
   {
      /// grey level
      union
      {
         unsigned char  uc;
         short          s;
         unsigned short us;
         int            i;
         unsigned int   ui;
         double         d;
         float          f;
         Gan_Bool       b;
         void          *p;

#ifdef GAN_UINT8
         gan_ui8 ui8;
#endif
#ifdef GAN_UINT16
         gan_ui16 ui16;
#endif
#ifdef GAN_UINT32
         gan_ui32 ui32;
#endif
      } gl;

      /// grey level with alpha channel
      union
      {
         Gan_GLAPixel_uc uc;
         Gan_GLAPixel_s  s;
         Gan_GLAPixel_us us;
         Gan_GLAPixel_i  i;
         Gan_GLAPixel_ui ui;
         Gan_GLAPixel_d  d;
         Gan_GLAPixel_f  f;

#ifdef GAN_UINT8
         Gan_GLAPixel_ui8 ui8;
#endif
#ifdef GAN_UINT16
         Gan_GLAPixel_ui16 ui16;
#endif
#ifdef GAN_UINT32
         Gan_GLAPixel_ui32 ui32;
#endif
      } gla;

      /// RGB colour
      union
      {
         Gan_RGBPixel_uc uc;
         Gan_RGBPixel_s  s;
         Gan_RGBPixel_us us;
         Gan_RGBPixel_i  i;
         Gan_RGBPixel_ui ui;
         Gan_RGBPixel_d  d;
         Gan_RGBPixel_f  f;

#ifdef GAN_UINT8
         Gan_RGBPixel_ui8 ui8;
#endif
#ifdef GAN_UINT16
         Gan_RGBPixel_ui16 ui16;
#endif
#ifdef GAN_UINT32
         Gan_RGBPixel_ui32 ui32;
#endif
      } rgb;

      /// RGB colour with alpha channel
      union
      {
         Gan_RGBAPixel_uc uc;
         Gan_RGBAPixel_s  s;
         Gan_RGBAPixel_us us;
         Gan_RGBAPixel_i  i;
         Gan_RGBAPixel_ui ui;
         Gan_RGBAPixel_d  d;
         Gan_RGBAPixel_f  f;

#ifdef GAN_UINT8
         Gan_RGBAPixel_ui8 ui8;
#endif
#ifdef GAN_UINT16
         Gan_RGBAPixel_ui16 ui16;
#endif
#ifdef GAN_UINT32
         Gan_RGBAPixel_ui32 ui32;
#endif
      } rgba;

      /// 2D vector field
      union
      {
         Gan_Vector2_f f;
         Gan_Vector2   d;
         Gan_Vector2_s s;
         Gan_Vector2_i i;
      } vfield2D;

      /// 3D vector field
      union
      {
         Gan_Vector3_f f;
         Gan_Vector3   d;
         Gan_Vector3_s s;
         Gan_Vector3_i i;
      } vfield3D;
   } data;
} Gan_Pixel;

unsigned char  gan_pixel_rgbuc_to_yuc ( Gan_RGBPixel_uc *rgb );
unsigned short gan_pixel_rgbuc_to_yus ( Gan_RGBPixel_uc *rgb );
unsigned int   gan_pixel_rgbuc_to_yui ( Gan_RGBPixel_uc *rgb );
unsigned char  gan_pixel_rgbus_to_yuc ( Gan_RGBPixel_us *rgb );
unsigned short gan_pixel_rgbus_to_yus ( Gan_RGBPixel_us *rgb );
float          gan_pixel_rgbui_to_yf  ( Gan_RGBPixel_ui *rgb );
float          gan_pixel_rgbf_to_yf   ( Gan_RGBPixel_f  *rgb );
double         gan_pixel_rgbd_to_yd   ( Gan_RGBPixel_d  *rgb );
unsigned char  gan_pixel_rgbauc_to_yuc ( Gan_RGBAPixel_uc *rgba );
unsigned short gan_pixel_rgbauc_to_yus ( Gan_RGBAPixel_uc *rgba );
unsigned int   gan_pixel_rgbauc_to_yui ( Gan_RGBAPixel_uc *rgba );
unsigned char  gan_pixel_rgbaus_to_yuc ( Gan_RGBAPixel_us *rgba );
unsigned short gan_pixel_rgbaus_to_yus ( Gan_RGBAPixel_us *rgba );
float          gan_pixel_rgbaui_to_yf  ( Gan_RGBAPixel_ui *rgba );
float          gan_pixel_rgbaf_to_yf   ( Gan_RGBAPixel_f  *rgba );
double         gan_pixel_rgbad_to_yd   ( Gan_RGBAPixel_d  *rgba );

Gan_Bool gan_image_convert_pixel ( Gan_Pixel *inpix,
                                   Gan_ImageFormat format, Gan_Type type,
                                   Gan_Pixel *outpix, int *error_code );
double gan_image_convert_scale ( Gan_Type gtFromType, Gan_Type gtToType );
Gan_Bool gan_image_offset_pixel_q ( Gan_Pixel *pixel, double offset,
                                    Gan_Pixel *offset_pixel );
Gan_Bool gan_image_scale_pixel_q ( Gan_Pixel *pixel, double scale,
                                   Gan_Pixel *scaled_pixel );
Gan_Bool gan_image_divide_pixel_q ( Gan_Pixel *pixel, double scale,
                                    Gan_Pixel *scaled_pixel );
Gan_Bool gan_image_fill_zero_pixel ( Gan_Pixel *pixel );

/**
 * \brief Macro: Convert a pixel field from \c unsigned \c short to \c unsigned \c char type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
unsigned char gan_pixel_us_to_uc ( unsigned short x );
#else
#if (SIZEOF_SHORT-SIZEOF_CHAR == 1)
#define gan_pixel_us_to_uc(x) ((unsigned char)((x) / 0x101))
#else
#error "Illegal sizes"
#endif /* #if (SIZEOF_SHORT-SIZEOF_CHAR == 1) */
#endif

/**
 * \brief Macro: Convert a pixel field from \c unsigned \c int to \c unsigned \c char type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
unsigned char gan_pixel_ui_to_uc ( unsigned x );
#else
#if (SIZEOF_INT-SIZEOF_CHAR == 3)
#define gan_pixel_ui_to_uc(x) ((unsigned char)((x) / 0x1010101))
#else
#error "Illegal sizes"
#endif /* #if (SIZEOF_INT-SIZEOF_CHAR == 3) */
#endif

/**
 * \brief Macro: Convert a pixel field from \c float to \c unsigned \c char type.
 *
 * Overflow is handled.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
unsigned char gan_pixel_f_to_uc ( float x );
#else
#define gan_pixel_f_to_uc(x) \
   (((x) < 0.0F) ? 0\
                 : (((x) < 1.0F) ? (unsigned char) ((x)*(GAN_UCHAR_MAXF+1.0F))\
                                 : (unsigned char) UCHAR_MAX))
#endif

/**
 * \brief Macro: Convert a pixel field from \c double to \c unsigned \c char type.
 *
 * Overflow is handled.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
unsigned char gan_pixel_d_to_uc ( double x );
#else
#define gan_pixel_d_to_uc(x) \
   (((x) < 0.0) ? 0\
                : (((x) < 1.0) ? (unsigned char) ((x)*(GAN_UCHAR_MAXD+1.0)) \
                               : (unsigned char) UCHAR_MAX))
#endif

/**
 * \brief Macro: Convert a pixel field from \c unsigned \c char to \c unsigned \c short type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
unsigned short gan_pixel_uc_to_us ( unsigned char x );
#else
#if (SIZEOF_SHORT-SIZEOF_CHAR == 1)
#define gan_pixel_uc_to_us(x) ((unsigned short) ((x) * 0x101))
#else
#error "Illegal sizes"
#endif /* #if (SIZEOF_SHORT-SIZEOF_CHAR == 1) */
#endif

/**
 * \brief Macro: Convert a pixel field from \c unsigned \c int to \c unsigned \c short type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
unsigned short gan_pixel_ui_to_us ( unsigned x );
#else
#if (SIZEOF_INT-SIZEOF_SHORT == 2)
#define gan_pixel_ui_to_us(x) ((unsigned short)((x) / 0x10001))
#else
#error "Illegal sizes"
#endif /* #if (SIZEOF_INT-SIZEOF_SHORT == 2) */
#endif

/**
 * \brief Macro: Convert a pixel field from \c int to \c short type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
short gan_pixel_i_to_s ( int x );
#else
#if (SIZEOF_INT-SIZEOF_SHORT == 2)
#define gan_pixel_i_to_s(x) ((short)((x) / 0x10002))
#else
#error "Illegal sizes"
#endif /* #if (SIZEOF_INT-SIZEOF_SHORT == 2) */
#endif

/**
 * \brief Macro: Convert a pixel field from \c float to \c unsigned \c short type.
 *
 * Overflow is handled.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
unsigned short gan_pixel_f_to_us ( float x );
#else
#define gan_pixel_f_to_us(x) \
   (((x) < 0.0F) ? 0\
                 : (((x) < 1.0F) ? (unsigned short)((x)*(GAN_USHRT_MAXF+1.0F))\
                                 : (unsigned short)USHRT_MAX))
#endif

/**
 * \brief Macro: Convert a pixel field from \c double to \c unsigned \c short type.
 *
 * Overflow is handled.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
unsigned short gan_pixel_d_to_us ( double x );
#else
#define gan_pixel_d_to_us(x) \
   (((x) < 0.0) ? 0\
                : (((x) < 1.0 ) ? (unsigned short) ((x)*(GAN_USHRT_MAXD+1.0)) \
                                : (unsigned short) USHRT_MAX))
#endif

/**
 * \brief Macro: Convert a pixel field from \c unsigned \c char to \c unsigned \c int type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
unsigned int gan_pixel_uc_to_ui ( unsigned char x );
#else
#if (SIZEOF_INT-SIZEOF_CHAR == 3)
#define gan_pixel_uc_to_ui(x) ((unsigned) ((x) * 0x1010101))
#else
#error "Illegal sizes"
#endif /* #if (SIZEOF_INT-SIZEOF_CHAR == 3) */
#endif

/**
 * \brief Macro: Convert a pixel field from \c unsigned \c short to \c unsigned \c int type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
unsigned int gan_pixel_us_to_ui ( unsigned short x );
#else
#if (SIZEOF_INT-SIZEOF_SHORT == 2)
#define gan_pixel_us_to_ui(x) ((unsigned) ((x) * 0x10001))
#else
#error "Illegal sizes"
#endif /* #if (SIZEOF_INT-SIZEOF_SHORT == 2) */
#endif

/**
 * \brief Macro: Convert a pixel field from \c short to \c int type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
int gan_pixel_s_to_i ( short x );
#else
#if (SIZEOF_INT-SIZEOF_SHORT == 2)
#define gan_pixel_s_to_i(x) ((int) ((x) * 0x10002))
#else
#error "Illegal sizes"
#endif /* #if (SIZEOF_INT-SIZEOF_SHORT == 2) */
#endif

/**
 * \brief Macro: Convert a pixel field from \c float to \c unsigned \c int type.
 *
 * Overflow is handled.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
unsigned int gan_pixel_f_to_ui ( float x );
#else
#define gan_pixel_f_to_ui(x) \
   (((x) < 0.0F) ? 0\
                 : (((x) < 1.0F) ? (unsigned) ((x)*(GAN_UINT_MAXF+1.0F)) \
                                 : (unsigned) UINT_MAX))

#endif

/**
 * \brief Macro: Convert a pixel field from \c double to \c unsigned \c int type.
 *
 * Overflow is handled.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
unsigned int gan_pixel_d_to_ui ( double x );
#else
#define gan_pixel_d_to_ui(x) \
   (((x) < 0.0) ? 0\
                : (((x) < 1.0) ? (unsigned) ((x)*(GAN_UINT_MAXD+1.0)) \
                               : (unsigned) UINT_MAX))
#endif

/**
 * \brief Macro: Convert a pixel field from \c unsigned \c char to \c float type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
float gan_pixel_uc_to_f ( unsigned char x );
#else
#define gan_pixel_uc_to_f(x) ((float)(x)/GAN_UCHAR_MAXF)
#endif

/**
 * \brief Macro: Convert a pixel field from \c unsigned \c short to \c float type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
float gan_pixel_us_to_f ( unsigned short x );
#else
#define gan_pixel_us_to_f(x) ((float)(x)/GAN_USHRT_MAXF)
#endif

/**
 * \brief Macro: Convert a pixel field from \c unsigned \c int to \c float type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
float gan_pixel_ui_to_f ( unsigned int x );
#else
#define gan_pixel_ui_to_f(x) ((float)(x)/GAN_UINT_MAXF)
#endif

/**
 * \brief Macro: Convert a pixel field from \c unsigned \c char to \c double type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
double gan_pixel_uc_to_d ( unsigned char x );
#else
#define gan_pixel_uc_to_d(x) ((double)(x)/GAN_UCHAR_MAXD)
#endif

/**
 * \brief Macro: Convert a pixel field from \c unsigned \c short to \c double type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
double gan_pixel_us_to_d ( unsigned short x );
#else
#define gan_pixel_us_to_d(x) ((double)(x)/GAN_USHRT_MAXD)
#endif

/**
 * \brief Macro: Convert a pixel field from \c unsigned \c int to \c double type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
double gan_pixel_ui_to_d ( unsigned int x );
#else
#define gan_pixel_ui_to_d(x) ((double)(x)/GAN_UINT_MAXD)
#endif
   
/**
 * \brief Macro: Convert a pixel field from \c float to \c int type.
 *
 * Overflow is handled.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
int gan_pixel_f_to_i ( float x );
#else
#define gan_pixel_f_to_i(x) \
   (((x) < -1.0F) ? INT_MIN\
                 : (((x) < 1.0F) ? (int) ((x)*(GAN_INT_MAXF+1.0F)) \
                                 : INT_MAX))

#endif

/**
 * \brief Macro: Convert a pixel field from \c double to \c int type.
 *
 * Overflow is handled.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
int gan_pixel_d_to_i ( double x );
#else
#define gan_pixel_d_to_i(x) \
   (((x) < -1.0) ? INT_MIN\
                : (((x) < 1.0) ? (int) ((x)*(GAN_INT_MAXD+1.0)) \
                               : INT_MAX))
#endif

/**
 * \brief Macro: Convert a pixel field from \c float to \c short type.
 *
 * Overflow is handled.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
short gan_pixel_f_to_s ( float x );
#else
#define gan_pixel_f_to_s(x) \
   (((x) < -1.0F) ? SHRT_MIN\
                 : (((x) < 1.0F) ? (int) ((x)*(GAN_SHRT_MAXF+1.0F)) \
                                 : SHRT_MAX))

#endif

/**
 * \brief Macro: Convert a pixel field from \c double to \c short type.
 *
 * Overflow is handled.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
short gan_pixel_d_to_s ( double x );
#else
#define gan_pixel_d_to_s(x) \
   (((x) < -1.0) ? SHRT_MIN\
                : (((x) < 1.0) ? (int) ((x)*(GAN_SHRT_MAXD+1.0)) \
                               : SHRT_MAX))
#endif

/**
 * \brief Macro: Convert a pixel field from \c short to \c float type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
float gan_pixel_s_to_f ( short x );
#else
#define gan_pixel_s_to_f(x) ((float)(x)/GAN_SHRT_MAXF)
#endif

/**
 * \brief Macro: Convert a pixel field from \c short to \c double type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
double gan_pixel_s_to_d ( short x );
#else
#define gan_pixel_s_to_d(x) ((double)(x)/GAN_SHRT_MAXD)
#endif
   
/**
 * \brief Macro: Convert a pixel field from \c int to \c float type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
float gan_pixel_i_to_f ( int x );
#else
#define gan_pixel_i_to_f(x) ((float)(x)/GAN_INT_MAXF)
#endif

/**
 * \brief Macro: Convert a pixel field from \c int to \c double type.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
double gan_pixel_i_to_d ( int x );
#else
#define gan_pixel_i_to_d(x) ((double)(x)/GAN_INT_MAXD)
#endif
   
/**
 * \brief Macro: Convert pixel.
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Converts \a inpix to the given \a format and \a type, producing an ouput
 * pixel in \a outpix.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_image_convert_pixel_q ( Gan_Pixel *inpix,
                                     Gan_ImageFormat format, Gan_Type type,
                                     Gan_Pixel *outpix );
#else
#define gan_image_convert_pixel_q(inpix,format,type,outpix) \
           gan_image_convert_pixel(inpix,format,type,outpix,NULL)
#endif

/**
 * \brief Macro: Convert pixel in-place.
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Converts \a pix to the given \a format and \a type, overwriting \a pix with
 * the result.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_image_convert_pixel_i ( Gan_Pixel *pix,
                                     Gan_ImageFormat format, Gan_Type type );
#else
#define gan_image_convert_pixel_i(pix,format,type) \
           gan_image_convert_pixel_q(pix,format,type,pix)
#endif

/**
 * \brief Macro: Offset pixel in-place.
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Adds all the channels of the given \a pixel to the given \a offset
 * factor, overwriting \a pixel with the result.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_image_offset_pixel_i ( Gan_Pixel *pixel, double offset );
#else
#define gan_image_offset_pixel_i(pix,o) gan_image_offset_pixel_q(pix,o,pix)
#endif

/**
 * \brief Macro: Scale pixel in-place.
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Multiplies all the channels of the given \a pixel by the given \a scale
 * factor, overwriting \a pixel with the result.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_image_scale_pixel_i ( Gan_Pixel *pixel, double scale );
#else
#define gan_image_scale_pixel_i(pix,s) gan_image_scale_pixel_q(pix,s,pix)
#endif

/**
 * \brief Macro: Divide pixel by scalar pixel in-place.
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Divides all the channels of the given \a pixel by the given \a scale
 * factor, overwriting \a pixel with the result.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_image_divide_pixel_i ( Gan_Pixel *pixel, double scale );
#else
#define gan_image_divide_pixel_i(pix,s) gan_image_divide_pixel_q(pix,s,pix)
#endif

/**
 * \}
 */

/**
 * \}
 */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef _GAN_PIXEL_H */
