/**
 * File:          $RCSfile: endian_io.h,v $
 * Module:        Little and big-endian I/O for architecture independent I/O
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.3 $
 * Last edited:   $Date: 2002/09/30 12:26:20 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef _GAN_ENDIAN_IO_H
#define _GAN_ENDIAN_IO_H

#include <gandalf/common/misc_defs.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \addtogroup Common
 * \{
 */

/**
 * \addtogroup CommonEndianIO
 * \{
 */

#ifdef GAN_INT16
Gan_Bool gan_fread_lendian_i16  ( FILE *fp, gan_i16 *aui16, size_t nitems );
Gan_Bool gan_fwrite_lendian_i16 ( FILE *fp, gan_i16 *aui16, size_t nitems );
Gan_Bool gan_fread_lendian_ui16  ( FILE *fp, gan_ui16 *aui16, size_t nitems );
Gan_Bool gan_fwrite_lendian_ui16 ( FILE *fp, gan_ui16 *aui16, size_t nitems );
#endif /* #ifdef GAN_INT16 */

#ifdef GAN_INT32
Gan_Bool gan_fread_lendian_i32  ( FILE *fp, gan_i32 *aui32, size_t nitems );
Gan_Bool gan_fwrite_lendian_i32 ( FILE *fp, gan_i32 *aui32, size_t nitems );
Gan_Bool gan_fread_lendian_ui32  ( FILE *fp, gan_ui32 *aui32, size_t nitems );
Gan_Bool gan_fwrite_lendian_ui32 ( FILE *fp, gan_ui32 *aui32, size_t nitems );
#endif /* #ifdef GAN_INT32 */

Gan_Bool gan_fread_lendian_f  ( FILE *fp, float *af, size_t nitems );
Gan_Bool gan_fwrite_lendian_f ( FILE *fp, float *af, size_t nitems );
Gan_Bool gan_fread_lendian_d  ( FILE *fp, double *ad, size_t nitems );
Gan_Bool gan_fwrite_lendian_d ( FILE *fp, double *ad, size_t nitems );

/* macros for standard types */

/**
 * \brief Macro: Reads an array of signed short integers from a file.
 * \param fp Pointer to a binary file opened for reading
 * \param as Array of integers
 * \param nitems The number of integers to be read into the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Reads an array of signed short integers from a file.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_fread_lendian_s  ( FILE *fp, short *as, size_t nitems );
#else
#if (SIZEOF_SHORT == 2)
#define gan_fread_lendian_s(fp,pl,ni)\
          gan_fread_lendian_i16(fp,(gan_i16 *)(pl),ni)
#else
#error "unsupported size"
#endif
#endif /* #ifdef GAN_GENERATE_DOCUMENTATION */

/**
 * \brief Macro: Writes an array of signed short integers to a file.
 * \param fp Pointer to a binary file opened for reading
 * \param as Array of integers
 * \param nitems The number of integers to be written from the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Writes an array of signed short integers to a file.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_fwrite_lendian_s  ( FILE *fp, short *as, size_t nitems );
#else
#if (SIZEOF_SHORT == 2)
#define gan_fwrite_lendian_s(fp,pl,ni)\
          gan_fwrite_lendian_i16(fp,(gan_i16 *)(pl),ni)
#else
#error "unsupported size"
#endif
#endif /* #ifdef GAN_GENERATE_DOCUMENTATION */

/**
 * \brief Macro: Reads an array of unsigned short integers from a file.
 * \param fp Pointer to a binary file opened for reading
 * \param aus Array of integers
 * \param nitems The number of integers to be read into the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Reads an array of unsigned short integers from a file.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_fread_lendian_us  ( FILE *fp, unsigned short *aus, size_t nitems );
#else
#if (SIZEOF_SHORT == 2)
#define gan_fread_lendian_us(fp,pul,ni)\
          gan_fread_lendian_ui16(fp,(gan_ui16 *)(pul),ni)
#else
#error "unsupported size"
#endif
#endif /* #ifdef GAN_GENERATE_DOCUMENTATION */

/**
 * \brief Macro: Writes an array of unsigned short integers to a file.
 * \param fp Pointer to a binary file opened for reading
 * \param aus Array of integers
 * \param nitems The number of integers to be written from the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Writes an array of unsigned short integers to a file.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_fwrite_lendian_us  ( FILE *fp, unsigned short *aus,
                                  size_t nitems );
#else
#if (SIZEOF_SHORT == 2)
#define gan_fwrite_lendian_us(fp,pul,ni)\
          gan_fwrite_lendian_ui16(fp,(gan_ui16 *)(pul),ni)
#else
#error "unsupported size"
#endif
#endif /* #ifdef GAN_GENERATE_DOCUMENTATION */

/**
 * \brief Macro: Reads an array of signed integers from a file.
 * \param fp Pointer to a binary file opened for reading
 * \param ai Array of integers
 * \param nitems The number of integers to be read into the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Reads an array of signed integers from a file.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_fread_lendian_i  ( FILE *fp, int *ai, size_t nitems );
#else
#if (SIZEOF_INT == 4)
#define gan_fread_lendian_i(fp,ai,ni)\
          gan_fread_lendian_i32(fp,(gan_i32 *)(ai),ni)
#else
#error "unsupported size"
#endif
#endif /* #ifdef GAN_GENERATE_DOCUMENTATION */

/**
 * \brief Macro: Writes an array of signed integers to a file.
 * \param fp Pointer to a binary file opened for reading
 * \param ai Array of integers
 * \param nitems The number of integers to be written from the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Writes an array of signed integers to a file.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_fwrite_lendian_i  ( FILE *fp, int *ai, size_t nitems );
#else
#if (SIZEOF_INT == 4)
#define gan_fwrite_lendian_i(fp,ai,ni)\
          gan_fwrite_lendian_i32(fp,(gan_i32 *)(ai),ni)
#else
#error "unsupported size"
#endif
#endif /* #ifdef GAN_GENERATE_DOCUMENTATION */

/**
 * \brief Macro: Reads an array of unsigned integers from a file.
 * \param fp Pointer to a binary file opened for reading
 * \param aui Array of integers
 * \param nitems The number of integers to be read into the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Reads an array of unsigned integers from a file.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_fread_lendian_ui  ( FILE *fp, unsigned *aui, size_t nitems );
#else
#if (SIZEOF_INT == 4)
#define gan_fread_lendian_ui(fp,pui,ni)\
          gan_fread_lendian_ui32(fp,(gan_ui32 *)(pui),ni)
#else
#error "unsupported size"
#endif
#endif /* #ifdef GAN_GENERATE_DOCUMENTATION */

/**
 * \brief Macro: Writes an array of unsigned integers to a file.
 * \param fp Pointer to a binary file opened for reading
 * \param aui Array of integers
 * \param nitems The number of integers to be written from the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Writes an array of unsigned integers to a file.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_fwrite_lendian_ui  ( FILE *fp, unsigned *aui, size_t nitems );
#else
#if (SIZEOF_INT == 4)
#define gan_fwrite_lendian_ui(fp,pui,ni)\
          gan_fwrite_lendian_ui32(fp,(gan_ui32 *)(pui),ni)
#else
#error "unsupported size"
#endif
#endif /* #ifdef GAN_GENERATE_DOCUMENTATION */

/**
 * \brief Macro: Reads an array of signed long integers from a file.
 * \param fp Pointer to a binary file opened for reading
 * \param al Array of integers
 * \param nitems The number of integers to be read into the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Reads an array of signed long integers from a file.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_fread_lendian_l  ( FILE *fp, long *al, size_t nitems );
#else
#if (SIZEOF_LONG == 4)
#define gan_fread_lendian_l(fp,pl,ni)\
          gan_fread_lendian_i32(fp,(gan_i32 *)(pl),ni)
#else
#error "unsupported size"
#endif
#endif /* #ifdef GAN_GENERATE_DOCUMENTATION */

/**
 * \brief Macro: Writes an array of signed long integers to a file.
 * \param fp Pointer to a binary file opened for reading
 * \param al Array of integers
 * \param nitems The number of integers to be written from the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Writes an array of signed long integers to a file.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_fwrite_lendian_l  ( FILE *fp, long *al, size_t nitems );
#else
#if (SIZEOF_LONG == 4)
#define gan_fwrite_lendian_l(fp,pl,ni)\
          gan_fwrite_lendian_i32(fp,(gan_i32 *)(pl),ni)
#else
#error "unsupported size"
#endif
#endif /* #ifdef GAN_GENERATE_DOCUMENTATION */

/**
 * \brief Macro: Reads an array of unsigned long integers from a file.
 * \param fp Pointer to a binary file opened for reading
 * \param aul Array of integers
 * \param nitems The number of integers to be read into the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Reads an array of unsigned long integers from a file.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_fread_lendian_ul  ( FILE *fp, unsigned long *aul, size_t nitems );
#else
#if (SIZEOF_LONG == 4)
#define gan_fread_lendian_ul(fp,pul,ni)\
          gan_fread_lendian_ui32(fp,(gan_ui32 *)(pul),ni)
#else
#error "unsupported size"
#endif
#endif /* #ifdef GAN_GENERATE_DOCUMENTATION */

/**
 * \brief Macro: Writes an array of unsigned long integers to a file.
 * \param fp Pointer to a binary file opened for reading
 * \param aul Array of integers
 * \param nitems The number of integers to be written from the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Writes an array of unsigned long integers to a file.
 */
#ifdef GAN_GENERATE_DOCUMENTATION
Gan_Bool gan_fwrite_lendian_ul  ( FILE *fp, unsigned long *aul,
                                  size_t nitems );
#else
#if (SIZEOF_LONG == 4)
#define gan_fwrite_lendian_ul(fp,pul,ni)\
          gan_fwrite_lendian_ui32(fp,(gan_ui32 *)(pul),ni)
#else
#error "unsupported size"
#endif
#endif /* #ifdef GAN_GENERATE_DOCUMENTATION */

/**
 * \}
 */

/**
 * \}
 */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef _GAN_ENDIAN_IO_H */
