/*
 *  Copyright (C) 2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "misc_callbacks.h"
#include "misc_gui.h"
#include "misc_string.h"
#include "eel-gconf-extensions.h"

/* system includes */
#include <ctype.h>
#include <string.h>
#include <libgnome/gnome-i18n.h>

/* Styles for tab labels */
GtkStyle *loading_text_style = NULL;
GtkStyle *new_text_style = NULL;

/**
 * misc_gui_label_set_accel_text: Sets a label's text with shortcut key
 * handling in the standard _<key> accel way
 */
void
misc_gui_label_set_accel_text (const gchar *text, GtkWidget *label,
			       GtkWidget *menu, GtkWidget *item)
{
	gint tmp_key = gtk_label_parse_uline (GTK_LABEL (label), text);
	GtkAccelGroup *menu_accels = 
		gtk_menu_ensure_uline_accel_group (GTK_MENU (menu));
	gtk_widget_add_accelerator (item, "activate_item", menu_accels,
			tmp_key, 0, 0);
}

/**
 * misc_gui_new_num_accel_label: Creates a label with a numbered/lettered
 * accel
 */
GtkWidget *
misc_gui_new_num_accel_label (gint num, gchar *origtext, gboolean lettersok, 
			      GtkWidget *menu, GtkWidget *item)
{
	gchar *text = misc_string_new_num_accel (num, origtext, lettersok);
	if (text == NULL)
		return gtk_label_new (origtext);
	else
	{
		GtkWidget *label = gtk_label_new ("");
		misc_gui_label_set_accel_text (text, label, menu, item);
		g_free (text);
		return label;
	}
}

/**
 * misc_gui_menu_position_under_widget:
 */
void
misc_gui_menu_position_under_widget (GtkMenu *menu, gint *x, gint *y, 
				     gpointer user_data)
{
	GtkWidget *w = GTK_WIDGET(user_data);
	gint width, height;
	gint screen_width, screen_height;
	GtkRequisition requisition;

	gdk_window_get_size (w->window, &width, &height);
	gdk_window_get_deskrelative_origin(w->window, x, y);
	*y = *y + height;

	gtk_widget_size_request (GTK_WIDGET (menu), &requisition);
      
	screen_width = gdk_screen_width ();
	screen_height = gdk_screen_height ();
	  
	*x = CLAMP (*x, 0, MAX (0, screen_width - requisition.width));
	*y = CLAMP (*y, 0, MAX (0, screen_height - requisition.height));
}

/**
 * misc_gui_colors_init: initialise (or update) global color styles
 */
void
misc_gui_colors_init (void)
{
	GtkWidget *label;
	GtkStyle  *rcstyle;
	char *tmp;

	/* create a dummy label so we can fetch the associated rc style */
	label = gtk_label_new ("");
	rcstyle = gtk_rc_get_style (label);

	/* this is needed for some (broken?) themes */
	if (rcstyle == NULL)
	{
		rcstyle = gtk_style_new ();
	}
	else
	{
		gtk_style_ref (rcstyle);
	}

	/* free previous styles (if any) */
	if (loading_text_style) gtk_style_unref (loading_text_style);
	if (new_text_style) gtk_style_unref (new_text_style);
	
	/* make styles */
	loading_text_style = gtk_style_copy (rcstyle);
	new_text_style = gtk_style_copy (rcstyle);

	/* load colours */
	tmp = eel_gconf_get_string (CONF_TABS_TABBED_LOADING_COLOR);
	gdk_color_parse (tmp, &(loading_text_style->fg[0]));
	gdk_color_parse (tmp, &(loading_text_style->fg[2]));
	g_free (tmp);

	tmp = eel_gconf_get_string (CONF_TABS_TABBED_NEW_COLOR);
	gdk_color_parse (tmp, &(new_text_style->fg[0]));
	gdk_color_parse (tmp, &(new_text_style->fg[2]));
	g_free (tmp);

	/* free */
	gtk_widget_destroy (label);
	gtk_style_unref (rcstyle);
}

/**
 * misc_gui_pixmap_data_new_from_file: create a PixmapData struct from an
 * image
 */
PixmapData *
misc_gui_pixmap_data_new_from_file (const gchar *file, gboolean prelight)
{
	GdkPixbuf *pixbuf;
	PixmapData *pixmap_data;

	/* load pixbuf from disk */
	pixbuf = gdk_pixbuf_new_from_file (file);

	/* check... */
	if (pixbuf == NULL)
	{
		/* warn */
		g_warning (_("`%s' does not exist or is not "
			   "a valid image file\n"), file);

		/* callers should handle this */
		return NULL;
	}

	/* threshold out alpha if possible */
	if (gdk_pixbuf_get_has_alpha       (pixbuf) &&
	    gdk_pixbuf_get_bits_per_sample (pixbuf) == 8 &&
	    gdk_pixbuf_get_colorspace      (pixbuf) == GDK_COLORSPACE_RGB &&
	    gdk_pixbuf_get_n_channels      (pixbuf) == 4)
	{
		misc_gui_pixbuf_threshold_alpha (pixbuf, prelight);
	}
	
	/* allocate space for a new pixmap */
	pixmap_data = g_new0 (PixmapData, 1);

	/* render the pixbuf into the pixmap data */
	gdk_pixbuf_render_pixmap_and_mask (pixbuf, &(pixmap_data->pixmap),
					   &(pixmap_data->mask), 100);

	/* lose the pixbuf */
	gdk_pixbuf_unref (pixbuf);
	
	/* return completed structure */
	return pixmap_data; 
}

/**
 * misc_gui_pixmap_data_free: free a PixmapData structure
 */
void
misc_gui_pixmap_data_free (PixmapData *data)
{
	if (data->pixmap)
		gdk_pixmap_unref (data->pixmap);
	if (data->mask)
		gdk_bitmap_unref (data->mask);
	memset (data, 0, sizeof (PixmapData));
	g_free (data);
}

/**
 * misc_gui_pixbuf_threshold_alpha: this function does the job of
 * thresolding alpha in a pixbuf, by compositing with the default theme
 * background colour, and marking alpha < 0.125 as alpha = 0 -- this looks
 * nice enough against most flat, gradient and pixmap themes
 */
void
misc_gui_pixbuf_threshold_alpha (GdkPixbuf *pixbuf, gboolean prelight)
{
	gint width, height, rowstride;
	guchar r, g, b, a;
	guchar tr, tg, tb;
	guchar *buffer;
	gint x, y, i;
	GdkColor bg;
	GtkWidget *dummy = gtk_button_new ();
	GtkStyle *rcstyle = gtk_rc_get_style (dummy);

	/* this is needed for some (broken?) themes */
	if (rcstyle == NULL)
	{
		rcstyle = gtk_style_new ();
	}
	else
	{
		gtk_style_ref (rcstyle);
	}

	/* get pixbuf properties */
	width  = gdk_pixbuf_get_width (pixbuf);
	height = gdk_pixbuf_get_height (pixbuf);
	buffer = gdk_pixbuf_get_pixels (pixbuf);
	rowstride = gdk_pixbuf_get_rowstride (pixbuf);
	
	/* get target colour to blend with based on style background */
	bg = prelight ? rcstyle->bg[2] : rcstyle->bg[0];
	tr = bg.red   >> 8;
	tg = bg.green >> 8;
	tb = bg.blue  >> 8;

	/* blend whole image */
	for (i = 0, y = 0; y < height; y++, i += rowstride - (width * 4))
	{
		for (x = 0; x < width; x++, i += 4)
		{
			/* get pixel */
			r = buffer[i + 0];
			g = buffer[i + 1];
			b = buffer[i + 2];
			a = buffer[i + 3];

			/* blend pixel */
			r = ((a * r) + ((255 ^ a) * tr)) >> 8;
			g = ((a * g) + ((255 ^ a) * tg)) >> 8;
			b = ((a * b) + ((255 ^ a) * tb)) >> 8;

			/* threshold alpha */
			if (a > 12)
			{
				a = 255;
			}
			else
			{
				a = 0;
			}
			
			/* writeback pixel */
			buffer[i + 0] = r;
			buffer[i + 1] = g;
			buffer[i + 2] = b;
			buffer[i + 3] = a;
		}
	}

	gtk_widget_destroy (dummy);
	gtk_style_unref (rcstyle);

}

/**
 * misc_gui_gtk_radio_button_get: get the active member of a radiobutton
 * group from one of the buttons in the group. This should be in GTK+!
 */
gint
misc_gui_gtk_radio_button_get (GtkRadioButton *radio_button)
{
	GtkToggleButton *toggle_button;
	gint i, length;
        GSList *list;
	
	/* get group list */
        list = gtk_radio_button_group (GTK_RADIO_BUTTON (radio_button));
        length = g_slist_length (list);

	/* iterate over list to find active button */
	for (i = 0; list != NULL; i++, list = g_slist_next (list))
	{
		/* get button and text */
		toggle_button = GTK_TOGGLE_BUTTON (list->data);
		if (gtk_toggle_button_get_active (toggle_button))
		{
			break;
		}
	}

	/* check we didn't run off end */
	g_assert (list != NULL);

	/* return index (reverse order!) */
	return (length - 1) - i;
}

/**
 * misc_gui_gtk_radio_button_set: set the active member of a radiobutton
 * group from one of the buttons in the group. This should be in GTK+!
 */
void
misc_gui_gtk_radio_button_set (GtkRadioButton *radio_button, gint index)
{
	GtkToggleButton *button;
	GSList *list;
	gint length;

	/* get the list */
        list = gtk_radio_button_group (GTK_RADIO_BUTTON (radio_button));

	/* check out the length */
        length = g_slist_length (list);

        /* new buttons are *preppended* to the list, so button added as first 
         * has last position in the list */
        index = (length - 1) - index;

	/* find the right button */
        button = GTK_TOGGLE_BUTTON (g_slist_nth_data (list, index));

	/* set it... this will de-activate the others in the group */
	if (gtk_toggle_button_get_active (button) == FALSE)
	{
		gtk_toggle_button_set_active (button, TRUE);
	}
}

/**
 * misc_gui_radio_menuitem_index: find the index of a give radio menuitem
 * within its group.
 */
gint
misc_gui_radio_menuitem_index (GtkRadioMenuItem *menuitem)
{
	GSList *list;
	gint index, length;

	list = GTK_RADIO_MENU_ITEM (menuitem)->group;
	index = g_slist_index (list, menuitem);
	length = g_slist_length (list);

	/* reverse order! */
	return ((length - 1) - index);
}

/*************************************************************************/
/* original code stolen from gtkaccellabel code, GTK+ CREDITS            */
/* modification necessary because gtkaccellabel doesn't work for what we */
/* want to do in the bookmark menu code                     -- jorn      */
/* N.B. Strings are not translated here as they aren't in original gtk   */
/* either                                                                */
/*************************************************************************/ 
gchar *
misc_gui_string_from_accel_data (GtkWidget *widget, guint accel_key, 
				 GdkModifierType accel_mods)
{
	GtkAccelLabel *al = GTK_ACCEL_LABEL (gtk_accel_label_new (""));
	GtkAccelLabelClass *class = 
		GTK_ACCEL_LABEL_CLASS (GTK_OBJECT (al)->klass);
	gchar *str = NULL;
      
	GString *gstring;
	gboolean had_mod = FALSE;
	
	gstring = g_string_new ("   ");

	if (accel_mods & GDK_SHIFT_MASK)
	{
		g_string_append (gstring, class->mod_name_shift);
		had_mod = TRUE;
	}
	if (accel_mods & GDK_CONTROL_MASK)
	{
		if (had_mod)
	    		g_string_append (gstring, class->mod_separator);
		g_string_append (gstring, class->mod_name_control);
		had_mod = TRUE;
	}
	if (accel_mods & GDK_MOD1_MASK)
	{
		if (had_mod)
			g_string_append (gstring, class->mod_separator);
		g_string_append (gstring, class->mod_name_alt);
		had_mod = TRUE;
	}
	      
	if (had_mod)
		g_string_append (gstring, class->mod_separator);
      	if (accel_key < 0x80 || (accel_key > 0x80 && 
	    accel_key <= 0xff && class->latin1_to_char))
	{
		switch (accel_key)
	    	{
		case ' ':
			g_string_append (gstring, "Space");
			break;
		case '\\':
			g_string_append (gstring, "Backslash");
			break;
		default:
			g_string_append_c (gstring, 
					   toupper (accel_key));
			break;
		}
	}
	else
	{
		gchar *tmp;
	  
		tmp = gtk_accelerator_name (accel_key, 0);
		if (tmp[0] != 0 && tmp[1] == 0)
			tmp[0] = toupper (tmp[0]);
		g_string_append (gstring, tmp);
		g_free (tmp);
	}

	str = gstring->str;
	g_string_free (gstring, FALSE);
	gtk_widget_unref (GTK_WIDGET (al));

	return str;
}

/*************************************************************************/
/* gnome_dialog_run code stolen from libgnomeui, modified because the    */
/* original is buggy.                        -- jorn                     */
/*************************************************************************/

struct GnomeDialogRunInfo {
  gint button_number;
  gint close_id, clicked_id, destroy_id;
  gboolean destroyed;
};

static void
gnome_dialog_shutdown_run (GnomeDialog* dialog,
                           struct GnomeDialogRunInfo* runinfo)
{
  if (!runinfo->destroyed) 
    {
      
      gtk_signal_disconnect(GTK_OBJECT(dialog),
                            runinfo->close_id);
      gtk_signal_disconnect(GTK_OBJECT(dialog),
                            runinfo->clicked_id);
  
      runinfo->close_id = runinfo->clicked_id = -1;
    }

    gtk_main_quit();
}

static void
gnome_dialog_setbutton_callback(GnomeDialog *dialog,
				gint button_number,
				struct GnomeDialogRunInfo *runinfo)
{
  if(runinfo->close_id < 0)
    return;

  runinfo->button_number = button_number;

  gnome_dialog_shutdown_run(dialog, runinfo);
}

static gboolean
gnome_dialog_quit_run(GnomeDialog *dialog,
		      struct GnomeDialogRunInfo *runinfo)
{
  if(runinfo->close_id < 0)
    return FALSE;

  gnome_dialog_shutdown_run(dialog, runinfo);

  return FALSE;
}

static void
gnome_dialog_mark_destroy(GnomeDialog* dialog,
                          struct GnomeDialogRunInfo* runinfo)
{
  runinfo->destroyed = TRUE;

  if(runinfo->close_id < 0)
    return;
  else gnome_dialog_shutdown_run(dialog, runinfo);
}

gint
misc_gui_gnome_dialog_run (GnomeDialog* dialog, gboolean close_after)
{
  struct GnomeDialogRunInfo ri = {-1,-1,-1,-1,FALSE};

  g_return_val_if_fail(dialog != NULL, -1);
  g_return_val_if_fail(GNOME_IS_DIALOG(dialog), -1);

  /* There are several things that could happen to the dialog, and we
     need to handle them all: click, delete_event, close, destroy */

  ri.clicked_id =
    gtk_signal_connect(GTK_OBJECT(dialog), "clicked",
		       GTK_SIGNAL_FUNC(gnome_dialog_setbutton_callback),
		       &ri);

  ri.close_id =
    gtk_signal_connect(GTK_OBJECT(dialog), "close",
		       GTK_SIGNAL_FUNC(gnome_dialog_quit_run),
		       &ri);

  ri.destroy_id = 
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
                       GTK_SIGNAL_FUNC(gnome_dialog_mark_destroy),
                       &ri);

  if ( ! GTK_WIDGET_VISIBLE(GTK_WIDGET(dialog)) )
    gtk_widget_show(GTK_WIDGET(dialog));

  gtk_main();
  
  if(!ri.destroyed) {

    gtk_signal_disconnect(GTK_OBJECT(dialog), ri.destroy_id);

    if(ri.close_id >= 0) /* We didn't shut down the run? */
      {
	gtk_signal_disconnect(GTK_OBJECT(dialog), ri.close_id);
	gtk_signal_disconnect(GTK_OBJECT(dialog), ri.clicked_id);
      }

    if (close_after)
      {
        gnome_dialog_close(dialog);
      }
  }

  return ri.button_number;
}

/*************************************************************************/
/* tree utility functions stolen from gtkctree code, GTK+ CREDITS        */
/* these are necessary because the original GtkCTree is a total f*ck-up, */
/* and has O(n^2) sorting complexity :-(                    -- MattA     */
/*************************************************************************/

static GtkCTreeNode *
gtk_ctree_last_visible (GtkCTree     *ctree,
			GtkCTreeNode *node)
{
  GtkCTreeNode *work;
  
  if (!node)
    return NULL;

  work = GTK_CTREE_ROW (node)->children;

  if (!work || !GTK_CTREE_ROW (node)->expanded)
    return node;

  while (GTK_CTREE_ROW (work)->sibling)
    work = GTK_CTREE_ROW (work)->sibling;

  return gtk_ctree_last_visible (ctree, work);
}

void
misc_gui_gtk_ctree_link (GtkCTree     *ctree,
			 GtkCTreeNode *node,
			 GtkCTreeNode *parent,
			 GtkCTreeNode *sibling,
			 gboolean      update_focus_row)
{
  GtkCList *clist;
  GList *list_end;
  GList *list;
  GList *work;
  gboolean visible = FALSE;
  gint rows = 0;
  
  clist = GTK_CLIST (ctree);

  for (rows = 1, list_end = (GList *)node; list_end->next;
       list_end = list_end->next)
    rows++;

  GTK_CTREE_ROW (node)->parent = parent;
  GTK_CTREE_ROW (node)->sibling = sibling;

  if (!parent || (parent && (gtk_ctree_is_viewable (ctree, parent) &&
			     GTK_CTREE_ROW (parent)->expanded)))
    {
      visible = TRUE;
      clist->rows += rows;
    }

  if (parent)
    work = (GList *)(GTK_CTREE_ROW (parent)->children);
  else
    work = clist->row_list;

  if (sibling)
    {
      if (work != (GList *)sibling)
	{
	  while (GTK_CTREE_ROW (work)->sibling != sibling)
	    work = (GList *)(GTK_CTREE_ROW (work)->sibling);
	  GTK_CTREE_ROW (work)->sibling = node;
	}

      if (sibling == GTK_CTREE_NODE (clist->row_list))
	clist->row_list = (GList *) node;
      if (GTK_CTREE_NODE_PREV (sibling) &&
	  GTK_CTREE_NODE_NEXT (GTK_CTREE_NODE_PREV (sibling)) == sibling)
	{
	  list = (GList *)GTK_CTREE_NODE_PREV (sibling);
	  list->next = (GList *)node;
	}
      
      list = (GList *)node;
      list->prev = (GList *)GTK_CTREE_NODE_PREV (sibling);
      list_end->next = (GList *)sibling;
      list = (GList *)sibling;
      list->prev = list_end;
      if (parent && GTK_CTREE_ROW (parent)->children == sibling)
	GTK_CTREE_ROW (parent)->children = node;
    }
  else
    {
      if (work)
	{
	  /* find sibling */
	  while (GTK_CTREE_ROW (work)->sibling)
	    work = (GList *)(GTK_CTREE_ROW (work)->sibling);
	  GTK_CTREE_ROW (work)->sibling = node;
	  
	  /* find last visible child of sibling */
	  work = (GList *) gtk_ctree_last_visible (ctree,
						   GTK_CTREE_NODE (work));
	  
	  list_end->next = work->next;
	  if (work->next)
	    list = work->next->prev = list_end;
	  work->next = (GList *)node;
	  list = (GList *)node;
	  list->prev = work;
	}
      else
	{
	  if (parent)
	    {
	      GTK_CTREE_ROW (parent)->children = node;
	      list = (GList *)node;
	      list->prev = (GList *)parent;
	      if (GTK_CTREE_ROW (parent)->expanded)
		{
		  list_end->next = (GList *)GTK_CTREE_NODE_NEXT (parent);
		  if (GTK_CTREE_NODE_NEXT(parent))
		    {
		      list = (GList *)GTK_CTREE_NODE_NEXT (parent);
		      list->prev = list_end;
		    }
		  list = (GList *)parent;
		  list->next = (GList *)node;
		}
	      else
		list_end->next = NULL;
	    }
	  else
	    {
	      clist->row_list = (GList *)node;
	      list = (GList *)node;
	      list->prev = NULL;
	      list_end->next = NULL;
	    }
	}
    }

  if (clist->row_list_end == NULL ||
      clist->row_list_end->next == (GList *)node)
    clist->row_list_end = list_end;

  if (visible && update_focus_row)
    {
      gint pos;
	  
      pos = g_list_position (clist->row_list, (GList *)node);
  
      if (pos <= clist->focus_row)
	{
	  clist->focus_row += rows;
	  clist->undo_anchor = clist->focus_row;
	}
    }
}

void
misc_gui_gtk_ctree_unlink (GtkCTree     *ctree, 
			   GtkCTreeNode *node,
			   gboolean      update_focus_row)
{
  GtkCList *clist;
  gint rows;
  gint level;
  gint visible;
  GtkCTreeNode *work;
  GtkCTreeNode *parent;
  GList *list;

  g_return_if_fail (ctree != NULL);
  g_return_if_fail (GTK_IS_CTREE (ctree));
  g_return_if_fail (node != NULL);

  clist = GTK_CLIST (ctree);
  
  if (update_focus_row && clist->selection_mode == GTK_SELECTION_EXTENDED)
    {
//    GTK_CLIST_GET_CLASS (clist)->resync_selection (clist, NULL);
      
      g_list_free (clist->undo_selection);
      g_list_free (clist->undo_unselection);
      clist->undo_selection = NULL;
      clist->undo_unselection = NULL;
    }

  visible = gtk_ctree_is_viewable (ctree, node);

  /* clist->row_list_end unlinked ? */
  if (visible &&
      (GTK_CTREE_NODE_NEXT (node) == NULL ||
       (GTK_CTREE_ROW (node)->children &&
	gtk_ctree_is_ancestor (ctree, node,
			       GTK_CTREE_NODE (clist->row_list_end)))))
    clist->row_list_end = (GList *) (GTK_CTREE_NODE_PREV (node));

  /* update list */
  rows = 0;
  level = GTK_CTREE_ROW (node)->level;
  work = GTK_CTREE_NODE_NEXT (node);
  while (work && GTK_CTREE_ROW (work)->level > level)
    {
      work = GTK_CTREE_NODE_NEXT (work);
      rows++;
    }

  if (visible)
    {
      clist->rows -= (rows + 1);

      if (update_focus_row)
	{
	  gint pos;
	  
	  pos = g_list_position (clist->row_list, (GList *)node);
	  if (pos + rows < clist->focus_row)
	    clist->focus_row -= (rows + 1);
	  else if (pos <= clist->focus_row)
	    {
	      if (!GTK_CTREE_ROW (node)->sibling)
		clist->focus_row = MAX (pos - 1, 0);
	      else
		clist->focus_row = pos;
	      
	      clist->focus_row = MIN (clist->focus_row, clist->rows - 1);
	    }
	  clist->undo_anchor = clist->focus_row;
	}
    }

  if (work)
    {
      list = (GList *)GTK_CTREE_NODE_PREV (work);
      list->next = NULL;
      list = (GList *)work;
      list->prev = (GList *)GTK_CTREE_NODE_PREV (node);
    }

  if (GTK_CTREE_NODE_PREV (node) &&
      GTK_CTREE_NODE_NEXT (GTK_CTREE_NODE_PREV (node)) == node)
    {
      list = (GList *)GTK_CTREE_NODE_PREV (node);
      list->next = (GList *)work;
    }

  /* update tree */
  parent = GTK_CTREE_ROW (node)->parent;
  if (parent)
    {
      if (GTK_CTREE_ROW (parent)->children == node)
	{
	  GTK_CTREE_ROW (parent)->children = GTK_CTREE_ROW (node)->sibling;
	  if (!GTK_CTREE_ROW (parent)->children)
	  {
//	    gtk_ctree_collapse (ctree, parent);
	  }
	}
      else
	{
	  GtkCTreeNode *sibling;

	  sibling = GTK_CTREE_ROW (parent)->children;
	  while (GTK_CTREE_ROW (sibling)->sibling != node)
	    sibling = GTK_CTREE_ROW (sibling)->sibling;
	  GTK_CTREE_ROW (sibling)->sibling = GTK_CTREE_ROW (node)->sibling;
	}
    }
  else
    {
      if (clist->row_list == (GList *)node)
	clist->row_list = (GList *) (GTK_CTREE_ROW (node)->sibling);
      else
	{
	  GtkCTreeNode *sibling;

	  sibling = GTK_CTREE_NODE (clist->row_list);
	  while (GTK_CTREE_ROW (sibling)->sibling != node)
	    sibling = GTK_CTREE_ROW (sibling)->sibling;
	  GTK_CTREE_ROW (sibling)->sibling = GTK_CTREE_ROW (node)->sibling;
	}
    }
}

/*************************************************************************/

/**
 * misc_gui_toolbar_append_separator: build a separator with correct
 * dimensions
 */

GtkWidget *
misc_gui_toolbar_append_separator (GtkToolbar *toolbar)
{
	GtkWidget *vsep = gtk_vseparator_new (), *hsep = gtk_hseparator_new ();

	gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar), vsep, NULL, NULL);
	gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar), hsep, NULL, NULL);
	
	gtk_signal_connect_after (GTK_OBJECT (toolbar), "orientation_changed", 
			    GTK_SIGNAL_FUNC
			   	(separator_toolbar_orientation_changed_cb),
			    vsep);
	gtk_signal_connect_after (GTK_OBJECT (toolbar), "orientation_changed", 
			    GTK_SIGNAL_FUNC
			   	(separator_toolbar_orientation_changed_cb),
			    hsep);
	
	if (GTK_TOOLBAR (toolbar)->orientation == GTK_ORIENTATION_HORIZONTAL)
	{
		gtk_widget_show (vsep);
		return vsep;
	}
	else
	{
		gtk_widget_show (hsep);
		return hsep;
	}
}

/**
 * Progress bar code used by new downloader
 */

#define HASH(colour, width) (GINT_TO_POINTER(((colour) << 8) | (width)))

/**
 * misc_gui_render_bar: render a minature progress bar complete with pixmap
 * cache
 */
GdkPixmap *
misc_gui_render_bar (guint32 colour, gint width)
{
	static GHashTable *pixmap_cache = NULL;
	GdkPixmap *pixmap;
	GdkPixbuf *pixbuf;
	guint32 *pixels;
	int x, y, i;

	/* initialise cache */
	if (pixmap_cache == NULL)
	{
		pixmap_cache = g_hash_table_new (g_direct_hash, 
						 g_direct_equal);
	}

	/* check our cache */
	pixmap = g_hash_table_lookup (pixmap_cache, HASH (colour, width));
	if (pixmap != NULL)
	{
		return pixmap;
	}

	/* make a new pixbuf */
	pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, 
				 BAR_WIDTH, BAR_HEIGHT);

	/* fill with background colour */
	pixels = (guint32 *)(gdk_pixbuf_get_pixels (pixbuf));
	for (i = 0; i < BAR_WIDTH * BAR_HEIGHT; i++)
	{
		pixels[i] = BAR_BACKGROUND;
	}

	/* add border */
	for (i = 0; i < BAR_WIDTH; i++)
	{
		pixels[i] = BAR_BORDER;
		pixels[i + ((BAR_HEIGHT - 1) * BAR_WIDTH)] = BAR_BORDER;
	}
	for (i = 0; i < BAR_HEIGHT; i++)
	{
		pixels[i * BAR_WIDTH] = BAR_BORDER;
		pixels[(i * BAR_WIDTH) + (BAR_WIDTH - 1)] = BAR_BORDER;
	}

	/* draw coloured bit */
	for (y = 1 ; y < BAR_HEIGHT - 1; y++)
	{
		for (x = 1; x < 1 + width; x++)
		{
			pixels[x + (y * BAR_WIDTH)] = colour;
		}
	}

	/* draw delineator(tm) */
	if (width > 0 && width < (BAR_WIDTH - 2))
	{
		for (y = 1 ; y < BAR_HEIGHT - 1; y++)
		{
			pixels[width + (y * BAR_WIDTH)] = BAR_LINE;
		}
	}

	/* render the pixbuf into the pixmap data */
	gdk_pixbuf_render_pixmap_and_mask (pixbuf, &pixmap, NULL, 100);

	/* lose the pixbuf */
	gdk_pixbuf_unref (pixbuf);

	/* add pixmap to cache */
	g_hash_table_insert (pixmap_cache, HASH (colour, width), pixmap);

	/* return the finished pixmap */
	return pixmap;
}

/**
 * misc_gui_gdk_window_is_visible: returns true if the given window is
 * visible, and false if it's withdrawn or iconified.  taken from a post by
 * Owen Taylor at
 * http://mail.gnome.org/archives/gtk-list/1998-June/msg00236.html
 */
gboolean
misc_gui_gdk_window_is_visible (GdkWindow *window)
{
	guint32 wm_state;
	guchar *data;

	if (!window) return FALSE;
	wm_state = gdk_atom_intern ("WM_STATE", TRUE);

	if (!gdk_property_get (window, wm_state, wm_state,
			       0, 4, FALSE, NULL, NULL, NULL, &data))
	{
		return FALSE;
	}
	else
	{
		guint32 state = *(guint32 *) data;

		switch (state)
		{
			case 0: return FALSE;
			case 1: return TRUE;
			case 2: /* intentional */
			default: return FALSE;
		}
	}
}

/**
 * misc_gui_new_popup_menu_lock_accels: create a new gnome menu, and lock all
 * the accels inside it, they don't work in popups anyway
 */
GtkWidget *
misc_gui_new_popup_menu_lock_accels (GnomeUIInfo *menu_uiinfo)
{
        GtkWidget *menu;
        gint i;

        /* create the context menu */
        menu = gnome_popup_menu_new (menu_uiinfo);

        for (i = 0; menu_uiinfo[i].type != GNOME_APP_UI_ENDOFINFO; i++)
	{
                if (!menu_uiinfo[i].widget) continue;

                gtk_widget_lock_accelerators (menu_uiinfo[i].widget);
        }

        return menu;
}
