/**
 * @file coco.c Core Connection API
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "galagod.h"
#include "filter.h"
#include "person-list.h"
#include "service-list.h"
#include "utils.h"


#define GALAGO_DBUS_COCO_PREFIX "/org/freedesktop/Galago/cocos"

struct _GalagodCoCoPrivate
{
	char *name;
	char *id;
	char *dbus_service_id;
	gboolean feed;

	GalagoContext *context;
};

static GHashTable *_cocos    = NULL;
static GHashTable *_coco_ids = NULL;


/**************************************************************************
 * Object/Class support
 **************************************************************************/
static void galagod_coco_finalize(GObject *gobject);

static GalagoObjectClass *parent_class = NULL;

G_DEFINE_TYPE(GalagodCoCo, galagod_coco, GALAGO_TYPE_OBJECT);

static void
galagod_coco_class_init(GalagodCoCoClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	gobject_class->finalize = galagod_coco_finalize;
}

static void
galagod_coco_init(GalagodCoCo *coco)
{
	coco->priv = g_new0(GalagodCoCoPrivate, 1);
	coco->priv->context = galago_context_new();
}

static void
galagod_coco_finalize(GObject *gobject)
{
	GalagodCoCo *coco = (GalagodCoCo *)gobject;
	const char *obj_path;

	obj_path = galago_object_get_dbus_path(GALAGO_OBJECT(coco));

	g_hash_table_remove(_cocos,    obj_path);
	g_hash_table_remove(_coco_ids, coco->priv->dbus_service_id);

	dbus_connection_unregister_object_path(galago_get_dbus_conn(),
										   obj_path);

	galagod_people_remove_with_coco(coco);
	galagod_services_remove_with_coco(coco);

	g_object_unref(coco->priv->context);

	if (coco->priv->name != NULL)
		g_free(coco->priv->name);

	if (coco->priv->id != NULL)
		g_free(coco->priv->id);

	if (coco->priv->dbus_service_id != NULL)
		g_free(coco->priv->dbus_service_id);

	g_free(coco->priv);

	G_OBJECT_CLASS(parent_class)->finalize(gobject);
}

static char *
normalize_name(const char *name)
{
	char *tmp;
	const char *s;
	char *d;

	tmp = g_new0(char, strlen(name) + 1);

	for (s = name, d = tmp; *s != '\0'; s++)
	{
		if (g_ascii_isalnum(*s) && *s != '-' && *s != '_')
			*d++ = *s;
	}

	*d = '\0';

	return tmp;
}

static char *
generate_obj_path(const char *id)
{
	g_return_val_if_fail(id != NULL, NULL);

	return g_strdup_printf("%s/%s", GALAGO_DBUS_COCO_PREFIX, id);
}

static char *
generate_id(const char *name)
{
	char *id = NULL, *nname;
	char *obj_path = NULL;
	int i = 0;

	nname = normalize_name(name);

	do
	{
		if (id != NULL)
			g_free(id);

		id = g_strdup_printf("%s%d", nname, ++i);
		obj_path = generate_obj_path(id);
	}
	while (g_hash_table_lookup(_cocos, obj_path) != NULL);

	g_free(obj_path);
	g_free(nname);

	return id;
}

GalagodCoCo *
galagod_coco_new(const char *name, gboolean feed, const char *dbus_id)
{
	GalagodCoCo *coco;
	char *obj_path;

	g_return_val_if_fail(name     != NULL, NULL);
	g_return_val_if_fail(*name    != '\0', NULL);
	g_return_val_if_fail(dbus_id  != NULL, NULL);
	g_return_val_if_fail(*dbus_id != '\0', NULL);

	coco = g_object_new(GALAGOD_TYPE_COCO, NULL);

	coco->priv->name            = g_strdup(name);
	coco->priv->feed            = feed;
	coco->priv->id              = generate_id(name);
	coco->priv->dbus_service_id = g_strdup(dbus_id);

	obj_path = generate_obj_path(coco->priv->id);
	galago_object_set_dbus_path(GALAGO_OBJECT(coco), obj_path);

	galago_context_push(coco->priv->context);
	galago_context_set_obj_path_prefix(obj_path);
	galago_context_pop();

	g_hash_table_insert(_cocos,    obj_path, coco);
	g_hash_table_insert(_coco_ids, coco->priv->dbus_service_id, coco);

	return coco;
}

GalagoContext *
galagod_coco_get_context(const GalagodCoCo *coco)
{
	g_return_val_if_fail(coco != NULL, NULL);

	return coco->priv->context;
}

const char *
galagod_coco_get_name(const GalagodCoCo *coco)
{
	g_return_val_if_fail(coco != NULL, NULL);

	return coco->priv->name;
}

const char *
galagod_coco_get_service_id(const GalagodCoCo *coco)
{
	g_return_val_if_fail(coco != NULL, NULL);

	return coco->priv->dbus_service_id;
}

const char *
galagod_coco_get_id(const GalagodCoCo *coco)
{
	g_return_val_if_fail(coco != NULL, NULL);

	return coco->priv->id;
}

gboolean
galagod_coco_is_feed(const GalagodCoCo *coco)
{
	g_return_val_if_fail(coco != NULL, FALSE);

	return coco->priv->feed;
}

GalagodCoCo *
galagod_cocos_get_with_id(const char *id)
{
	GalagodCoCo *coco;
	char *obj_path;

	g_return_val_if_fail(id != NULL, NULL);

	obj_path = generate_obj_path(id);
	coco = galagod_cocos_get_with_obj_path(obj_path);
	g_free(obj_path);

	return coco;
}

GalagodCoCo *
galagod_cocos_get_with_service_id(const char *service_id)
{
	g_return_val_if_fail(service_id != NULL, NULL);

	return g_hash_table_lookup(_coco_ids, service_id);
}

GalagodCoCo *
galagod_cocos_get_with_obj_path(const char *obj_path)
{
	g_return_val_if_fail(obj_path != NULL, NULL);

	return g_hash_table_lookup(_cocos, obj_path);
}

void
galagod_cocos_init(void)
{
	_cocos    = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
	_coco_ids = g_hash_table_new(g_str_hash, g_str_equal);
}

static void
destroy_each_coco(gpointer key, gpointer value, gpointer user_data)
{
	g_object_unref(value);
}

void
galagod_cocos_uninit(void)
{
	g_hash_table_foreach(_cocos, destroy_each_coco, NULL);
	g_hash_table_destroy(_cocos);
	g_hash_table_destroy(_coco_ids);

	_cocos = NULL;
}
