
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdarg.h>
#include <unistd.h>
#include <ncurses.h>

#define IO_C_DEF
#include "io.h"

/* Number of colors defined in io.h */
#define NUM_COLORS	8

/* Number of attributes defined in io.h */
#define NUM_ATTRS	7

/* Cursor definitions */
#define CURSOR_INVISIBLE	0
#define CURSOR_NORMAL		1

#define LCS_VL	(0xb3 | A_ALTCHARSET)	/*  */
#define LCS_RT	(0xb4 | A_ALTCHARSET)	/*  */
#define LCS_TR	(0xbf | A_ALTCHARSET)	/*  */
#define LCS_BL	(0xc0 | A_ALTCHARSET)	/*  */
#define LCS_BT	(0xc1 | A_ALTCHARSET)	/*  */
#define LCS_TT	(0xc2 | A_ALTCHARSET)	/*  */
#define LCS_LT	(0xc3 | A_ALTCHARSET)	/*  */
#define LCS_HL	(0xc4 | A_ALTCHARSET)	/*  */
#define LCS_CT	(0xc5 | A_ALTCHARSET)	/*  */
#define LCS_BR	(0xd9 | A_ALTCHARSET)	/*  */
#define LCS_TL	(0xda | A_ALTCHARSET)	/*  */

/* Maps color definitions onto their real definitions */
static int color_map[NUM_COLORS];

/* Maps attribute definitions onto their real definitions */
static int attr_map[NUM_ATTRS];

/* Current attribute used on screen */
static int out_attr;

/* Current color used on screen */
static int out_color;

const line_t line =
{
   .tl = LCS_TL,
   .tr = LCS_TR,
   .bl = LCS_BL,
   .br = LCS_BR,
   .lt = LCS_LT,
   .rt = LCS_RT,
   .tt = LCS_TT,
   .bt = LCS_BT,
   .ct = LCS_CT,
   .hl = LCS_HL,
   .vl = LCS_VL,
};

/* Key definitions go in here */
const int keytable[] =
{
   /*  0 */ KEY_F0,
   /*  1 */ KEY_IC,
   /*  2 */ KEY_DC,
   /*  3 */ KEY_ENTER,
   /*  4 */ '+',
   /*  5 */ '-',
   /*  6 */ '*',
   /*  7 */ '/',
   /*  8 */ KEY_C1,
   /*  9 */ KEY_DOWN,
   /* 10 */ KEY_C3,
   /* 11 */ KEY_LEFT,
   /* 12 */ KEY_B2,
   /* 13 */ KEY_RIGHT,
   /* 14 */ KEY_A1,
   /* 15 */ KEY_UP,
   /* 16 */ KEY_A3,
   /* 17 */ KEY_IC,
   /* 18 */ KEY_HOME,
   /* 19 */ KEY_PPAGE,
   /* 20 */ KEY_NPAGE,
   /* 21 */ KEY_END,
   /* 22 */ KEY_DC,
   /* 23 */ KEY_UP,
   /* 24 */ KEY_DOWN,
   /* 25 */ KEY_LEFT,
   /* 26 */ KEY_RIGHT
};

/*
 * Init & Close
 */

/* Initialize screen */
void io_open ()
{
   /* initialize display */
   initscr ();
   start_color ();
   curs_set (CURSOR_INVISIBLE);
   out_attr = A_NORMAL;
   out_color = COLOR_WHITE;
   noecho ();
   keypad (stdscr,TRUE);
   nodelay (stdscr,FALSE);

   /* map colors */
   color_map[COLOR_BLACK] = COLOR_BLACK;
   color_map[COLOR_RED] = COLOR_RED;
   color_map[COLOR_GREEN] = COLOR_GREEN;
   color_map[COLOR_YELLOW] = COLOR_YELLOW;
   color_map[COLOR_BLUE] = COLOR_BLUE;
   color_map[COLOR_MAGENTA] = COLOR_MAGENTA;
   color_map[COLOR_CYAN] = COLOR_CYAN;
   color_map[COLOR_WHITE] = COLOR_WHITE;

   /* map attributes */
   attr_map[ATTR_OFF] = A_NORMAL;
   attr_map[ATTR_BOLD] = A_BOLD;
   attr_map[ATTR_DIM] = A_DIM;
   attr_map[ATTR_UNDERLINE] = A_UNDERLINE;
   attr_map[ATTR_BLINK] = A_BLINK;
   attr_map[ATTR_REVERSE] = A_REVERSE;
   attr_map[ATTR_INVISIBLE] = A_INVIS;
}

/* Restore original screen state */
void io_close ()
{
   static const char buf[] = "\033[2J\033[1H";
   echo ();
   attrset (A_NORMAL);
   clear ();
   keypad (stdscr,TRUE);
   curs_set (CURSOR_NORMAL);
   refresh ();
   endwin ();
   write (STDOUT_FILENO,buf,sizeof (buf));
}

/*
 * Output
 */

/* Set color attributes */
void out_setattr (int attr)
{
   out_attr = attr_map[attr];
}

/* Set color */
void out_setcolor (int fg,int bg)
{
   out_color = (color_map[bg] << 3) | color_map[fg];
   init_pair (out_color,color_map[fg],color_map[bg]);
   attrset (COLOR_PAIR (out_color) | out_attr);
}

/* Move cursor to position (x,y) on the screen. Upper corner of screen is (0,0) */
void out_gotoxy (int x,int y)
{
   move (y,x);
}

/* Put a character on the screen */
void out_putc (wchar_t c)
{
   addch (c);
}

/* Put a string on the screen */
void out_puts (const char *s)
{
   addstr (s);
}

/* Put a string on the screen */
void out_printf (const char *format, ...)
{
   va_list ap;
   va_start (ap,format);
   vwprintw (stdscr,format,ap);
   va_end (ap);
}

/* Refresh screen */
void out_refresh ()
{
   refresh ();
}

/* Get the screen width */
int out_width ()
{
   return (COLS);
}

/* Get the screen height */
int out_height ()
{
   return (LINES);
}

/* Beep */
void out_beep ()
{
   beep ();
}

/* Clear screen */
void out_clear ()
{
   clear ();
   move (0,0);
}

/*
 * Input
 */

/* Read a character. */
int in_getc ()
{
   return (getch ());
}

/* Empty keyboard buffer */
void in_flush ()
{
   flushinp ();
}

