
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <string.h>

#include <debug/memory.h>

#include <abz/error.h>
#include <abz/aton.h>

#include "config.h"

#define ARRAYSIZE(x) (sizeof (x) / sizeof ((x)[0]))

/*
 * Initialize parser.
 */
void config_create (struct config *config)
{
   config->chain = NULL;
   config->table = NULL;
   config->from = OTHER;
   config->head = config->tail = NULL;
}

static void monitor_destroy (struct monitor **monitor)
{
   struct monitor *tmp;

   while (*monitor != NULL)
	 {
		tmp = *monitor, *monitor = (*monitor)->next;
		mem_free (tmp->label);
		if (tmp->description != NULL) mem_free (tmp->description);
		if (tmp->n) mem_free (tmp->network);
		mem_free (tmp);
	 }
}

/*
 * Clean up.
 */
void config_destroy (struct config *config)
{
   if (config->chain != NULL) mem_free (config->chain);
   if (config->table != NULL) mem_free (config->table);
   config->from = OTHER;
   monitor_destroy (&config->head);
   config->tail = NULL;
}

static int transition (struct config *config,state_t to)
{
   switch (config->from)
	 {
	  case OTHER:
		if (to == MSTMT)
		  {
			 abz_set_error ("chain, table, or monitor expected");
			 return (-1);
		  }
		break;

	  case MONITOR:
		if (to != MSTMT)
		  {
			 abz_set_error ("description, bandwidth, or network expected");
			 return (-1);
		  }
		break;

	  case MSTMT:
		break;
	 }

   config->from = to;

   return (0);
}

static void out_of_memory ()
{
   abz_set_error ("out of memory");
}

static int parse_chain (struct config *config,const struct tokens *tokens)
{
   if (transition (config,OTHER) < 0)
	 return (-1);

   if (tokens->argc != 2)
	 {
		abz_set_error ("usage: chain { INPUT | FORWARD | OUTPUT | <user-defined-chain> }");
		return (-1);
	 }

   if (config->chain != NULL)
	 {
		abz_set_error ("chain %s already defined",config->chain);
		return (-1);
	 }

   if ((config->chain = (char *) mem_alloc ((strlen (tokens->argv[1]) + 1) * sizeof (char))) == NULL)
	 {
		out_of_memory ();
		return (-1);
	 }

   strcpy (config->chain,tokens->argv[1]);

   return (0);
}

static int parse_table (struct config *config,const struct tokens *tokens)
{
   if (transition (config,OTHER) < 0)
	 return (-1);

   if (tokens->argc != 2)
	 {
		abz_set_error ("usage: chain { filter | nat | mangle | <user-defined-table> }");
		return (-1);
	 }

   if (config->table != NULL)
	 {
		abz_set_error ("table %s already defined",config->table);
		return (-1);
	 }

   if ((config->table = (char *) mem_alloc ((strlen (tokens->argv[1]) + 1) * sizeof (char))) == NULL)
	 {
		out_of_memory ();
		return (-1);
	 }

   strcpy (config->table,tokens->argv[1]);

   return (0);
}

static int check_tail (struct config *config)
{
   if (config->tail->description == NULL)
	 {
		abz_set_error ("no description defined for monitor section %s",config->tail->label);
		return (-1);
	 }

   if (!config->tail->n)
	 {
		abz_set_error ("no networks defined for monitor section %s",config->tail->label);
		return (-1);
	 }

   return (0);
}

static int parse_monitor (struct config *config,const struct tokens *tokens)
{
   struct monitor *tmp;
   char *label;

   if (transition (config,MONITOR) < 0)
	 return (-1);

   if (tokens->argc != 2)
	 {
		abz_set_error ("usage: monitor <label>");
		return (-1);
	 }

   for (tmp = config->head; tmp != NULL; tmp = tmp->next)
	 if (!strcmp (tmp->label,tokens->argv[1]))
	   {
		  abz_set_error ("monitor %s already defined",tokens->argv[1]);
		  return (-1);
	   }

   if (config->head != NULL)
	 if (check_tail (config) < 0)
	   return (-1);

   if ((tmp = mem_alloc (sizeof (struct monitor))) == NULL ||
	   (label = mem_alloc ((strlen (tokens->argv[1]) + 1) * sizeof (char))) == NULL)
	 {
		out_of_memory ();
		if (tmp != NULL) mem_free (tmp);
		return (-1);
	 }

   memset (tmp,0L,sizeof (struct monitor));
   tmp->label = label;
   strcpy (tmp->label,tokens->argv[1]);

   if (config->head != NULL)
	 config->tail->next = tmp, config->tail = config->tail->next;
   else
	 config->head = config->tail = tmp;

   return (0);
}

static int parse_description (struct config *config,const struct tokens *tokens)
{
   if (transition (config,MSTMT) < 0)
	 return (-1);

   if (tokens->argc != 2)
	 {
		abz_set_error ("usage: description <text-string>");
		return (-1);
	 }

   if (config->tail->description != NULL)
	 {
		abz_set_error ("description %s already defined for this monitor section",config->tail->description);
		return (-1);
	 }

   if ((config->tail->description = (char *) mem_alloc ((strlen (tokens->argv[1]) + 1) * sizeof (char))) == NULL)
	 {
		out_of_memory ();
		return (-1);
	 }

   strcpy (config->tail->description,tokens->argv[1]);

   return (0);
}

static int __rate (uint64_t *rate,const char *str)
{
   const char *tmp = str;
   uint64_t pow10 = 1;
   enum { BPS, KBPS, MBPS, KBIT, MBIT } suffix;

   *rate = 0;

   while (*tmp >= '0' && *tmp <= '9') tmp++;
   if (tmp == str) return (-1);

   if (*tmp == '\0')
	 suffix = BPS;
   else if (!strcmp (tmp,"kbps"))
	 suffix = KBPS;
   else if (!strcmp (tmp,"mbps"))
	 suffix = MBPS;
   else if (!strcmp (tmp,"kbit"))
	 suffix = KBIT;
   else if (!strcmp (tmp,"mbit"))
	 suffix = MBIT;
   else return (-1);

   while (--tmp >= str)
	 {
		*rate += (*tmp - '0') * pow10, pow10 *= 10;
		if (*rate > (uint32_t) -1) return (-1);
	 }
   if (!*rate) return (-1);

   switch (suffix)
	 {
	  case MBPS:
		*rate *= 1024;
	  case KBPS:
		*rate *= 1024;
	  case BPS:
		break;

	  case MBIT:
		*rate *= 1000;
	  case KBIT:
		*rate *= 1000;
		*rate >>= 3;
		break;
	 }

   return (0);
}

static int getrate (struct bandwidth *rate,int *argc,char ***argv)
{
   if (!*argc || __rate (&rate->rate,**argv) < 0)
	 return (-1);

   rate->peak = rate->rate;
   (*argc)--, (*argv)++;

   if (*argc && !strcmp (**argv,"peak"))
	 {
		(*argc)--, (*argv)++;

		if (!*argc || __rate (&rate->peak,**argv) < 0)
		  return (-1);

		(*argc)--, (*argv)++;
	 }

   return (0);
}

static int parse_bandwidth (struct config *config,const struct tokens *tokens)
{
   int argc = tokens->argc - 1;
   char **argv = tokens->argv + 1;
   int result = -1;

   if (transition (config,MSTMT) < 0)
	 return (-1);

   if (config->tail->input.rate)
	 {
		abz_set_error ("bandwidth already defined for this monitor section");
		return (-1);
	 }

   do
	 {
		if (argc == 1 || argc == 3)
		  {
			 if (!getrate (&config->tail->input,&argc,&argv))
			   {
				  config->tail->output = config->tail->input;
				  result = 0;
			   }
			 break;
		  }
		else
		  {
			 result = 0;
			 config->tail->input.rate = config->tail->output.rate = 0;

			 while (!result && argc)
			   {
				  if (!config->tail->input.rate && !strcmp (*argv,"incoming"))
					{
					   argc--, argv++;
					   result = getrate (&config->tail->input,&argc,&argv);
					   continue;
					}

				  if (!config->tail->output.rate && !strcmp (*argv,"outgoing"))
					{
					   argc--, argv++;
					   result = getrate (&config->tail->output,&argc,&argv);
					   continue;
					}

				  result = -1;
			   }

			 if (!result && config->tail->input.rate && config->tail->output.rate)
			   result = 0;
		  }
	 }
   while (0);

   if (result < 0)
	 {
		abz_set_error ("usage: bandwidth { <bw> | incoming <bw> outgoing <bw> }, where <bw> is <rate> [peak <rate>]");
		return (-1);
	 }

   return (0);
}

static int parse_network (struct config *config,const struct tokens *tokens)
{
   struct network network,*tmp;

   if (transition (config,MSTMT) < 0)
	 return (-1);

   if (tokens->argc != 2 || aton (&network,tokens->argv[1]) < 0)
	 {
		abz_set_error ("usage: network <network>");
		return (-1);
	 }

   if ((tmp = mem_realloc (config->tail->network,(config->tail->n + 1) * sizeof (struct network))) == NULL)
	 {
		out_of_memory ();
		return (-1);
	 }
   config->tail->network = tmp;

   config->tail->network[config->tail->n++] = network;

   return (0);
}

/*
 * Parse a command,
 *
 * Return 0 if successful, -1 if some error occurred. Call
 * abz_get_error() to see the error message.
 */
int config_parse (struct config *config,const struct tokens *tokens)
{
   static const struct
	 {
		const char *name;
		int (*parse) (struct config *,const struct tokens *);
	 } command[] =
	 {
		{ "chain", parse_chain },
		{ "table", parse_table },
		{ "monitor", parse_monitor },
		{ "description", parse_description },
		{ "bandwidth", parse_bandwidth },
		{ "network", parse_network }
	 };
   int i;

   abz_clear_error ();

   for (i = 0; i < ARRAYSIZE (command); i++)
	 if (!strcmp (command[i].name,tokens->argv[0]))
	   return (command[i].parse (config,tokens));

   abz_set_error ("unknown command %s",tokens->argv[0]);

   return (-1);
}

/*
 * Symantic analysis.
 *
 * Return 0 if successful, -1 if some error occurred. Call abz_get_error()
 * to see the error message.
 */
int config_analyze (struct config *config)
{
   abz_clear_error ();

   if (transition (config,OTHER) < 0)
	 return (-1);

   if (config->chain == NULL)
	 {
		abz_set_error ("chain not defined");
		return (-1);
	 }

   if (config->table == NULL)
	 {
		static const char filter[] = "filter";

		if ((config->table = (char *) mem_alloc ((strlen (filter) + 1) * sizeof (char))) == NULL)
		  {
			 out_of_memory ();
			 return (-1);
		  }

		strcpy (config->table,filter);
	 }

   if (config->head == NULL)
	 {
		abz_set_error ("no monitor sections defined");
		return (-1);
	 }

   return (check_tail (config));
}

