/***************************************

    This is part of frox: A simple transparent FTP proxy
    Copyright (C) 2000 James Hollingshead

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  ftp-cmds.c Parsing code for individual commands.
  
  ***************************************/

#include <fcntl.h>
#include "ftp-cmds.h"
#include "control.h"
#include "data.h"
#include "transdata.h"
#include "vscan.h"
#include "os.h"

void pasv_parse(sstr * cmd, sstr * arg);
void port_parse(sstr * cmd, sstr * arg);
void abor_parse(sstr * cmd, sstr * arg);
void xfer_command(sstr * cmd, sstr * arg);
void pasv_reply(sstr * msg);

void ftpcmds_init()
{
	static cmd_struct list[] = {	/*Pinched in part SUSE proxy suite! */
		{"PORT", port_parse},
		{"PASV", pasv_parse},
		{"ABOR", abor_parse},
		{"USER", send_command},
		{"PASS", send_command},
		{"ACCT", send_command},
		{"CWD", send_command},
		{"CDUP", send_command},
		{"SMNT", send_command},
		{"QUIT", send_command},
		{"REIN", send_command},
		{"TYPE", send_command},
		{"STRU", send_command},
		{"MODE", send_command},
		{"RETR", xfer_command},
		{"STOR", xfer_command},
		{"STOU", xfer_command},
		{"APPE", xfer_command},
		{"ALLO", send_command},
		{"REST", send_command},
		{"RNFR", send_command},
		{"RNTO", send_command},
		{"DELE", send_command},
		{"RMD", send_command},
		{"MKD", send_command},
		{"PWD", send_command},
		{"LIST", xfer_command},
		{"NLST", xfer_command},
		{"SITE", send_command},
		{"SYST", send_command},
		{"STAT", send_command},
		{"HELP", send_command},
		{"NOOP", send_command},
		{"SIZE", send_command},	/* Not found in RFC 959 */
		{"MDTM", send_command},
		{"MLFL", send_command},
		{"MAIL", send_command},
		{"MSND", send_command},
		{"MSOM", send_command},
		{"MSAM", send_command},
		{"MRSQ", send_command},
		{"MRCP", send_command},
		{"XCWD", send_command},
		{"XMKD", send_command},
		{"XRMD", send_command},
		{"XPWD", send_command},
		{"XCUP", send_command},
#if 0
		{"APSV", send_command},	/* As per RFC 1579      */
#endif
		{"", 0}
	};

	ftp_cmds = list;
}

/* ------------------------------------------------------------- **
** Parse the PORT command in arg and store the client's data listening
** port. Either send out a PASV instead, or open a port of our own
** and send this to the server in a rewritten PORT command.
** ------------------------------------------------------------- */
void port_parse(sstr * cmd, sstr * arg)
{
	int code;
	sstr *msg;

	info->client_data.address = extract_address(arg);

	if (!config_portok(&info->client_data.address)) {
		send_cmessage(500, "Bad PORT command");
		return;
	}

	if (config.apconv) {
		info->mode = APCONV;
		debug("Rewriting PORT command to PASV\n");

		send_ccommand("PASV", "");
		get_message(&code, &msg);

		info->server_data.address = extract_address(msg);
		if (!config_pasvok(&info->server_data.address)) {
			send_cmessage(500, "Remote server error. PORT failed");
			return;
		} else {
			debug("Rewriting 227 reply.\n");
			send_cmessage(200, "PORT command OK.");
			return;
		}
	} else {
		sstr *newbuf;
		int a1, a2, a3, a4, p1, p2;
		struct sockaddr_in listenaddr;
		socklen_t len;

		if (info->mode == PASSIVE && info->listen != -1)
			il_free();
		info->mode = ACTIVE;
		if (info->listen != -1) close(info->listen);
		info->listen = -1;

		len = sizeof(listenaddr);
		getsockname(info->server_control.fd,
			    (struct sockaddr *) &listenaddr, &len);
		listenaddr.sin_port = 0;
		listenaddr.sin_family = AF_INET;

		info->listen = listen_on_socket(&listenaddr, ACTV);

		if (info->listen == -1) {
			send_cmessage(451, "Proxy unable to comply.");
			return;
		}

		n2com(listenaddr, &a1, &a2, &a3, &a4, &p1, &p2);

		newbuf = sstr_init(40);
		sstr_apprintf(newbuf, "%d,%d,%d,%d,%d,%d", a1, a2, a3, a4,
			      p1, p2);

		debug("  Rewritten PORT command:\n");

		send_command(cmd, newbuf);
		sstr_free(newbuf);
	}
}

/* ------------------------------------------------------------- **
** Intercepted a PASV command. 
**
** Parse the 227 reply message. Either: a) We are transparently
** proxying the data connection - send the 227 through unchanged, and
** do a intercept_listen() for when the client tries to connect. b) We
** aren't - listen on a port of our own and rewrite the 227 with that.
** ------------------------------------------------------------- */
void pasv_parse(sstr * cmd, sstr * arg)
{
	int a1, a2, a3, a4, p1, p2;
	struct sockaddr_in tmp;
	int code;
	sstr *msg, *newbuf;

	debug("  Intercepted a PASV command\n");

	info->mode = PASSIVE;
	close(info->listen);
	info->listen = -1;

	send_command(cmd, arg);
	get_message(&code, &msg);

	info->server_data.address = extract_address(msg);
	if (!config_pasvok(&info->server_data.address)) {
		send_cmessage(500, "Bad passive command from server");
		return;
	}

	if (config.transdata) {
		get_local_address(&tmp);
		info->listen =
		    intercept_listen(info->server_data.address, tmp, PASV);
		if (info->listen != -1) {
			send_message(227, msg);
			info->mode = PASSIVE;
			return;
		}
		debug("Intercept_listen failed. Rewriting 227 reply instead\n");
	}

	get_local_address(&tmp);
	tmp.sin_port = 0;

	info->listen = listen_on_socket(&tmp, PASV);

	if (info->listen == -1) {
		send_cmessage(451, "Screwed up pasv command.");
		return;
	}

	n2com(tmp, &a1, &a2, &a3, &a4, &p1, &p2);

	newbuf = sstr_init(60);
	sstr_apprintf(newbuf, "Entering Passive Mode (%d,%d,%d,%d,%d,%d)",
		      a1, a2, a3, a4, p1, p2);

	debug("  Rewritten 227 reply:\n");

	send_message(227, newbuf);
	info->mode = PASSIVE;
	sstr_free(newbuf);
}

/* ------------------------------------------------------------- **
** Intercepted an ABOR -- we need to send telnet IPs etc.
** ------------------------------------------------------------- */
void abor_parse(sstr * cmd, sstr * arg)
{
	int code;
	close(info->server_data.fd);
	close(info->client_data.fd);
	info->server_data.fd = info->client_data.fd = -1;
	info->state = NEITHER;

	get_message(&code, NULL);
	send_cmessage(426, "Transfer aborted. Data connection closed.");
	send_cmessage(226, "Abort successful");
	return;
}

/* ------------------------------------------------------------- **
** Commands that require a data stream.
** ------------------------------------------------------------- */
void xfer_command(sstr * cmd, sstr * arg)
{
	if (info->mode == APCONV) {
		debug2("Connecting to both data streams for %s command\n",
		       sstr_buf(cmd));
		if(connect_client_data() == -1) {
			send_cmessage(425, "Can't open data connection");
			return;
		}

		if(connect_server_data() == -1) {
			send_cmessage(425, "Can't open data connection");
			return;
		}
	}

	if (!sstr_casecmp2(cmd, "RETR") ||
	    !sstr_casecmp2(cmd, "LIST") || !sstr_casecmp2(cmd, "NLST"))
		info->state = DOWNLOAD;
	else
		info->state = UPLOAD;
	send_command(cmd, arg);

	if(!sstr_casecmp2(cmd, "RETR"))
		vscan_new(0);
}
