/***************************************

    This is part of frox: A simple transparent FTP proxy
    Copyright (C) 2000 James Hollingshead

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  cache.c -- Generic code for caching.

Overview:

   cache_init() redirects USER in ftp_cmds to come here. If user is
   anonymous we redirect all the other stuff we are going to need to
   parse or intercept for caching to come here too.

   We intercept RETR commands, and get all the info we need about the
   filename (size, last modification date, and uri).

   We call the specific cache code (in squidcache.c or localcache.c)
   with all this information, and it has the opportunity to return a
   file descriptor which will return the file.

   Specific cache code gets called with all incoming data in case it
   wishes to either alter it (ie. strip HTTP headers), or store it.

  Current Problems include: 
      o STAT does nothing useful.
      o If we start doing anything with USER or 220 cmds then we have
        the potential to interact with non-transparent proxying code.

  ***************************************/

#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "common.h"
#include "control.h"
#include "ftp-cmds.h"
#include "transdata.h"
#include "cache.h"

void cache_user(sstr * cmd, sstr * arg);
void cache_stat(sstr * cmd, sstr * arg);
void cache_abor(sstr * cmd, sstr * arg);
void cache_mode(sstr * cmd, sstr * arg);
void cache_stru(sstr * cmd, sstr * arg);
void cache_type(sstr * cmd, sstr * arg);
void cache_cwd(sstr * cmd, sstr * arg);
void cache_rest(sstr * cmd, sstr * arg);
void cache_retr(sstr * cmd, sstr * arg);
int setup_fileinfo(sstr * cmd, sstr * arg);
void strip2bs(sstr *p);

extern void abor_parse(sstr * cmd, sstr * arg);
extern void xfer_command(sstr * cmd, sstr * arg);

static enum {
	NOTHING = FALSE,
	GOT_SIZE,
	GOT_MDTM,
	GOT_CWD,
	CLOSE
} expecting = NOTHING;

static int user_ind, mode_ind, cwd_ind, cdup_ind;
static int stru_ind, type_ind, rest_ind, retr_ind;

static int nooping = FALSE;
static int type = 0;		/*Ascii or Binary */
static int noop = 0;		/*No of NOOPs sent */
static time_t last_noop;
static int offset = 0;

struct filedetails {
	sstr *host;
	sstr *path;
	sstr *filename;
	int size;
	sstr *mdtm;
};

struct filedetails fileinfo = { NULL, NULL, NULL, 0, NULL};

#define NOOP_INTERVAL 30	/*Seconds */

static cmd_struct xfer_list[] = {
	{"ABOR", cache_abor},
	{"STAT", cache_stat},
	{"", NULL}
};
cmd_struct *main_cmdlist;
sstr *strictpath;

static int (*geninit) (void);
static int (*retr_start) (const sstr * host, const sstr * file,
			  const sstr * mdtm, int size, int offset, int type);
void (*inc_data) (sstr * inc);
static int (*retr_end) (void);

/* ------------------------------------------------------------- **
** Set up the function pointers to the cache module specified in
** cachemod.
** ------------------------------------------------------------- */
int cache_geninit(void)
{
	if (!config.cachemod)
		return (0);
	if (!strcasecmp(config.cachemod, "Local")) {
		geninit = l_geninit;
		retr_start = l_retr_start;
		inc_data = l_inc_data;
		retr_end = l_retr_end;
	} else if (!strcasecmp(config.cachemod, "HTTP")) {
		geninit = s_geninit;
		retr_start = s_retr_start;
		inc_data = s_inc_data;
		retr_end = s_retr_end;
	} else {
		write_log(ERROR, "Unrecognised cache module %s",
			  config.cachemod);
		return (-1);
	}

	if (geninit == NULL) {
		write_log(ERROR, "Cache module %s is not compiled in",
			  config.cachemod);
		free(config.cachemod);
		config.cachemod=NULL;
		return (-1);
	}
	if (!fileinfo.host)
		fileinfo.host = sstr_init(MAX_LINE_LEN);
	if (!fileinfo.path)
		fileinfo.path = sstr_init(MAX_LINE_LEN);
	if (!fileinfo.filename)
		fileinfo.filename = sstr_init(MAX_LINE_LEN);
	if (!fileinfo.mdtm)
		fileinfo.mdtm = sstr_init(MAX_LINE_LEN);

	if(geninit()==-1) {
		write_log(ERROR, "Failed initialisation of cache");
		free(config.cachemod);
		config.cachemod = NULL;
		return -1;
	}
	return 0;
}

/* ------------------------------------------------------------- **
** Find and store the ftp_cmds that we may need to modify.
** Modify the ftp_cmds structure so that we get called on login.
** ------------------------------------------------------------- */
void cache_init(void)
{
	int i;

	for (i = 0; *(ftp_cmds + i)->name != 0; i++) {
		if (!strcasecmp((ftp_cmds + i)->name, "USER"))
			user_ind = i;
		else if (!strcasecmp((ftp_cmds + i)->name, "MODE"))
			mode_ind = i;
		else if (!strcasecmp((ftp_cmds + i)->name, "STRU"))
			stru_ind = i;
		else if (!strcasecmp((ftp_cmds + i)->name, "TYPE"))
			type_ind = i;
		else if (!strcasecmp((ftp_cmds + i)->name, "REST"))
			rest_ind = i;
		else if (!strcasecmp((ftp_cmds + i)->name, "RETR"))
			retr_ind = i;
		else if (!strcasecmp((ftp_cmds + i)->name, "CDUP"))
			cdup_ind = i;
		else if (!strcasecmp((ftp_cmds + i)->name, "CWD"))
			cwd_ind = i;
	}

	(ftp_cmds + user_ind)->cmd = cache_user;
	main_cmdlist = ftp_cmds;
}

/* ------------------------------------------------------------- **
** If login is anonymous then redirect the cmds we need to intercept
** ------------------------------------------------------------- */
void cache_user(sstr * cmd, sstr * arg)
{
	send_command(cmd, arg);

	if (sstr_casecmp2(arg, "ftp")
	    && sstr_casecmp2(arg, "anonymous"))
		return;

	(ftp_cmds + mode_ind)->cmd = cache_mode;
	(ftp_cmds + stru_ind)->cmd = cache_stru;
	(ftp_cmds + type_ind)->cmd = cache_type;
	(ftp_cmds + rest_ind)->cmd = cache_rest;
	(ftp_cmds + retr_ind)->cmd = cache_retr;
	if (config.strictcache) {
		strictpath = sstr_init(BUF_LEN);
		(ftp_cmds + cdup_ind)->cmd = cache_cwd;
		(ftp_cmds + cwd_ind)->cmd = cache_cwd;
	}
}

/* ------------------------------------------------------------- **
** Commands to intercept at all times. TODO Consider moving stru and
** mode functions to ftp-cmds.c They are pretty much obsolete, and
** probably not used by anyone any more. 
** ------------------------------------------------------------- */
void cache_mode(sstr * cmd, sstr * arg)
{
	if (sstr_getchar(arg, 0) == 'S')
		send_cmessage(200, "Command okay");
	else
		send_cmessage(504, "Only stream mode implemented");
}

void cache_stru(sstr * cmd, sstr * arg)
{
	if (sstr_getchar(arg, 0) == 'F')
		send_cmessage(200, "Command okay");
	else
		send_cmessage(504, "Only file structure implemented");
}

void cache_type(sstr * cmd, sstr * arg)
{
	if (sstr_getchar(arg, 0) == 'I') {
		type = 1;
		send_command(cmd, arg);
	} else if (!sstr_casecmp2(arg, "AN")
		   || !sstr_casecmp2(arg, "A")) {
		type = 0;
		send_command(cmd, arg);
	} else
		send_cmessage(504, "Only types I and AN implemented");
}

void cache_rest(sstr * cmd, sstr * arg)
{

	debug("cache.c intercepted REST\n");
	offset = sstr_atoi(arg);

	send_command(cmd, arg);
	/*send_message(350,"Setting restart point. Send RETR to initiate transfer."); */
}

void cache_cwd(sstr * cmd, sstr * arg)
{
	int code;
	sstr *msg;

	send_command(cmd, arg);
	get_message(&code, &msg);
	send_message(code, msg);

	if (code > 299)
		return;


	if (sstr_getchar(cmd, 1) == 'D') /*CDUP*/
		sstr_ncat2(strictpath, "..", 2);
	else { /*CWD*/ 
		sstr_escapebs(arg);
		sstr_cat(strictpath, arg);
	}
	sstr_ncat2(strictpath, "/", 1);
	debug2("Strictpath = \"%s\"\n", sstr_buf(strictpath));
}

/* ------------------------------------------------------------- **
** Commands to intercept during transfer
** ------------------------------------------------------------- */
void cache_stat(sstr * cmd, sstr * arg)
{
	send_cmessage(213, "Retrieving file through cache");
}

void cache_abor(sstr * cmd, sstr * arg)
{
	int code;
	while (noop > 0) {
		get_message(&code, NULL);
		noop--;
	}
	nooping = FALSE;

	close(info->client_data.fd);
	close(info->server_data.fd);
	info->server_data.fd = info->client_data.fd = -1;
	info->state = NEITHER;

	ftp_cmds=main_cmdlist;

	debug("cache.c intercepted ABOR\n");
	send_cmessage(426, "transfer aborted");
	send_cmessage(226, "Closing data connection");
}

/* ------------------------------------------------------------- **
** The important bit. Intercept RETR commands, find out all we need to
** about the file, and then pass all the info to the caching function. 
** ------------------------------------------------------------- */
void cache_retr(sstr * cmd, sstr * arg)
{
	int i=-1, code;
	sstr *msg;

	if(setup_fileinfo(cmd, arg) == 0)
		i = retr_start(fileinfo.host, fileinfo.path, fileinfo.mdtm,
			       fileinfo.size, offset, type);

	if (i == -1) {
		sstr *tmp;
		tmp = sstr_init(10);
		nooping = FALSE;
		if (offset != 0) {	/*Send another REST since we sent loads
					   * of rubbish since the last one. */
			sstr_apprintf(tmp, "%d", offset);
			offset = 0;
			send_ccommand("REST", sstr_buf(tmp));
			get_message(&code, &msg);
			if (code != 350) {
				send_cmessage(503, "Can't do REST after all!");
				sstr_free(tmp);
				return;
			}
		}
		sstr_cpy2(tmp, "RETR");
		xfer_command(tmp, fileinfo.filename);
		sstr_free(tmp);
		return;
	}

	/*Cache is retrieving the file - it will deal with REST, so
          reset it to 0 just in case*/
	if (offset != 0) {
		send_ccommand("REST", "0");
		get_message(&code, &msg);
		offset = 0;
	}

	/*Set up everything as if this were a normal connection */
	if (info->server_data.fd != -1) 
		close(info->server_data.fd);
	info->server_data.fd = i;
	sstr_empty(info->client_data.buf);
	sstr_empty(info->server_data.buf);
	if (info->client_data.fd == -1 && info->mode != PASSIVE) {
		if (config.transdata) {
			struct sockaddr_in tmp = info->server_control.address;
			tmp.sin_port = htons(20);
			info->client_data.fd =
			    transp_connect(info->client_data.address, tmp);
		} else
			info->client_data.fd =
			    connect_to_socket(info->client_data.address, ACTV);
	}

	main_cmdlist = ftp_cmds;
	ftp_cmds = xfer_list;

	nooping = TRUE;
	time(&last_noop);
}

int setup_fileinfo(sstr * cmd, sstr * arg)
{
	int code;
	sstr *msg;

	sstr_cpy(fileinfo.filename, arg);
	if (config.strictcache)
		sstr_escapebs(fileinfo.filename);

	send_ccommand("SIZE", sstr_buf(arg));
	get_message(&code, &msg);
	if (code / 100 != 2) {
		debug("SIZE not accepted - aborting caching\n");
		return (-1);
	}
	fileinfo.size = sstr_atoi(msg);
	debug2("Cache: Filesize is %d\n", fileinfo.size);

	send_ccommand("MDTM", sstr_buf(fileinfo.filename));
	get_message(&code, &msg);
	if (code / 100 != 2) {
		debug("MDTM not accepted - aborting caching\n");
		return (-1);
	}
	sstr_cpy(fileinfo.mdtm, msg);
	debug2("Cache: MDTM is %s\n", sstr_buf(fileinfo.mdtm));

	if (!config.strictcache) {
		if(sstr_getchar(fileinfo.filename, 0) != '/') {
			send_ccommand("PWD", "");
			get_message(&code, &msg);
			if (sstr_getchar(msg, 0) != '"')
				sstr_token(msg, NULL, "\"", 0);
			sstr_token(msg, fileinfo.path, "\"", 0);
			if (sstr_getchar(fileinfo.path,
					 sstr_len(fileinfo.path) - 1) != '/')
				sstr_ncat2(fileinfo.path, "/", 1);
		} else { /* Absolute path given in filename*/
			sstr_empty(fileinfo.path);
		}
	} else {
		sstr_ncpy2(fileinfo.path, "/", 1);
		sstr_cat(fileinfo.path, strictpath);
	}

	if(config.usefqdn) sstr_cpy(fileinfo.host, info->server_name);
	else sstr_cpy2(fileinfo.host,
		       inet_ntoa(info->server_control.address.sin_addr));

	if (ntohs(info->server_control.address.sin_port) != 21)
		sstr_apprintf(fileinfo.host, ":%d",
			      ntohs(info->server_control.address.sin_port));

	sstr_cat(fileinfo.path, fileinfo.filename);
	if(!config.strictcache) strip2bs(fileinfo.path);
	expecting = NOTHING;
	info->state = DOWNLOAD;

	return 0;
}

/* ------------------------------------------------------------- **
** Deal with server replies if we need to. 
** ------------------------------------------------------------- */
int cache_parsed_reply(int code, sstr * msg)
{
	if (!config.cachemod)
		return (FALSE);
	if (noop > 0) {
		if (code > 0)
			noop--;
		debug("Got NOOP reply\n");
		return (TRUE);
	}
	return FALSE;
}

void cache_inc_data(sstr * buf)
{
	if (!config.cachemod)
		return;

	if (nooping && time(NULL) - last_noop > NOOP_INTERVAL) {
		write(info->server_control.fd, "NOOP\r\n", 6);
		noop++;
		time(&last_noop);
		debug("Sent NOOP\n");
	}
	inc_data(buf);
}

int cache_close_data(void)
{
	int code;
	sstr *msg;

	if (!config.cachemod)
		return (-1);

	while (noop > 0) {
		get_message(&code, &msg);
		noop--;
	}

	nooping = FALSE;
	ftp_cmds = main_cmdlist;

	return (retr_end());
}

/*Strip multiple "//" from string */
void strip2bs(sstr *p)
{
	int i;
	for(i=0;i<sstr_len(p)-1;i++){
		while(sstr_getchar(p, i)=='/' && sstr_getchar(p, i+1)=='/')
			sstr_split(p, NULL, i, 1);
	}
}





