package fix;

import java.io.DataOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.Socket;
import java.net.URL;
import java.net.UnknownHostException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;

import javax.net.ssl.KeyManager;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;

/**
 * A Client for HTTP/HTTPS Connections.
 * 
 * With this class you can send and receive HTTP-MIME-messages
 * 
 * $Date: 2010-02-04 12:01:01 +0100 (Thu, 04 Feb 2010) $
 * $Revision: 30 $
 * $Author: roth $
 * 
 * @author Dominik Greibl
 * 
 * Copyright (C) 2008 Dominik Greibl
 * Copyright (C) 2009 Sebastian Roth
 * 
 * All Rights Reserved
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

public class HttpClient {

	private Socket socket = null;
	private String NEWLINE = "\r\n";
	public DataOutputStream out = null;
	private ErrorWindow er = null;
	private static boolean error = false;
	private static SSLSocketFactory sslSocketFactory = null;
	private boolean debug = false;

	/**
	 * The standard constructor for this class.
	 * 
	 * @param e
	 *            The instance of ErrorWindow the errors are displayed with.
	 * @param d
	 *            The debug value, if true the debugging is enabled.
	 */
	public HttpClient(ErrorWindow e, boolean d) {
		debug = d;
		er = e;
	}

	/**
	 * Establishes a connection to the given host URL by using
	 * <code>httpConnect()</code> or <code>httpsConnect()</code>
	 * 
	 * @param host
	 *            URL to connect to
	 */
	public void connect(String host) {
		String[] parsed = parseURL(host);
		int p = Integer.parseInt(parsed[2]); // p is the port to connect to
		/** Evaluates the protocol * */
		if (parsed[0].startsWith("https")) {
			httpsConnect(parsed[1], p);
		} else if (parsed[0].startsWith("http")) {
			httpConnect(parsed[1], p);
		} else {
			er.setMess("Wrong protocol: Please use HTTP or HTTPS.", null);
		}
		/** Creates the OutputStream for the connection * */
		try {
			out = new DataOutputStream(socket.getOutputStream());
		} catch (IOException e) {
			er.setMess("No Connection to Server.", e);
		}
	}

	/**
	 * Establishes a HTTP-Connection using a Socket
	 * 
	 * @param host
	 *            The hostname to connect to
	 */
	public void httpConnect(String host, int port) {
		try {
			socket = new Socket(host, port);
		} catch (UnknownHostException e) {
			er.setMess("Host-Server is unknown.", e);
		} catch (IOException e) {
			er.setMess("Could not connect to HTTP-Server: " + e.getMessage(), e);
		}
	}

	/**
	 * Establishes a HTTPS-Connection using a Socket.
	 * 
	 * @param host
	 *            The host name to connect to
	 */
	public void httpsConnect(String host, int port) {
		try {
			socket = getSocketFactory().createSocket(host, port);
		} catch (IOException e) {
			er.setMess("Could not connect to HTTPS-Server.", e);
		}
	}

	/**
	 * Sends a HTTP-MIME-Message to the Server.
	 * 
	 * @param HTTP
	 *            the message to send
	 */
	public void send(String HTTP) {
		try {
			out.writeBytes(HTTP);
			System.out.print(HTTP);
		} catch (IOException e) {
			if (!error) {
				er.setMess("Error while sending header: " + e.getMessage(), e);
				error = true;
			}
		}
	}

	/**
	 * Sends a HTTP-MIME-Message to the Server.
	 * 
	 * @param HTTP
	 *            the message to send
	 */
	public void send(byte[] HTTP) {
		try {
			out.write(HTTP);
			if (debug)
				for (byte b : HTTP) {
					System.out.print((char) b);
				}
		} catch (IOException e) {
			if (!error) {
				er.setMess("Error while sending header: " + e.getMessage(), e);
				error = true;
			}
		}
	}

	/**
	 * Sends a newline (\r\n) to the Server using the send method
	 */
	public void nl() {
		send(NEWLINE);
	}

	/**
	 * Parses the given URL address and returns an array existing of: [0]:
	 * Protocol (http, https) [1]: Host-Address [2]: the Host-Port as a String
	 * [3]: the Requested File.
	 * 
	 * If the protocol is not http oder https a {@link IllegalArgumentException}
	 * will be thrown
	 * 
	 * @param address
	 *            the URL address to be parsed
	 * @return String[] as described above
	 * 
	 */
	public String[] parseURL(String address) {
		String[] response = new String[4];
		try {
			URL url;
			url = new URL(address);

			response[0] = url.getProtocol();
			response[1] = url.getHost();
			response[2] = "" + url.getPort();
			response[3] = url.getFile();

			if (!response[0].equals("http") && !response[0].equals("https")) {
				throw new IllegalArgumentException("Protocol must be http or https.");
			}
			if (Integer.parseInt(response[2]) == -1)
				response[2] = "" + getPort(response[0]);
		} catch (MalformedURLException e) {
			er.setMess("Malformed Server URL: " + address, e);
			// System.out.println(address);
		}
		return response;
	}

	/**
	 * Gets the port number from the given protocol.
	 * 
	 * @param protocol
	 * @return port 443 if the protocol is https 80 else
	 */
	public int getPort(String protocol) {
		if (protocol.equals("https"))
			return 443;
		else
			return 80;
	}

	/**
	 * Gets the Socket for use in other classes
	 * 
	 * @return socket
	 */
	public Socket getSocket() {
		return socket;
	}

	/**
	 * Returns a SSL Factory instance that accepts all server certificates. use:
	 * 
	 * <pre>
	 * SSLSocket sock = (SSLSocket) getSocketFactory.createSocket(host, port);
	 * </pre>
	 * 
	 * @return An SSL-specific socket factory.
	 */
	public final SSLSocketFactory getSocketFactory() {
		if (sslSocketFactory == null) {
			try {
				TrustManager[] tm = new TrustManager[] { new NaiveTrustManager() };
				SSLContext context = SSLContext.getInstance("SSL");
				context.init(new KeyManager[0], tm, new SecureRandom());

				sslSocketFactory = (SSLSocketFactory)context.getSocketFactory();

			} catch (KeyManagementException e) {
				er.setMess("No SSL algorithm support: ", e);
			} catch (NoSuchAlgorithmException e) {
				er.setMess("Exception when setting up the Naive key management.", e);
			}
		}
		return sslSocketFactory;
	}

}