/*
 * Copyright (C) 2016-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* External interrupt/event controller (EXTI) */

/*
 * For most of the comments, infos, ... see:
 * ST - RM0383 - Reference Manual - STM32F411xC/E advanced ARM-based
 * 32-bit MCUs
 */

#define DEBUG_CONTROL_FLOW	1

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

struct {
	uint8_t state_in[23];

	/* Interrupt mask register (EXTI_IMR) */
	uint8_t imr[23];

	/* Event mask register (EXTI_EMR) */
	uint8_t emr[23];

	/* Rising trigger selection register (EXTI_RTSR) */
	uint8_t rtsr[23];

	/* Falling trigger selection register (EXTI_FTSR) */
	uint8_t ftsr[23];

	/* Pending register (EXTI_PR) */
	uint8_t pr[23];

	uint32_t reg[0x1000 >> 2];
} NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
NAME_(inN_set)(struct cpssp *cpssp, int pin, unsigned int val);
/*forward*/ static void
NAME_(reset)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

static void
NAME_(irq_update)(struct cpssp *cpssp, int pin)
{
	int irq;

	irq = cpssp->NAME.pr[pin] & cpssp->NAME.imr[pin];

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: pin=%d, irq=%d\n", __FUNCTION__, pin, irq);
	}

	NAME_(irqN_set)(cpssp, pin, irq);
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	unsigned int n;

	assert(bs == 0b1111); /* FIXME */

	addr &= 0x3ff;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x000:
		/* Interrupt mask register (EXTI_IMR) */
		for (n = 0; n < 23; n++) {
			cpssp->NAME.imr[n] = (val >> n) & 1;
			NAME_(irq_update)(cpssp, n);
		}
		break;
	case 0x004:
		/* Event mask register (EXTI_EMR) */
		for (n = 0; n < 23; n++) {
			cpssp->NAME.emr[n] = (val >> n) & 1;
		}
		break;
	case 0x008:
		/* Rising trigger selection register (EXTI_RTSR) */
		for (n = 0; n < 23; n++) {
			cpssp->NAME.rtsr[n] = (val >> n) & 1;
		}
		break;
	case 0x00c:
		/* Falling trigger selection register (EXTI_FTSR) */
		for (n = 0; n < 23; n++) {
			cpssp->NAME.ftsr[n] = (val >> n) & 1;
		}
		break;
	case 0x014:
		/* Pending register (EXTI_PR) */
		for (n = 0; n < 23; n++) {
			cpssp->NAME.pr[n] &= ~((val >> n) & 1);
			NAME_(irq_update)(cpssp, n);
		}
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);

		cpssp->NAME.reg[addr >> 2] = val;
		break;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	unsigned int n;

	assert(bs == 0b1111); /* FIXME */

	addr &= 0x3ff;

	switch (addr) {
	case 0x000:
		/* Interrupt mask register (EXTI_IMR) */
		*valp = 0;

		for (n = 0; n < 23; n++) {
			*valp |= cpssp->NAME.imr[n] << n;
		}
		break;
	case 0x004:
		/* Event mask register (EXTI_EMR) */
		*valp = 0;

		for (n = 0; n < 23; n++) {
			*valp |= cpssp->NAME.emr[n] << n;
		}
		break;
	case 0x008:
		/* Rising trigger selection register (EXTI_RTSR) */
		*valp = 0;

		for (n = 0; n < 23; n++) {
			*valp |= cpssp->NAME.rtsr[n] << n;
		}
		break;
	case 0x00c:
		/* Falling trigger selection register (EXTI_FTSR) */
		*valp = 0;

		for (n = 0; n < 23; n++) {
			*valp |= cpssp->NAME.ftsr[n] << n;
		}
		break;
	case 0x014:
		/* Pending register (EXTI_PR) */
		*valp = 0;

		for (n = 0; n < 23; n++) {
			*valp |= cpssp->NAME.pr[n] << n;
		}
		break;
	default:
		*valp = cpssp->NAME.reg[addr >> 2];

		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, *valp=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(inN_set)(struct cpssp *cpssp, int pin, unsigned int val)
{
	int irq;

	assert(0 <= pin && pin < 23);
	assert(val != SIG_STD_LOGIC_U);
	assert(val != SIG_STD_LOGIC_X);

	if (val == SIG_STD_LOGIC_Z) {
		val = 0;
	} else {
		val = 1000 <= SIG_mV(val);
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: pin=%d, val=%d\n", __FUNCTION__, pin, val);
	}

	if (! cpssp->NAME.state_in[pin]
	 && val
	 && cpssp->NAME.rtsr[pin]) {
		/* Rising edge detected and rising edge selected. */
		irq = 1;
	} else if (cpssp->NAME.state_in[pin]
		&& ! val
		&& cpssp->NAME.ftsr[pin]) {
		/* Falling edge detected and falling edge selected. */
		irq = 1;
	} else {
		/* No edge or edge not selected. */
		irq = 0;
	}
	cpssp->NAME.pr[pin] |= irq;

	NAME_(irq_update)(cpssp, pin);

	cpssp->NAME.state_in[pin] = val;
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	unsigned int n;

	/* Interrupt mask register (EXTI_IMR) */
	for (n = 0; n < 23; n++) {
		cpssp->NAME.imr[n] = 0;
	}

	/* Event mask register (EXTI_EMR) */
	for (n = 0; n < 23; n++) {
		cpssp->NAME.emr[n] = 0;
	}

	/* Rising trigger selection register (EXTI_RTSR) */
	for (n = 0; n < 23; n++) {
		cpssp->NAME.rtsr[n] = 0;
	}

	/* Falling trigger selection register (EXTI_FTSR) */
	for (n = 0; n < 23; n++) {
		cpssp->NAME.ftsr[n] = 0;
	}

	/* Pending register (EXTI_PR) */
	for (n = 0; n < 23; n++) {
		cpssp->NAME.pr[n] = 0;
	}

	for (n = 0; n < 23; n++) {
		NAME_(irq_update)(cpssp, n);
	}
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	unsigned int pin;

	for (pin = 0; pin < 23; pin++) {
		cpssp->NAME.state_in[pin] = 0;
	}
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
