/*
 * $Id: chip_intel_8042.c,v 1.54 2009-06-03 11:34:03 vrsieh Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "glue-log.h"
#include "glue-main.h"

#include "chip_intel_8042.h"

struct cpssp {
	/*
	 * Config
	 */

	/*
	 * Signals
	 */
	unsigned int state_power;
	struct sig_boolean_or *port_int1;
	struct sig_boolean_or *port_int12;
	struct sig_ps2_main *port_kbd_ps2;
	struct sig_ps2_main *port_mouse_ps2;
	struct sig_boolean *port_a20gate;
	struct sig_boolean *port_kbc_reset_hash_;

	/*
	 * State
	 */
#define STATE
#define NAME		kbc
#define NAME_(x)	kbc_ ## x
#define SNAME		"kbc"
#include "arch_kbd.c"
#undef SNAME
#undef NAME_
#undef NAME
#undef STATE
};

/*
 * Output Functions
 */
static void
kbc_a20gate_set(struct cpssp *cpssp, unsigned int val)
{
	sig_boolean_set(cpssp->port_a20gate, cpssp, val);
}

static void
kbc_n_reset_set(struct cpssp *cpssp, unsigned int n_val)
{
	sig_boolean_set(cpssp->port_kbc_reset_hash_, cpssp, n_val);
}

static void
kbc_int1_set(struct cpssp *cpssp, unsigned int val)
{
	sig_boolean_or_set(cpssp->port_int1, cpssp, val);
}

static void
kbc_intC_set(struct cpssp *cpssp, unsigned int val)
{
	if (cpssp->port_int12) {
		sig_boolean_or_set(cpssp->port_int12, cpssp, val);
	}
}

static void
kbc_kbd_send(struct cpssp *cpssp, uint8_t byte)
{
	sig_ps2_send(cpssp->port_kbd_ps2, cpssp, byte);
}

static void
kbc_kbd_clkrunning(struct cpssp *cpssp, int val)
{
	sig_ps2_clkrunning(cpssp->port_kbd_ps2, cpssp, val);
}

static void
kbc_mouse_send(struct cpssp *cpssp, uint8_t byte)
{
	if (cpssp->port_mouse_ps2) {
		sig_ps2_send(cpssp->port_mouse_ps2, cpssp, byte);
	}
}

static void
kbc_mouse_clkrunning(struct cpssp *cpssp, int val)
{
	if (cpssp->port_mouse_ps2) {
		sig_ps2_clkrunning(cpssp->port_mouse_ps2, cpssp, val);
	}
}

/*
 * Internal Architecture
 */
#define BEHAVIOR
#define NAME		kbc
#define NAME_(x)	kbc_ ## x
#define SNAME		"kbc"
#include "arch_kbd.c"
#undef SNAME
#undef NAME_
#undef NAME
#undef BEHAVIOR

/*
 * Input Functions
 */
static void
chip_intel_8042_kbd_recv(void *_css, uint8_t byte)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	if (! cpssp->state_power) {
		faum_log(FAUM_LOG_INFO, "8042", "keyboard",
				"Discarding keyboard data "
				"(machine not powered on yet!)\n");
		return;
	}

	kbc_kbd_recv(cpssp, byte);
}

static void
chip_intel_8042_mouse_recv(void *_css, uint8_t byte)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	if (! cpssp->state_power) {
		faum_log(FAUM_LOG_INFO, "8042", "mouse",
				"Discarding mouse data "
				"(machine not powered on yet!)\n");
		return;
	}

	kbc_mouse_recv(cpssp, byte);
}

static int
chip_intel_8042_inb(void *_css, uint8_t *valp, unsigned long port)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	*valp = kbc_inb(cpssp, (port >> 2) & 1);
	return 0;
}

static int
chip_intel_8042_outb(void *_css, uint8_t val, unsigned long port)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	kbc_outb(cpssp, val, (port >> 2) & 1);
	return 0;
}

static void
chip_intel_8042_power_set(void *_css, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _css;
	
	cpssp->state_power = val;

	sig_boolean_set(cpssp->port_kbc_reset_hash_, cpssp, val);
}

static void
chip_intel_8042_n_reset_set(void *_css, unsigned int n_val)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	if (! n_val) {
		kbc_reset(cpssp);
	}
}

void *
chip_intel_8042_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_boolean *port_power,
	struct sig_boolean *port_reset_hash_,
	struct sig_cs *port_cs,
	struct sig_isa_bus_main *port_isa_bus,
	struct sig_boolean_or *port_int1,
	struct sig_boolean_or *port_int12,
	struct sig_ps2_main *port_kbd_ps2,
	struct sig_ps2_main *port_mouse_ps2,
	struct sig_boolean *port_a20gate,
	struct sig_boolean *port_kbc_reset_hash_
)
{
	static const struct sig_boolean_funcs power_funcs = {
		.set = chip_intel_8042_power_set,
	};
	static const struct sig_boolean_funcs reset_hash__funcs = {
		.set = chip_intel_8042_n_reset_set,
	};
	static const struct sig_cs_funcs cs_funcs = {
		.readb = chip_intel_8042_inb,
		.writeb = chip_intel_8042_outb,
	};
	static const struct sig_ps2_main_funcs kbd_ps2_funcs = {
		.recv = chip_intel_8042_kbd_recv,
	};
	static const struct sig_ps2_main_funcs mouse_ps2_funcs = {
		.recv = chip_intel_8042_mouse_recv,
	};
	struct cpssp *cpssp;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	/* Out */
	cpssp->port_int1 = port_int1;
	sig_boolean_or_connect_out(port_int1, cpssp, 0);

	cpssp->port_int12 = port_int12;
	if (port_int12) {
		sig_boolean_or_connect_out(port_int12, cpssp, 0);
	}

	cpssp->port_a20gate = port_a20gate;
	sig_boolean_connect_out(port_a20gate, cpssp, 0);

	cpssp->port_kbc_reset_hash_ = port_kbc_reset_hash_;
	sig_boolean_connect_out(port_kbc_reset_hash_, cpssp, 0);

	/* Call */
	sig_cs_connect(port_cs, cpssp, &cs_funcs);

	cpssp->port_kbd_ps2 = port_kbd_ps2;
	sig_ps2_main_connect(port_kbd_ps2, cpssp, &kbd_ps2_funcs);

	cpssp->port_mouse_ps2 = port_mouse_ps2;
	if (port_mouse_ps2) {
		sig_ps2_main_connect(port_mouse_ps2, cpssp, &mouse_ps2_funcs);
	}

	/* In */
	cpssp->state_power = 0;
	sig_boolean_connect_in(port_power, cpssp, &power_funcs);
	sig_boolean_connect_in(port_reset_hash_, cpssp, &reset_hash__funcs);

	return cpssp;
}

void
chip_intel_8042_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	free(cpssp);
}
