/*
 * $Id: test.c,v 1.19 2010-02-26 15:00:16 potyra Exp $
 *
 * Copyright (C) 2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <unistd.h>
#include <sys/io.h>
#include <stdio.h>
#include <assert.h>
#include <stdint.h>
#include <string.h>
#include <sys/time.h>
#include <signal.h>
#include <inttypes.h>

static unsigned int base_port = 0xe800;

/* just to avoid compiler optimisation */
double used_variable;

static volatile int timer_expired = 0;

static void 
sig_vtalarm(int signo)
{
	timer_expired = -1;
}

static void
delay_loop(unsigned long long factor)
{
	unsigned long long i;
	double x, y, z;

	x = 1.0;
	y = 3.14;

	for (i = 0; i < factor; i++) {
		z = x * y;
		x += 0.1;
		y /= 2.0;
	}

	used_variable = z;
}

static unsigned long long
calibrate_delay_loop(void)
{
	int ret;
	unsigned long long factor;
	struct itimerval timer = {
		.it_interval = {
			.tv_sec = 1,
			.tv_usec = 0
		},
		.it_value = {
			.tv_sec = 1,
			.tv_usec = 0
		}
	};

	fprintf(stderr, "calibrating timer...");
	signal(SIGVTALRM, &sig_vtalarm);

	/* enable timer */
	ret = setitimer(ITIMER_VIRTUAL, &timer, NULL);
	assert(0 <= ret);

	factor = 0;
	while (! timer_expired) {
		delay_loop(1);
		factor++;
	}
	
	/* disable timer again */
	timer.it_interval.tv_sec = 0;
	timer.it_interval.tv_usec = 0;
	ret = setitimer(ITIMER_VIRTUAL, &timer, NULL);
	assert(0 <= ret);
	
	fprintf(stderr, " done.\n");
	return factor;
}

static void
play_sample(int16_t sample, unsigned long long delay)
{
	int16_t w;
	int16_t v = 0;

	/*  h   oct 3    out, ton on */
	w = 6 | 3 << 4 | 1 << 6 | 1 << 7;
	/* modulate 2 bit volume */
	sample >>= 6;
	sample &= ~((1 << 7) |(1 << 9));

	sample >>= 14;
	switch (sample & 0x3) {
	case 0:
		v = 3;
		break;
		
	case 1:
		v = 2;
		break;

	case 2:
		v = 1;
		break;

	case 3:
		v = 0;
		break;

	default:
		assert(0);
	}

	w |= v << 8;

	outw(w, base_port);
	delay_loop((delay / 48000) * 2);
}

static void
play_wav(const char *name, unsigned long long delay)
{
	FILE *f;
	static int16_t buf[32000];
	int ret;
	int i;

	f = fopen(name, "r");
	assert(f != NULL);

	while (! feof(f)) {
		ret = fread(buf, 
			sizeof(int16_t), 
			sizeof(buf)/sizeof(int16_t),
			f);

		if (ret <= 0) {
			break;
		}

		for (i = 0; i < ret; i++) {
			play_sample(buf[i], delay);
		}
	}

	fclose(f);

	/* disable all */
	outw(0, base_port);
}

static void
play_note(
	char note, 
	unsigned short divisor, 
	unsigned short octave, 
	unsigned short volume
)
{
	int16_t w;
	const int usec_full_note = 1800000;
	const float inter_note_delay_percent = 0.05;

	switch (note) {
	case 'c': w = 0; break;
	case 'd': w = 1; break;
	case 'e': w = 2; break;
	case 'f': w = 3; break;
	case 'g': w = 4; break;
	case 'a': w = 5; break;
	case 'h': w = 6; break;
	default: return;
	}

	fprintf(stderr, "playing %c %d %d\n", note, octave, divisor);

	assert(octave <= 3);
	w |= octave << 4;

	assert(volume <= 3);
	w |= volume << 8;
	
	/* enable out signal */
	w |= 1 << 6;
	/* enable generator */
	w |= 1 << 7;

	outw(w, base_port);

	/* delay */
	usleep(usec_full_note * (1 - inter_note_delay_percent) / divisor);

	/* disable out */
	w &= ~(1 << 6);
	/* disable generator */
	w &= ~(1 << 5);
	/* disable volume */
	w &= ~((1 << 8) | (1 << 9));

	outw(w, base_port);
	/* delay */
	usleep(usec_full_note * inter_note_delay_percent);
}


/* Note: "Die Gedanken sind frei" is in the public domain, since the
 *       copyright expired a long time ago.
 */
static void
play_dgsf(void)
{
	play_note('g', 8, 1, 3);
	play_note('g', 8, 1, 2);

	play_note('c', 4, 2, 3);
	play_note('c', 4, 2, 3);
	play_note('e', 8, 2, 3);
	play_note('c', 8, 2, 3);

	play_note('g', 2, 1, 3);
	play_note('g', 4, 1, 3);

	play_note('f', 4, 1, 3);
	play_note('d', 4, 1, 3);
	play_note('g', 4, 1, 3);

	play_note('e', 4, 1, 3);
	play_note('c', 4, 1, 3);
	play_note('g', 8, 1, 3);
	play_note('g', 8, 1, 3);

	play_note('c', 4, 2, 3);
	play_note('c', 4, 2, 3);
	play_note('e', 8, 2, 3);
	play_note('c', 8, 2, 3);

	play_note('g', 2, 1, 3);
	play_note('g', 4, 1, 3);

	play_note('f', 4, 1, 3);
	play_note('d', 4, 1, 3);
	play_note('g', 4, 1, 3);

	play_note('e', 4, 1, 3);
	play_note('c', 4, 1, 3);
	play_note('c', 4, 2, 3);

	play_note('h', 4, 1, 3);
	play_note('d', 4, 2, 3);
	play_note('d', 4, 2, 3);

	play_note('c', 4, 2, 3);
	play_note('e', 4, 2, 3);
	play_note('e', 4, 2, 3);

	play_note('h', 4, 1, 3);
	play_note('d', 4, 2, 3);
	play_note('d', 4, 2, 3);

	play_note('c', 4, 2, 3);
	play_note('e', 4, 2, 3);
	play_note('c', 4, 2, 3);

	play_note('a', 4, 1, 3);
	play_note('a', 4, 1, 3);
	play_note('c', 8, 2, 3);
	play_note('a', 8, 1, 3);

	play_note('g', 2, 1, 3);
	play_note('g', 8, 1, 3);
	play_note('e', 8, 2, 3);

	play_note('e', 8, 2, 3);
	play_note('d', 8, 2, 3);
	play_note('c', 4, 2, 3);
	play_note('h', 4, 1, 3);

	play_note('c', 2, 2, 3);
}

static int
self_test(void)
{
	int i;
	int ret;

	/* test only low byte, and avoid bits 6 and 7 since these enable sound
	 * generation */
	for (i = 0; i < 63; i++) {
		outw(i, base_port);
		ret = inw(base_port);

		if (ret != i) {
			fprintf(stderr, "Register value error i=%d, ret=%d\n",
				i, ret);
			return 1;
		}
	}

	fprintf(stderr, "self-test completed successfully\n");
	return 0;
}

int
main(int argc, char **argv)
{
	int ret;
	int do_self_test = 0;
	const char *play_file = NULL;

	if (1 < argc) {
		sscanf(argv[1], "%x", &base_port);
		argc--;
		argv++;
	}
	if (1 < argc) {
		if (strcmp(argv[1], "--self-test") == 0) {
			do_self_test = 1;

			argc--;
			argv++;
		}
	}

	if (1 < argc) {
		if (strcmp(argv[1], "--play") == 0) {
			argc--;
			argv++;

			if (argc < 2) {
				fprintf(stderr, "which file?\n");
				return 1;
			}

			play_file = argv[1];

			argc--;
			argv++;
		}
	}

	fprintf(stderr, "Using %x as base port.\n", base_port);

	ret = iopl(3);
	if (ret < 0) {
		perror("Could not obtain IO-Permisisions");
		return 1;
	}

	if (do_self_test) {
		ret = self_test();
		return ret;
	}

	if (play_file) {
		unsigned long long del;
		del = calibrate_delay_loop();
		play_wav(play_file, del);
		return 0;
	}

	play_dgsf();
	fprintf(stderr, "Playing done.\n");
	return 0;
}
