/*
 * farsight-session.c - Source for core API sessions
 *
 * Farsight Voice+Video library
 * Copyright (c) 2005 INdT.
 *   @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>,
 * Copyright 2005 Philippe Khalaf <burger@speedy.org>
 * Copyright 2005,2006 Collabora Ltd.
 * Copyright 2005,2006 Nokia Corp.
 *   @author Rob Taylor <rob.taylor@collabora.co.uk>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * SECTION:farsight-session
 * @short_description: An object representing an audio/video session which
 * may contain a number of streams. 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "farsight-session.h"
#include "farsight-plugin.h"
#include "farsight-marshal.h"

/* Signals */
enum
{
  ERROR,
  LAST_SIGNAL
};

struct _FarsightSessionPrivate
{
  GList *streams;
  gboolean disposed;
};

#define FARSIGHT_SESSION_GET_PRIVATE(o)  \
   (G_TYPE_INSTANCE_GET_PRIVATE ((o), FARSIGHT_TYPE_SESSION, FarsightSessionPrivate))

static void farsight_session_class_init (FarsightSessionClass * klass);
static void farsight_session_init (FarsightSession * session);
static void farsight_session_dispose (GObject * object);
static void farsight_session_finalize (GObject * object);

static void stream_weak_notify (FarsightSession *session,
                                GObject *object);

static GObjectClass *parent_class = NULL;
static guint farsight_signals[LAST_SIGNAL] = { 0 };

GType
farsight_session_get_type (void)
{
  static GType type = 0;

  if (type == 0) {
    static const GTypeInfo info = {
      sizeof (FarsightSessionClass),
      NULL,
      NULL,
      (GClassInitFunc) farsight_session_class_init,
      NULL,
      NULL,
      sizeof (FarsightSession),
      0,
      (GInstanceInitFunc) farsight_session_init
    };

    type = g_type_register_static (G_TYPE_OBJECT,
        "FarsightSession", &info, 0);
  }

  return type;
}

static void
farsight_session_class_init (FarsightSessionClass * klass)
{
  GObjectClass *gobject_class;

  gobject_class = (GObjectClass *) klass;
  parent_class = g_type_class_peek_parent (klass);

  /**
   * FarsightSession::error:
   * @self: #FarsightSession that emmitted the signal
   * @type: #FarsightSessionError type of error
   * @message: Error message
   *
   * This signal is emitted in any error condition
   */
  farsight_signals[ERROR] = g_signal_new ("error",
      G_TYPE_FROM_CLASS (klass),
      G_SIGNAL_RUN_LAST,
      0,
      NULL,
      NULL,
      farsight_marshal_VOID__INT_STRING,
      G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_STRING);

  gobject_class->dispose = farsight_session_dispose;
  gobject_class->finalize = farsight_session_finalize;

  g_type_class_add_private (klass, sizeof (FarsightSessionPrivate));
}

static void
farsight_session_init (FarsightSession * session)
{
  /* member init */
  /*maybe we should put the plugin in priv? */
  session->plugin = NULL;
  session->priv = FARSIGHT_SESSION_GET_PRIVATE (session);
  session->priv->disposed = FALSE;
}

static void
farsight_session_dispose (GObject * object)
{
  FarsightSession *session = FARSIGHT_SESSION (object);

  if (session->priv->disposed) {
    /* If dispose did already run, return. */
    return;
  }

  /* Make sure dispose does not run twice. */
  session->priv->disposed = TRUE;

  parent_class->dispose (object);
}

static void
farsight_session_finalize (GObject * object)
{
  parent_class->finalize (object);
}

/**
 * farsight_session_list_supported_codecs:
 * @session: a #FarsightSession
 *
 * Lists all codecs this session is cabable of handling.
 *
 * Returns: #GList of #FarsightCodec
 */
G_CONST_RETURN GList *
farsight_session_list_supported_codecs (FarsightSession * session)
{
  FarsightSessionClass *klass = FARSIGHT_SESSION_GET_CLASS (session);

  if (klass->list_supported_codecs) {     
    return klass->list_supported_codecs (session);
  } else {
    g_warning ("list_supported_codecs not defined for %s", 
        G_OBJECT_TYPE_NAME (session));
  }
  return NULL;
}


/**
 * farsight_session_create_stream:
 * @session: a #FarsightSession
 * @media_type: a #FarsightMediaType for this stream
 * @dir: a #FarsightStreamDirection for this stream
 *
 * Creates a #FarsightStream on this session.
 *
 * Returns: a new #FarsightStream
 */
FarsightStream *
farsight_session_create_stream (FarsightSession *session, 
                                FarsightMediaType media_type, 
                                FarsightStreamDirection dir)
{
  FarsightSessionClass *klass = FARSIGHT_SESSION_GET_CLASS (session);
  FarsightStream *stream = NULL;

  if (klass->create_stream) {
    stream = klass->create_stream (session, media_type, dir);

    session->priv->streams = g_list_append (
            session->priv->streams,
            stream);
    
    g_object_weak_ref (G_OBJECT (stream), 
            (GWeakNotify) stream_weak_notify, session);
    g_object_ref (session);
  } else {
    g_warning ("create_stream not defined for %s", 
        G_OBJECT_TYPE_NAME (session));
  }
  return stream;
}

/**
 * farsight_session_factory_make:
 * @session_id: a string representing the session to load
 *
 * Called by the client to create a new farsight session object. It then 
 * initialises this object to the given session. The string given must be 
 * a valid session plugin that exists.
 *
 * Return value: the #FarsightSession that has been created, NULL if an error
 * occurs.
 **/
FarsightSession *
farsight_session_factory_make (const gchar *session_id)
{
  FarsightSession *session;
  FarsightPlugin *plugin;

  g_return_val_if_fail (session_id != NULL, NULL);

  session = FARSIGHT_SESSION(farsight_plugin_create(session_id, "session", 
                                                    &plugin));

  if (!session) {
    g_print("Session not supported\n");
    return NULL;
  }

  session->plugin = plugin;

  return session;
}

/**
 * farsight_session_destroy:
 * @session: a #FarsightSession
 *
 * Destroy this session and all resources allocated by it. 
 * This function should be used instead of g_object_unref in order
 * to destroy the session.
 **/
void
farsight_session_destroy (FarsightSession *session)
{
  GList *ls;
  while ((ls = session->priv->streams)) {
      g_object_unref ((FarsightStream *) (ls->data));
  }

  g_object_unref (session);
}

static void 
stream_weak_notify (FarsightSession *session,
                    GObject *object)
{
  session->priv->streams = g_list_remove (session->priv->streams, object);
  g_object_unref (session);
}
