/*
 * conn.cpp -- part of the ezbounce IRC proxy.  
 * 
 * (C) 1998-2001 Murat Deligonul
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */                                     


#include "autoconf.h" 

#include <sys/types.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef HAVE_SYS_IOCTL_H
    #include <sys/ioctl.h>
#endif
#ifdef HAVE_SYS_TIME_H
    #include <sys/time.h>
#endif
#ifdef HAVE_STDARG_H
    #include <stdarg.h>
#endif
#ifdef HAVE_POLL_H
#   include <poll.h>
#endif      
#ifdef HAVE_SYS_POLL_H
#   include <sys/poll.h>
#endif

#include <sys/socket.h>
#include <sys/resource.h>
#include <arpa/inet.h>
#include <netdb.h> 
#include <errno.h>

#include "ezbounce.h"
#include "conn.h"
#include "dcc.h"
#include "ruleset.h"
#include "general.h"
#include "lib_mdidentd.h"
#include "messages.h"


extern bool port_in_set(u_short, const char *);

static int mk_printable_random_pass(char *, int);

/* 
 * Commands from clients 
 */
enum {
    USER, NICK, PASS, CONN, ADMIN, STATUS, KILL, DIE, REHASH, DIENOW, CANCEL,
    HELP, EZBOUNCE, INTERFACE,VHOSTS, WRITE, DETACH, REATTACH, DISCONNECT, IDENT, AUTOPASS, LOG,
    DCCSEND, MOTD, QUIT
};

/* 
 * Commands from irc server - checked when reattaching or connecting
 */
enum {
    PING = 100, INVAL, WHOIS_NICK, WHOIS_CHANNELS, WHOIS_SERVER, MODE_USER,
    CREATED, MYINFO, PRIVMSG, NUMERIC_005
};

const struct conn::command_set conn::user_commands[] = {
    /* str       int-id    req_flag           bad_flag               desc */
    /* -------------------------------------------------------------------------------------*/
    { "USER",    USER,     0,                 USERED | CONNECTING,   NULL},
    { "NICK",    NICK,     0,                 CONNECTING,   NULL},
    { "PASS",    PASS,     0,                 PWED | CONNECTING,     NULL},
    { "CONN",    CONN,     REGISTERED,        BOUNCED | CONNECTING,  "Connects to a server: CONN <server>[:port] [pass]"},
    { "ADMIN" ,  ADMIN,    REGISTERED,        ADMINED | CONNECTING,  "Attempts to become administrator"},
    { "AUTOPASS",AUTOPASS, REGISTERED,        CONNECTING,            "Change auto-detach password"},
    { "CANCEL" , CANCEL,   REGISTERED | CONNECTING,0,    NULL},
    { "HELP",    HELP,     REGISTERED,        CONNECTING,            NULL},
    { "EZBOUNCE",EZBOUNCE, REGISTERED | BOUNCED, CONNECTING,         "Lets you issue commands directly to proxy while you are "
                                                                     "connected to an irc server: /quote EZBOUNCE <command> " 
                                                                     "[args]. Prevents them from being relayed to IRC server."
                                                                     " Not all commands will work through this method."},
    { "EZB",     EZBOUNCE, REGISTERED | BOUNCED, CONNECTING,         "An alias for the ezbounce command."},
    { "INTERFACE",INTERFACE,REGISTERED,        BOUNCED | CONNECTING, "Lets you bind your connection to a different "
                                                                     "interface (virtual host): INTERFACE <address>"},
    { "VHOSTS",  VHOSTS,   REGISTERED,         0,                    "Displays available virtual hosts to use"},
    { "VHOST",   INTERFACE,REGISTERED,         BOUNCED | CONNECTING, NULL },
    { "DETACH",  DETACH,   REGISTERED/* | BOUNCED */, DETACHED | CONNECTING, "Detach connection from proxy. You must use this as /quote ezb detach <password>"},
    { "REATTACH",REATTACH, REGISTERED,        /* BOUNCED | */CONNECTING, "Reattach a detached connection"},
    { "DISCONNECT", DISCONNECT, REGISTERED | BOUNCED,   CONNECTING,  "Force disconnection from IRC server"},
    { "IDENT",  IDENT,     REGISTERED,         CONNECTING,           "Set a fake ident reply (if possible)" },
    { "LOG",    LOG,       REGISTERED,         CONNECTING,           "Controls logging while detached: use '/quote [ezb] LOG HELP' for more info"},

    { "QUIT",   QUIT,      REGISTERED | BOUNCED, CONNECTING,         "(/quote ezb quit) send a 'quit' to the server" },
    
#if __DEBUG__
    { "DCCSEND", DCCSEND,  REGISTERED,         CONNECTING,           "Murat's DCCSend debugger thing" },
#endif
    { "MOTD",    MOTD,     REGISTERED,         CONNECTING,           "Message of the day"},

    { "STATUS", STATUS,    REGISTERED | ADMINED, CONNECTING,         "Displays info on who's connected and to where"},
    { "KILL",   KILL,      REGISTERED | ADMINED, CONNECTING,         "Boots a user off a server: KILL <id> <reason>"},
    { "DIE" ,   DIE,       REGISTERED | ADMINED, CONNECTING,         "Terminates the server gracefully"},
    { "DIENOW", DIENOW,    REGISTERED | ADMINED, CONNECTING,         "Immediately terminates the server"},
    { "REHASH", REHASH,    REGISTERED | ADMINED, CONNECTING,         "Reloads the configuration file"},
    { "WRITE",  WRITE,     REGISTERED | ADMINED, CONNECTING,         "Send message to user(s). WRITE <user_id | all> <message>"}
}; 

const struct conn::command_set conn::from_server[] = {
    {"PRIVMSG", PRIVMSG,        0,              REATTACHING, NULL},
    {"NICK",    NICK,           0,              DETACHED, NULL},
    {"421",     INVAL,          REATTACHING,    0, NULL},
    {"311",     WHOIS_NICK,     REATTACHING,    0, NULL},
    {"319",     WHOIS_CHANNELS, REATTACHING,    0, NULL},
    {"312",     WHOIS_SERVER,   REATTACHING,    0, NULL},
    {"221",     MODE_USER,      REATTACHING,    0, NULL }, /* Last received for reattach */
    {"003",     CREATED,        0,              GOTSERVINFO, NULL },
    {"004",     MYINFO,         0,              GOTSERVINFO, NULL },
    {"005",     NUMERIC_005,    0,              GOTSERVINFO2, NULL}
};


conn *conn::pFirst = 0;
unsigned conn::num_active = 0;
unsigned conn::current_id = 0;

linkedlist<dcc_list_t> dcc_list_t::list;

static int mk_printable_random_pass(char * buffer, int size)
{
    srand(time(NULL));
    for (int x = 0; x < size - 1; x++)
        /* Valid range is ASCII 33 to 126, inclusive */
        buffer[x] = (rand()%(126 - 33)) + 33;

    buffer[size - 1] = 0;
    return 1;
} 

inline bool conn::do_auto_detach(void)
{
    if (!user.autopass)
    {
        printlog("Cannot auto-detach client %s: no password\n", addr());
        return 0;
    }
    printlog("Auto-detaching %s (%s) -- password is: %s \n",
                      addr(), strerror(errno), user.autopass);
    detach(user.autopass, "automatically detached");
    return 1;                               
}

conn::conn(int fd2accept) 
{
    socklen_t size = sizeof(struct sockaddr_in);
    int fd_client = accept(fd2accept, (struct sockaddr *) &client_saddr, &size);
    if (fd_client < 0)
    {
        printlog("failure in accept(): %s\n", strerror(errno));
        return;
    }

    if (pFirst)
    {
        conn *c;
        for (c = pFirst;c->pNext;c = c->pNext)
            ;
        c->pNext = this;
    }
    else
        pFirst = this;
    pNext = NULL;

    id = current_id++;

    client = new psock(this,fd_client);
    server = NULL;
    client_buff = new dynbuff(config.min_clientQ, config.max_clientQ);
    server_buff = NULL;
    reattach = NULL;
    log = NULL;
    
    nonblocking(client->fd);

    user.irc_nick = my_strdup( "(unknown)");
    user.fulladdr = new char[40];
    sprintf(user.fulladdr, "%lu:(unknown)@%s", id, inet_ntoa(client_saddr.sin_addr));
    
        
    last_recved = connect_time = ircproxy_time();
    stat = 0;
    log_options = config.log_options;
    failed_passwords = failed_admins = 0;
    my_dccs = 0;
    loglist = 0;
    
    user.iface = config.iface_conn;                  
    
    struct sockaddr_in sin;
    size = sizeof(sin);
    getsockname(client->fd, (struct sockaddr *) &sin, &size);
    memcpy(&local_saddr, &client_saddr, sizeof (sockaddr_in));
    local_saddr.sin_port = sin.sin_port;

    ++num_active;

    printlog("CONNECTION FROM: %s (assigned id %d)\n", inet_ntoa(client_saddr.sin_addr), id);
}

conn::~conn()                   
{
    on_server_disconnect();
    on_client_disconnect(0);
    disown_dccs(0);
    delete reattach;
    num_active--;

    if (pFirst == this)
        pFirst = pNext;
    else
    {
        for (conn *c = pFirst; c; c = c->pNext)
        {
            if (c->pNext == this)
            {
                c->pNext = pNext;
                break;
            }
        }
    }      
}


int conn::check_server(int events)
{
    int ret;
    char dummy;
    
    if (!server && !client)
        return CONN_KILLME;        /* dead object */
    if (!server)
        return CONN_NOTCONNECTED;  /* client has not connected to any irc server */
    if (!events)
        return 1;
    
    /*
     * We use this trick to check for two things:
     * 1) If asnyc connect attempt worked, and to get reason for failure.
     * 2) If connection was closed (although this can be done in the big 
     *      big switch statement below)
     */
    ret = recv(server->fd, &dummy, 1, MSG_PEEK);
    
    /* FIXME: ugly kludge: trying to track an infinite poll() loop */
    if ((events & POLLERR) || (events & POLLHUP))
        ret = 0;

    if (checkf(CONNECTING) && (events & POLLOUT))
    {
        if (ret < 0 && errno != EAGAIN)
        {
            on_server_connect(errno);     /*  Will call on_server_disconnect() and stuff */
            return 1;                  
        }
        on_server_connect(0);
        return CONN_LINK_ACTIVATED;
    }

    /* Check if there is actually data waiting */
    if (ret < 0 && errno == EAGAIN)
        return CONN_NOWAITING;
    else if ((!ret) || ((ret < 0) && errno != EAGAIN))
    {
        /* The call to recv returned 0 or 
           there was another error trying to read */
        on_server_disconnect();
        if (checkf(DETACHED))
            return CONN_KILLME;
        send_client(msg_disconnected);
        return CONN_DISCONNECTED;
    }

    /* get from server  
       .. check results   */
    switch (server_buff->add(server->fd))
    {
    case DYNBUFF_ERR_READ:
    case DYNBUFF_ERR_WRITE:
        on_server_disconnect();
        if (checkf(DETACHED))
            return CONN_KILLME;
        send_client(msg_disconnected);
        return CONN_DISCONNECTED;

    case DYNBUFF_ERR_FULL:
        return CONN_FULL;

    case DYNBUFF_ERR_MEM:
        printlog("FATAL: memory allocation failure -- aborting server\n");
        abort();
        
    case 0:
        break;

    default:
        int q = parse_incoming();
        return (q == -1) ? CONN_KILLME : ((q == 0) ? 1 : q);
    }
    return 0;
}

/* 
 * Check client's socket. Parse and relay.
 */
int conn::check_client(int events)
{
    if (checkf(DETACHED) && (!client_buff))
        return 1;                             /* detached -- no client, nothing to do */
    if (!server && !client)
        return CONN_KILLME;
    if (!events)
        return 0;

    /*
     * We now check for disconnections and EAGAIN crap here, no 
     * recv() with msg_peek anymore
     */
    switch (client_buff->add(client->fd))
    {
    case 0: 
        if (recv_test(client->fd) == -1)
        {

        case DYNBUFF_ERR_READ:
        case DYNBUFF_ERR_WRITE:
            if ( (config.flags & ENABLE_AUTO_DETACH)
                 && (!checkf(REATTACHING) && checkf(BOUNCED)))
            {
                if (do_auto_detach())
                    return 1;
            }
        }
        

        /* If client died, terminate server connection as well */
        printlog("Disconnect: %s (while reading).\n", addr());
        on_server_disconnect();
        on_client_disconnect(); 
        return CONN_KILLME;

    case DYNBUFF_ERR_FULL:
        return CONN_FULL;

    case DYNBUFF_ERR_MEM:
        printlog("FATAL: failed to allocate some memory -- bailing out.\n");
        abort();
            
    default:
        last_recved = ircproxy_time();

        /* 
         * Check for commands now ..
         * parse() will relay it
         */
        int q = parse();
        return (q == -1) ? CONN_KILLME : ((q == 0) ? 1 : q);
    }
    return 0;
}


/* 
 * When stat & BOUNCED, we check for the ezbounce command, and do
 *  what it says. If there isn't any, we relay the text to the irc serv.
 *
 * return values for parse() and parse_incoming():
 *   1: OK
 *  -1: something occured requring destruction of the object
 */
int conn::parse(void)
{
    char line[512], command[12];

    while (1)
    {
        u_long len;
        if (!client_buff)
            break;
        /* special args: don't leave line, require crlf at end
           of line */
        switch ((len = client_buff->get_line(0, line, sizeof(line), 0, 1)))
        {
        case -1:
            return 1;
        case 0:
            continue;
        default:
            gettok(line, command,12,' ', 1);
            switch (do_command(command, no_leading(line + strlen(command))))
            {
            case -1:
                return -1;
            case 0:
                /* There was nothing to parse, relay to server if needed */
                if (checkf(BOUNCED))
                {
                    send(server->fd, line, len, 0);
                    send(server->fd, "\r\n", 2, 0);
                }
            default:
                continue;
            }
        }
    }  
    return 1;
}


/* 
 *  Checks and stuff does for various non-admin commands
 *   that can be issued to proxy.
 *
 * Only return 0 if you don't want the command to be acknowledged 
 *   as recongnized. (It will be sent directly to irc server)
 *  Return -1 if object should be destroyed asap.
 *   
 */
int conn::do_command(char const * command, char const * args)
{
    #define isadmin() (stat & ADMINED)
    /*
     * Speed hack: we don't want to bother checking all commands
     * from user while connected, as we will have to look thru 15+
     * commands. So we check if its the ezbounce command now,
     * if not, drop it. 
     * Speed Hack #2: Also need to catch CTCPs here, so we can
     *  do DCC proxying.
     */ 
    const struct command_set * c;
    char ezb_cmd[25] = "";
    if (checkf(BOUNCED))
    {
        if ((config.flags & ENABLE_AUTO_DETACH) && strcasecmp(command, "quit") == 0)
        {
            do_auto_detach();
            return 1;
        }
        
        if (strcasecmp(command, "ezbounce") == 0 || strcasecmp(command,"ezb") == 0)
        {
            /* Handle ezbounce command here */
            if (!gettok(args, ezb_cmd, sizeof(ezb_cmd), ' ', 1))
                return 0;
            command = ezb_cmd;
            args = no_leading(args + strlen(ezb_cmd));
        } 
        else if ((config.flags & ENABLE_OUTGOING_DCC_PROXYING) && strcasecmp(command, "PRIVMSG") == 0)
        {
            /* If we get here, the message is in the form
             * PRIVMSG target :[CTCP] xxxx
             */
            char address[256]; /* ctcp[15] -- we'll be cheap and store this in ezb_cmd */;
            if (!(gettok(args, ezb_cmd, sizeof(ezb_cmd), ' ', 2)) || !(ezb_cmd[1] == '\001'))
                return 0;
            
            if (!gettok(args, address, sizeof(address), ' ', 1))
                return 0;
            
            args = no_leading(args + strlen(ezb_cmd) + strlen(address) + 2);
            return do_ctcp(0, &ezb_cmd[2], user.irc_nick, address, args);
        }
        else
            return 0;
    }
    else if (stat & BOUNCED)
        return 0;               /* Not ezbounce command, bail out */

    c = match_command(1, command);

    if (!c)
        return 0;
    /* 
     * We have another function to do admin-specific commands.
     * We don't need to check if we are administrator, match_command
     * does that..
     */
    else if (c->req_flags & ADMINED)
        return do_admin_command(c, args);

    switch (c->id)
    {
    case EZBOUNCE:
        send_client("Double ezbounce command; not allowed.\r\n");
        return 1;

    case CANCEL:
        on_server_connect(0x29A);
        return 0;
        
    case DISCONNECT:
        on_server_disconnect();
        send_client("Ok, broke your connection to the irc server.\r\n");
        printlog("DISCONNECT: %s from IRC server by his request.\n",
                          addr());
        return 1;

    case DETACH:
    {
        /* We'll check for this here, to be more user-friendly */
        if (!checkf(BOUNCED))
        {
            send_client("DETACH: you must be connected to an irc server first.\r\n"); 
            return 1;
        }
        if (!(config.flags & ENABLE_DETACH_COMMAND) && !isadmin())
        {
            send_client("Sorry, the Detach command has been disabled here\r\n");
            return 1;
        }
        
        char password[24];
        if (!gettok(args, password, sizeof(password), ' ', 1))
        {
            send_client("DETACH: you must supply a password\r\n");
            return 1;
        }
        if (!detach(password, args + strlen(password)))
            send_client("Detach failed.\r\n");
        return 1;
    }
    
    case REATTACH:
    {
        /* We'll check for this here, to be more user-friendly */
        if (checkf(BOUNCED))
        {
            send_client("REATTACH: You must not be connected "
                        "to an irc server to use this command\r\n");
            return 1;
        }
            
        if (!(config.flags & ENABLE_DETACH_COMMAND) && !isadmin())
        {
            send_client("The detach and reattach commands have been disabled here\r\n");
            return 1;
        }
        
        char password[24], id[15];   
        if (!(gettok(args, password, sizeof(password), ' ', 2) || isadmin()) ||
            !gettok(args, id, sizeof(id), ' ', 1))
        {
            send_client("REATTACH: usage: reattach <id> <password>\r\n");
            return 1;
        }
        int __id = atoi(id);
        if (!reattach_start(password, __id))
           send_client("Reattach failed\r\n");
        return 1;
    }

    case USER:
        /* Check for non-blank non-whitespace valid arguments */
        if (!gettok(args, NULL, 0, ' ', 1))
            return 1;
        user.usercmd = my_strdup(args);
        setf(USERED);
        if (checkf(USERED) && checkf(NICKED))
        {
            /* init rulesets and stuff here and greet user */
            if (init_rulesets() != 1)
                return -1;  /* -1 return: terminate this object */
            if (!checkf(PWED))
                send_client(msg_welcome, user.irc_nick);  
            if (strcmp(config.password, "") == 0 || checkf(PWED))
                on_client_connect(0);
            
        }
        return 1;

        /* end of user */
    case NICK:
    {
        char nickbuff[NICKNAME_LENGTH + 1];
        if (!gettok(args, nickbuff, sizeof nickbuff, ' ', 1))
            return 1;

        /* Allow user to change nick on the proxy */
        if (checkf(USERED) && checkf(NICKED) && checkf(PWED))
        {
            fdprintf(client->fd, ":%s!unknown@%s NICK :%s\r\n", user.irc_nick, inet_ntoa(client_saddr.sin_addr), nickbuff);
            printlog("NICK CHANGE: %s --> %s\n", addr(), nickbuff);
            delete[] user.irc_nick;
            user.irc_nick = my_strdup(nickbuff);
            return 1;
        }
        
        /* First time */
        delete[] user.irc_nick;
        user.irc_nick = my_strdup(nickbuff);
        setf(NICKED);

        if (checkf(USERED) && checkf(NICKED))
        {
             /* Init rulesets and stuff here and greet user */
             if (init_rulesets() != 1)
                 return -1;
             if (!checkf(PWED))
                send_client(msg_welcome, user.irc_nick);
             if (strcmp(config.password, "") == 0 || checkf(PWED))
                 on_client_connect(0);             
         }
         return 1;
    }

    case PASS:
    {
        char pw[PASSWORD_LENGTH + 1];
        if (!gettok(args, pw, sizeof(pw), ' ', 1))
            return 1;
        if (strcmp(pw, config.password) == 0) 
        {   
            args += strlen(pw) + 1;
            if (*args)
                user.detachpass = my_strdup(args);    
            setf(PWED);
            if (checkf(USERED) && checkf(NICKED))
            {
                send_client(msg_password_accepted); 
                on_client_connect(0);
            }
        }
        else 
        {
            send_client(msg_password_incorrect);
            printlog("Incorrect password from %s\n", addr());
            if (++failed_passwords >= config.max_failed_passwords && config.max_failed_passwords)
            {
               send_client(msg_too_many_failures);
               printlog("... disconnected client for giving too many incorrect passwords!\n");
               on_client_disconnect(0);
               return -1;
            }
        }
        return 1;
        /* end of pass */
    }
       
    case HELP:
        send_client(msg_helpheader);
        for (unsigned z = 0; z < sizeof(user_commands) / sizeof(command_set); z++)
            if (user_commands[z].desc)
                if (((user_commands[z].req_flags & ADMINED) && (stat & ADMINED)) 
                 || !(user_commands[z].req_flags & ADMINED))
                    send_client(msg_helpitem, strtrunc(user_commands[z].cmd, 11), 
                                user_commands[z].desc);
        return 1;             

    case CONN:
        do_conn_command(args);
        return 1;
     
    case VHOSTS:
        if (!(config.flags & ENABLE_VHOST_COMMAND) && !isadmin())
        {
            send_client("This command has been disabled here.\r\n");
            return 1;
        }
        send_client("Available virtual hosts:\n");
        
        char buff[256];
        for (unsigned m = 0; m < config.vhost_table->num(); m++)
        {
           config.vhost_table->get(m, buff, sizeof(buff));
           send_client("%s\n", buff);
        }
        send_client("End of list.\n");
        return 1;        
    
    case INTERFACE:
    {
        if (!(config.flags & ENABLE_VHOST_COMMAND) && !isadmin())
        {
           send_client("This command has been disabled here.\r\n");
           return 1;
        }
        
        char interface[256];
        if (!gettok(args, interface, sizeof(interface), ' ', 1))
        {
            /*
             * No arguments given: set vhost to default
             * interface
             */
            user.iface = config.iface_conn;
            send_client(msg_interface_set, "default", inet_ntoa(config.iface_conn));
            return 1;
        }
        /*
         * Arguments given: check if the vhost exists in the table
         * and then set it if so
         * (ignore whats on the table for admin)
         */
        if (!checkf(ADMINED) && !config.vhost_table->have("all") 
           && !config.vhost_table->have(interface))
        {
            send_client("Host is on not on my vhost list.\n");
            return 1;
        }
        
        /* Resolve interface and fill it into client->iface */
        struct in_addr in;
        switch (fill_in_addr(interface, &in))
        {
        case 0:
            send_client(msg_interface_failed, strerror(errno));
            break;
        case 1:
            memcpy(&user.iface, &in, sizeof(in));
            send_client(msg_interface_set, interface, inet_ntoa(user.iface));
               break;
        case -1:
            send_client(msg_interface_failed, "unknown host\n");
        default:                  
            break;
        }
        return 1;
    }
    /*
     * Set a fake ident. Fake idents require MDIDENTD to be running
     */
    case IDENT:
    {
        if (!(config.flags & ENABLE_FAKE_IDENTS) && !isadmin())
        {
            send_client("Sorry, fake idents have been disabled here.\r\n");
            return 1;
        }
     
        char id[20];
        if (gettok(args, id, sizeof(id), ' ', 1))
        {
            copy_fake_ident(id);
            send_client("Ok, set your fake ident to %s\r\n", id);
        } else
            send_client(msg_not_enuff_args, "IDENT");
        return 1;
    }
       
    case ADMIN:
    {
        char tmp[NICKNAME_LENGTH], tmp2[PASSWORD_LENGTH];
        gettok(args, tmp, sizeof(tmp), ' ', 1 );
        gettok(args, tmp2, sizeof(tmp2), ' ', 2);
        if (config.admins->authorized(tmp,tmp2,inet_ntoa(client_saddr.sin_addr)))
        {
           send_client(msg_admin_accepted, user.irc_nick);
           printlog("Granted admin privilages to %s\n", addr());
           setf(ADMINED);
           /* Make user exempt from rulesets. This means unregister, and then wipe out
            * the list */
           unregister_rulesets(rule_set::TO);
           unregister_rulesets(rule_set::FROM);
           free(rules.objs);
           rules.objs = 0;
           rules.num = 0;
        }
        else 
        {
           send_client(msg_admin_incorrect);
           printlog("Failed admin attempt from %s\n", addr());
           
           if (++failed_admins >= config.max_failed_admins && config.max_failed_admins)
           {
                send_client(msg_too_many_failures);
                printlog("... disconnected client for reaching max failed admin attempts\n");
                on_client_disconnect();
                return -1;
            }
        }
        return 1;
    }

    case AUTOPASS:
    /* Password to use in case we auto-detach */ 
    { 
        if (!(config.flags & ENABLE_AUTO_DETACH))
            return 0;
        char tmp[PASSWORD_LENGTH + 1];
        if (gettok(args, tmp, sizeof(tmp), ' ', 1))
        {
            delete[] user.autopass;
            user.autopass = my_strdup(tmp);
        }
        send_client("Your magic number for this session: %d\r\n", id);
        send_client("Your auto-detach pass: %s\r\n", user.autopass);
        
        return 1;
    }
    
    
    case LOG:
    {
        if (!(config.flags & LOG_OPTIONS) && !checkf(ADMINED))
        {
            send_client("Detached-logging has been disabled here.\r\n");
            return 1;
        }

        char cmd[12] = "", cmd_args[50] = "";
        gettok(args, cmd, sizeof(cmd), ' ', 1);
        gettok(args, cmd_args, sizeof(cmd_args),' ', 2, 1);

        do_log_command(cmd, cmd_args);
    }
    return 1;

#if 0
    case DCCSEND:
    {
        char filename[120];

        if (!gettok(args, filename, sizeof (filename), ' ', 1))
            return 1;
        if (!dcc_send_file(filename))
            send_client("Couldn't send!\r\n");
    }
    return 1;
#endif

    case MOTD:
        if (config.motd_file)
            show_motd();
        else
            send_client("There is no MOTD.\r\n");
        return 1;

    case QUIT:
        /* When auto-detach enabled, use this to send quit
           to IRC server */
        fdprintf(server->fd, "QUIT :%s\r\n", args);
        return 1;

    default:
        printlog("No handler for valid command %s\n", c->cmd);
        break;
    }
    return 0;
    #undef isadmin
}

int conn::do_admin_command(const struct conn::command_set * pc, const char *args)
{
    if (!pc)
        return 0;
    int fd_client = client->fd;

    switch (pc->id)
    {
    /*
     * Send a message to a single user or everybody
     * using the proxy. Unfortunately there is no 
     * way for the users to respond back ;)
     */
    case WRITE:
    {
        char id[10];
        char second_arg[128];
        if (!gettok(args, second_arg, sizeof(second_arg), ' ', 2, 1)
            || !gettok(args, id, sizeof(id), ' ', 1))
        {
            send_client("Usage: WRITE <userid/ALL> <message>\n");
            return 1;
        }
        if (strcasecmp(id, "ALL") == 0)
        {
            char buff[72];
            sprintf(buff, "*** GLOBAL MESSAGE FROM %s ***", user.irc_nick);
            broadcast(buff);
            broadcast(second_arg);
        }
        else {
            conn * c = NULL; 
            if ((isdigit(id[0])) && (c = find_by_id(atoi(id))) && (c->client))
                c->send_client("Message from %s: %s\n", user.irc_nick, second_arg);
            else if (c && !c->client)
                send_client("WRITE: Can't deliver message: user is detached or dead.\r\n");
            else 
                send_client("Can't find user by that id\n");
        }
    }
    return 1;     

    case DIENOW: 
    case DIE:
        if (*args)
        {        
            send_client((pc->id == DIENOW ? msg_dieing_now : msg_dieing));
            ircproxy_die((pc->id == DIENOW), args);
        }
        else
            send_client(msg_not_enuff_args, pc->cmd);
        return 1;

    /* 
     * Dump information about the proxy
     */
    case STATUS:
    {
        char timebuff[100];
        duration(ircproxy_time() - start_time, 1, timebuff, sizeof(timebuff));
        send_client(msg_status_uptime, timestamp(), timebuff);
#ifdef HAVE_GETRUSAGE 
        {
            struct rusage ru;
            char buff[100];
            getrusage(RUSAGE_SELF, &ru);
            time_t mins = (int) ((ru.ru_utime.tv_sec + ru.ru_stime.tv_sec) / 60);
            time_t sex = (int) ((ru.ru_utime.tv_sec + ru.ru_stime.tv_sec) - (mins * 60));
            sprintf(buff, "%02d:%02d", (int) mins, (int) sex);  
            send_client(msg_status_cputime, buff);
        }
#endif
        send_client(msg_status_connections, num_active);
        send_client("Active DCC Connections: %d\n", dcc_list_t::list.num());
        fdprintf(fd_client, msg_status_listheader, user.irc_nick);   
        for (conn *c = pFirst; c; c = c->pNext)
        {
            char str_stat[9] = "?";
            int z = 0;
            if (c->stat & BOUNCED)    str_stat[z++] = 'b';
            if (c->stat & CONNECTING) str_stat[z++] = 'c';
            if ((c->stat & NICKED) && 
                (c->stat & USERED))  str_stat[z++] = 'r';
            if (c->stat & ADMINED)    str_stat[z++] = 'a';
            if (c->stat & PWED)       str_stat[z++] = 'p';
            if (c->stat & DETACHED)   str_stat[z++] = 'd';
            if (c->stat & REATTACHING) str_stat[z++] = 'e';
            if (!c->stat)              str_stat[z++] = 'w';        /* (waiting) */        
            if (c->dead())
            {
                strcpy(str_stat, "(zombie)");
                z = 8;
            }
                        
            str_stat[z] = 0;

            duration(ircproxy_time() - c->connect_time, 0, timebuff, sizeof(timebuff));          
            /* ":" "info!ezb" " NOTICE %s :ID    TIME  NICK       FROM         TO             STAT           VHOST\n"; */
            fdprintf(fd_client,":%s NOTICE %s :%-6d", ezbounce_guy, user.irc_nick,
                     c->id);
            fdprintf(fd_client,"%-6s",timebuff,6);
            fdprintf(fd_client,"%-10s ", (c->user.irc_nick ? strtrunc(c->user.irc_nick, 10) : "(none)"));
            fdprintf(fd_client,"%-15s ",inet_ntoa(c->client_saddr.sin_addr));
            fdprintf(fd_client,"%-15s ",(c->server) ? 
                     inet_ntoa(c->serv_saddr.sin_addr) :
                     "(n/a)");
            fdprintf(fd_client,"%-9s",   str_stat, 9);
            fdprintf(fd_client,"%-15s\r\n", inet_ntoa(c->user.iface));
            
        } 
        return 1;
    }

    case KILL:
    {
        char uid[10];
        char reason[256];
        conn *victim;
        unsigned i;

        if (!gettok(args, uid, sizeof(uid), ' ', 1) || 
            !gettok(args, reason, sizeof(reason), ' ', 2))
            return send_client(msg_not_enuff_args,"KILL"), 1;
        i = (unsigned int) atoi(uid);
        
        if (i == id) 
        {
            printlog("%s tried to kill himself\n", addr());
            send_client("Suicides not allowed.\r\n");
            return 1;
        }
        if ((victim = conn::find_by_id(i))) 
        {
            const char * reason2 = args + strlen(uid) + 1;
            victim->kill(user.irc_nick, reason2);
            send_client(msg_killed, i, victim->user.irc_nick);
            printlog("KILLED: %s killed by %s: %s\n", 
                              victim->addr(), addr(), reason2);
        }
        else
            send_client(msg_cant_kill, i);
        return 1;
    }

    case REHASH:
        printlog("Attempting to reload configuration file..\r\n");

        switch (reload_config_file(&config, config.configfile))
        {
        case 1:
            printlog("Reload successful\n");
            send_client("REHASH successful\r\n");
            set_dns_timeout(config.max_dns_wait_time);
            break;
        case 0:
            printlog("Reload failed -- errors loading file: %s\n", strerror(errno));
            send_client("REHASH failed: %s\r\n", strerror(errno));
            break;
        default:
            printlog("Reload failed.\n");
            send_client("REHASH failed -- check log file for details\r\n");
            break;
        }

    default:
        return 1;
    }
    return 0;           
}    

/*
 *  Handles the connection process to a server.
 *  Checks if rule sets permit it, and handles registering
 *    w/ the rule sets.
 *  
 */
int conn::do_connect(const char *where, u_short port, const char *pass)
{
    struct sockaddr_in sin;
    int s;
    if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0)
        return 0;
    memset(&sin, 0, sizeof(sin));

    sin.sin_addr = user.iface;
    sin.sin_family = AF_INET;
    if ((bind(s, (const struct sockaddr *) &sin, sizeof(sin)) < 0)
        || (!resolve_address(where, &sin)))
    {
        close(s);
        return -1;
    }

    sin.sin_family = AF_INET;
    sin.sin_port   = htons(port);
    nonblocking(s);
    
    /*
     * Right about to connect.. we have local port and remote? port
     * so we register the fake ident if needed
     */
    if (((config.flags & ENABLE_FAKE_IDENTS) || (checkf(ADMINED))) && user.fake_ident)
    {
 	    int x = sizeof(struct sockaddr_in);
	    struct sockaddr_in sin2;

	    getsockname(s, (struct sockaddr *) &sin2, (socklen_t *) &x);
        x = register_fake_ident(MDIDENTD_PATH, user.fake_ident, ntohs(sin2.sin_port), port);
	    if (x != IDENT_OK)
        {
            send_client("Error registering fake ident '%s' for you (but continuing anyway): %s\r\n", 
                        user.fake_ident, ident_error(x));
            printlog("Fake ident FAILED: '%s' for %s: %s\n",
                              user.fake_ident, addr(), ident_error(x));
        } else 
            printlog("Fake ident registered: '%s' for %s\n",
                              user.fake_ident, addr());
    }

    switch (connect(s, (struct sockaddr *) &sin, sizeof(sin)))
    {
    case -1:
        /* Non blocking connect in progress or something f*cked up */
        if (errno != EINPROGRESS)
            break;

        server = new psock(this, s);
        setf(CONNECTING);
        user.port = port;
        if (strlen(pass))
            user.ircpass = my_strdup(pass);
        
        goto success;

    case 0:
        /* Connection worked immediately, this *might* happen, and we'll
         * handle it to be safe */
        server = new psock(this, s);
        user.port = port;
        if (strlen(pass))
            user.ircpass = my_strdup(pass);
        
        on_server_connect(0);

        goto success;
        
    }
    close(s);
    return 0;

success:
    server->set_events(POLLIN | POLLOUT);
    for (unsigned z = 0; z < rules.num; z++)
       rules.objs[z]->register_connection(rule_set::TO,inet_ntoa(client_saddr.sin_addr), where, port);
    setf(TO_RULESETS_REGISTERED);
    return 1;
}    

/*
 *   Calls rule_set::find_matching_sets(..)
 *  Then it checks to see if this client is permitted to connect 
 *  on this port.
 *  
 *  Return values: 
 *     err_killme:    i'm not allowed here, better destroy me
 *     1         :    i'm allowed
 *     
 *  FIXME: there are two checks for unauthorized connections,
 *     and both appear to be needed....
 */  
int conn::init_rulesets()
{
    struct sockaddr_in sin;
    socklen_t size = sizeof(sin);
    char * address = my_strdup(inet_ntoa(client_saddr.sin_addr));

    getsockname(client->fd, (struct sockaddr *) &sin, &size);
    if (!rule_set::find_matching_sets(address, ntohs(sin.sin_port), &rules))
    {
        send_client(msg_not_authorized, address);
        printlog("Connection DENIED: from %s on port %d (unauthorized, killing)\n", address, ntohs(sin.sin_port));
        on_client_disconnect(0);
        delete[] address;
        return CONN_KILLME;
    }
    
    /* 
     *  Go through rules and make sure we're 
     *  not banned or anything
     */
    int permitted_flag = 0, q;
    char reason[230];
    for (unsigned z = 0; z < rules.num; z++)
    {
        if ((q = rules.objs[z]->is_allowed(address, ntohs(sin.sin_port), reason, sizeof(reason))) == -1)
        {
            send_client(msg_banned, reason, reason);
            printlog("Connection DENIED: from %s on port %d\n", address, ntohs(sin.sin_port));
            on_client_disconnect(0);   
            delete[] address;
            return CONN_KILLME;
        }
        else if (q == 1)
            permitted_flag = 1;
    }
    
    if (permitted_flag)
    {
        for (unsigned z = 0; z < rules.num; ++z)
            rules.objs[z]->register_connection(rule_set::FROM, address, NULL, ntohs(sin.sin_port));
        setf(FROM_RULESETS_REGISTERED);
    }
    else {
        printlog("Connection UNAUTHORIZED: from %s on port %d\n", address, 
                          ntohs(sin.sin_port));
        send_client(msg_banned, "No Authorization", "No Authorization");
        on_client_disconnect(0);
        permitted_flag = CONN_KILLME;
    }
    delete[] address;
    return permitted_flag;
}

/* static */ conn * conn::find_by_id(unsigned i)
{
    for (conn *c = pFirst; c; c = c->pNext)
        if (c->id == i)
            return c;
    return NULL;
}

void conn::kill(const char *killer, const char *reason)
{
    if (client)
    {
        send_client("You were killed by %s: %s\r\n", killer, reason);
        fdprintf(client->fd, "ERROR :Closing Link: %s (Killed)\r\n", addr());
    }
    on_server_disconnect(); 
    on_client_disconnect();
}

/*
 * Go thru rules and make sure he can connect there.
 * Display nice little warning message if can't.
 */
bool conn::can_connect(const char *towhere, u_short port)
{
    char reasonbuff[200];
    bool permitted = 0; 
    
    if (checkf(ADMINED))
        return 1;

    /* First, check for self connection attempts if needed */
    if (config.flags & PREVENT_SELF_CONNECTS)
    {
        struct sockaddr_in sin;
        bool resolved = resolve_address(towhere, &sin);
        /*
         * Deny access if:
         * port matches one of the ones we're listening on
         * AND:
         *    client is trying to connect to 127.*
         *    OR
         *    client is trying to connect to interface we are listening on
         */
        if ((port_in_set(port, config.ports))    
            && ((resolved && sin.sin_addr.s_addr == config.iface.s_addr) ||
                (resolved && wild_match("127.*", inet_ntoa(sin.sin_addr)))
               )
           )
        {
            send_client(msg_cannot_connect, towhere, "Self-connects are not permitted");
            return 0;
        }
    }

    for (unsigned z =0; z < rules.num; z++)
    {
        switch (rules.objs[z]->is_allowed_to(inet_ntoa(client_saddr.sin_addr), towhere, port, 
                                             reasonbuff, sizeof(reasonbuff)))
        {
        case -1:
            send_client(msg_cannot_connect, towhere, reasonbuff);
            return 0;
        case 1:                  // make sure rule sets let us
            permitted = 1;       // connect to this address.
        default:
            break;
        }
    }
    if (!permitted)
        send_client(msg_cannot_connect, towhere, "Host and/or port is not on my allowed-to list");
    return permitted;
}

/* static */ void conn::broadcast(const char *msg)
{
    for (conn *c = pFirst; c; c = c->pNext)
        if (c->client)
            c->send_client("%s\r\n", msg);
}

/* 
 * Called socket gets connected, err != 0 in case it failed.
 *  use 0x29A for err for user abort.
 */
int conn::on_server_connect(int err)
{
    socklen_t size = sizeof(serv_saddr);
    if (!err)
    {
        /* If connection ok, update internal variables */
        getpeername(server->fd, (struct sockaddr *) &serv_saddr, &size);
        
        /* Create the server buffer */
        server_buff = new dynbuff(config.min_serverQ, config.max_serverQ);
        if (!server_buff)
        {
            printlog("Connect attempt FAILED: %s to %s:%d: %s\n", addr(), user.server, user.port,
                          "Memory allocation error");
            send_client(msg_conn_failed, user.server, "Memory allocation error");
            goto out;
        }
        
        clearf(CONNECTING);
        setf(BOUNCED);
        server->set_events(POLLIN);
        
        client_buff->flush();                      
        send_client(msg_conn_successful, user.server);
        /* Now register with the server */
        if (user.ircpass)
            fdprintf(server->fd, "PASS :%s\n", user.ircpass);
        fdprintf(server->fd, "USER %s\nNICK :%s\n", user.usercmd, user.irc_nick);
        
        printlog("Connect SUCCESSFUL: %s to %s:%d\n", 
                          addr(), user.server, user.port);
        
        delete[] user.ircpass;
        user.ircpass = NULL;
        return 1;
    }
    
out:
    const char * error = (err == 0x29A) ? "cancel request from user" : strerror(errno);
    send_client(msg_conn_failed, user.server, error);
    printlog("Connect attempt FAILED: %s to %s:%d: %s\n", addr(), user.server, user.port,
                          error);
    on_server_disconnect();
    return 1;
}

/*
 *  0 return if idle time limit exceeded.
 *  Idle time limits do not apply to admins.
 *  Idle time limits do not apply when when connected to an irc server
 */
bool conn::check_idle(time_t now)
{
    if (dead() || checkf(ADMINED) || checkf(BOUNCED) || checkf(DETACHED))
        return 1;
    /* User not registered */
    if (!checkf(USERED) || !checkf(NICKED))
    {
        if (now - connect_time > config.max_register_time 
            && config.max_register_time != 0)
        {
            kill("The Server", "You failed to register in time!");
            return 0;
        }
    }
    else if (checkf(USERED) && checkf(NICKED))
    {
        if (now - last_recved > config.max_idle 
            && config.max_idle != 0)
        {
            kill("The Server", "Idle time limit exceeded.");
            return 0;
        }
    }
    return 1;
}

/* problems: **third** function to look like fdprintf */
int conn::send_client(const char *message, ...)
{
    static char buffer[2048];
    static const char *hdr = ":" ezbounce_guy " NOTICE "; /*"[nickname] :"; */
    int len;
    buffer[0] = 0;
    va_list ap;
    va_start(ap, message); 
    len = vsprintf(buffer, message, ap);
    va_end(ap);

    write(client->fd, hdr, strlen(hdr));
    write(client->fd, user.irc_nick, strlen(user.irc_nick));
    write(client->fd, " :", 2);
    return (write(client->fd, buffer, len));
}

/* Allow sending of multi-lined messages. \a is the seperator. Be sure to include
 * \r\n. Parses line by line and sends it out. NOTE! Don't use % !! */
int conn::send_client_multiline(const char * message)
{
    char line[128];
    int cur = 1;
    while (gettok(message, line, sizeof(line), '\a', cur++))
        send_client(line);
    return 1;
}

/*
 * Detach client from proxy but keep his connection to server alive 
 * 
 * client must:
 *    be connected to a server
 *    not already be detached
 *   (we assume caller has checked so)
 */

int conn::detach(const char * password, const char * awaymsg)
{
    /* get rid of these */
    delete[] user.ircpass;
    delete[] user.usercmd;
    delete client_buff;
    user.ircpass = user.usercmd = NULL;
    client_buff = NULL;

    user.detachpass = my_strdup(password);

    /* 
     * Setup logging now. Must have proper options or be admin. Must have log_options
     * Inform user of errors and wonderful options
     */
    if (checkf(ADMINED) || (config.flags & ENABLE_PRIVATE_LOGGING) || (config.flags & ENABLE_PUBLIC_LOGGING))
    {
        if (log_options)
        {
            int result;
            log = new logfile(config.logdir, user.irc_nick, id, log_options, 
                              (checkf(ADMINED) ? 0 : config.max_logsize), password, &result);
            if (result < 0)
            {
                send_client("Tried to setup logging but failed. Error is: %s\r\n", strerror(errno));
                printlog("Failed log setup for %s: %s\n", addr(), strerror(errno));
                delete log;
                log = 0;
            }
            else if (result == 0)
            {
                send_client("Some errors setting up logs, but trying to continue. Error was: %s\r\n", strerror(errno));
                printlog("Errors during log setup for %s: %s (but continuing)\n", addr(), strerror(errno));
            } 
            else
            {
                char buff[15];
                logfile::intflags_to_char(log_options, buff);
                send_client("Session will be logged.\r\n");
                send_client("Logging options are: %s.\r\n", buff);
                send_client("Use '/quote ezb log send all' to retrieve log files after reattach...\r\n");
                printlog("Log setup OK for %s\n", addr());
            }
        }
    }

    send_client("Ok, detached... To reattach, you must use /quote REATTACH %d %s\r\n", id, password);
    send_client("Closing connection to you...\r\n");
    fdprintf(client->fd, "ERROR :Closing Link: %s (Closing connection)\r\n", addr());
    printlog("Detach SUCCESFUL: %s from connection to %s:%d\n",
                      addr(), user.server, user.port);

    disown_dccs(0);
    /* Pretend user has disconnected. Protect the FROM rulesets,
     * because they are needed on reattach  
     * We have to reset some flags as on_client_disconnect() zeroes out stat */
    clearf(FROM_RULESETS_REGISTERED);
    on_client_disconnect(0);
    setf(FROM_RULESETS_REGISTERED);
    setf(DETACHED | USERED | NICKED | PWED | BOUNCED);
    if (awaymsg && *awaymsg)
        fdprintf(server->fd, "AWAY :%s\r\n", awaymsg);
    return 1;
}

/*
 * The reattaching process:
 *
 * 1) Client requests reattach
 * 2) We send a garbage, invalid command to the server. We used to 
 *     send TIME but ircd-hybrid will not always reply to it!!
 * 3) Error reply from the bad command has nickname of user and name of server, 
 *     we store those
 * 4) We send WHOIS <user>
 * 5) We get full address of user and channels he's on
 * 6) We simulate connection to server by sending numerics 000-003
 * 7) We then send a dummy motd
 * 8) And finally a bunch of JOINS
 */
int conn::reattach_start(const char * password, int id_to_attach)
{
    conn * c2 = find_by_id(id_to_attach);
    if (!c2 || (c2 && !c2->checkf(DETACHED) || !c2->server || c2->reattach))
    {
        send_client("REATTACH: Nonexistant or bad target userid.\n");
        return 0;
    }
    if (strcasecmp(password, c2->user.detachpass))
    {
        send_client("REATTACH: password incorrect.\n");
        return 0;
    }

    /* Prepare both sides for reattach-info-gathering process */
    reattach = new rinfo;
    c2->reattach = new rinfo;
    reattach->target = c2;
    c2->reattach->target = this;
    setf(REATTACHING);
    c2->setf(REATTACHING);

    send_client("REATTACH: Please wait, gathering info on irc session...\n");
    printlog("Reattach attempt: %s -----> %s on %s:%d\n",
                      addr(), c2->addr(), c2->user.server, c2->user.port);
    if (checkf(ADMINED))
        printlog("Reattach: Used executive privileges for this admin.\n");
    /* send the bad command */
    fdprintf(c2->server->fd, "EGAVAELC\r\n");
    return 1;
}

/*
 * Responds to server pings, handles numeric stuff 
 * does incoming dcc proxying.
 */
int conn::parse_incoming(void)
{
    char line[512], command[512];
    long len;
    const struct command_set *pc = NULL;
   
    while (1)
    {
        /* special arguments: don't leave line, require crlf at end
           of line */
        switch ((len = server_buff->get_line(0, line, sizeof(line), 0, 1)))
        {
        case -1:    /* Nothing left in buffer */
            return 1;   
        case 0:     /* Line is empty.. try the next one ? */
            continue;
        default:
        {
            /* Log it (FIXME: need to checkf(DETACHED)?? */
            if (log)
                log->write(line);

            gettok(line, command,sizeof(command),' ', 2);
    
            pc = match_command(2, command);
            /*
             *
             * Ok... the consolidated thing is setup.. It appears to be working fine..
             * However it has caused a rather large hit on efficiency. Should none
             * of the commands match, it will unsuccesfully loop through all the items.
             */
            if (pc)
            {
                switch (pc->id)
                {       

                case NICK:
                    /* One day it occurred to me that we ought to 
                     * keep track of nickname changes internally */
                    char temp[NICKNAME_LENGTH + 5];
                    char from[NICKNAME_LENGTH + 5];
                    gettok(line, from, sizeof(from), ' ', 1);
                    gettok(no_leading(from), temp, sizeof(temp), '!' , 1);
                    
                    if (strcasecmp(temp, user.irc_nick) == 0 && gettok(line, from, sizeof(from), ' ', 3))
                    {
                        printlog("NICK CHANGE: %s ----> %s\n", addr(), no_leading(from));
                        delete[] user.irc_nick;
                        delete[] user.fulladdr;
                        user.irc_nick = my_strdup(no_leading(from));
                        user.fulladdr = new char[10 + strlen(user.irc_nick) + 18];
                       
                        sprintf(user.fulladdr, "%lu:%s@%s", id, user.irc_nick, inet_ntoa(client_saddr.sin_addr));
                    }
                    break;


                case CREATED:
                    /* Get the real nickname of the user as we connect */
                    char nick[NICKNAME_LENGTH];
                    gettok(line, nick, sizeof(nick), ' ', 3);
                    if (strcasecmp(nick, user.irc_nick))
                    {
                        printlog("NICK CHANGE: %s ----> %s\n", addr(), no_leading(nick));
                        delete[] user.irc_nick;
                        delete[] user.fulladdr;
                        user.irc_nick = my_strdup(no_leading(nick));
                        user.fulladdr = new char[10 + strlen(user.irc_nick) + 18];
                        
                        sprintf(user.fulladdr, "%lu:%s@%s", id, user.irc_nick, inet_ntoa(client_saddr.sin_addr));
                    }

                    gettok(line, command, sizeof(command), ' ', 7, 1);
                    if (user.servercreated)
                        delete[] user.servercreated;
                    user.servercreated = my_strdup(command);
                    break;


                case MYINFO:
                    gettok(line, command, sizeof(command), ' ', 5);
                    if (user.serverversion)
                        delete[] user.serverversion;
                    user.serverversion = my_strdup(command);
                    gettok(line, command, sizeof(command), ' ', 6, 1);
                    if (user.servermodes)
                        delete[] user.servermodes;
                    user.servermodes = my_strdup(command);
                    setf(GOTSERVINFO);
                    break;

                case NUMERIC_005:
                    gettok(line, command, sizeof(command), ' ', 4, 1);
                    if (user.server005)
                        delete[] user.server005;
                    user.server005 = my_strdup(command);
                    setf(GOTSERVINFO2);


                case PRIVMSG:
                {
                    /* Ensure that we really need to handle this.. */
                    if ((checkf(DETACHED) || (config.flags & ENABLE_INCOMING_DCC_PROXYING))
                         && gettok(line, command, sizeof(command), ' ', 4) &&
                         command[1] == '\001')
                     {
                        char target[NICKNAME_LENGTH + 3], source[NICKNAME_LENGTH + 3];
                        gettok(line, target, sizeof(target), ' ', 3);
                        gettok(line, source, sizeof(source), ' ', 1);
                        if ((config.flags & ENABLE_INCOMING_DCC_PROXYING)
                            && !checkf(DETACHED) && !checkf(REATTACHING))
                        {
                            /* Possibly a DCC -- have do_ctcp handle it (will relay too) */
                            char args[60];
                            gettok(line, args, sizeof(args), ' ', 5, true);
                            if (!do_ctcp(true, &command[2], no_leading(source), no_leading(target), no_leading(args)))
                                break;
                            continue;
                        }

                        if (checkf(DETACHED))
                        {
                            /* Check for and reply to CTCP PINGs  -- only once per 5 secs */
                            if (ircproxy_time() - last_recved <= 5
                                || strcmp(no_leading(command), "\001PING") != 0)
                                continue;   /* Detached -- cannot relay */
                            last_recved = ircproxy_time(); 
    
                            /* Argument after the PING */
                            if (!gettok(line, temp, sizeof(temp), ' ', 5, 1))
                                continue;
                            char * t = strchr(source, '!');
                            if (t)
                                *t = 0;

                            /* And reply */
                            fdprintf(server->fd, "NOTICE %s :\001PING %s\r\n",
                                no_leading(source), temp);
                            continue; 
                        }
                    }
                    break;
                }

                /* Reattach info gatherers */
                case INVAL:
                    /*
                    * inval command reply, get nickname
                    * once we have it WHOIS ourselves
                    */
                    gettok(line, command, sizeof(command), ' ', 3);
                    reattach->nick = my_strdup(command);
                    fdprintf(server->fd, "WHOIS %s\r\n", command);
                    continue;

                case WHOIS_NICK:
                    /* 
                    * We could get nick, but we get ident and hostname from here.. 
                    */
                    gettok(line, command, sizeof(command), ' ', 6);
                    reattach->hostname = my_strdup(command);
                    gettok(line, command, sizeof(command), ' ', 5);
                    reattach->ident = my_strdup(command);
                    continue;

                case WHOIS_CHANNELS:
                    gettok(line, command, sizeof(command), ' ', 5, 1);
                    reattach->channels = my_strdup(no_leading(command));
                    continue;

                case WHOIS_SERVER:
                    gettok(line, command, sizeof(command), ' ', 5);
                    reattach->server = my_strdup(command);
		            fdprintf(server->fd, "MODE %s\r\n", reattach->nick);
                    continue;

	            case MODE_USER:
		            gettok(line, command, sizeof(command), ' ', 4, 1);
		            reattach->umode = my_strdup(command);
                    reattach->target->reattach_complete();
                    continue;

                default:
                    break;

                } /* switch block */
            } /* if (pc) */

            if (!checkf(REATTACHING) && !checkf(DETACHED)) {
                /* Wasn't handled or intercepted: relay */
                send(client->fd, line, len, 0);
                send(client->fd, "\r\n", 2, 0);
                continue;
            }

            /* Below is to respond to server pings. Obviously we are detached */
            char temp[10];
            gettok(line, command,sizeof(command),' ', 2);
            /* Could be server-ping */
            gettok(line, temp, sizeof(temp), ' ', 1);
            if (strcasecmp(temp, "ping") == 0)
            {
                fdprintf(server->fd, "PONG :%s\r\n",no_leading(command)); 
                /* DEBUG("(id: %d) PONG -> %s\n",
                                        id, no_leading(command)); */
            }
        } /* default block */
        } /* .. of this switch statement */
    }     /* while-loop */ 
    return 1;
}

/*
 * This is to be called after all information from server 
 * has been retrieved about irc session.
 * Acts as attacher.
 */
int conn::reattach_complete(void)
{
    conn * c2 = reattach->target;
    int fd_client = client->fd;

    /* It's simpler to do the text sending first so do that */

    /* Sync nickname first -- we do it twice, to be safe */
    fdprintf(fd_client, ":%s!%s NICK :%s\r\n", user.irc_nick, addr(), c2->reattach->nick);
    /*
     * first: send fake numerics and pretend client has connected to an irc server
     */
    fdprintf(fd_client, ":%s 001 %s :Welcome to the Internet Relay Network %s\r\n",
             c2->reattach->server, c2->reattach->nick, c2->reattach->nick);
    fdprintf(fd_client, ":%s 002 %s :Your host is %s, running version %s\r\n",
             c2->reattach->server, c2->reattach->nick, c2->reattach->server, c2->user.serverversion);
    fdprintf(fd_client, ":%s NOTICE %s :Your host is %s running version %s\r\n", 
             c2->reattach->server, c2->reattach->nick, c2->reattach->server, c2->user.serverversion);
    fdprintf(fd_client, ":%s 003 %s :This server was created %s\r\n",
             c2->reattach->server, c2->reattach->nick, c2->user.servercreated);
    fdprintf(fd_client, ":%s 004 %s %s %s %s\r\n",
             c2->reattach->server, c2->reattach->nick, c2->reattach->server, c2->user.serverversion, c2->user.servermodes);
    fdprintf(fd_client, ":%s 005 %s\r\n", 
             c2->reattach->server, c2->user.server005);
    fdprintf(fd_client, ":%s NOTICE %s :Your host is %s and %s\r\n", 
             c2->reattach->server, c2->reattach->nick, c2->reattach->server, EZBOUNCE_VERSION);
    fdprintf(fd_client, ":%s 375 %s :-%s Message of the Day -\r\n",
             c2->reattach->server, c2->reattach->nick, c2->reattach->server);
    fdprintf(fd_client, ":%s 372 %s :This is a dummy MOTD\r\n",
             c2->reattach->server, c2->reattach->nick);
    fdprintf(fd_client, ":%s 376 %s :End of /MOTD command.\r\n",
             c2->reattach->server, c2->reattach->nick);
    fdprintf(c2->server->fd, "LUSERS\r\n");    
    fdprintf(c2->server->fd, "AWAY\r\n");

    fdprintf(fd_client, ":%s!%s NICK :%s\r\n", user.irc_nick, addr(), c2->reattach->nick);
    fdprintf(fd_client, ":%s MODE %s :%s\r\n", c2->reattach->nick, c2->reattach->nick, c2->reattach->umode);
                                
    /* 
     * Simulate joins 
     */
    char channel[128];
    char * channel_ptr;
    int curchan = 1;
    while (c2->reattach->channels && gettok(c2->reattach->channels, channel, sizeof(channel), ' ', curchan++))
    {
        /*
         * Check for and remove leading +,@ or %.
         * Replacing it with a space is not enough because some clients will 
         * think '#channel' is different from ' #channel'
         * So.. use a seperate pointer and increment if needed. 
         */
        channel_ptr = channel;
        if (channel[0] == '+' || channel[0] == '@' || channel[0] == '%' || channel[0] == '*' || channel[0] == '^')
            channel_ptr++;
        fdprintf(fd_client, ":%s!%s@%s JOIN :%s\r\n",
                 c2->reattach->nick, c2->reattach->ident, c2->reattach->hostname,
                 channel_ptr);
        fdprintf(c2->server->fd, "TOPIC %s\r\n", channel_ptr);
        fdprintf(c2->server->fd, "NAMES %s\r\n", channel_ptr);
    }

    printlog("Reattach SUCCESFUL: %s ----> id %d; is now %s on server %s\n",
                  addr(), c2->id, c2->reattach->nick, c2->reattach->server);

    /*
     * Transfer settings.
     *  Actual reattaching happens here..
     */
    c2->clearf(REATTACHING);
    c2->clearf(DETACHED);
    c2->setf(BOUNCED);
    c2->setf(GOTSERVINFO | GOTSERVINFO2);
    
    delete[] c2->user.irc_nick;
    c2->user.irc_nick = c2->reattach->nick;
    c2->reattach->nick = 0;         /* So to not deallocate it */

    /* Restrict length if needed */
    if (strlen(user.irc_nick) > NICKNAME_LENGTH)
        user.irc_nick[strlen(user.irc_nick)] = 0;

    memcpy(&c2->client_saddr, &this->client_saddr, sizeof(client_saddr)); 
    delete[] c2->user.fulladdr;
    c2->user.fulladdr = new char[10 + strlen(c2->user.irc_nick) + 18];
    sprintf(c2->user.fulladdr, "%lu:%s@%s", c2->id, c2->user.irc_nick, inet_ntoa(c2->client_saddr.sin_addr));
    
    /*
     * What we do w/ rulesets on detach:
     *  Keep them alive. It is easier that way.
     *  On reattach, the user simply unregisters his and he becomes
     *  the target so rulesets are taken care of there.
     * Remember that the from rulesets of the detached target match the address
     * of the original user and not necessarily the user attempting to reattach.
     * So, we keep the original address and port of connection in local_saddr.
     */
    unregister_rulesets(rule_set::FROM);
    stat = 0;
   
    c2->client_buff = this->client_buff;
    this->client_buff = 0;
    c2->client        = this->client;
    c2->client->owner = c2;
    this->client      = 0;

    disown_dccs(c2);

    on_client_disconnect();
    
    /* Do the logfiles now */
    if (c2->log)
    {
        const char *names[2];
        c2->log->dump("****** Reattach Completed *******\n");
        c2->log->stop();
        
        c2->log->get_filenames(names);

        c2->send_client("-------------------------------------------------\r\n");
        c2->send_client("- Ended log files\r\n");
        c2->send_client("- You may RETRIEVE your log files by typing\r\n");
        c2->send_client("- /quote ezb LOG SEND all\r\n");
        c2->send_client("-  \r\n");
        c2->send_client("- Type /quote ezb log help for logging help\r\n");
        c2->send_client("-------------------------------------------------\r\n");
        delete c2->log;
        c2->log = 0;
        if (c2->loglist)
            for (unsigned n = 0; n < c2->loglist->num; n++)
                delete[] c2->loglist->objs[n];
        delete c2->loglist;
        c2->loglist =  new obj_set<char>;
        if (names[0])
            c2->loglist->add(my_strdup(names[0]));
        if (names[1])
            c2->loglist->add(my_strdup(names[1]));
    }

    c2->send_client("Reattach successful: you are now %s!%s@%s on %s:%d\r\n",
            c2->user.irc_nick, c2->reattach->ident, c2->reattach->hostname,
            c2->reattach->server, c2->user.port);

    /* Clean up */
    delete c2->reattach;
    delete reattach;
    reattach = NULL;
    c2->reattach = NULL;
    
    return 1;
}

/*
 * Cleanup mess if reattach process is interrupted 
 * by either side or both sides disconnecting
 */
int conn::reattach_cancel()
{
    if (reattach && checkf(DETACHED))
    {
        /* 
         * This is a detached connection. 
         * A client was trying to reattach to it.
         * But the detached connection's connection to the irc server died
         */                                      
        if (reattach->target)
            reattach->target->send_client("Reattach failed: connection to irc server was lost\r\n");
        printlog("Reattach FAILED: (id: %d) to (id: %d): connection to server was lost\n",
                          reattach->target->id, id);
        reattach->target->clearf(REATTACHING);
        delete reattach->target->reattach;
        reattach->target->reattach = NULL;
        clearf(REATTACHING);
        clearf(DETACHED);   
        delete reattach;
        reattach = NULL;

    }
    else if (reattach && !checkf(DETACHED) && checkf(REATTACHING))
    {
        /* 
         * This is a client who was trying to reattach but died 
         * (we can't send anything because client was lost) 
         */
        /* reattach->target->stat &= ~REATTACHING; */
        reattach->target->clearf(REATTACHING);
        clearf(REATTACHING);
        printlog("Reattach FAILED: (id: %d) to (id: %d): client was lost\n",
                          id, reattach->target->id);
        delete reattach->target->reattach;
        reattach->target->reattach = NULL;
        delete reattach;
        reattach = NULL;
    }
    return 1;
}

/*
 * Use what as fake ident.
 * If what is not supplied, access user.user_info
 * and get the [xxx]@blah.com.
 */
bool conn::copy_fake_ident(const char * what)
{
    char id[20];
    if (what || gettok(user.usercmd, id, sizeof(id), ' ', 1))
    {
        delete user.fake_ident;
        user.fake_ident = (what) ? my_strdup(what) : my_strdup(id);
        return 1;
    }
    return 0;
}

/*
 * simple = just match text, no flag-related crap
 * which: 1 for user_commands (from the user)
 *        2 for stuff coming from IRC server
 */
const struct conn::command_set * conn::match_command(int which, const char * cmd, bool simple)
{
    const struct command_set * table = 0;
    unsigned size = 0;
    if (which == 1)
    {
        table = &user_commands[0];
        size = sizeof user_commands;
    }
    else if (which == 2)
    {
        table = &from_server[0];
        size = sizeof from_server;
    }
    
    for (unsigned q = 0; q < (size / sizeof(conn::command_set)); ++q)
    {
        if (strcasecmp(cmd, table[q].cmd) == 0)
        {
            /* No required flags for this command or we meet all required flags */
            if ((simple) || (!table[q].req_flags)
                || ((stat & table[q].req_flags) == table[q].req_flags))
                /* We don't have any of the bad flags */
                if ((simple) || (!table[q].bad_flags) || 
                    !((stat & table[q].bad_flags) & table[q].bad_flags))
                    /* This is acceptable */
                    return &table[q];
            return NULL;
        }

    }
    return NULL;
}

/*
 * Handle trapped CTCPs and do any special things we want for 
 * them. Right now it's only used for DCC proxying.
 * Handles both outgoing and incoming CTCPs, as long as you
 * tell it that ;)
 *
 * Return:
 * 0 - not handled; relay to IRC server
 * 1 - handled; will not relay to IRC server
 */
int conn::do_ctcp(bool incoming, const char * ctcp, const char * source, 
                  const char * target, const char * args)
{
    int out_fd = server->fd, in_fd = client->fd;
    if (incoming)   /* Coming from IRC server to the proxy */
    {
        in_fd = server->fd;
        out_fd = client->fd;
    } 

    if (strcasecmp(ctcp, "DCC") == 0)
    {
        static const char *dcc_types[] = {
            "SEND",
            "CHAT",
            "TSEND",
            /* Gay VIRC extensions */
            "TVIDEO",
            "TVOICE",
            /* mIRC DCC RESUME is NOT yet supported */
            NULL
        };
        /* 
         * Format is:
         * DCC [TYPE] [FILENAME] [IP] [PORT] [size]
         * FIXME: test with corrupted input
         */
        char arg1[8], arg2[256];
        unsigned long ip, size;
        unsigned short port;

        int num_args = sscanf(args, "%7s %255s %lu %hu %lu",
                              arg1,arg2, &ip, &port, &size);
        if (num_args < 4)
            return 0;

        /* Find out first what type of dcc this is */
        int t = 0;
        do {
            if (strcasecmp(arg1, dcc_types[t]) == 0)
                break;
        } while (++t && dcc_types[t]);
        if (!dcc_types[t])
            return 0;
    	else
        {
            unsigned short plisten;
            struct sockaddr_in sin;
            socklen_t dummy = sizeof(sin);
            
            /* 
             * We want to bind to and send the ip of the interface that 
             * we are either connected to IRC to (if outgoing)
             * or the client connected to us on (if its incoming)
             */
            getsockname(out_fd, (struct sockaddr *) &sin, &dummy);
            sin.sin_port = 0;
            ip = htonl(ip);
            
            struct in_addr in_ip = { ip };

            if (!incoming)
            {
                if (ip != client_saddr.sin_addr.s_addr)
                {
                    char * tmp = my_strdup(inet_ntoa(in_ip));

                    printlog("DCC Proxy: Warning: IP of sender (%s) doesn't match IP in CTCP DCC (%s), using sender's IP\n",
                             inet_ntoa(client_saddr.sin_addr), 
                             tmp);
                    in_ip.s_addr = ip = client_saddr.sin_addr.s_addr;
                    delete[] tmp;
                }
            }
            
            dcc * d = new dccpipe(ntohl(ip),port, &sin, &plisten);
            if (!d || !plisten)
            {
                printlog("DCC proxy FAILED: (%s) for %s: %s:%hu (errno=%s)" ,
                                  dcc_types[t],
                                  addr(), inet_ntoa(in_ip), port, strerror(errno));
                delete d;
                return 0;
            }
            printlog("DCC Proxy STARTED: (%s) for %s: (sender) %s:%hu\n",
                                  dcc_types[t],
                                  addr(), inet_ntoa(in_ip), port);
            printlog("     (me) %s:%hu \n", inet_ntoa(sin.sin_addr), plisten);
                
            (void) new dcc_list_t(this, d, NULL);
            
            if (incoming)   /* From IRC server to proxy */
                fdprintf(out_fd, ":%s PRIVMSG %s :\001DCC %s %s %lu %hu",
                         source, target, dcc_types[t], arg2, ntohl(sin.sin_addr.s_addr), 
                         ntohs(sin.sin_port));
            else
                fdprintf(out_fd, "PRIVMSG %s :\001DCC %s %s %lu %hu",
                         target, dcc_types[t], arg2, ntohl(sin.sin_addr.s_addr), ntohs(sin.sin_port));
            
            if (num_args == 5)
                fdprintf(out_fd, " %lu", size);
            fdprintf(out_fd, "\001\r\n");
        }
        return 1;
    }
    return 0;
}

/*
 * Handle proxied dcc events. Take care of logging, etc
 *
 * For DCC Proxying we need to intercept CTCP DCCs from the client.
 *  Once that happens create the DCC object, and it will wait for conncetions
 *  With the information we got about what IP and port its listening on,
 *  relay that to the intended receiver
 * 
 * Here we poll the dcc socks and let them do their job
 * if one of them dies, destroy it and remove it from the list.
 */
/* static */ int conn::do_dcc(dcc_list_t * dt, int r, struct sockaddr_in * sin)
{
    conn * c = dt->owner;
    const char * owner;
    if (c)
        owner = c->addr();
    else 
        owner = "[orphan]";
    
    switch (r)
    {
    case DCC_PIPE_TIMEDOUT:
        printlog("DCC: Proxy session timed out for %s\n", owner);
        if (c)
            c->send_client("DCC: Proxy session timed out\n");
        goto delete_dcc;

    case DCC_SEND_TIMEDOUT:
    case DCC_SEND_COMPLETE:
        if (r == DCC_SEND_TIMEDOUT)
        {
            printlog("DCC: TIMEOUT: Send of file %s to %s timed out after 90 secs\n", dt->filename, owner);
            if (c)
                c->send_client("DCC Send Timed Out: %s\r\n", nopath(dt->filename));
        }
        else {
            printlog("DCC: Send complete of file %s to %s\n", dt->filename, owner);
            if (c)
                c->send_client("DCC Send complete: %s\r\n", nopath(dt->filename));
        
            if (dt->islog)
            {   
                unlink(dt->filename);
                logfile::release(dt->filename);
                printlog("DCC: ... unlocking and deleteing finished log file dcc send.\n");
            }
        }
        goto delete_dcc;
        
    case DCC_CLOSED:
    case DCC_ERROR:
        if (dt->filename)
        {
            printlog("DCC: Send of %s to %s incomplete: %s\n", dt->filename, owner, strerror(errno));
            if (c)
                c->send_client("DCC Send of %s incomplete: %s\r\n", nopath(dt->filename), strerror(errno));
                
        } else
        {
            printlog("DCC: Closing DCC session of %s: %s\n", owner, strerror(errno));
            if (c)
                c->send_client("DCC: Closing DCC Session: %s\r\n", strerror(errno));
        }
       
delete_dcc:
        delete dt->ptr;
        dcc_list_t::list.remove(dt);
        if (c)
            c->my_dccs->remove(dt);
        delete dt;
        break;
        

    case DCC_PIPE_ESTABLISHED:
        printlog("DCC: Full DCC link established for proxied DCC of %s\n", 
                              owner);
        break;

    case DCC_ACCEPTED:
        printlog("DCC: Connection from receipient accepted for proxied DCC of %s\n",
                              owner);
        printlog("DCC: Receipient is from %s:%hu\n", inet_ntoa(sin->sin_addr), ntohs(sin->sin_port));
        break;

    case DCC_SEND_ESTABLISHED:
        printlog("DCC: Send established to %s (%s:%hu)\n", owner, 
                             inet_ntoa(sin->sin_addr), ntohs(sin->sin_port));
        break;

    case 1:
        break;
    
    }
    return 1;
}

/* static */ void conn::kill_dccs(void)
{
    list_iterator<dcc_list_t> i(&dcc_list_t::list);
    do {
        dcc_list_t * dt = i.get();
        if (!dt)
            break;
        delete dt->ptr;
        delete dt;
    } while (++i);
}

void conn::disown_dccs(conn * target)
{
    if (!my_dccs)
        return;
    if (target)
    {
        if (!target->my_dccs)
            target->my_dccs = new linkedlist<dcc_list_t>;
    }
    printlog("DEBUG: Transfering DCC ownership of %p to %p\n", this, target);
    list_iterator<dcc_list_t> i(my_dccs);
    do {
        dcc_list_t * dt = i.get();
        if (!dt)
            break;
        dt->owner = target;
        if (target)
            target->my_dccs->add(dt);
    } while (++i);

    delete my_dccs;
    my_dccs = 0;
}

int conn::unregister_rulesets(char type)
{
    if (type == rule_set::FROM && checkf(FROM_RULESETS_REGISTERED))
    {
        for (unsigned z = 0; z < rules.num; ++z)
            rules.objs[z]->unregister_connection(rule_set::FROM, inet_ntoa(local_saddr.sin_addr), NULL, ntohs(local_saddr.sin_port));
        clearf(FROM_RULESETS_REGISTERED);
        return 1;
    } 
    else if (type == rule_set::TO && checkf(TO_RULESETS_REGISTERED))
    {
        for (unsigned q = 0; q < rules.num; q++)
            rules.objs[q]->unregister_connection(rule_set::TO, inet_ntoa(local_saddr.sin_addr), 
                user.server, ntohs(serv_saddr.sin_port));
        clearf(TO_RULESETS_REGISTERED);
        return 1;
    }
    return 0;
}

int conn::on_client_disconnect(int /* error -- not used right now.. */)
{
    unregister_rulesets(rule_set::FROM);    /* Function does all needed checks */

    if (client) 
    {
        reattach_cancel();
        delete client;
        client = 0;
    }

    if (loglist)
        for (unsigned n = 0; n < loglist->num; n++)
            delete[] loglist->objs[n];
    delete loglist;
    delete client_buff;
    delete[] user.usercmd;
    delete[] user.ircpass;

    loglist = 0;
    user.usercmd = user.ircpass = NULL;
    client_buff = NULL;

    clearf(PWED | NICKED | USERED | ADMINED | REATTACHING | CONNECTING);
    return 1;
}

int conn::on_server_disconnect(int /* error -- not used right now.. */)
{
    if (server)
    { 
        reattach_cancel();
        delete server;
        server = 0;
    }
    
    unregister_rulesets(rule_set::TO);  /* Does all needed checks */

    delete[] user.server;
    delete server_buff;
    
    if (log)
    {
        /* Welp.. user disappeared */
        log->dump("******* Connection to server lost *******\n");
        log->stop();
        delete log;
        log = 0;
    }

    user.server = NULL;
    server_buff = NULL;
    clearf(BOUNCED | CONNECTING);
    return 1;
}

/*
 * Not really what it seems.. 
 * The constructor is the true on_client_connect.. 
 * This one greets the user and informs them of special crap
 * and sends motd and such
 */
int conn::on_client_connect(int /* error */)
{
    delete[] user.fulladdr;
    user.fulladdr = new char[10 + strlen(user.irc_nick) + 18];
    sprintf(user.fulladdr, "%lu:%s@%s", id, user.irc_nick, inet_ntoa(client_saddr.sin_addr));

    if (config.motd_file)
        show_motd();
    send_client("Use /quote CONN <server>:[port] [pass] to connect and"
                " /quote HELP for info on other commands\n");
    setf(PWED);
    /* If auto-fake-ident... store first part of user into fake_ident */
    if (config.flags & AUTO_FAKE_IDENT)
        copy_fake_ident();
    
    /* Auto-detach crap */
    if (config.flags & ENABLE_AUTO_DETACH)
    {
        user.autopass = new char[10];
        mk_printable_random_pass(user.autopass, 9);
        send_client("Auto-Detach is enabled here -- your magic number and password for this"
                    " session: %d %s\r\n", id, user.autopass);
        send_client("You will be automatically detached unless you do '/quote ezb quit'\r\n");
    }
    /* automatic server connection */
    if (config.autoserver || user.detachpass)
    {
        char * s = (user.detachpass) ? user.detachpass : config.autoserver;
        send_client("Automatically connecting you to: %s\r\n", s);
        printlog("CONNECT: Attempting auto connect of %s to %s\n", addr(), s);
        do_conn_command(s);
        delete user.detachpass;
        user.detachpass = 0;
    }

    return 1;
}


/*
 * Very simple. Load the file into a dynbuff and dump it out line
 * by line. There are no tricks here to conserve memory. MOTDs 
 * ought to be short and sweet. We will therefore restrict the buffer
 * size to a max of 64kB. If the admin wants to use a 50MB
 * file, too bad.
 *
 * We don't transmit these with the MOTD numeric right now.
 */
int conn::show_motd(void) 
{
    send_client("Message of the day: \r\n");
    
    if (!config.motd_file)
        return 0;
    
    int fd = open(config.motd_file, O_RDONLY);
    if (fd < 0)
    {
        printlog("Request for MOTD file %s failed: %s\r\n", config.motd_file, 
                           strerror(errno));
        return 0;
    }

    dynbuff d(512, 65536);
    char line[512];
    int len;

    d.add(fd);
    close(fd);
    

    while (1)
    {
        /* special arguments: don't leave line, DON'T require crlf at end
          of line */
        switch ((len = d.get_line(0, line, sizeof(line), 0, 0)))
        {
        case -1:    /* Nothing left in buffer */
            goto out;   
        case 0:     /* Line is empty */
            send_client(" \r\n");
            continue;
        default:
            send_client("%s\r\n", line);
        }
    }

out:
    send_client("  \r\n");
    send_client("---\r\n");
    return 1;
}


int conn::do_log_command(const char * cmd, const char * args)
{
    char buff[20]; 
    /* Status Command */
    if (!cmd || !*cmd) /* FIXME: really need both checks ? */
    {                                          
        logfile::intflags_to_char(log_options, buff);
        send_client("Your current log options are: %s\r\n", buff);
        send_client("Use /quote [ezb] LOG HELP for description of options\r\n");
        send_client("Use /quote [ezb] LOG LIST to list any log files ready for sending\r\n");
    }
    else if (strcasecmp(cmd, "LIST") == 0)
    {
        if (!loglist)
            send_client("LOG LIST: nothing to list.\r\n");
        else
            goto list_logs;
    }
    else if (strcasecmp(cmd, "SEND") == 0) 
    {
        unsigned int idx = (unsigned int) atoi(args);
        unsigned int num2send = 1;
        if (!loglist)
        {
            send_client("LOG SEND: There are no log files to send. Try '/quote [ezb] log find'\r\n");
            return 1;
        } 
        if (strcasecmp(args, "all") == 0)
        {
            send_client("LOG SEND: Send all requested: I will only send two at a time...\r\n");
            num2send = 2;
            idx = 1;
        }
        else if (!idx || idx > loglist->num)
        {
            send_client("LOG SEND: Log idx %d out of range: try log send <1-%d>\r\n", idx, loglist->num);
            return 1;
        }
        idx--;
        unsigned int x;
        for (x = 0; (x < num2send) && (idx < loglist->num); idx++)
        {
            if (logfile::is_locked(loglist->objs[idx]))
            {
                send_client("LOG SEND: Unable to send log file %d: logfile is locked\r\n", idx + 1);
                continue;
            }

            dcc * d = dcc_send_file(loglist->objs[idx], logfile::fixup_logname(loglist->objs[idx]), NULL, 1);
            if (!d)
            {
                send_client("LOG SEND: Unable to send log file %d: %s\r\n", idx + 1, strerror(errno));
                continue;
            }
            logfile::lock(loglist->objs[idx]);
            printlog("LOG SEND: Sent logfile %s to %s\n", loglist->objs[idx], addr());
            send_client("LOG SEND: Sent %s...\r\n", loglist->objs[idx]);
            x++;
        }
        send_client("Sent %d log files...\r\n", x);
    } 
    else if (strcasecmp(cmd, "SET") == 0)
    {
        if (!checkf(ADMINED) && !(config.flags & LOG_OPTIONS))
        {
            send_client("LOG SET: Error: Detached-Logging is disabled here.\r\n");
            return 1;
        }
        /* Set the log options. Reverse lookup the log options if the
         * user stuck any invalid ones in there */
        int tmp = logfile::charflags_to_int(args);
        if (!(tmp & logfile::LOG_ALL))
        {
            send_client("LOG SET: Error: you did not specify a A,C,P option, assuming you want to log everything\n");
            tmp |= logfile::LOG_ALL;
        }
        /* Enforce config file restrictions */
        if (!checkf(ADMINED))
        {
            if (!(config.flags & ENABLE_PRIVATE_LOGGING) && (tmp & logfile::LOG_PRIVATE))
            {       
                send_client("LOG SET: Error: logging of private messages is disabled here.\r\n");
                tmp &= ~logfile::LOG_PRIVATE;
            }
            if (!(config.flags & ENABLE_PUBLIC_LOGGING) && (tmp & logfile::LOG_PUBLIC))
            {
                send_client("LOG SET: Error: logging of public messages is disabled here.\r\n");
                tmp &= ~logfile::LOG_PUBLIC;
            }
            if (!(config.flags & ENABLE_SEPERATE_LOGGING) && (tmp & logfile::LOG_SEPERATE))
            {
                send_client("LOG SET: Error: logging to seperate files is disabled here.\r\n");
                tmp &= ~logfile::LOG_SEPERATE;
            }
        }
        
        log_options = tmp;
        logfile::intflags_to_char(log_options, buff);
        send_client("Set your log options to: %s\r\n", buff);
        send_client("Use /quote HELP LOG for description of options\r\n");
    }

    /* Look up old log files */
    else if (strcasecmp(cmd, "FIND") == 0)
    {
        char _pass[PASSWORD_LENGTH], _id[9];
        int num;

        if (!gettok(args, _id, sizeof _id, ' ', 1)
            || !gettok(args, _pass, sizeof(_pass), ' ', 2))
        {
            send_client("Usage: /quote [ezb] LOG FIND <id> <password>\r\n");
            return 1;
        }
        send_client("LOG FIND: Searching...\r\n");

        /* List is destroyed each and every time */
        if (loglist)
            for (unsigned n = 0; n < loglist->num; n++)
                delete[] loglist->objs[n];
        delete loglist;
        loglist = new obj_set<char>;

        num = logfile::find_log_files(config.logdir, user.irc_nick, 
                                         _pass, atoi(_id), loglist, 10);
        if (num)
        {
            send_client("Found %d matching log files: \r\n", num);

list_logs:        
            for (unsigned i = 0; i < loglist->num; i++)
                send_client("   %d. %s\r\n", i+1, loglist->objs[i]);
            send_client("---\r\n");
            send_client("Use /quote [ezb] LOG SEND <#> to retrieve a log file.\r\n");
            return 1;
        }

        send_client("LOG FIND: Couldn't find anything...\r\n");
        send_client("LOG FIND: Be sure to check your logfile send list (with /quote [ezb] LOG LIST)\r\n");
    }

    else if (strcasecmp(cmd, "HELP") == 0)
    {
        static const char *log_help = 
            "-------------------------------------------\r\n\a"
            "- Available LOG Commands:\r\n\a"
            "-\r\n\a"
            "- SET <any combination of tfpcans>\r\n\a"
            "-\r\n\a"
            "-      Sets logging options for this session\r\n\a"
            "-      p - log private messages\r\n\a"
            "-      c - log channel messages\r\n\a"
            "-      a - log both channel and private\r\n\a"
            "-      n - don't log anything\r\n\a"
            "-      s - log to seperate files\r\n\a"
            "-      t - timestamp logs\r\n\a"
            "-      f - show full addresses in *public* messages\r\n\a"
            "-\r\n\a"
            "-\r\n\a"
            "- SEND <# or ALL>\r\n\a"
            "-\r\n\a"
            "-     Sends any logfiles to you that are ready to be sent.\r\n\a"
            "-     You specify which one to receive or to receive two at a time.\r\n\a"
            "-\r\n\a"
            "-\r\n\a"
            "- FIND <id> <password>\r\n\a"
            "-\r\n\a"
            "-     Allows you to recover log files from old detached\r\n\a"
            "-     sessions that you were not able to retrieve\r\n\a"
            "-     ID and Password refer to the magic number and password\r\n\a"
            "-     of the session whose logs you want to retrieve\r\n\a"
            "-\r\n\a"
            "-\r\n\a"
            "- LIST\r\n\a"
            "-     Lists the results of your previous LOG FIND search and \r\n\a"
            "-     and any other log files that are ready to be sent.\r\n\a"
            "-\r\n\a"
            "-\r\n\a"
            "- Use /LOG without any options to view current log settings\r\n\a"
            "-------------------------------------------\r\n\a";

        send_client_multiline(log_help);
    }
    return 1;
}

int conn::do_conn_command(const char * args)
{
    char server[256] = "",
          port[6] = "", 
          pass[10] = "";
    int tmp;
    u_short p;

    if (!gettok(args, server, sizeof(server), ' ', 1))
    {
        send_client(msg_not_enuff_args, "CONN");  
        return 1;
    }
    if (gettok(server, port, sizeof(port), ':', 2))
        *strchr(server,':') = 0;
    gettok(args, pass, sizeof(pass), ' ', 2);
    p = (strcmp(port, "") == 0) ? 6667: (u_short) atoi(port);
    
    /* Let's see if we CAN connect to this place */
    if (!can_connect(server, p))
    {
        printlog("Connection attempt DENIED: %s to %s:%d\n",
                         addr(), server, p);
        return 1;
    }
    /* we can now try to connect.. */
    user.server = my_strdup(server);
    printlog("Connection attempt: %s to %s:%d\n",
                      addr(), server, p);
    if ((tmp = do_connect(server, p,pass)) == 1)
    {
        send_client(msg_attempting_to_connect, server, p);
        client_buff->flush();
    }
    else 
    {
       const char *err;
       if (tmp == -1)
           err = "unknown host";
       else 
           err = strerror(errno);
       send_client(msg_conn_failed, server, err);
       printlog("Connection attempt FAILED: %s to %s:%d (failed prematurely): %s\n", addr(),
                        server, p, err);
    }
    return 1;
}

dcc * conn::dcc_send_file(const char * file, const char * send_as, struct sockaddr_in * psin, bool islog)
{
    unsigned short port;
    struct sockaddr_in sin;
    unsigned long filesize;
    socklen_t len = sizeof(sin);
    memset(&sin, 0, sizeof(sin));
    

    getsockname(client->fd, (struct sockaddr *) &sin, &len);
    sin.sin_port = 0;

    dccsend * d = new dccsend(file, &sin, &port, &filesize);
    if (!port)
    {
        printlog("DCC Send: Error sending %s: %s\n", file, strerror(errno));
        delete d;
        return 0;
    }
    printlog("DCC SEND: Sending %s to %s (id: %d).\n", file, addr(), id);

    send_client("Ok.. sending %s.. waiting on port %d\n", file, port);
    
    dcc_list_t * dt = new dcc_list_t(this, d, file);
    if (islog)
        dt->islog = 1;

    
    fdprintf(client->fd, ":%s PRIVMSG %s :\001DCC SEND %s %lu %hu %lu\001\r\n",
             ezbounce_guy, user.irc_nick, (send_as ? remove_path(send_as) : remove_path(file)), ntohl(sin.sin_addr.s_addr), 
                 ntohs(sin.sin_port), filesize);

    if (psin)
        memcpy(psin, &sin, sizeof(sin));
    return d;
}


dcc_list_t::dcc_list_t(conn * owner, dcc * ptr, const char * filename)
{
    this->owner = owner;
    this->ptr = ptr;  
    this->filename = my_strdup(filename);
    ptr->dcc_list_entry = this;
    list.add(this);
    if (owner)
    {
        if (!owner->my_dccs)
            owner->my_dccs = new linkedlist<dcc_list_t>;
        owner->my_dccs->add(this);
    }
    islog = 0;
    DEBUG("DCC_LIST_T: constructing %p filename %s\n", this, filename);
}

inline dcc_list_t::~dcc_list_t(void)
{
    DEBUG("DCC_LIST_T: destructing %p\n", this);
    delete[] filename;
}

    
int conn::psock::event_handler(const struct pollfd *)
{
    /* One of the socks has an event pending. 
     * Might as well check both. Always check server first. 
     * This is kinda dangerous. We might be deleted by our 
     * owner. There is no guarantee we will even exist in some of the
     * next few lines, so save the pointer!!! 
     *
     * Also, clear the data ready flags for the other socket pair.
     * We don't want to be called again. */
    conn * c = owner;
    int e = c->server ? c->server->revents() : 0;
    /* DEBUG("In socket event handler for %p - fd: %d revents: %d\n", c, pfd->fd, pfd->revents); */
    if (e)
    {   
        c->server->set_revents(0); 
        DEBUG("     ...Calling check_server: %d\n", e);
        switch (c->check_server(e))
        {
        case CONN_DISCONNECTED:
            printlog("DISCONNECT: %s lost connection to server: %s.\n", 
                     c->addr(), strerror(errno));
            if (config.flags & DROP_ON_DISCONNECT)
            {
                printlog("... removed client from proxy\n");
                delete c;
                goto out;
            }
            break;

        case CONN_KILLME:
            printlog("DISCONNECT: %s lost connection to server: %s -- killing.\n", 
                     c->addr(), strerror(errno));
            delete c;
            goto out;

        case CONN_FULL:
            if (config.flags & KILL_WHEN_FULL)
            {
                printlog("KILLING: %s -- full input queue(s)\n",
                        c->addr());
                delete c;
            }
            goto out;

        default:
            break;
        }
    }

    e = c->client ? c->client->revents() : 0;
    if (e)
    {
        c->client->set_revents(0);
        DEBUG("    .. in check_client(): %d\n", e);
        switch (c->check_client(e))
        {
        case CONN_LINK_DEACTIVATED:
            printlog("DISCONNECT: %s lost connection to server: %s.\n", 
                 c->addr(), strerror(errno));
            if (config.flags & DROP_ON_DISCONNECT)
            {
                printlog("... removed client from proxy\n");
                delete c;
                goto out;
            }
            break;

        case CONN_DISCONNECTED:
        case CONN_KILLME:
            delete c;
            goto out;

        case CONN_FULL:
            if (config.flags & KILL_WHEN_FULL)
            {
                printlog("KILLING: %s -- full input queue(s).\n",
                         c->addr());  
                delete c;
            }
            goto out;
            
        default:
            break;
        }
    } 

out:
    return 1;
}

int dcc::dccsock::event_handler(const struct pollfd *)
{
    struct sockaddr_in sin;
    int ret = owner->poll(&sin);
    if (owner->receiver && owner->receiver->sock)
        owner->receiver->sock->set_revents(0);
    if (owner->sender && owner->sender->sock)
        owner->sender->sock->set_revents(0);
    if (owner->lsock)
        owner->lsock->set_revents(0);
    
    conn::do_dcc(owner->dcc_list_entry, ret, &sin);
    return 1;
}

conn::uinfo::uinfo() 
{
    usercmd = autopass = server = ircpass = detachpass = 
        fake_ident = irc_nick = serverversion = servercreated = 
        server005 = servermodes = fulladdr = NULL;
    port = 0;
    iface.s_addr = 0;
}

conn::uinfo::~uinfo()
{
    delete[] usercmd;
    delete[] ircpass;
    delete[] server;
    delete[] detachpass;
    delete[] fake_ident;
    delete[] irc_nick;
    delete[] autopass;
    delete[] serverversion;
    delete[] servercreated;
    delete[] servermodes;
    delete[] server005;
    delete[] fulladdr;
}


/* static */ void conn::do_timers(time_t time_ctr)
{
    srand(time_ctr);
    conn * c = pFirst, * c2 = 0;
    do {
        /* Kill stale connections */
        if (c->dead()) 
        {
            c2 = c->next();
            DEBUG("deleting dead object 0x%lx\n", c);
            delete c;
            c = c2;
        }
         
        /* Check for idlers */
        else if (!c->check_idle(time_ctr)) 
        {
            c2 = c->next();
            printlog("KILLING: %s for exceeding idle time limit\n", c->addr());
            delete c;
            c = c2;
        }

        /* Anti idle for detached connections */
        else if (c->checkf(DETACHED) && !c->checkf(REATTACHING))
        {
            if ((random()%4) > 1)
            {
                static const char *anti_idle[5] = {
                    "2]raga2 :hey, whats up?",
                    "~habib :ASDFASFJAD GKJDFKJ AKS JASKDJ ASKLDJ AKLSJDLKA SJDLKA SJD KASJD LASKDJ",
                    "987987 :SOMEBODY SET UP US THE BOMB",
                    "!!!!! :s disturbing",
                    ".das :main screen"
                };
                fdprintf(c->server->fd, "PRIVMSG %s\r\n", anti_idle[random()%5]);
            }
            c->server_buff->optimize();
            c = c->next();
        }
        else
        {
            /* shrink the buffers if possible */
            if (c->server_buff)
                c->server_buff->optimize();
            if (c->client_buff)
                c->client_buff->optimize();
            c = c->next();
        }
    } while (c);
}
