/*
    ext2_inode_relocator.c -- ext2 inode relocator
    Copyright (C) 1998, 1999, 2000 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char _ext2_inode_relocator_c[] = "$Id: ext2_inode_relocator.c,v 1.16 2004/09/30 14:01:41 sct Exp $";

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>	/* for S_ISDIR */
#include "ext2.h"






struct ext2_reference
{
	blk_t			 block;
	off_t			 offset;
};

struct ext2_inode_entry
{
	ino_t			 ino;
	ino_t			 dest;
	unsigned		 numreferences:16;
	unsigned		 isdir:1;
	struct ext2_reference	*ref;
};

struct ext2_inode_relocator_state
{
	int			 usedentries;
	int			 resolvedentries;
	struct ext2_inode_entry	*inode;
	struct ext2_reference	*last;
};





static struct ext2_inode_entry *findit(struct ext2_inode_relocator_state *state,
				       ino_t inode)
{
	int			 min_e;
	int			 max_e;
	struct ext2_inode_entry *retv;
	int			 t;
	blk_t			 tval;

	max_e = state->usedentries - 1;
	min_e = 0;
	retv = NULL;

 repeat:
	if (min_e > max_e)
		goto out;

	t = (min_e + max_e) >> 1;
	tval = state->inode[t].ino;

	t--;
	if (tval > inode)
		max_e = t;

	t += 2;
	if (tval < inode)
		min_e = t;

	t--;

	if (tval != inode)
		goto repeat;

	retv = &state->inode[t];

 out:
	return retv;
}

static int addref(struct ext2_fs *fs, struct ext2_inode_relocator_state *state,
		  ino_t ino, blk_t blockno, off_t offset)
{
	struct ext2_inode_entry *ent;
	int i;

	if ((ent = findit(state, ino)) == NULL)
		return 1;

	for (i = 0; i < ent->numreferences; i++)
		if (!ent->ref[i].block)
			break;

	if (i == ent->numreferences) {
		fprintf(stderr,
			"%s: found an inode with a incorrect link count\n"
			"run e2fsck first.\n", fs->prog);
		return 0;
	}

	if (i == ent->numreferences - 1)
		state->resolvedentries++;

	ent->ref[i].block = blockno;
	ent->ref[i].offset = offset;

	return 1;
}

static int doblock(struct ext2_fs *fs, struct ext2_inode_relocator_state *state,
		   blk_t blockno)
{
	struct ext2_buffer_head *bh;
	off_t			 offset;

	bh = ext2_bread(fs, blockno);

	offset = 0;
	do {
		struct ext2_dir_entry_2 *ptr;

		ptr = (struct ext2_dir_entry_2 *)(bh->data + offset);

		if (ptr->name_len)
			if (!addref(fs, state, ptr->inode, blockno, offset))
				return 0;

		offset += ptr->rec_len;
	} while (offset < fs->blocksize);

	ext2_brelse(bh, 0);
	return 1;
}

static int doindblock(struct ext2_fs *fs,
		      struct ext2_inode_relocator_state *state, blk_t blockno)
{
	struct ext2_buffer_head *bh;
	blk_t			 blk;
	int			 i;

	bh = ext2_bread(fs, blockno);

	for (i = 0; i < fs->blocksize >> 2; i++)
		if ((blk = ((__u32 *)bh->data)[i]) != 0)
			if (!doblock(fs, state, blk))
				return 0;

	ext2_brelse(bh, 0);
	return 1;
}

static int dodindblock(struct ext2_fs *fs,
		       struct ext2_inode_relocator_state *state, blk_t blockno)
{
	struct ext2_buffer_head *bh;
	blk_t			 blk;
	int			 i;

	bh = ext2_bread(fs, blockno);

	for (i = 0; i < fs->blocksize >> 2;i++)
		if ((blk = ((__u32 *)bh->data)[i]) != 0)
			if (!doindblock(fs, state, blk))
				return 0;

	ext2_brelse(bh, 0);
	return 1;
}

static int dotindblock(struct ext2_fs *fs,
		       struct ext2_inode_relocator_state *state, blk_t blockno)
{
	struct ext2_buffer_head *bh;
	blk_t			 blk;
	int			 i;

	bh = ext2_bread(fs, blockno);

	for (i = 0; i < fs->blocksize >> 2;i++)
		if ((blk = ((__u32 *)bh->data)[i]) != 0)
			if (!dodindblock(fs, state, blk))
				return 0;

	ext2_brelse(bh, 0);
	return 1;
}

static int doinode(struct ext2_fs *fs, struct ext2_inode_relocator_state *state,
		   ino_t ino)
{
	struct ext2_inode tmp;
	struct ext2_inode *inode = &tmp;
	int		  i;


	ext2_read_inode(fs, ino, inode);

	if (S_ISDIR(inode->i_mode)) {
		blk_t blk;

		for (i = 0; i < EXT2_NDIR_BLOCKS; i++)
			if ((blk = inode->i_block[i]))
				if (!doblock(fs, state, blk))
					return 0;

		if ((blk = inode->i_block[EXT2_IND_BLOCK]))
			if (!doindblock(fs, state, blk))
				return 0;

		if ((blk = inode->i_block[EXT2_DIND_BLOCK]))
			if (!dodindblock(fs, state, blk))
				return 0;

		if ((blk = inode->i_block[EXT2_TIND_BLOCK]))
			if (!dotindblock(fs, state, blk))
				return 0;
	}

	return 1;
}

static int doscangroup(struct ext2_fs *fs,
		       struct ext2_inode_relocator_state *state, int group)
{
	struct ext2_buffer_head *bh;
	int			 i;
	int			 offset;

	if (fs->flags & FL_VERBOSE)
		printf(" scanning group %i.... ", group);

	bh = ext2_bread(fs, fs->gd[group].bg_inode_bitmap);
	offset = group * fs->sb.s_inodes_per_group + 1;

	for (i = 0; i < fs->sb.s_inodes_per_group; i++)
		if (check_bit(bh->data, i)) {
			if (!doinode(fs, state, offset + i)) {
				ext2_brelse(bh, 0);
				return 0;
			}

			if (state->resolvedentries == state->usedentries)
				break;
		}

	ext2_brelse(bh, 0);

	if (fs->flags & FL_VERBOSE)
		printf("%i/%i inodes resolved\r",
			state->resolvedentries, state->usedentries);

	return 1;
}

static int doscan(struct ext2_fs *fs, struct ext2_inode_relocator_state *state)
{
	int i;

	if (!doscangroup(fs, state, 0))
		return 0;

	if (state->resolvedentries != state->usedentries)
		for (i = fs->numgroups - 1; i > 0; i--) {
			if (!doscangroup(fs, state, i))
				return 0;

			if (state->resolvedentries == state->usedentries)
				break;
		}

	if (fs->flags & FL_VERBOSE)
		printf("\n");

	return 1;
}







static void ext2_inode_relocator_copy(struct ext2_fs *fs,
				      struct ext2_inode_relocator_state *state)
{
	int i;

	for (i = 0; i < state->usedentries; i++) {
		struct ext2_inode inode;
		struct ext2_inode_entry *entry;

		entry = &state->inode[i];

		if (!ext2_get_inode_state(fs, entry->ino) ||
		    ext2_get_inode_state(fs, entry->dest))
			fprintf(stderr, "inode bitmap error\n");

		ext2_read_inode(fs, entry->ino, &inode);
		ext2_write_inode(fs, entry->dest, &inode);

		entry->isdir = S_ISDIR(inode.i_mode) ? 1 : 0;
	}

	if (fs->flags & FL_SAFE)
		ext2_sync(fs);
}

static void ext2_inode_relocator_finish(struct ext2_fs *fs,
					struct ext2_inode_relocator_state *state)
{
	int i;

	for (i = 0; i < state->usedentries; i++) {
		struct ext2_inode_entry *entry;

		entry = &state->inode[i];
		ext2_set_inode_state(fs, entry->dest, 1, 1);
		ext2_set_inode_state(fs, entry->ino, 0, 1);
		ext2_zero_inode(fs, entry->ino);
	}

	if (fs->flags & FL_SAFE)
		ext2_sync(fs);
}

static int ext2_inode_relocator_ref(struct ext2_fs *fs,
				    struct ext2_inode_relocator_state *state)
{
	int		i;
	static int	numerrors = 0;

	for (i = 0; i < state->usedentries; i++) {
		struct ext2_inode_entry *entry;
		int			 j;
		__u32			 t;

		entry = &state->inode[i];
		t = entry->dest;

		for (j = 0; j < entry->numreferences; j++) {
			struct ext2_buffer_head *bh;

			bh = ext2_bread(fs, entry->ref[j].block);

			if ((*((__u32 *)(bh->data + entry->ref[j].offset))) !=
			    entry->ino) {
				fprintf(stderr,
					"inode %li ref error! "
					"(->%li, [%i]={%i, %li})\n",
					entry->ino, entry->dest, j,
					entry->ref[j].block,
					entry->ref[j].offset);
				ext2_brelse(bh, 0);

				if (numerrors++ < 4)
					continue;

				fprintf(stderr, "all is not well!\n");
				return 0;
			}

			*((__u32 *)(bh->data + entry->ref[j].offset)) = t;
			bh->dirty = 1;

			ext2_brelse(bh, 0);
		}

		if (entry->isdir) {
			int oldgroup;
			int newgroup;

			oldgroup = (entry->ino - 1) / fs->sb.s_inodes_per_group;
			newgroup = (entry->dest- 1) / fs->sb.s_inodes_per_group;

			fs->gd[oldgroup].bg_used_dirs_count--;
			fs->gd[newgroup].bg_used_dirs_count++;

			fs->metadirty = EXT2_META_GD;
		}
	}

	if (fs->flags & FL_SAFE)
		ext2_sync(fs);

	return 1;
}

static int ext2_inode_relocator_grab_inodes(struct ext2_fs *fs,
					    struct ext2_inode_relocator_state *state)
{
	int i;
	int ptr;

	ptr = 0;

	for (i = 0; i < fs->numgroups; i++)
		if (fs->gd[i].bg_free_inodes_count) {
			struct ext2_buffer_head *bh;
			int j;
			int offset;

			bh = ext2_bread(fs, fs->gd[i].bg_inode_bitmap);
			offset = i * fs->sb.s_inodes_per_group + 1;

			for (j = i ? 0 : EXT2_GOOD_OLD_FIRST_INO;
			     j < fs->sb.s_inodes_per_group; j++) {
				if (!check_bit(bh->data, j)) {
					state->inode[ptr++].dest = offset + j;

					if (ptr == state->usedentries) {
						ext2_brelse(bh, 0);
						return 1;
					}
				}
			}

			ext2_brelse(bh, 0);
		}

	fprintf(stderr, "not enough free inodes!\n");

	return 0;
}

static int ext2_inode_relocator_flush(struct ext2_fs *fs,
				      struct ext2_inode_relocator_state *state)
{
	if (!state->usedentries)
		return 1;

	if (!doscan(fs, state))
		return 0;

	if (!ext2_inode_relocator_grab_inodes(fs, state))
		return 0;

	ext2_inode_relocator_copy(fs, state);
	ext2_inode_relocator_ref(fs, state);
	ext2_inode_relocator_finish(fs, state);

	state->usedentries = 0;
	state->resolvedentries = 0;
	state->last = (struct ext2_reference *)fs->relocator_pool_end;

	if (fs->flags & FL_SAFE)
		ext2_sync(fs);

	return 1;
}

static int ext2_inode_relocator_mark(struct ext2_fs *fs,
				     struct ext2_inode_relocator_state *state,
				     ino_t ino)
{
	struct ext2_inode	 inode;
	struct ext2_inode_entry *ent;
	void			*adv;
	void			*rec;
	int			 i;

	ext2_read_inode(fs, ino, &inode);

	adv = state->inode + state->usedentries + 1;
	rec = state->last - inode.i_links_count;

	if (adv >= rec)
		ext2_inode_relocator_flush(fs, state);

	state->last -= inode.i_links_count;

	ent = &state->inode[state->usedentries];
	ent->ino = ino;
	ent->dest = 0;
	ent->numreferences = inode.i_links_count;
	ent->ref = state->last;

	for (i = 0; i < ent->numreferences; i++) {
		ent->ref[i].block = 0;
		ent->ref[i].offset = 0;
	}

	state->usedentries++;

	return 1;
}


int ext2_inode_relocate(struct ext2_fs *fs)
{
	int i;
	struct ext2_inode_relocator_state state;

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	state.usedentries = 0;
	state.resolvedentries = 0;
	state.inode = (struct ext2_inode_entry *)fs->relocator_pool;
	state.last = (struct ext2_reference *)fs->relocator_pool_end;

	for (i = fs->newgroups; i < fs->numgroups; i++) {
		struct ext2_buffer_head *bh;
		int			 j;
		int			 offset;

		bh = ext2_bread(fs, fs->gd[i].bg_inode_bitmap);
		offset = i * fs->sb.s_inodes_per_group + 1;

		for (j = 0; j < fs->sb.s_inodes_per_group; j++)
			if (check_bit(bh->data, j))
				ext2_inode_relocator_mark(fs, &state, offset+j);

		ext2_brelse(bh, 0);
	}

	ext2_inode_relocator_flush(fs, &state);

	return 1;
}

