/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/


/**********************************************************
*
*  File: tmain.c
*
*  Contents:  main() for Surface Evolver
*/

#include "include.h"

int tty_flag;  /* to do tty interface instead of menus */

#if defined(NeXTapp) || defined(MOTIF) || defined(MAC_APP) || defined(WIN32S) || defined(MAC_CW)
#define main old_main
#endif

#ifdef SHARED_MEMORY
void m_set_idlist()
{ proc_ids[m_get_myid()] = getpid(); }
#endif

#if defined(PVM) || defined(MPI)
#define main old_main
#endif

int main(argc,argv)
int argc;
char *argv[];
{ 
#ifdef NeXTapp
  int no_Next_graph = 0;
#endif
  msgmax = 200;
  msg = mycalloc(1,msgmax);

#ifdef CLK_TCK
     clock();  /* initialize clock */
#endif         

  outfd = stdout;
  sprintf(msg,"Surface Evolver %s\n\n",VERSION);
  outstring(msg);
#ifdef LONGDOUBLE
  sprintf(msg,"Compiled for %d byte long double.\n\n",sizeof(REAL));
  outstring(msg);
#endif

#ifdef SGI_MULTI
  procs_requested = 1;  /* default at John's request.  m_get_numprocs(); */
#endif


  print_express(NULL,0); /* just to initialize string allocation */

  /* parse command line options */
  if ( argc > 0 )
  {
     argv++; argc--;
     while (  argc && (argv[0] != NULL) && (argv[0][0] == '-') )
     { switch ( argv[0][1] )
       {  
          case 'a': auto_convert_flag = (argv[0][2]=='-') ? 0 : 1; break;
          case 'q': option_q = (argv[0][2]=='-') ? 0 : 1; break; 
          case 'e': echo_flag = 1; break;
          case 't': tty_flag = 1;
                break;
          case 'u': tty_flag = 1; 
#ifdef NeXTapp
                 no_Next_graph = 1;
#endif
                break;
          case 'f' : /* commands from file */
                 cmdfilename = argv[0]+2;
                 if ( cmdfilename[0] == 0 ) /* probably a space inserted */
                    { cmdfilename = *++argv; argc--; }
                 break;
          case 'd' :  /* parser debug */
                 yydebug = 1;
                 break;
          case 'i' : match_id_flag = 1; break;
          case 'p' : procs_requested = atoi(argv[0]+2); 
#ifdef SGI_MULTI
                if ( procs_requested < 1 )
                  { kb_error(1321, "-p with nonpositive number. Procs set to 1.\n",
                                    WARNING);
                     procs_requested = 1;
                  }
#else
                             kb_error(1322,"-p option not effective.  This Evolver not compiled for SGI multiprocessor.\n", WARNING);

#endif
                             break; 
          case 'x' : exit_after_error = 1; break;
          case 'w' : exit_after_warning = exit_after_error = 1; break;
          case 'm' : memdebug = 1; break;
          default:
                 sprintf(msg,"Illegal option: %s\n",argv[0]); outstring(msg);
          case 'h' :
                 outstring("Legal options: \n"); 
                 outstring("  -ffilename          take commands from file\n"); 
                 outstring("  -i                     use id numbers as in datafile\n"); 
                 outstring("  -q                     convert to named quantities\n"); 
                 outstring("  -a                     auto convert to named quantities when needed\n"); 
                 outstring("  -t                     terminal version on NeXT\n"); 
                 outstring("  -x                     exit after error\n"); 
                 outstring("  -w                     exit after warning\n"); 
                 outstring("  -d                     parser debugging on\n"); 
                 outstring("  -m                     memory debugging on\n"); 
#ifdef SGI_MULTI
                 outstring("  -pn                    use n processes \n"); 
#endif
                 outstring("  -h                     print this help\n"); 
                 break;
            }
        argv++; argc--;
     }
  }

#ifdef SGI_MULTI
  sprintf(msg,"Using %d processes on %d processors.\n\n",
      procs_requested,m_get_numprocs()); 
  outstring(msg);
  m_set_procs(procs_requested);
  if ( m_get_numprocs() > 1 )
  { int n;
     /* set up list of locks available for critical sections */
     usconfig(CONF_INITSIZE,200*_MAXLOCKS);
     usconfig(CONF_ARENATYPE,US_SHAREDONLY);
     usconfig(CONF_INITUSERS,4+m_get_numprocs());
     lock_arena = usinit(lock_arena_name);
     if ( lock_arena == NULL ) { perror(lock_arena_name); exit(2); }
     for ( n = 0 ; n < _MAXLOCKS ; n++ )
     { locklist[n] = usnewlock(lock_arena);
        if ( locklist[n] == NULL )
        { fprintf(stderr,"lock allocation failure on lock %d.\n",n);
          perror("usnewlock");
          exit(2);
        }
     }
     m_fork(m_set_idlist);
     m_park_procs();
     mpflag = M_INACTIVE;
  }
#endif
  nprocs = procs_requested;

  signal(SIGINT,catcher);    /* to catch user interrupt */     
#ifdef SIGUSR1
  signal(SIGUSR1,catcher);    /* to catch user interrupt */     
#endif
#ifdef SIGTERM
  signal(SIGTERM,catcher);    /* to catch user interrupt, dump and kill  */     
#endif
#ifdef SIGHUP
  signal(SIGHUP,catcher);    /* to catch user interrupt, dump and kill  */     
#endif
#ifdef SIGPIPE
  signal(SIGPIPE,catcher);    /* to catch broken pipe */     
#endif

  scoeff_init();
  vcoeff_init();  
  init_view();
  reset_web();  /* in case no datafile on command line */

  if ( argc && argv &&  argv[0] && argv[1] )
     kb_error(1323,"Extra command line arguments ignored.\n",WARNING);


  /* command sources stack */
#if defined (NeXTapp) || defined(MOTIF)
  if ( tty_flag && no_Next_graph ) 
      push_commandfd(stdin,"stdin");
#else
  push_commandfd(stdin,"stdin");
#endif

  if ( cmdfilename )    /* trap back to here if error and skip cmdfilename */
  { if ( !setjmp(jumpbuf) )    
          push_commandfd(NULL,cmdfilename);
  }

  if ( setjmp(loadjumpbuf) )
  { /* LOAD command returns here */
    if ( list && (list != permlist)) 
    { myfree((char*)list); list = NULL; }/* plug memory leak */
    startup(loadfilename);
  }
  else
  {
#ifndef NeXTapp
  if ( setjmp(jumpbuf) )    /* return here after datafile errors */
     { 
        startup(NULL);
     }
  else
  if ( argc > 0 ) startup(argv[0]);
  else startup(NULL);
#else
  if ( argc > 0 ) startup(argv[0]);
#endif
  }
  datafile_flag = 0;

#ifdef NeXTapp
  if ( !no_Next_graph ) go_display_flag = 1;
#endif

#if !defined (NeXTapp) && !defined(MOTIF)
  while ( setjmp(jumpbuf) != 0 );    /* return here after commandfile  errors */
  exec_commands();    /* command read and execute loop */
  my_exit(0);
#else
  while ( setjmp(jumpbuf) != 0 );    /* return here after commandfile  errors */
  exec_commands();    /* command read and execute loop */
  outstring("Enter command: "); /* get initial prompt */
    /* fall through and return to outer event loop */
#endif

  return 0; /* success return code */
}

/********************************************************************
*
*  function: my_exit()
* 
*  purpose: graceful exit from program
*/

void my_exit(code)
int code;
{
  if ( OOGL_flag ) End_OOGL();

#ifdef SIGTERM
  signal(SIGTERM,SIG_DFL);
#endif

#ifdef SGI_MULTI
  if ( nprocs > 1)
     { m_rele_procs();
       m_kill_procs();  /* kill any parallel threads */
     }
#endif

#if defined(PVM)
  my_pvm_exit();
#endif

#if defined(MPI)
  my_pvm_exit();
#endif


  exit(code);
}

/********************************************************************
*
*  function: exec_commands()
*
*  purpose: reads and executes commands from input.
*              pops command file stack whenever end of file.
*/

void exec_commands()
{ /* main event loop of program */
  while ( commandfd )
  {
    char response[200];

    temp_free_all(); /* stray memory blocks */
    free_discards(); /* from previous cycle */
    memset(response,0,sizeof(response));
#ifdef NeXTapp
    if ( commandfd == NULL ) break;
#endif
    if ( prompt("Enter command: ",response,sizeof(response)) == EOF ) 
      pop_commandfd();
    else 
      old_menu(response);
  }
}

/********************************************************************
*
*  function: exec_file()
*
*  purpose: reads and executes commands from a file.
*              to be used by read "filename" command.
*              can be used in the middle of executing a command.
*/

void exec_file(fd,name)
FILE *fd;  /* file, if already opened, like stdin */
char *name; /* file name, if not already opened */
{ int old_read_depth = read_depth;
  push_commandfd(fd,name);
  do  /* main event loop of program */
  {
    char response[200];

    temp_free_all(); /* stray memory blocks */
    free_discards(); /* from previous cycle */
    memset(response,0,sizeof(response));
#ifdef NeXTapp
    if ( commandfd == NULL ) break;
#endif
    if ( prompt("Enter command: ",response,sizeof(response)) == EOF ) 
      pop_commandfd();
    else 
      old_menu(response);
  }
  while ( read_depth > old_read_depth ); 
}

/****************************************************************
*
* Function: startup()
*
* Purpose:  Start new datafile.
*
*****************************************************************/

void startup(file_name)
char *file_name;  /* NULL if need to ask for name */
{
  char *name = file_name;
  char response[100];
  FILE *newfd;

  datafile_flag = 0;
  /* close leftover input files */
  for ( ; read_depth > (cmdfilename ? 2 : 1) ; read_depth-- ) 
      fclose(cmdfile_stack[read_depth-1].fd);

#ifdef LISTOPEN
/* find which files open */
{
#include <sys/stat.h>
  int n;
  struct stat buf;
  for ( n = 0 ; n < 20 ; n++ )
  { int retval = fstat(n,&buf);
     if ( retval == 0 )
      printf("fildes %d: inode %d:%d  size: %d \n",n,buf.st_dev,buf.st_ino,
         buf.st_size);
  }
}
#endif

file_retry:
  if ( name == NULL )
     { char *c;
        prompt("Enter new datafile name (none to continue, q to quit): ",response,sizeof(response));
        c = strchr(response,'\n');
        if ( c ) *c = 0;
        if ( (strcmp(response,"q") == 0) || (strcmp(response,"quit")==0)
              || (strcmp(response,"bye")==0) || (strcmp(response,"exit")==0))
             my_exit(0);
        else if ( !response[0] ) return; /* continue same */
        name = response;
     }
  newfd = path_open(name);
  if (newfd == NULL)
  { if ( name[0] )
     { sprintf(msg,"Cannot open datafile %s.\n",name);
        erroutstring(msg);
        if ( exit_after_error ) my_exit(1);
        name = NULL;
        while ( read_depth > (cmdfilename ? 2 : 1)  ) pop_commandfd(); 
        goto file_retry;
     }
     return; /* continue with old */
  }
  reset_web(); 
if (memdebug) memory_report();
  strncpy(datafilename,name,sizeof(datafilename));
  push_commandfd(newfd,name); /* start #include stack */

#ifdef __WIN32__
if ( heapcheck() < 0 )
  kb_error(1324,"Internal error: Corrupt heap.\n",UNRECOVERABLE);

#endif

  datafile_flag = 1;  /* so parser knows */
  cmdptr = 0;
  initialize();
  reset_view();  
  datafile_flag = 0;
  if ( read_depth > 0 )
        cmdfile_stack[read_depth-1].line = line_no;

  if ( datafile_view_flag )
    { int i,j;
      REAL sum;

      for ( sum = 0.0, i = 0 ; i < 3 ; i++ )
         for ( j = 0 ; j < 3 ; j++ ) sum += fabs(view[i][j]);
      overall_size = 3/sum;
      
    }
  else
    { 
      resize();
    }

  if ( parse_errors ) 
  { while ( (read_depth > 1) ) pop_commandfd();
     kb_error(1325,"Invalid datafile.  Surface may not be in a consistent state.\n",RECOVERABLE);
  }

  run_checks();
  calc_content(Q_FIXED|Q_INFO|Q_RENORMALIZE);
  if ( web.torus_flag ) fix_volconst();
  calc_pressure();
  calc_energy();  /* just to get initial total area */
  target_length = web.total_area; /* for square curvature string model */
  if ( OOGL_flag ) ask_wrap_display();
  update_display();
}

#ifdef __convex_SPP
/*****************************************************************************
*
* function: m_fork(), Convex Exemplar version
*
* purpose: spawn parallel tasks
*
*/

void m_fork(void (*func)(), int arg1, int arg2)
{ int i;
#pragma _CNX loop_parallel(ivar=i)
  for ( i = 0 ; i < nprocs ; i++ )
     (*func)(arg1,arg2);

}
#endif

