/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/**********************************************************************
*
*  File: tiex.c
*/

#include "include.h"

/********************************************************************
*
*  Function: curvature_forces()
*
*  Purpose:  Forces as function of curvatures.
*            Finds mean and gaussian curvature at each vertex,
*            and normal.
*
*/

struct gvert { REAL angle;
               REAL area;
               REAL star_area;
               REAL normal[MAXCOORD];
               REAL force[MAXCOORD]; /* for mean curvature */
             };
static struct gvert  *gverts;

void curvature_forces_init(mode,mi)
int mode;
struct method_instance *mi;
{ 
  vertex_id v[3];
  edge_id e_id;
  facet_id f_id;
  facetedge_id fe;
  int fixcount;
  int i,j;
  REAL area;
  REAL side[3][MAXCOORD];
  REAL ss[3];
  REAL st[3];
  REAL angle;
  REAL c;
  struct gvert  *gv,*vc[3];
  REAL normal[MAXCOORD];

  if ( gverts ) myfree((char*)gverts);
  gverts = (struct gvert*)mycalloc(web.skel[VERTEX].max_ord+1,
                                            sizeof(struct gvert));

  /* accumulate angles and stuff around vertices */
  FOR_ALL_FACETS(f_id)
    { fe = get_facet_fe(f_id);
      fixcount = 0;
      for ( i = 0; i < FACET_VERTS ; i++,fe=get_next_edge(fe) )
        { e_id = get_fe_edge(fe);
          get_edge_side(e_id,side[i]);
          v[i] = get_edge_tailv(e_id);
          if ( get_vattr(v[i]) & (FIXED|BOUNDARY) ) fixcount++;
        }
        for ( i = 0 ; i < FACET_VERTS ; i++ )
         { ss[i] = SDIM_dot(side[i],side[i]);
            st[i] = SDIM_dot(side[i],side[(i+2)%3]);
         }
        area = 0.5*sqrt(ss[0]*ss[1]-st[1]*st[1]);
        cross_prod(side[0],side[1],normal);
        for ( i = 0 ; i < FACET_VERTS ; i++ )
        { REAL dd = ss[i]*ss[(i+2)%3];
          gv = gverts + ordinal(v[i]);
          if ( dd > 0.0 )
          { c = -st[i]/sqrt(ss[i]*ss[(i+2)%3]);
             if ( fabs(c) <= 1.0 )
             { angle = acos(c);
                gv->angle += angle;
             }
          } 
          gv->area  += area/3;
          gv->star_area  += area/(3-fixcount);
          for ( j = 0 ; j < SDIM ; j++ )
            gv->normal[j] += normal[j];
        }

        /* for mean curvature */

        for ( i = 0 ; i < FACET_VERTS ; i++ )
          vc[i] = gverts + ordinal(v[i]);

        for ( i = 0 ; i < SDIM ; i++ )
         { 
            vc[0]->force[i] += (ss[1]*side[2][i]-st[2]*side[1][i])/4/area;
            vc[1]->force[i] += (ss[2]*side[0][i]-st[0]*side[2][i])/4/area;
            vc[2]->force[i] += (ss[0]*side[1][i]-st[1]*side[0][i])/4/area;
         }
    }
}
 
REAL curvature_forces_energy(v_info)
struct qinfo *v_info;
{ return 0.0; /* fake it */
}

REAL curvature_forces(v_info)
struct qinfo *v_info;
{ REAL K; /* gauss curvature of vertex */
  REAL H; /* mean curvature of vertex, average of sectional curvatures */
  struct gvert *vg = gverts + ordinal(v_info->id);
  REAL norm;
  REAL f; /* function of H and K */
  int i;

  if ( get_vattr(v_info->id) & (FIXED|BOUNDARY) ) return 0.0;

  /* normalize normal */
  norm = sqrt(SDIM_dot(vg->normal,vg->normal));
  for ( i = 0 ; i < SDIM ; i++ )
     vg->normal[i] /= norm;

  K = (2*M_PI - vg->angle)/vg->area;

  H = SDIM_dot(vg->force,vg->normal)/2/vg->area;

  /* fix here for function of H and K */
  f =  (H>0.0) ? exp(K) : -exp(K);    /* to detect misorientation */
  /* end of fixing */

  /* force as multiple of normal */
  for ( i = 0 ; i < SDIM ; i++ )
     v_info->grad[0][i] = -f*vg->normal[i];
 
  return 0.0;  /* fake energy */
}
