/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/


/**********************************************************
*
*  File: query.c
*
*  Contents:  query interpreter for Surface Evolver
*/

#include "include.h"
#include "ytab.h"

int commandverb;

/***********************************************************************
  History mechanism.
  Command texts are stored in linear array history_space[], separated by
  nulls, most recent first.  history_list[] contains indices of starts
  of commands, -1 for no command.
  history_number is permanent counter of commands.
  history_count is number of currently saved commands.
**************************************************************************/


/**************************************************************************
*  
* function: new_history()
*
* purpose:  Appends command to history list, queue fashion.
*/

void new_history(text)
char *text;
{ int length;
  int k;

  if ( history_space == NULL )
    { history_space = mycalloc(HISTORYSPACE,1);
      for ( k = 0 ; k < MAXHISTORY ; k++ ) history_list[k] = -1; /* empty */
      history_count = 0;
    }
  length = strlen(text)+1;
  if ( length >= HISTORYSPACE-5 ) 
     { kb_error(1585,"Command too long to save in history list.\n",WARNING);

        return;
     }
  kb_memmove(history_space+length,history_space,HISTORYSPACE-length);
  for ( k = history_count-1 ; k >= 0 ; k-- )
    { /* adjust pointers in history list */
      history_list[k+1] = history_list[k] + length;
      if ( history_list[k+1] >= HISTORYSPACE )
         { history_list[k+1] = -1; history_count--; }
    }
  strcpy(history_space,text);
  history_list[0] = 0;
  if ( history_count < MAXHISTORY-1 ) history_count++;
  history_number++;
}

/**************************************************************************
*  
* function: old_history()
*
* purpose:  Checks commands beginning with ! for history list match.
*/

void old_history(text)
char *text;
{ char *h;
  int k;
  int hnum;

  if ( history_space == NULL )
    { history_space = mycalloc(HISTORYSPACE,1);
      for ( k = 0 ; k < MAXHISTORY ; k++ ) history_list[k] = -1; /* empty */
      history_count = 0;
    }

  if (text[1] == '!' ) 
    {
      if ( history_count > 0 ) 
      { k = 0; goto do_old;
      }
      else { kb_error(1586,"History list is empty.\n",WARNING); return; }

    }
  else if ( isdigit(text[1]) )
     { hnum = atoi(text+1);
       k = history_number - hnum;
       if ( (k >= 0) && (k < history_count) )
          goto do_old;
       else kb_error(1587,"Command not in history list.\n",RECOVERABLE);
     }
  else for ( k = 0 ; k < history_count ; k++ )
     { h = history_space+history_list[k]; 
       if ( strlen(h) < strlen(text+1) ) continue; /* prevent compare overrun */
       if ( strncmp(h,text+1,strlen(text+1)) == 0 )
          goto do_old;
     }
  kb_error(1588,"Command not found in history.\n",RECOVERABLE);


do_old:
  h = history_space + history_list[k];
  outstring(h);outstring("\n");
  command(h);  /* reinserts fulltext copy in history list */
  return;
}

/**************************************************************************
*  
* function: catfulltext()
*
* purpose:  catenates line to full text of command
*/

void catfulltext(stuff)
char *stuff;
{

  if ( fulltextsize > MAXCMDSIZE-3 ) return; /* too long */
  if ( fulltextsize > 0 ) 
     { strcat(fulltext,"  "); /* indentation */
        fulltextsize += 2;
     }
  strncpy(fulltext+fulltextsize,stuff,MAXCMDSIZE-fulltextsize);
  fulltextsize = strlen(fulltext);
  if ( fulltext[fulltextsize-1] != '\n' ) fulltext[fulltextsize++] = '\n';
  fulltext[fulltextsize] = '\0';

}

/**************************************************************************
*  
* function: command()
*
* purpose:  Executes a text command string.
*/

int command(text)
char *text;
{
  struct expnode qnode;  /* for query expression */
  int retval;
  int old_scope;

  qnode.root = NULL;
  qnode.start = NULL;

  listmax = 30;
  list = (struct treenode *)mycalloc(listmax,sizeof(struct treenode));
  listtop = 1;
  loopdepth = 0;
  parse_error_flag = 0;
  lists_flag = 0;
  cmdptr = text;
  old_scope = begin_scope();
  if ( setjmp(cmdbuf) )
  { set_scope(old_scope);  
    cmdptr = NULL;  
    myfree((char *)list); 
    list = NULL;
    return 0; 
  }
  verb_flag = 1; /* tell lex we need a verb first */
  old_global_count = web.global_count; /* for error recovery */
  fulltextsize = 0; 
  catfulltext(text);
  retval = yybegin();  /* 0 for accept, 1 for error */
  cmdptr = NULL;

  if ( (retval == 1) || (parse_error_flag) )
  { myfree((char *)list); list = NULL; set_scope(old_scope); return 0; }

  qnode.start=list;
  qnode.root = list + listtop - 1;
  if ( qnode.root->type == 0 ) /* empty command */
  { myfree((char *)qnode.start); list = NULL; set_scope(old_scope); return 0; }
  if ( qnode.root->type != CMDLIST_ )  /* expression or something */
     kb_error(1589,"Illegal command.\n",COMMAND_ERROR);


  list[0] = list[listtop-1];  /* root also in first spot */
  if ( list[0].left ) list[0].left += listtop - 1;
  if ( list[0].right ) list[0].right += listtop - 1;

  /* put DONE marker after root */
  list[listtop++].type = FINISHED;

  if ( logfd )
  fprintf(logfd,"%s\n",text);
  
  /* add to history list */
  if ( strlen(fulltext) < MAXCMDSIZE-4 ) 
    new_history(fulltext);

  /* initialize current element to NULLID */
  aggregate_depth = 0; calc_quant_flag = 0;
  eval(&qnode,NULL,NULLID);

  qnode.flag = USERCOPY; /* so free_expr will work */
  free_expr(&qnode); list = NULL;
  set_scope(old_scope);
  return 1;
}

