/*
 *   (C) Copyright IBM Corp. 2006
 *   (C) Copyright ALT Linux Ltd. 2006
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: FAT FSIM
 * File: evms2/engine/plugins/fat/utils.c
 */

#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/wait.h>
#include <plugin.h>

#include "utils.h"
#include "fatfsim.h"

#define LLONG_MAX	0x8fffffffffffffffLL
#define FAT_BLOCK_SIZE	512

/*
 * FIXME: correct values
 */
#define MAXVOL_FAT16	(1024*1024*4)
#define MAXVOL_FAT32	INT32_MAX

/*
 * is_boot_sector_fat() checks whether the buffer at boot_sector is a valid
 * FAT boot sector. Returns TRUE if it is valid and FALSE if not.
 */
static boolean is_boot_sector_fat(const fat_boot_sector *boot)
{
	LOG_ENTRY();

	if (!strncmp((char *)boot->system_id, "MSDOS", 5) ||
	    !strncmp((char *)boot->system_id, "MSWIN", 5) ||
	    !strncmp((char *)boot->system_id, "MTOOL", 5) ||
	    !strncmp((char *)boot->system_id, "mkdosfs", 7) ||
	    !strncmp((char *)boot->system_id, "kmkdosfs", 8) ||
	    /* Michal Svec: created by fdformat, old msdos utility for
	     * formatting large (1.7) floppy disks.
	     */
	    !strncmp((char *)boot->system_id, "CH-FOR18", 8) ||
	    !strncmp((char *)boot->fat.vi.fs, "FAT12   ", 8) ||
	    !strncmp((char *)boot->fat.vi.fs, "FAT16   ", 8) ||
	    !strncmp((char *)boot->fat32.vi.fs, "FAT32   ", 8)) {
		LOG_EXIT_BOOL(TRUE);
		return TRUE;
	}

	LOG_EXIT_BOOL(FALSE);
	return FALSE;
}

int fill_private_data(int isfat16,
		      logical_volume_t *vol,
		      const fat_boot_sector *boot)
{
	private_data_t *pd = vol->private_data;
	u_int16_t sectors;

	LOG_ENTRY();

	pd->logical_sector_size = DISK_TO_CPU16(boot->sector_size);
	if (!pd->logical_sector_size) {
		LOG_DETAILS("Logical sector size is zero.");
	}
	pd->cluster_size = boot->cluster_size * pd->logical_sector_size;
	if (!pd->cluster_size) {
		LOG_DETAILS("Cluster size is zero.");
	}
	if (boot->fats != 2 && boot->fats != 1) {
		LOG_DETAILS("Currently, only 1 or 2 FATs are supported, "
			    "not %d.\n", boot->fats);
	}
	sectors = DISK_TO_CPU16(boot->sectors);
	pd->fs_size = sectors ? sectors : DISK_TO_CPU32(boot->total_sect);
	
	/* FIXME: make correct MAX VOLUME size. */
	pd->max_vol_size = isfat16 ? MAXVOL_FAT16 : MAXVOL_FAT32;
	pd->max_fs_size = pd->fs_size;

	pd->vol_id = isfat16 ? DISK_TO_CPU32(boot->fat.vi.volume_id)
			     : DISK_TO_CPU32(boot->fat32.vi.volume_id);

	pd->vol_name = isfat16 ? EngFncs->engine_strdup((char *)boot->fat.vi.volume_label)
			       : EngFncs->engine_strdup((char *)boot->fat32.vi.volume_label);

	LOG_EXIT_INT(0);
	return 0;
}

/*
 * Read the boot sector from the volume and validate it.
 */
int get_fat_boot(int isfat16, logical_volume_t *ev, fat_boot_sector *boot)
{
	int rc = EINVAL;
	int fd;

	if (boot == NULL) {
		LOG_CRITICAL("Failed to allocate memory for a boot info.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	fd = EngFncs->open_volume(ev, O_RDONLY);
	if (fd < 0) {
		rc = -fd;
		LOG_SERIOUS("Failed to open volume %s. Error code is %d: %s\n",
			    ev->name, rc, EngFncs->strerror(rc));
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* Try to read primary boot sector. */
	rc = EngFncs->read_volume(ev, fd, boot, FAT_BLOCK_SIZE, 0);
	if (rc == FAT_BLOCK_SIZE) {
		if (is_boot_sector_fat(boot)) {
			if (isfat16 ||
			    (!boot->fat_length && boot->fat32.length)) {
				rc = 0;
			}
		} else {
			LOG_DETAILS("Primary boot sector is not valid.");
		}
	} else {
		LOG_WARNING("Unable to read primary boot sector.");
	}

	EngFncs->close_volume(ev, fd);
	/* FIXME: trim space chars */
	if (isfat16) {
		boot->fat.vi.volume_label[11] = '\0';
	} else {
		boot->fat32.vi.volume_label[11] = '\0';
	}

	LOG_EXIT_INT(rc);
	return rc;
}


int clear_fat_boot_sectors(logical_volume_t *ev)
{
	int rc = 0;
	int fd;
	void *block;
	int32_t bytes_written;

	LOG_ENTRY();

	block = EngFncs->engine_alloc(FAT_BLOCK_SIZE);
	if (block == NULL) {
		LOG_CRITICAL("Can't get a buffer for writing.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	fd = EngFncs->open_volume(ev, O_WRONLY);
	if (fd < 0) {
		rc = -fd;
		LOG_SERIOUS("Failed to open volume %s. Error code is %d: %s\n",
			    ev->name, rc, EngFncs->strerror(rc));
		EngFncs->engine_free(block);
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* Wipe out the primary boot sector. */
	bytes_written = EngFncs->write_volume(ev, fd, block, FAT_BLOCK_SIZE, 0);
	if (bytes_written != FAT_BLOCK_SIZE) {
		LOG_WARNING("Failed to clear the primary boot sector on "
			    "volume %s.\n", ev->name);
		rc = -bytes_written;
	}

	EngFncs->close_volume(ev, fd);
	EngFncs->engine_free(block);

	LOG_EXIT_INT(rc);
	return rc;
}


int try_run(char *prog_name)
{
	int rc;
	char *argv[3];
	pid_t pidm;
	int status;
	int fds[2];

	/* Open a pipe to catch the program output that we don't care about. */
	rc = pipe(fds);
	if (rc) {
		rc = errno;
		LOG_SERIOUS("Could not opening a pipe. Error code is %d: %s\n",
			    rc, strerror(rc));
		LOG_EXIT_INT(rc);
		return rc;
	}

	argv[0] = prog_name;
	argv[1] = NULL;

	pidm = EngFncs->fork_and_execvp(NULL, argv, NULL, NULL, NULL);
	if (pidm != -1) {
		waitpid(pidm, &status, 0);
		if (WIFEXITED(status)) {
			LOG_DEFAULT("%s completed with exit code %d.\n",
				    prog_name, WEXITSTATUS(status));
		} else {
			LOG_WARNING("%s did not exit normally.\n", prog_name);
			rc = EINTR;
		}
	} else {
		rc = errno;
		LOG_DEFAULT("Unable to run %s. Error code is %d: %s\n",
			    prog_name, rc, EngFncs->strerror(rc));
	}

	close(fds[0]);
	close(fds[1]);

	LOG_EXIT_INT(rc);
	return rc;
}


int get_fs_limits(logical_volume_t *vol,
		  sector_count_t *min_fs_size,
		  sector_count_t *max_fs_size,
		  sector_count_t *max_vol_size)
{
	int rc = 0;
	char *argv[5];
	pid_t pidm;
	int status;
	int fds[2];
	char *buffer = NULL;
	char *p = NULL;
	u_int64_t min_fs_size_bytes = 0;
	u_int64_t max_fs_size_bytes = 0;

	LOG_ENTRY();

	if (!have_fatresize) {
		LOG_DETAILS("The fatresize utility is not installed.\n");
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	*min_fs_size = vol->fs_size;
	*max_fs_size = vol->fs_size;
	*max_vol_size = vol->fs_size;

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (buffer == NULL) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	status = pipe(fds);
	if (status < 0) {
		EngFncs->engine_free(buffer);
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	/* Run fatresize -i to get the smallest volume size. */
	argv[0] = "fatresize";
	argv[1] = "-i";
	argv[2] = vol->dev_node;
	argv[3] = NULL;

	fcntl(fds[0], F_SETFL, fcntl(fds[0], F_GETFL, 0) | O_NONBLOCK);
	fcntl(fds[1], F_SETFL, fcntl(fds[1], F_GETFL, 0) | O_NONBLOCK);

	pidm = EngFncs->fork_and_execvp(vol, argv, NULL, fds, fds);
	if (pidm != -1) {
		waitpid(pidm, &status, 0);
		if (WIFEXITED(status)) {
			read(fds[0], buffer, MAX_USER_MESSAGE_LEN);
			rc = WEXITSTATUS(status);
			LOG_DETAILS("%s completed with exit code %d \n", argv[0], rc);
		} else {
			/*
			 * The process didn't exit. It must have been
			 * interrupted by a signal.
			 */
			rc = EINTR;
		}
	} else {
		LOG_SERIOUS("Failed to fork and exec %s.  Error code is %d: %s\n",
			    argv[0], rc, EngFncs->strerror(rc));
		rc = errno;
	}

	if (rc == 0) {
		if ((p = strstr(buffer, "Min size:"))) {
			p += 10;
			min_fs_size_bytes = atoll(p);
		}
		if ((p = strstr(buffer, "Max size:"))) {
			p += 10;
			max_fs_size_bytes = atoll(p);
		}
		if (min_fs_size_bytes != 0) {
			/* fatresize reports the size in bytes. */
			*min_fs_size = min_fs_size_bytes >> EVMS_VSECTOR_SIZE_SHIFT;
		}
		if (max_fs_size_bytes != 0) {
			/* fatresize reports the size in bytes. */
			*max_fs_size = max_fs_size_bytes >> EVMS_VSECTOR_SIZE_SHIFT;
			*max_vol_size = *max_fs_size;
		}
	}

	EngFncs->engine_free(buffer);
	
	close(fds[0]);
	close(fds[1]);

	LOG_EXIT_INT(rc);
	return rc;
}

void free_private_data(logical_volume_t *vol)
{
	private_data_t *pd = vol->private_data;

	LOG_ENTRY();

	if (pd != NULL) {
		if (pd->vol_name != NULL) {
			EngFncs->engine_free(pd->vol_name);
		}
		EngFncs->engine_free(pd);
		vol->private_data = NULL;
	}

	LOG_EXIT_VOID();
}

int resize_fat(int isfat16, logical_volume_t *volume, sector_count_t *new_size)
{
	int rc = 0;
	char *argv[6];
	char ascii_new_size[16];
	pid_t pidm;
	int status;
	int fds[2];
	char *buffer;
	fat_boot_sector *boot;
	private_data_t *pd = volume->private_data;

	LOG_ENTRY();

	if (!have_fatresize) {
		MESSAGE("The fatresize utility is not installed on this "
			"machine. The FAT16/FAT32 FSIM uses fatresize to "
			"expand the FAT16/FAT32 file system on the volume. "
			"Get the latest version of the FAT utilities from "
			"http://sourceforge.net/projects/fatresize/\n");
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (buffer == NULL) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	status = pipe(fds);
	if (status < 0) {
		rc = errno;
		EngFncs->engine_free(buffer);
		LOG_EXIT_INT(rc);
		return rc;
	}

	argv[0] = "fatresize";
	argv[1] = "-vvv";
	argv[2] = "-s";

	/* Get the ASCII version of the new size in KB. */
	sprintf(ascii_new_size, "%"PRIu64, (*new_size) >> 1);
	strcat(ascii_new_size, "k");
	argv[3] = ascii_new_size;

	argv[4] = volume->dev_node;
	argv[5] = NULL;

	fcntl(fds[0], F_SETFL, fcntl(fds[0], F_GETFL, 0) | O_NONBLOCK);
	fcntl(fds[1], F_SETFL, fcntl(fds[1], F_GETFL, 0) | O_NONBLOCK);

	pidm = EngFncs->fork_and_execvp(volume, argv, NULL, fds, fds);
	if (pidm != -1) {
		waitpid(pidm, &status, 0);
		if (WIFEXITED(status)) {
			rc = WEXITSTATUS(status);
			if (rc == 0) {
				LOG_DETAILS("%s completed with exit code %d \n",
					    argv[0], rc);
			} else {
				LOG_ERROR("%s completed with exit code %d \n",
					  argv[0], rc);
			}
		} else {
			/* The process didn't exit. It must have been
			 * interrupted by a signal.
			 */
			rc = EINTR;
		}
	} else {
		rc = errno;
		LOG_SERIOUS("Failed to fork and exec %s. Error code is %d: %s\n",
			    argv[0], rc, EngFncs->strerror(rc));
	}

	close(fds[0]);
	close(fds[1]);

	EngFncs->engine_free(buffer);
	
	if (rc == 0) {
		/* Get the new sizes into the private data. */
		memset(pd, 0, sizeof(*pd));

		boot = EngFncs->engine_alloc(sizeof(fat_boot_sector));
		if (boot == NULL) {
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		rc = get_fat_boot(isfat16, volume, boot);
		if (rc == 0) {
			rc = fill_private_data(isfat16, volume, boot);
		}
		*new_size = pd->fs_size;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

