/*
 *   (C) Copyright IBM Corp. 2006
 *   (C) Copyright Novell, Inc. 2006 All Rights Reserved
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of version 2 of the GNU General Public License as
 *   published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: OCFS2 FSIM
 * File: evms2/engine/plugins/ocfs2/ocfs2.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/wait.h>

#include <plugin.h>

#include "ocfs2.h"

/*****************************************************************************
 * functions prototypes
 ****************************************************************************/
static int fsim_rw_diskblocks(logical_volume_t *volume, int dev_ptr,
			      u_int64_t disk_offset, int32_t disk_count,
			      void *data_buffer, int mode);
static void fsim_set_mkfs_options(option_array_t *options,
				  char **argv, char *vol_name);
static void fsim_set_fsck_options(option_array_t *options, char **argv,
				  logical_volume_t *volume);

/*****************************************************************************
 * global variables
 ****************************************************************************/
plugin_record_t *my_plugin_record = &oc2_plugin_record;
engine_functions_t *EngFncs = NULL;

/*****************************************************************************
 * variables needed across functions
 ****************************************************************************/
static int oc2_version = 0;	/* version of oc2fs utils. */
static char oc2_version_text[32];  /* version of oc2fs utils. */

/*
 *--------------------------------------------------------------------------
 *   Support Routines
 *--------------------------------------------------------------------------
 */

/******************************************************************************
 * NAME: fs_init_mkfs_acceptable_objects
 *
 * FUNCTION:
 * 	Initialize mkfs task acceptable objects by enumerating volumes, finding
 * 	those that have no FSIM claiming them and are of the proper size and
 * 	adding them to the acceptable objects list.
 *
 * PARAMETERS:
 *	context
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int fs_init_mkfs_acceptable_objects(task_context_t *context)
{
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;
	logical_volume_t *volume;
	int rc;

	LOG_ENTRY();

	rc = EngFncs->get_volume_list(NULL, NULL, 0, &global_volumes);
	if (!rc) {
		LIST_FOR_EACH(global_volumes, vol_list_iter, volume) {
			/* Only allow mkfs on unformatted volumes. */
			if (!volume->file_system_manager &&
			    !EngFncs->is_mounted(volume->dev_node, NULL) &&
			    volume->vol_size > MINOCFS2) {
				EngFncs->insert_thing(context->acceptable_objects,
						      volume, INSERT_BEFORE, NULL);
			}
		}
		EngFncs->destroy_list(global_volumes);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/******************************************************************************
 * NAME: fsim_get_volume_limits
 *
 * FUNCTION:
 * 	Get the size limits for this volume.
 *
 * PARAMETERS:
 *	sb_ptr
 *	min_size
 *	max_volume_size
 *	max_object_size
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int fsim_get_volume_limits(struct _ocfs2_dinode *sb_ptr,
				  sector_count_t *min_size,
				  sector_count_t *max_volume_size,
				  sector_count_t *max_object_size)
{
	LOG_ENTRY();

	/* 32bits of 4k blocks */
	*max_volume_size = (sector_count_t)0xffffffff * (sector_count_t)8;
	*max_object_size = (sector_count_t)0xffffffff * (sector_count_t)8;
	*min_size = 16 * (1 << sb_ptr->id2.i_super.s_clustersize_bits);

	LOG_EXIT_INT(0);
	return 0;
}

/******************************************************************************
 * NAME: free_args
 *
 * FUNCTION: free args
 *
 * PARAMETERS:
 *	argv
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static inline void free_args(char **argv)
{
	int i;

	for (i = 0; argv[i] != NULL; i++) {
		EngFncs->engine_free(argv[i]);
	}
}

/******************************************************************************
 * NAME: fsim_mkfs
 *
 * FUNCTION: Format the volume
 *
 * PARAMETERS:
 *	volume
 *	options
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int fsim_mkfs(logical_volume_t *volume, option_array_t *options)
{
	char *argv[MKFS_OPTIONS_COUNT + 7]; /* NULL-terminated array. */
	char *buffer;
	pid_t pidm;
	int fds2[2];  /* pipe for stderr and stdout 0=read, 1=write */
	int bytes_read;
	int status;
	int rc;

	LOG_ENTRY();

	rc = pipe(fds2);
	if (rc) {
		rc = errno;
		goto out1;
	}

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (!buffer) {
		rc = ENOMEM;
		goto out2;
	}

	fsim_set_mkfs_options(options, argv, volume->dev_node);

	pidm = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidm == -1) {
		rc = EIO;
		goto out3;
	}

	fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL, 0) | O_NONBLOCK);

       	while (!waitpid(pidm, &status, WNOHANG)) {
		bytes_read = read(fds2[0], buffer, MAX_USER_MESSAGE_LEN);
		if (bytes_read > 0) {
			LOG_DEFAULT("mkfs output:\n%s", buffer);
			/* clear out message */
			memset(buffer, 0, bytes_read);
		}
		/* Don't hog all the cpu. */
		usleep(10000);
	}

	if (!WIFEXITED(status) || WEXITSTATUS(status) == ENOENT) {
		rc = EINTR;
		goto out3;
	}

	do {
		bytes_read = read(fds2[0], buffer, MAX_USER_MESSAGE_LEN);
		if (bytes_read > 0) {
			LOG_DEFAULT("mkfs output:\n%s", buffer);
		}
	} while (bytes_read > 0);

	LOG_DEFAULT("mkfs completed with rc = %d \n", status);
	rc = WEXITSTATUS(status);

out3:
	free_args(argv);
	EngFncs->engine_free(buffer);

out2:
	close(fds2[0]);
	close(fds2[1]);

out1:
	LOG_EXIT_INT(rc);
	return rc;
}

/******************************************************************************
 * NAME: fsim_set_mkfs_options
 *
 * FUNCTION: Build options array (argv) for mkfs.ocfs2
 *
 * PARAMETERS:
 *      options   - options array passed from EVMS engine
 *      argv      - mkfs options array
 *      vol_name  - volume name on which program will be executed
 *
 * RETURNS:
 *	none
 *
 *****************************************************************************/
static void fsim_set_mkfs_options(option_array_t *options,
				  char **argv,
				  char *vol_name)
{
	int opt_count = 0;
	int i;

	LOG_ENTRY();

	argv[opt_count++] = EngFncs->engine_strdup("mkfs.ocfs2");

	for (i = 0; i < options->count; i++) {
		if (options->option[i].number == MKFS_SETVOL_INDEX) {
			if (options->option[i].value.s) {
				argv[opt_count++] = EngFncs->engine_strdup("-L");
				argv[opt_count++] = EngFncs->engine_strdup(options->option[i].value.s);
			}
		}
	}

	argv[opt_count++] = EngFncs->engine_strdup("-F");
	argv[opt_count++] = EngFncs->engine_strdup("-x");
	argv[opt_count++] = EngFncs->engine_strdup(vol_name);
	argv[opt_count] = NULL;

	LOG_EXIT_VOID();
	return;
}


/******************************************************************************
 * NAME: fsim_fsck
 *
 * FUNCTION: Run fsck on the volume
 *
 * PARAMETERS:
 *      options   - options array passed from EVMS engine
 *      vol_name  - volume on which program will be executed
 *
 * RETURNS:
 *	0 = SUCCESSFUL
 *
 *****************************************************************************/
static int fsim_fsck(logical_volume_t *volume, option_array_t *options)
{
	char *argv[FSCK_OPTIONS_COUNT + 7];
	char *buffer;
	pid_t pidf;
	int status;
	int fds1[2];  /* Pipe for stdin 0=read 1=write */
	int fds2[2];  /* Pipe for stderr and stdout 0=-read, 1=write */
	int bytes_read;
	int rc;

	LOG_ENTRY();

	rc = pipe(fds1);
	if (rc) {
		rc = errno;
		goto out1;
	}

	rc = pipe(fds2);
	if (rc) {
		rc = errno;
		goto out2;
	}

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (!buffer) {
		rc = ENOMEM;
		goto out3;
	}

	fsim_set_fsck_options(options, argv, volume);

	pidf = EngFncs->fork_and_execvp(volume, argv, fds1, fds2, fds2);
	if (pidf == -1) {
		rc = EIO;
		goto out4;
	}

	/* Fork and execute the correct program.
	 * WARNING: Do Not close read handle of stdin or
	 * you will cause a SIGPIPE if you write after the
	 * child process has gone away.
	 */
	write(fds1[1], "Yes\n", 4);

	/* Wait for child to complete */
	fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL, 0) | O_NONBLOCK);

	while (!waitpid(pidf, &status, WNOHANG)) {
		bytes_read = read(fds2[0], buffer, MAX_USER_MESSAGE_LEN);
		if (bytes_read > 0) {
			/* Send newline to take default
			 * on prompt if there is one.
			 */
			write(fds1[1], "\n", 1);
			MESSAGE(_("fsck output:\n%s"), buffer);
			/* Clear out message. */
			memset(buffer, 0, bytes_read);
		}
		/* Don't hog all the cpu. */
		usleep(10000);
	}

	if (!WIFEXITED(status) || WEXITSTATUS(status) == ENOENT) {
		rc = EINTR;
		goto out4;
	}

	do {
		bytes_read = read(fds2[0], buffer, MAX_USER_MESSAGE_LEN);
		if (bytes_read > 0) {
			MESSAGE(_("fsck output:\n\n%s"), buffer);
		}
	} while (bytes_read > 0);

	LOG_DEFAULT("fsck completed with rc = %d\n", status);
	rc = WEXITSTATUS(status);

out4:
	free_args(argv);
	EngFncs->engine_free(buffer);
out3:
	close(fds2[0]);
	close(fds2[1]);
out2:
	close(fds1[0]);
	close(fds1[1]);
out1:
	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: set_fsck_options
 *
 * FUNCTION: Build options array (argv) for oc2fsck
 *
 * PARAMETERS:
 *      options   - options array passed from EVMS engine
 *      argv      - fsck options array
 *      volume    - volume on which program will be executed
 *
 * RETURNS:
 *	argv
 *
 *****************************************************************************/
static void fsim_set_fsck_options(option_array_t *options,
				  char **argv,
				  logical_volume_t *volume)
{
	int opt_count = 0;

	LOG_ENTRY();

	argv[opt_count++] = EngFncs->engine_strdup("fsck.ocfs2");
	argv[opt_count++] = EngFncs->engine_strdup(volume->dev_node);
	argv[opt_count] = NULL;

	LOG_EXIT_VOID();
}

/******************************************************************************
 * NAME: fsim_get_dinode_block
 *
 * FUNCTION:
 *
 * PARAMETERS:
 *	volume
 *	fd
 * 	block
 *	b_ptr
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int fsim_get_dinode_block(logical_volume_t *volume,
				 int fd,
				 int block,
				 struct _ocfs2_dinode *b_ptr)
{
	int rc, offset;

	LOG_ENTRY();

	offset = block * 2048;
	rc = fsim_rw_diskblocks(volume, fd, offset, DINODE_SIZE, b_ptr, GET);

	LOG_EXIT_INT(rc);
	return rc;
}

/******************************************************************************
 * NAME: fsim_get_oc2_oc2fs_super_block
 *
 * FUNCTION: Get and validate a JFS oc2fs_super_block
 *
 * PARAMETERS:
 *      vol_name   - volume name from which to get the ocfs2_super_block
 *      sb_ptr     - pointer to ocfs2_super_block
 *
 * RETURNS:
 *      (0) for success
 *      != 0 otherwise
 *
 *****************************************************************************/
static int fsim_get_ocfs2_super_block(logical_volume_t *volume,
				      struct _ocfs2_dinode *sb_ptr)
{
	int count = 0;
	int rc = 1;
	int fd;

	LOG_ENTRY();

	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) {
		rc = -fd;
		LOG_EXIT_INT(rc);
		return rc;
	}

	while (count < 10) {
		rc = fsim_get_dinode_block(volume, fd, count, sb_ptr);
		if (!rc) {
			if (!strncmp((const char *)sb_ptr->h_signature, "OCFSV2", 8)) {
				LOG_DEBUG("OCFSV2 super block found\n");
				rc = 0;
				break;
			}
		}
		count++;
	}

	EngFncs->close_volume(volume, fd);
		
	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: fsim_checkif_ocfs_disk_hdr_block
 *
 * FUNCTION: Get 1st block on disk
 *
 * PARAMETERS:
 *      volname   - volume name from which to get the hdr_block
 *
 * RETURNS:
 *      (0) for success
 *      != 0 otherwise
 *
 *****************************************************************************/
static int fsim_checkif_ocfs_hdr_block(logical_volume_t *volume)
{
	struct _ocfs_vol_disk_hdr *sb_ptr;
	int rc, fd;

	LOG_ENTRY();

	sb_ptr = EngFncs->engine_alloc(HDR_SIZE);
	if (!sb_ptr) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) {
		rc = -fd;
		EngFncs->engine_free(sb_ptr);
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* Get 1st block.  */
	rc = fsim_rw_diskblocks(volume, fd, 0, HDR_SIZE, sb_ptr, GET);
	if (!rc) {
		/* See if 1st block belongs to OCFS2  */
		if (strncmp((const char *)sb_ptr->signature,
			    OCFS2_HDR_SIGNATURE, strlen(OCFS2_HDR_SIGNATURE))) {
			rc = EINVAL;
		}
	}

	EngFncs->close_volume(volume, fd);
	EngFncs->engine_free(sb_ptr);

	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: fsim_rw_diskblocks
 *
 * FUNCTION: Read or write specific number of bytes for an opened device.
 *
 * PARAMETERS:
 *      dev_ptr         - file handle of an opened device to read/write
 *      disk_offset     - byte offset from beginning of device for start of
 *                        disk block read/write
 *      disk_count      - number of bytes to read/write
 *      data_buffer     - On read this will be filled in with data read from
 *                        disk; on write this contains data to be written
 *      mode            - GET (read) or PUT (write)
 *
 * RETURNS:
 *      FSIM_SUCCESS (0) for success
 *      ERROR       (-1) can't lseek
 *      EINVAL
 *      EIO
 *
 *****************************************************************************/
static int fsim_rw_diskblocks(logical_volume_t *volume,
			      int dev_ptr,
			      u_int64_t disk_offset,
			      int32_t disk_count,
			      void *data_buffer,
			      int mode)
{
	int rc = 0;
	size_t bytes = 0;

	LOG_ENTRY();

	switch (mode) {
	case GET:
		bytes = EngFncs->read_volume(volume, dev_ptr, data_buffer,
					     disk_count, disk_offset);
		break;

	case PUT:
		bytes = EngFncs->write_volume(volume, dev_ptr, data_buffer,
					      disk_count, disk_offset);
		break;

	default:
		rc = EINVAL;
		break;
	}

	if (!rc && bytes != disk_count) {
		rc = EIO;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/******************************************************************************
 * NAME: fsim_test_version
 *
 * FUNCTION:
 * 	test mkfs.ocfs2 version.
 *  	It must be at version 1.1.2  or greater to function properly with this
 *	FSIM. The -V option is used to get the version.
 *    	If it succeeds, the installed version is OK.  If
 *    	it fails, pass back the exit code (should be EINVAL) of the failure.
 *
 * PARAMETERS:
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int fsim_test_version(void)
{
	char *argv[3];
	char *ver, *end;
	char *buffer;
	pid_t pidm;
	int fds2[2];  /* Pipe for stderr and stdout 0=read, 1=write */
	int status;
	int bytes_read;
	int rc;

	LOG_ENTRY();

	rc = pipe(fds2);
	if (rc) {
		rc = errno;
		goto out1;
	}

	buffer = EngFncs->engine_alloc(1024);
	if (!buffer) {
		rc = ENOMEM;
		goto out2;
	}

	argv[0] = "mkfs.ocfs2";
	argv[1] = "-V";
	argv[2] = NULL;

	pidm = EngFncs->fork_and_execvp(NULL, argv, NULL, fds2, fds2);
	if (pidm == -1) {
		rc = EIO;
		goto out3;
	}

	LOG_DEBUG("Waiting for process %d to exit.\n", pidm);
	fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL, 0) | O_NONBLOCK);
	waitpid(pidm, &status, 0);

	if (!WIFEXITED(status) || WEXITSTATUS(status) == ENOENT) {
		goto out3;
	}

	LOG_DEBUG("Reading output from process %d on fd %d.\n", pidm, fds2[0]);
	bytes_read = read(fds2[0], buffer, 1024);
	if (bytes_read > 0) {
		ver = strstr(buffer, "1.1");
		if (!ver) {
			rc = EINVAL;
		} else {
			oc2_version = 2;
			end = strstr(ver, "\n");
			if (end) {
				strncpy(oc2_version_text, ver, min(end-ver, 31));
			} else {
				strncpy(oc2_version_text, ver, 2);
			}
		}
	}

out3:
	EngFncs->engine_free(buffer);
out2:
	close(fds2[0]);
	close(fds2[1]);
out1:
	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 *                  Start Of EVMS Plugin Functions
 *            (exported to engine via function table)
 *****************************************************************************/


/******************************************************************************
 * NAME: oc2_setup
 *
 * FUNCTION: setup this plugin
 *
 * PARAMETERS:
 *	engine_function_table
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_setup(engine_functions_t *engine_function_table)
{
	int rc;

	EngFncs = engine_function_table;
	LOG_ENTRY();

	/* Run 'mkfs.ocfs2 -V' to get the version of the OCFS tools. */
	rc = fsim_test_version();

	if (oc2_version < 2) {
		LOG_DEBUG("OCFS2 file system tools not found.\n");
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: oc2_cleanup
 *
 * FUNCTION:  Exit the plugin, free up any private data
 *
 * PARAMETERS:
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static void oc2_cleanup(void)
{
	int rc;
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;
	logical_volume_t *volume;

	LOG_ENTRY();

	rc = EngFncs->get_volume_list(my_plugin_record, NULL,
				      0, &global_volumes);
	if (!rc) {
		LIST_FOR_EACH(global_volumes, vol_list_iter, volume) {
			EngFncs->engine_free(volume->private_data);
		}
		EngFncs->destroy_list(global_volumes);
	}

	LOG_EXIT_VOID();
}


/******************************************************************************
 * NAME: oc2_probe
 *
 * FUNCTION: Does this plugin manage the file system on this volume
 *		Return 0 for yes
 * PARAMETERS:
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_probe(logical_volume_t *volume)
{
	int rc;

	LOG_ENTRY();

	/* Get and validate oc2fs_super_block */
	LOG_DEBUG("Checking for OCFS2 on volume %s", volume->name);
	rc = fsim_checkif_ocfs_hdr_block(volume);

	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: ocs_can_mkfs
 *
 * FUNCTION: Can we make the OCFS2 file system on this volume
 *
 * PARAMETERS:
 *	volume
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_can_mkfs(logical_volume_t *volume)
{
	int rc = 0;

	LOG_ENTRY();

	/*****************************************************
	 *  FUTURE - ensure mkfs.oc2 exists                  *
	 *           match version with available functions  *
	 *****************************************************/

	if (oc2_version < 2) {
		LOG_DEBUG("OCFS2 file system tools not found.\n");
		rc = ENOSYS;
	}

	if (volume->vol_size < MINOCFS2) {
		LOG_DEBUG("Volume %s is smaller than minimum OCFS2 size.\n",
			  volume->name);
		rc = ENOSPC;
	}

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't format. */
		rc = EBUSY;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: oc2_can_unmkfs
 *
 * FUNCTION: Can we unmake the OCFS2 file system?
 *
 * PARAMETERS:
 *	volume
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_can_unmkfs(logical_volume_t *volume)
{
	int rc = 0;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: oc2_can_fsck
 *
 * FUNCTION: Can we do a file system check on this volume
 *
 * PARAMETERS:
 *	volume
 *	
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_can_fsck(logical_volume_t *volume)
{
	int rc = 0;

	LOG_ENTRY();

	if (oc2_version < 2) {
		LOG_DEBUG("OCFS2 file system tools not found.\n");
		rc = ENOSYS;
	}

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't fsck. */
		rc = EBUSY;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: oc2_get_fs_size
 *
 * FUNCTION: Get the current size of this volume
 *
 * PARAMETERS:
 *	volume
 *
 * RETURNS:
 *	size
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_get_fs_size(logical_volume_t *volume,
			   sector_count_t *size)
{
	int rc;
	struct _ocfs2_dinode *sb_ptr;

	LOG_ENTRY();

	sb_ptr = EngFncs->engine_alloc(DINODE_SIZE);
	if (!sb_ptr) {
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* Re-read super block and store in private data. */
	rc = fsim_get_ocfs2_super_block(volume, sb_ptr);
	if (rc) {
		EngFncs->engine_free(sb_ptr);
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* The size of the filesystem is the number of clusters
	 * times the clustersize divided by 512.
	 */
	*size = sb_ptr->i_clusters *
		(1 << sb_ptr->id2.i_super.s_clustersize_bits) /
		EVMS_VSECTOR_SIZE;

	LOG_DEBUG("minor_rev_level: %i\n", sb_ptr->id2.i_super.s_minor_rev_level);
	LOG_DEBUG("block_size_bits: %i\n", sb_ptr->id2.i_super.s_blocksize_bits);
	LOG_DEBUG("max_slots: %i\n", sb_ptr->id2.i_super.s_max_slots);
	LOG_DEBUG("clustersize_bits: %i\n", sb_ptr->id2.i_super.s_clustersize_bits);
	LOG_DEBUG("clusters: %i\n", sb_ptr->i_clusters);
	LOG_DEBUG("filesystem size: %"PRIu64"\n", *size);

	EngFncs->engine_free(sb_ptr);

	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: oc2_get_fs_limits
 *
 * FUNCTION: Get the size limits for this volume
 *
 * PARAMETERS:
 *	volume
 *	min_size
 * 	max_volume_size
 *	max_object_size
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_get_fs_limits(logical_volume_t *volume,
			     sector_count_t *min_size,
			     sector_count_t *max_volume_size,
			     sector_count_t *max_object_size)
{
	struct _ocfs2_dinode *sb_ptr;
	int rc;

	LOG_ENTRY();

	sb_ptr = EngFncs->engine_alloc(DINODE_SIZE);
	if (!sb_ptr) {
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/*
	 * Re-read superblock to set up private data
	 */
	rc = fsim_get_ocfs2_super_block(volume, sb_ptr);
	if (!rc) {
		rc = fsim_get_volume_limits(sb_ptr, min_size,
					    max_volume_size, max_object_size);
	}

	LOG_DEBUG("volume name: %s\n", volume->name);
	LOG_DEBUG("file size: %"PRIu64"\n", volume->fs_size);
	LOG_DEBUG("min_size: %"PRIu64"\n", *min_size);
	LOG_DEBUG("max_volume_size: %"PRIu64"\n", *max_volume_size);
	LOG_DEBUG("max_object_size: %"PRIu64"\n", *max_object_size);
	LOG_DEBUG("fssize: %"PRIu64"\n", volume->fs_size);
	LOG_DEBUG("vol_size: %"PRIu64"\n", volume->vol_size);

	EngFncs->engine_free(sb_ptr);

	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: oc2_unmkfs
 *
 * FUNCTION: unmkfs the volume
 *
 * PARAMETERS:
 *	volume
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_unmkfs(logical_volume_t *volume)
{
	struct ocfs_vol_disk_hdr *ocfs_hdr;
	int rc, fd;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	fd = EngFncs->open_volume(volume, O_RDWR|O_EXCL, 0);
	if (fd < 0) {
		rc = -fd;
		LOG_EXIT_INT(rc);
		return rc;
	}

	ocfs_hdr = EngFncs->engine_alloc(HDR_SIZE);
	if (!ocfs_hdr) {
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return rc;
	}

	memset(ocfs_hdr, 0, HDR_SIZE);

	rc = fsim_rw_diskblocks(volume, fd, 0, HDR_SIZE, ocfs_hdr, PUT);

	EngFncs->engine_free(ocfs_hdr);

	EngFncs->close_volume(volume, fd);

	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: oc2_mkfs
 *
 * FUNCTION: format the volume
 *
 * PARAMETERS:
 *	volume
 *	options
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_mkfs(logical_volume_t *volume, option_array_t *options)
{
	int rc = 0;

	LOG_ENTRY();

	if (oc2_version < 2) {
		MESSAGE(_("OCFS2 does not appear to be installed or has the "
			  "wrong version. The mkfs.ocfs2 utility must be "
			  "installed\n"));
		LOG_EXIT_INT(E_CANCELED);
		return E_CANCELED;
	}

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* Don't format if mounted. */
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	rc = fsim_mkfs(volume, options);

	/* Probe to set up private data */
	if (!rc) {
		rc = oc2_probe(volume);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/******************************************************************************
 * NAME:  oc2_discard
 *
 * FUNCTION:  Forget about this volume.
 *		Don't remove the files system.  Just clean up any data
 *		structures associated with it
 *
 * PARAMETERS:
 *	volume
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_discard(logical_volume_t *volume)
{
	LOG_ENTRY();

	EngFncs->engine_free(volume->private_data);
	volume->private_data = NULL;

	LOG_EXIT_INT(0);
	return 0;
}

/******************************************************************************
 * NAME: oc2_fsck
 *
 * FUNCTION: Run fsck on the volume
 *
 * PARAMETERS:
 *	volume
 *	options
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_fsck(logical_volume_t *volume, option_array_t *options)
{
	int rc = EINVAL;

	LOG_ENTRY();

	if (oc2_version < 2) {
		MESSAGE(_("OCFS2 does not appear to be installed or has the "
			  "wrong version. The fsck.ocfs2 utility must be "
			  "installed.\n"));
		LOG_EXIT_INT(E_CANCELED);
		return E_CANCELED;
	}

	rc = fsim_fsck(volume, options);

	/*
	 * If oc2fsck returns FSCK_CORRECTED, the
	 * file system is clean, so return FSCK_OK.
	 */
	if (rc == FSCK_CORRECTED) {
		rc = FSCK_OK;
	} else if (rc == -1) {
		/*
		 * The value of FSCK_CORRECTED is the same as
		 * EPERM, so fsim_fsck will return -1 for EPERM.
		 */
		rc = EPERM;
	}

	/*
	 * If the volume is mounted, oc2fsck checked READ ONLY
	 * regardless of options specified.  If the check was READ ONLY
	 * and errors were found, let the user know how to fix them.
	 */
	if (rc && EngFncs->is_mounted(volume->dev_node, NULL)) {
		MESSAGE(_("%s is mounted. oc2fsck (fsck.oc2fs) checked the "
			  "volume READ ONLY and found errors but did not fix "
			  "them. Unmount %s and run oc2fsck (fsck.oc2fs) "
			  "again to repair the file system."),
			volume->dev_node, volume->dev_node);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: oc2_get_option_count
 *
 * FUNCTION: Return the total number of support options for the specified task
 *
 * PARAMETERS:
 *	context
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_get_option_count(task_context_t *context)
{
	int count;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_mkfs:
		count = MKFS_OPTIONS_COUNT;
		break;
	case EVMS_Task_fsck:
		count = FSCK_OPTIONS_COUNT;
		break;
	default:
		count = -1;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}

/******************************************************************************
 * NAME: oc2_init_task
 *
 * FUNCTION:
 *	Fill in the initial list of acceptable objects. Fill in the minimum and
 * 	maximum nuber of objects that must/can be selected.  Set up all initial
 * 	values in the option_descriptors in the context record for the given
 *	task.  Some fields in the option_descriptor may be dependent on a
 *	selected object.  Leave such fields blank for now, and fill in during
 *	the set_objects call.
 *
 * PARAMETERS:
 *	context
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_init_task(task_context_t *context)
{
	option_desc_array_t *od = context->option_descriptors;
	value_list_t *value_list;
	int rc = 0;
	int i;

	LOG_ENTRY();

	switch (context->action) {
	
	case EVMS_Task_mkfs:

		if (oc2_version < 2) {
			LOG_WARNING("OCFS2 file system tools not found.\n");
			rc = E_CANCELED;
			break;
		}

		rc = fs_init_mkfs_acceptable_objects(context);
		if (rc) {
			break;
		}

		od->count = MKFS_OPTIONS_COUNT;

		/* Set version display option */
		i = MKFS_VERSION_INDEX;
		od->option[i].name = EngFncs->engine_strdup(_("version"));
		od->option[i].title = EngFncs->engine_strdup(_("Version of mkfs.ocfs2"));
		od->option[i].tip = EngFncs->engine_strdup(_("Informational only: this is the current version of OCFS2 utitlities found on your system."));
		od->option[i].type = EVMS_Type_String;
		od->option[i].min_len = 1;
		od->option[i].max_len = 32;
		od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[i].value.s = EngFncs->engine_alloc(32 + 1);
		strncpy(od->option[i].value.s, oc2_version_text, 31);

		/* Set Volume Label option */
		i = MKFS_SETVOL_INDEX;
		od->option[i].name = EngFncs->engine_strdup(_("vollabel"));
		od->option[i].title = EngFncs->engine_strdup(_("Volume Label"));
		od->option[i].tip = EngFncs->engine_strdup(_("Set the volume label for the file system."));
		od->option[i].type = EVMS_Type_String;
		od->option[i].min_len = 1;
		od->option[i].max_len = 16;
		od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
				      EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		od->option[i].value.s = EngFncs->engine_alloc(16 + 1);
		if (oc2_version < 2) {
			od->option[i].flags |= EVMS_OPTION_FLAGS_INACTIVE;
		}

		/* Log size option */
		i = MKFS_SETLOGSIZE_INDEX;
		od->option[i].name = EngFncs->engine_strdup(_("logsize"));
		od->option[i].title = EngFncs->engine_strdup(_("Log Size"));
		od->option[i].tip = EngFncs->engine_strdup(_("Set log size (in kilobytes).  Default log size is 8193 4k Blocks."));
		od->option[i].type = EVMS_Type_Unsigned_Int32;
		od->option[i].unit = EVMS_Unit_Kilobytes;
		od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
				      EVMS_OPTION_FLAGS_INACTIVE;
		od->option[i].constraint_type = EVMS_Collection_Range;
		od->option[i].constraint.range = EngFncs->engine_alloc(sizeof(value_range_t));
		od->option[i].constraint.range->min.ui32 = 2052;
		od->option[i].constraint.range->max.ui32 = 130996;
		od->option[i].constraint.range->increment.ui32 = 4;
		od->option[i].value.ui32 = 32772;

		context->min_selected_objects = 1;
		context->max_selected_objects = 1;
		break;

	case EVMS_Task_fsck:

		if (oc2_version < 2) {
			LOG_WARNING("OCFS2 file system tools not found ");
			rc = E_CANCELED;
			break;
		}

		od->count = FSCK_OPTIONS_COUNT;

		od->option[0].constraint.list = EngFncs->engine_alloc(3 * sizeof(value_t) +
								      sizeof(value_list_t));
		value_list = od->option[0].constraint.list;
		if (!value_list) {
			LOG_EXIT_INT(rc);
			return(rc);
		}

		if (EngFncs->is_mounted(context->volume->dev_node, NULL)) {
			value_list->count = 1;
			value_list->value[0].s = EngFncs->engine_strdup(_("Check Read-Only"));
		} else {
			value_list->count = 3;
			value_list->value[0].s = EngFncs->engine_strdup(_("Check Read-Only"));
			value_list->value[1].s = EngFncs->engine_strdup(_("Fix"));
			value_list->value[2].s = EngFncs->engine_strdup(_("Rebuild Tree"));
		}

		/* MODE option */
		i = FSCK_MODE_INDEX;
		od->option[i].name = EngFncs->engine_strdup(_("mode"));
		od->option[i].title = EngFncs->engine_strdup(_("Mode"));
		od->option[i].tip = EngFncs->engine_strdup(_("Mode of operation"));
		od->option[i].max_len = EVMS_VOLUME_NAME_SIZE;
		od->option[i].min_len = 1;
		od->option[i].type = EVMS_Type_String;
		/* If volume is mounted, only possible oc2fsck options is READONLY */
		od->option[i].constraint_type = EVMS_Collection_List;
		od->option[i].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);
		strcpy(od->option[i].value.s, FSCK_READONLY);

		context->min_selected_objects = 0;
		context->max_selected_objects = 0;

		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/******************************************************************************
 * NAME: oc2_set_option
 *
 * FUNCTION:
 *	Examine the specified value, and determine if it is valid for the task
 *	and option_descriptor index. If it is acceptable, set that value in the
 *	appropriate entry in the option_descriptor. The value may be adjusted
 *	if necessary/allowed. If so, set the effect return value accordingly.
 *
 * PARAMETERS:
 *	context
 *	index
 *	value
 *	effect
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_set_option(task_context_t *context,
			  u_int32_t index,
			  value_t *value,
			  task_effect_t *effect)
{
	int rc = 0;

	LOG_ENTRY();

	/* Parameter check */
	if (!context || !value || !effect) {
		rc = EFAULT;
		LOG_EXIT_INT(rc);
		return rc;
	}

	switch (context->action) {
	
	case EVMS_Task_mkfs:
		switch (index) {
		
		case MKFS_VERSION_INDEX:
			/* User typed over version field option set?
			 * Ignore and reload from the option descriptor.
			 */
			*effect |= EVMS_Effect_Reload_Options;
			break;

		case MKFS_SETVOL_INDEX:
			/* 'set volume label' option set? */
			strncpy(context->option_descriptors->option[index].value.s, value->s, 16);
			break;

		case MKFS_SETLOGSIZE_INDEX:
			/* 'set log size' option set? */
			context->option_descriptors->option[index].value.ui32 = value->ui32;
			break;

		default:
			break;
		}
		break;

	case EVMS_Task_fsck:
		switch (index) {
		
		case FSCK_MODE_INDEX:
			/* If mounted, only allow read-only */
			if (EngFncs->is_mounted(context->volume->dev_node, NULL)) {
				if (!strcmp(value->s, FSCK_FIX) ||
				    !strcmp(value->s, FSCK_REBUILD_TREE)) {
					strcpy(context->option_descriptors->option[FSCK_MODE_INDEX].value.s, FSCK_READONLY);
					MESSAGE(_("Only Check mode is alllowed when the volume is mounted."));
					*effect |= EVMS_Effect_Reload_Options;
				}
			} else {
				if (!strcmp(value->s, FSCK_FIX) ||
				    !strcmp(value->s, FSCK_REBUILD_TREE) ||
				    !strcmp(value->s, FSCK_READONLY)) {
					strcpy(context->option_descriptors->option[FSCK_MODE_INDEX].value.s, value->s);
				}
				if (strcmp(value->s, FSCK_READONLY)) {
					MESSAGE(_("Selecting a mode other than read only will cause default "
						"responses to be given to any questions asked by the oc2fsck "
						"(fsck.oc2fs) utitlity. If you do not wish to run in this "
						"mode, you must run oc2fsck (fsck.oc2fs) from the command line."));
				}
			}
			break;
		default:
			break;
		}
		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: oc2_set_volumes
 *
 * FUNCTION:
 *	Validate the volumes in the selected_objects list in the task context.
 *	Remove from the selected objects lists any volumes which are not
 *	acceptable.  For unacceptable volumes, create a declined_handle_t
 *	structure with the reason why it is not acceptable, and add it to the
 *	declined_volumes list.  Modify the acceptable_objects list in the task
 *	context as necessary based on the selected objects and the current
 *	settings of the options.  Modify any option settings as necessary based
 *	on the selected objects.  Return the appropriate task_effect_t settings
 *	if the object list(s), minimum or maximum objects selected, or option
 *	settings have changed.
 *
 * PARAMETERS:
 *	context
 *	declined_volumes
 *	effect
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_set_volumes(task_context_t *context,
			   list_anchor_t declined_volumes,
			   task_effect_t *effect)
{
	int rc = 0;
	logical_volume_t *vol;

	LOG_ENTRY();

	if (context->action == EVMS_Task_mkfs) {
		/* Get the selected volume */
		vol = EngFncs->first_thing(context->selected_objects, NULL);
		if (vol) {
			if (EngFncs->is_mounted(vol->dev_node, NULL)) {
				/* If mounted, can't mkfs.oc2. */
				rc = EBUSY;
			} else if (vol->vol_size < MINOCFS2) {
				MESSAGE("The size of volume %s is %"PRIu64
					" bytes. mkfs.oc2fs requires a minimum "
					"of %u bytes to build the OCFS2 file "
					"system.",
					vol->name,
					vol->vol_size << EVMS_VSECTOR_SIZE_SHIFT,
					MINOCFS2);
				rc = ENOSPC;
			}
		} else {
			rc = ENODATA;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/******************************************************************************
 * NAME: oc2_get_volume_info
 *
 * FUNCTION:
 * Return any additional information that you wish to provide about the
 * volume.  The Engine privides an external API to get the information
 * stored in the logical_volume_t.  This call is to get any other
 * information about the volume that is not specified in the
 * logical_volume_t.  Any piece of information you wish to provide must be
 * in an extended_info_t structure.  Use the Engine's engine_alloc() to
 * allocate the memory for the extended_info_t.  Also use engine_alloc() to
 * allocate any strings that may go into the extended_info_t.  Then use
 * engine_alloc() to allocate an extended_info_array_t with enough entries
 * for the number of exteneded_info_t structures you are returning.  Fill
 * in the array and return it in *info.
 * If you have extended_info_t descriptors that themselves may have more
 * extended information, set the EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag
 * in the extended_info_t flags field.  If the caller wants more information
 * about a particular extended_info_t item, this API will be called with a
 * pointer to the storage_object_t and with a pointer to the name of the
 * extended_info_t item.  In that case, return an extended_info_array_t with
 * further information about the item.  Each of those items may have the
 * EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag set if you desire.  It is your
 * resposibility to give the items unique names so that you know which item
 * the caller is asking additional information for.  If info_name is NULL,
 * the caller just wants top level information about the object.
 *
 * PARAMETERS:
 *	volume
 *	name
 *	info
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/

/*--------------------------------------------------------------
 * Return specific volume information -- shows up under the more
 * button.
 *
 * FIXME: This doesn't seem to be returning useful info.
 *-------------------------------------------------------------*/
static int oc2_get_volume_info(logical_volume_t *volume,
			       char *info_name,
			       extended_info_array_t **info)
{
	int rc = EINVAL;
	extended_info_array_t *Info;

	LOG_ENTRY();

	if (info_name) {
		/* There is no more information about any of the descriptors. */
		LOG_EXIT_INT(rc);
		return rc;
	}

	oc2_get_fs_limits(volume, &volume->min_fs_size,
			  &volume->max_vol_size, &volume->max_fs_size);

	Info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     4 * sizeof(extended_info_t));
	if (!Info) {
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return rc;
	}

	Info->count = 4;

	Info->info[0].name = EngFncs->engine_strdup(_("Signature"));
	Info->info[0].title = EngFncs->engine_strdup(_("Signature"));
	Info->info[0].desc = EngFncs->engine_strdup(_("Signature Identifying disk as OCFS2"));
	Info->info[0].type = EVMS_Type_String;
	Info->info[0].value.s = EngFncs->engine_strdup(_("OCFSV2"));

	Info->info[1].name = EngFncs->engine_strdup(_("Version"));
	Info->info[1].title = EngFncs->engine_strdup(_("Version Number"));
	Info->info[1].desc = EngFncs->engine_strdup(_("Version Number"));
	Info->info[1].type = EVMS_Type_Unsigned_Int16;
	Info->info[1].value.ui16 = 2;

	Info->info[2].name = EngFncs->engine_strdup(_("VolLabel"));
	Info->info[2].title = EngFncs->engine_strdup(_("Volume Label"));
	Info->info[2].desc = EngFncs->engine_strdup(_("File system volume label"));
	Info->info[2].type = EVMS_Type_String;
	Info->info[2].value.s = EngFncs->engine_strdup(_("Unknown"));

	Info->info[3].name = EngFncs->engine_strdup(_("UsableSize"));
	Info->info[3].title = EngFncs->engine_strdup(_("File System Size"));
	Info->info[3].desc = EngFncs->engine_strdup(_("Size of the file system"));
	Info->info[3].type = EVMS_Type_Unsigned_Int64;
	Info->info[3].unit = EVMS_Unit_Sectors;
	Info->info[3].value.ui64 = 1000 * 1024;

	*info = Info;
	rc = 0;

	LOG_EXIT_INT(rc);
	return rc;
}

/******************************************************************************
 * NAME: oc2_get_plugin_info
 *
 * FUNCTION: Return information about the plugin
 *
 * PARAMETERS:
 *	descriptor_name
 *	info
 *
 * RETURNS:
 *      (0) for success
 *
 *****************************************************************************/
static int oc2_get_plugin_info(char *descriptor_name,
			       extended_info_array_t **info)
{
	extended_info_array_t *Info;
	char required_engine_api_version_string[64];
	char required_fsim_api_version_string[64];
	char version_string[64];
	int rc = EINVAL;

	LOG_ENTRY();

	if (!info || descriptor_name) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* Init to no info returned. */
	*info = NULL;

	Info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     6 * sizeof(extended_info_t));
	if (!Info) {
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return rc;
	}

	Info->count = 6;

	sprintf(version_string, "%d.%d.%d",
		MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);

	sprintf(required_engine_api_version_string, "%d.%d.%d",
		my_plugin_record->required_engine_api_version.major,
		my_plugin_record->required_engine_api_version.minor,
		my_plugin_record->required_engine_api_version.patchlevel);

	sprintf(required_fsim_api_version_string, "%d.%d.%d",
		my_plugin_record->required_plugin_api_version.fsim.major,
		my_plugin_record->required_plugin_api_version.fsim.minor,
		my_plugin_record->required_plugin_api_version.fsim.patchlevel);

	Info->info[0].name = EngFncs->engine_strdup(_("Short Name"));
	Info->info[0].title = EngFncs->engine_strdup(_("Short Name"));
	Info->info[0].desc = EngFncs->engine_strdup(_("A short name given to this plugin"));
	Info->info[0].type = EVMS_Type_String;
	Info->info[0].value.s = EngFncs->engine_strdup(my_plugin_record->short_name);

	Info->info[1].name = EngFncs->engine_strdup(_("Long Name"));
	Info->info[1].title = EngFncs->engine_strdup(_("Long Name"));
	Info->info[1].desc = EngFncs->engine_strdup(_("A long name given to this plugin"));
	Info->info[1].type = EVMS_Type_String;
	Info->info[1].value.s = EngFncs->engine_strdup(my_plugin_record->long_name);

	Info->info[2].name = EngFncs->engine_strdup(_("Type"));
	Info->info[2].title = EngFncs->engine_strdup(_("Plugin Type"));
	Info->info[2].desc = EngFncs->engine_strdup(_("Plugin responsible for OCFS2 fs."));
	Info->info[2].type = EVMS_Type_String;
	Info->info[2].value.s = EngFncs->engine_strdup(_("File System Interface Module"));

	Info->info[3].name = EngFncs->engine_strdup(_("Version"));
	Info->info[3].title = EngFncs->engine_strdup(_("Plugin Version"));
	Info->info[3].desc = EngFncs->engine_strdup(_("The version number of the plugin."));
	Info->info[3].type = EVMS_Type_String;
	Info->info[3].value.s = EngFncs->engine_strdup(version_string);

	Info->info[4].name = EngFncs->engine_strdup(_("Required Engine Services Version"));
	Info->info[4].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
	Info->info[4].desc = EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  It will not run on older versions of the Engine services."));
	Info->info[4].type = EVMS_Type_String;
	Info->info[4].value.s = EngFncs->engine_strdup(required_engine_api_version_string);

	Info->info[5].name = EngFncs->engine_strdup(_("Required Engine FSIM API Version"));
	Info->info[5].title = EngFncs->engine_strdup(_("Required Engine FSIM API Version"));
	Info->info[5].desc = EngFncs->engine_strdup(_("This is the version of the Engine FSIM API that this plug-in requires.  It will not run on older versions of the Engine FSIM API."));
	Info->info[5].type = EVMS_Type_String;
	Info->info[5].value.s = EngFncs->engine_strdup(required_fsim_api_version_string);

	*info = Info;
	rc = 0;

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Table of standard plugin APIs for the OCFS2 plugin.
 */
static fsim_functions_t fsim_ops = {
	.setup_evms_plugin	= oc2_setup,
	.cleanup_evms_plugin	= oc2_cleanup,
	.probe			= oc2_probe,
	.get_fs_size		= oc2_get_fs_size,
	.get_fs_limits		= oc2_get_fs_limits,
	.can_mkfs		= oc2_can_mkfs,
	.can_unmkfs		= oc2_can_unmkfs,
	.can_fsck		= oc2_can_fsck,
	.mkfs			= oc2_mkfs,
	.discard		= oc2_discard,
	.unmkfs			= oc2_unmkfs,
	.fsck			= oc2_fsck,
	.get_option_count	= oc2_get_option_count,
	.init_task		= oc2_init_task,
	.set_option		= oc2_set_option,
	.set_volumes		= oc2_set_volumes,
	.get_volume_info	= oc2_get_volume_info,
	.get_plugin_info	= oc2_get_plugin_info,
};

/*
 * Plugin record of the OCFS2 file system
 */
plugin_record_t oc2_plugin_record = {
	.id = EVMS_OCFS2_PLUGIN_ID,
	.version = {
		.major		= MAJOR_VERSION,
		.minor		= MINOR_VERSION,
		.patchlevel	= PATCH_LEVEL
	},
	.required_engine_api_version = {
		.major		= 15,
		.minor		= 0,
		.patchlevel	= 0
	},
	.required_plugin_api_version = {
		.fsim = {
			.major		= 11,
			.minor		= 0,
			.patchlevel	= 0
		}
	},
	.short_name = EVMS_OCFS2_PLUGIN_SHORT_NAME,
	.long_name = EVMS_OCFS2_PLUGIN_LONG_NAME,
	.oem_name = EVMS_NOVELL_OEM_NAME,
	.functions = {
		.fsim = &fsim_ops
	},
	.container_functions = NULL
};

/*
 * evms_plugin_recods is the well-known symbol name that the engine looks for
 * when dynamically loading the plugins.
 */
plugin_record_t *evms_plugin_records[] = {
	&oc2_plugin_record,
	NULL
};

