/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <frontend.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "dialog.h"
#include "clist.h"
#include "selwin.h"
#include "task.h"
#include "plugin.h"
#include "enum.h"
#include "readable.h"
#include "logging.h"

/**
 *	copy_function_info - makes a copy of a function_info_t struct
 *	@info: the function_info_t structure to copy
 *	@handle: the object handle the function is for
 *
 *	This routine produces a copy of the given function_info_t
 *	and places it along with the object handle in our plugin_func_info
 *	structure we return.
 */
struct plugin_func_info *copy_function_info(function_info_t *info, object_handle_t handle)
{
	struct plugin_func_info *new_info;
	
	new_info = g_new0(struct plugin_func_info, 1);
	new_info->handle = handle;
	new_info->info.function = info->function;

	if (info->title) {
		new_info->info.title = g_strdup(info->title);
	}
	if (info->verb) {
		new_info->info.verb = g_strdup(info->verb);
	}
	if (info->help) {
		new_info->info.help = g_strdup(info->help);
	}
	log_debug("%s: returning %p.\n", __FUNCTION__, new_info);

	return new_info;
}

/**
 *	free_plugin_func_info - frees all resources associated with plugin function info
 *	@info: the plugin function info structure to free
 *
 *	This routine frees the strings and the copy of the function_info_t structure
 *	allocated with copy_function_info.
 */
void free_plugin_func_info(struct plugin_func_info *info)
{
	log_debug("%s: Freeing function info %p.\n", __FUNCTION__, info);
	g_free(info->info.title);
	g_free(info->info.verb);
	g_free(info->info.help);
	g_free(info);
}

/**
 *	on_delete_plugin_menu_item - invoked when a plugin function menu item is deleted
 *	@item: the menu item being deleted
 *
 *	This routine is a menuitem_delete_cb function invoked when a menu item for
 *	a plugin function is deleted. This allows us to free the structure we allocated
 *	in association with the menu item.
 */
void on_delete_plugin_menu_item(struct menu_item *item)
{
	free_plugin_func_info(item->user_data);
}

/**
 *	on_plugin_function_menuitem_activated - invoked when a plugin function menu item is activated
 *	@item: the menu item activated
 *
 *	This routine is invoked when a menu item for a plugin function/task is activated.
 *	We use information from the plugin_func_info associated with the menu item to
 *	start up the task.
 */
int on_plugin_function_menuitem_activated(struct menu_item *item)
{
	struct dialog_window *dialog;
	struct plugin_func_info *info = item->user_data;

	dialog = create_task_dialog(NULL, info->handle, info->info.function,
				info->info.title, info->info.verb, info->info.help);
	if (dialog != NULL) {
		process_dialog_list_events(dialog->list);
		delete_dialog_list(dialog->list->list);
	}

	return 0;
}

/**
 *	append_plugin_function_menu_items - append menu items for actions that are plugin specific
 *	@menu: the popup menu
 *	@handle: the object handle
 *
 *	This routine queries the plugin for additional functions specific
 *	to the plugin for the thing in question. It then appends menu items
 *	for each function to the menu supplied.
 */
void append_plugin_function_menu_items(struct popup_menu *menu, object_handle_t handle)
{
	function_info_array_t *functions;

	if (evms_get_plugin_functions(handle, &functions) == 0) {
		int i;
		gchar *label;
		struct plugin_func_info *info;
		struct menu_item *item;

		for (i = 0; i < functions->count; i++) {
			int is_inactive;
			
			info = copy_function_info(&(functions->info[i]), handle);
			is_inactive = functions->info[i].flags & EVMS_FUNCTION_FLAGS_INACTIVE;
            
	    		if (is_inactive)
				label = g_strdup_printf(_("%s (unavailable)"), info->info.title);
			else
				label = g_strdup_printf("%s...", info->info.title);
			item = append_menu_item(menu, label, NULL, 0, FALSE,
					(menuitem_activate_cb)on_plugin_function_menuitem_activated,
					info);
			set_menu_item_delete_cb(item, (menuitem_delete_cb)on_delete_plugin_menu_item);
			
			/* Make the menu item insensitive to events if it is inactive */
			if (is_inactive)
				 set_menu_item_sensitivity(item, FALSE);
			g_free(label);
		}
		evms_free(functions);
	}
}

/**
 *	filter_plugins_for_task - a clist_filter_func that keeps plugins that can do the given task
 *	@handle: the plugin handle
 *	@user_data: contains a task action
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given plugin can initiate the requested task.
 */
int filter_plugins_for_task(object_handle_t handle, void *user_data)
{
	return !(can_plugin_do_action(handle, GPOINTER_TO_UINT(user_data)));
}

/**
 *	get_plugin_clist_enum_func - returns the enum function to get plugins wanted
 *	@type: the plugin type (EVMS_SEGMENT_MANAGER, etc.)
 *	@support_containers: TRUE if we want any plugin that supports containers
 *
 *	This routine returns the enum function that should be invoked when attempting
 *	to populate the plugin selection list. If the support_containers parameter is
 *	TRUE, then the function that enumerates all plugins that support contains is 
 *	returned and the given plugin code type is ignored.
 */
clist_enum_func get_plugin_clist_enum_func(evms_plugin_code_t type, gboolean support_containers)
{
	clist_enum_func enum_func;

	if (support_containers == TRUE) {
		enum_func = enum_container_plugins;
	} else {
		switch (type) {
		case EVMS_DEVICE_MANAGER:
			enum_func = enum_device_manager_plugins;
			break;
		case EVMS_SEGMENT_MANAGER:
			enum_func = enum_segment_manager_plugins;
			break;
		case EVMS_REGION_MANAGER:
			enum_func = enum_region_manager_plugins;
			break;
		case EVMS_FEATURE:
			enum_func = enum_feature_plugins;
			break;
		case EVMS_FILESYSTEM_INTERFACE_MODULE:
			enum_func = enum_fsim_plugins;
			break;
		default:
			enum_func = enum_all_plugins;
			break;
		}
	}
	return enum_func;
}

/**
 *	create_plugin_selection_window - create a dialog with a list of plugins for a certain task
 *	@title: the window title
 *	@type: the type of plugin wanted (EVMS_SEGMENT_MANAGER, etc.)
 *	@support_containers: TRUE if we want any plugin that supports containers
 *	@action: the task action we'ld like the plug-in to do
 *
 *	This routine creates a selection dialog window containing a list of plugins
 *	for us to use in initiating a task.
 */
struct selwin *create_plugin_selection_window(char *title, evms_plugin_code_t type,
						gboolean support_containers, task_action_t action)
{
	struct selwin *selwin;
	clist_enum_func enum_func;

	enum_func = get_plugin_clist_enum_func(type, support_containers);

	selwin = create_selection_window(title, NULL, NULL, NULL, NULL, NULL, NULL, NULL);

	set_clist_column_count(selwin->clist, 2);
	set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 0.05),
				0,
				CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(selwin->clist, 1, calc_clist_column_width(selwin->clist, 0.95),
				get_clist_column_end(selwin->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, "");
	print_clist_column_title(selwin->clist, 1, make_plugin_type_readable_string(type, FALSE));

	set_menu_item_visibility(((struct dialog_window *)selwin)->prev_button, FALSE);

	clist_populate(selwin->clist, enum_func, filter_plugins_for_task,
			format_standard_item, NULL, GUINT_TO_POINTER(action));

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	return selwin;
}

/**
 *	on_delete_plugin_dialog - callback invoked when dialog is to be deleted
 *	@dialog: the plugin selection dialog
 *
 *	This routine is invoked when the plugin selection dialog is deleted. We
 *	take care of freeing any dynamically allocated entries in the hash table
 *	before freeing the hash table associated with the dialog.
 */
void on_delete_plugin_dialog(struct dialog_window *dialog)
{
	GHashTable *hash_table = dialog->user_data;

	if (hash_table != NULL) {
		g_free(g_hash_table_lookup(hash_table, "title"));
		g_hash_table_destroy(hash_table);
	}
}

/**
 *	process_plugin_task - process the dialogs for a task initiated by selecting a plugin
 *	@title: the window title
 *	@type: the type of plugin wanted (EVMS_SEGMENT_MANAGER, etc.)
 *	@support_containers: TRUE if we want any plugin that supports containers
 *	@action: the task action we'ld like the plug-in to do
 *
 *	This routine manages the walking a user through a task initiated by first selecting
 *	the plugin for the task. The correct series of dialogs is then created and processed
 *	through to the completion of the task.
 */
void process_plugin_task(char *title, evms_plugin_code_t type, gboolean support_containers,
				task_action_t action)
{
	struct dialog_list dialogs;
	struct dialog_window *dialog;

	dialog = (struct dialog_window *)create_plugin_selection_window(title, type,
									support_containers, action);
	/*
	 * Allocate a hash table for storing multiple pieces of data needed by the
	 * activation callback. Save off the task action and title for starters.
	 */
	dialog->user_data = g_hash_table_new(g_str_hash, g_str_equal);
	g_hash_table_insert(dialog->user_data, "title", g_strdup(title));
	g_hash_table_insert(dialog->user_data, "action", GUINT_TO_POINTER(action));

	set_menu_item_activate_cb(dialog->next_button,
				(menuitem_activate_cb)on_init_task_menuitem_activated);
	set_dialog_delete_cb(dialog, (dialog_delete_cb)on_delete_plugin_dialog);

	init_dialog_list(&dialogs);
	append_dialog_to_list(dialog, &dialogs);
	process_dialog_list_events(&dialogs);
	delete_dialog_list(dialogs.list);
}

/**
 *	get_parent_plugin_for_object - return parent's plugin handle
 *	@object: handle of object for the query
 *
 *	This routine returns the plugin handle corresponding to the
 *	plugin managing the first immediate parent object for the
 *	object given.
 */
plugin_handle_t get_parent_plugin_for_object(object_handle_t object)
{
	int rc;
	plugin_handle_t plugin = 0;
	handle_object_info_t *info;
	
	rc = evms_get_info(object, &info);
	if (rc == 0) {
		handle_array_t *users = info->info.object.parent_objects;
		
		if (users->count > 0) {
			handle_object_info_t *parent_info;
			
			rc = evms_get_info(users->handle[0], &parent_info);
			if (rc == 0) {
				plugin = parent_info->info.object.plugin;
				evms_free(parent_info);
			} else {
				log_error("%s: evms_get_info() returned error %d for first parent object.\n",
					__FUNCTION__, rc);
			}
		}
		evms_free(info);
	} else {
        	log_error("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
	}
	return plugin;
}

/**
 *	get_parent_plugin_type - return parent object plugin type
 *	@object: handle of object for the query
 *
 *	This routine attempts to return the plugin type for the plugin
 *	managing the first immediate parent object for the object given.
 */
plugin_type_t get_parent_plugin_type(object_handle_t object)
{
	int rc;
	plugin_type_t type;
	plugin_handle_t plugin_handle;
	handle_object_info_t *info;

	plugin_handle = get_parent_plugin_for_object(object);
	
	rc = evms_get_info(plugin_handle, &info);
	if (rc == 0) {
        	type = GetPluginType(info->info.plugin.id);
        	evms_free(info);
	} else {
		type = EVMS_NO_PLUGIN;
		log_error("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
	}
	return type;
}

/**
 *	get_parent_plugin_type_string - get string corresponding to the parent object's plugin type
 *	@object: handle of object for the query
 *	@lowercase: TRUE if first letter of each word should be lowercased
 *
 *	This routine returns the plugin type string for the plugin managing the
 *	first immediate parent object for the given object.
 */
inline char *get_parent_plugin_type_string(object_handle_t object, gboolean lowercase)
{
	return make_plugin_type_readable_string(get_parent_plugin_type(object), lowercase);
}

/**
 *	on_show_plugin_funcs_button_activated - after plugin task selected, process task dialogs
 *	@item: the menu item that was activated
 *
 *	This routine is invoked after a plugin function has been selected
 *	proceeding to display the task dialog.
 */
int on_show_plugin_funcs_button_activated(struct menu_item *item)
{
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;
	struct plugin_func_info *curr_selection, *prev_selection;

	prev_selection = dialog->user_data;
	curr_selection = get_selected_data(selwin->clist);

	if (curr_selection != prev_selection) {
		GList *next_dialog;
		struct dialog_window *new_dialog;
		/*
		 * Since the selected plugin task has changed we need to delete any
		 * dialogs that we may have built for the add feature task.
		 */
		next_dialog = get_next_dialog(dialog);
		if (next_dialog != NULL)
			delete_dialog_list(next_dialog);

		new_dialog = create_task_dialog(dialog->list, 
						curr_selection->handle,
						curr_selection->info.function,
						curr_selection->info.title,
						curr_selection->info.verb,
						curr_selection->info.help);
		if (new_dialog != NULL) {
			dialog->user_data = curr_selection;
			dialog->list->current = get_next_dialog(dialog);
		}
	} else {
		dialog->list->current = get_next_dialog(dialog);
	}

	return 0;
}

/**
 *	get_other_clist_enum_func - returns the enum function to get things of the given type
 *	@type: the object type
 *
 *	This routine returns the enum function that should be invoked when attempting
 *	to populate the plugin tasks selection list.
 */
clist_enum_func get_other_clist_enum_func(object_type_t type)
{
	clist_enum_func enum_func;

	switch (type) {
		case CONTAINER:
			enum_func = enum_containers;
			break;
		case VOLUME:
			enum_func = enum_volumes;
			break;
		case PLUGIN:
			enum_func = enum_all_plugins;
			break;
		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
		default:
			enum_func = enum_data_objects;
			break;
	}
	return enum_func;
}

/**
 *	filter_things_with_plugin_funcs - filter things that have plugin functions available
 *	@handle: the thing's handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing has any private plugin functions available for it.
 */
int filter_things_with_plugin_funcs(engine_handle_t handle, void *user_data)
{
	int rc;

	function_info_array_t *functions;

	rc = evms_get_plugin_functions(handle, &functions);
	if (rc == 0) {
		if (functions->count <= 0)
			rc = EPERM;
		evms_free(functions);
	}
	return rc;
}

/**
 *	on_delete_plugin_clist_item - invoked when a plugin function clist item is deleted
 *	@item: the clist item being deleted
 *
 *	This routine is a clist_item_delete_cb function invoked when a clist item for
 *	a plugin function is deleted. This allows us to free the structure we allocated
 *	in association with the clist item.
 */
void on_delete_plugin_clist_item(struct clist_item *item)
{
	free_plugin_func_info(item->user_data);
}

/**
 *	populate_clist_with_plugin_funcs - append items to a clist for each plugin function available for a handle
 *	@clist: the selection list
 *	@handle: the object handle
 *
 *	This routine populates the given clist with information on each plugin function available
 *	for the handle given.
 */
void populate_clist_with_plugin_funcs(struct clist *clist, object_handle_t handle)
{
	function_info_array_t *functions;

	if (evms_get_plugin_functions(handle, &functions) == 0) {
		int i;

		for (i = 0; i < functions->count; i++) {
			GPtrArray *text;
			int is_inactive;
			struct clist_item *item;
			struct plugin_func_info *info;

			is_inactive = functions->info[i].flags & EVMS_FUNCTION_FLAGS_INACTIVE;
			info = copy_function_info(&(functions->info[i]), handle);

			text = g_ptr_array_new();
			g_ptr_array_add(text, g_strdup(" "));
			if (is_inactive)
				g_ptr_array_add(text, g_strdup_printf(_("%s (unavailable)"), info->info.title));
			else
				g_ptr_array_add(text, g_strdup(info->info.title));
    			item = append_item(clist, text, info, (clist_item_delete_cb)on_delete_plugin_clist_item);
			g_ptr_array_free(text, TRUE);

			if (is_inactive)
				set_item_selectable(item, FALSE);
		}
		evms_free(functions);
	}
}

/**
 *	create_plugin_func_dialog - create selection list dialog with plugin functions for a handle
 *	@handle: the object handle
 *
 *	This routine creates a standard selection dialog window populated with the names of
 *	plugin function tasks that can be activated for a given handle.
 */
struct dialog_window *create_plugin_func_dialog(object_handle_t handle)
{
	struct selwin *selwin;
	struct dialog_window *dialog;

	selwin = create_selection_window(_("Plug-in Tasks Available"),
					NULL, NULL, NULL,
					(menuitem_activate_cb)on_show_plugin_funcs_button_activated,
					NULL, NULL, NULL);


	set_clist_column_count(selwin->clist, 2);
	set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 0.05),
				0,
				CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(selwin->clist, 1, calc_clist_column_width(selwin->clist, 0.95),
				get_clist_column_end(selwin->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, "");
	print_clist_column_title(selwin->clist, 1, _("Task Description"));

	populate_clist_with_plugin_funcs(selwin->clist, handle);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	dialog = (struct dialog_window *)selwin;

	return dialog;
}

/**
 *	list_plugin_funcs_button_activated - display a dialog with plugin functions for selected thing
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Next button is activated after selecting
 *	a object, volume, container or plugin to display the plugin functions available
 *	to it. 
 */
int list_plugin_funcs_button_activated(struct menu_item *item)
{
	struct selwin *selwin = item->user_data;
	object_handle_t prev_selection, curr_selection;
	struct dialog_window *dialog = item->user_data;

	curr_selection = get_selected_handle(selwin->clist);
	prev_selection = GPOINTER_TO_UINT(dialog->user_data);

	if (curr_selection != prev_selection) {
		GList *next_dialog;
		struct dialog_window *new_dialog;
		/*
		 * Since the selected thing has changed we need to delete any
		 * dialogs that we may have built for the plugin funcs task.
		 */
		next_dialog = get_next_dialog(dialog);
		if (next_dialog != NULL)
			delete_dialog_list(next_dialog);

		new_dialog = create_plugin_func_dialog(curr_selection);

		if (new_dialog != NULL) {
			dialog->user_data = GUINT_TO_POINTER(curr_selection);
			append_dialog_to_list(new_dialog, dialog->list);
			dialog->list->current = get_next_dialog(dialog);
		}
	} else {
		dialog->list->current = get_next_dialog(dialog);
	}

	return 0;
}

/**
 *	actions_show_things_with_plugin_funcs_menuitem_activated - display thing's that have plugin functions
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to display a selection list of things that have plugin
 *	functions available. We populate the clist with the object type supplied in the
 *	item->user_data.
 */
int actions_show_things_with_plugin_funcs_menuitem_activated(struct menu_item *item)
{
	struct selwin *selwin;
	clist_enum_func enum_func;
	struct dialog_list dialogs;
	struct dialog_window *dialog;
	object_type_t type = GPOINTER_TO_UINT(item->user_data);

	enum_func = get_other_clist_enum_func(type);

	selwin = create_selection_window(_("Access Plug-in Specific Tasks"),
					NULL, NULL, NULL,
					(menuitem_activate_cb)list_plugin_funcs_button_activated,
					NULL, NULL, NULL);

	dialog = (struct dialog_window *)selwin;

	if (type == PLUGIN) {
		set_clist_column_count(selwin->clist, 2);
		set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 0.05),
				0,
				CLIST_TEXT_JUSTIFY_CENTER);
		set_clist_column_info(selwin->clist, 1, calc_clist_column_width(selwin->clist, 0.95),
				get_clist_column_end(selwin->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);

		print_clist_column_title(selwin->clist, 0, "");
		print_clist_column_title(selwin->clist, 1, _("Plug-in"));
	} else {
		char *column_title;

		if (type == CONTAINER) 
			column_title = _("Storage Container");
		else if (type == VOLUME)
			column_title = _("Logical Volume");
		else
			column_title = _("Storage Object");

		print_clist_column_title(selwin->clist, 0, " ");
		print_clist_column_title(selwin->clist, 1, column_title);
		print_clist_column_title(selwin->clist, 2, _("Size"));
	}

	set_menu_item_visibility(dialog->prev_button, FALSE);

	clist_populate(selwin->clist, enum_func, filter_things_with_plugin_funcs,
			format_standard_item, NULL, NULL);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	init_dialog_list(&dialogs);
	append_dialog_to_list(dialog, &dialogs);
	process_dialog_list_events(&dialogs);
	delete_dialog_list(dialogs.list);

	return 0;
}
