/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: reiserfs.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <plugin.h>
#include <fcntl.h>
#include <sys/wait.h>

#include "reiserfsim.h"

static int fsim_rw_diskblocks( logical_volume_t *,int, int64_t, int32_t, void *, int );
static int reiser_version = 0;	// version of reiserfs utils.
static char reiser_version_text[32];  // version of reiserfs utils.

plugin_record_t * my_plugin_record = &reiser_plugin_record;
engine_functions_t *EngFncs = NULL;


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                                   Common Routines                                    +
+                                                                                      +
+--------------------------------------------------------------------------------------*/

/*
 * Get the size limits for this volume.
 */
static int fsim_get_volume_limits(     struct reiserfs_super_block * sb_ptr,
				       sector_count_t    * min_size,
				       sector_count_t    * max_volume_size,
				       sector_count_t    * max_object_size)
{
	int rc = 0;
	LOG_ENTRY();

	*max_volume_size = (sector_count_t)0xffffffff * (sector_count_t)8;  // 32bits of 4k blocks
	*max_object_size = (sector_count_t)0xffffffff * (sector_count_t)8;  // 32bits of 4k blocks
	*min_size = (get_sb_block_count(sb_ptr) - get_sb_free_blocks(sb_ptr)) * (get_sb_block_size(sb_ptr)/EVMS_VSECTOR_SIZE);

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * NAME: set_mkfs_options
 *
 * FUNCTION: Build options array (argv) for mkreiserfs
 *
 * PARAMETERS:
 *      options   - options array passed from EVMS engine
 *      argv      - mkfs options array
 *      vol_name  - volume name on which program will be executed
 *
 */
static void set_mkfs_options( option_array_t * options, char ** argv, char * vol_name )
{
	int i, opt_count = 0;
	char buffer[EVMS_VOLUME_NAME_SIZE + 10];

	LOG_ENTRY();

	argv[opt_count++] = "mkreiserfs";

	    /* 'force/quiet' option -ff*/
	argv[opt_count++] = "-ff";

	for ( i=0; i<options->count; i++ ) {

		if (!options->option[i].is_number_based) {
			if ( !strcmp(options->option[i].name, "vollabel")
			     && (reiser_version >= 2)
                             && options->option[i].value.s ) {
				/* 'set volume name' option */
				options->option[i].number = MKFS_SETVOL_INDEX;
                        } else if (!strcmp(options->option[i].name, "logsize")
				   && options->option[i].value.ui ) {
				/* 'set log size' option */
				options->option[i].number = MKFS_SETLOGSIZE_INDEX;
                        } else {
				/* unknown. ignore. */
				continue;
                        }
                }

		switch (options->option[i].number) {
			
			case MKFS_SETVOL_INDEX:
				/* 'set volume name' option */
				if ( options->option[i].value.s ) {
					argv[opt_count++] = "-l";
					argv[opt_count++] = options->option[i].value.s;
                                }
				break;

			case MKFS_SETLOGSIZE_INDEX:
				/* 'set log size' option */
				if ( options->option[i].value.ui ) {
					sprintf( buffer, "%s%u", "-s", options->option[i].value.ui/4 );
					argv[opt_count++] = buffer;
                                }
				break;

			default:
				break;
				
		}
	}

	argv[opt_count++] = vol_name;
	/* NULL terminate the argument array */
	argv[opt_count] = NULL;

	LOG_EXIT_VOID();
	return;
}


/*
 * Format the volume.
 */
static int fsim_mkfs(logical_volume_t * volume, option_array_t * options )
{
	int     rc = EINVAL;
	char   *argv[MKFS_OPTIONS_COUNT + 7];
	pid_t   pidm;
	int     fds2[2];  // pipe for stderr and stdout 0=read,1=write
	int     bytes_read;
	char    *buffer = NULL;
	int     status;

	LOG_ENTRY();
	rc = pipe(fds2);
	if (rc) {
		rc = errno;
		LOG_EXIT_INT(rc);
		return(rc);
	}
	if (!(buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN))) {
		close(fds2[0]);
		close(fds2[1]);
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return(rc);
	}

	set_mkfs_options( options, argv, volume->dev_node );

	pidm = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidm != -1) {
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		while (!(waitpid( pidm, &status, WNOHANG ))) {
			bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				LOG_DEFAULT("mkfs output: \n%s",buffer); // on prompt if there is one
				memset(buffer,0,bytes_read); //clear out message
			}
			usleep(10000); // don't hog all the cpu
		}
		if (WIFEXITED(status)) {
			do {
				bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
				if (bytes_read > 0) {
					LOG_DEFAULT("mkfs output: \n%s",buffer);
				}
			} while (bytes_read > 0);
			LOG_DEFAULT("mkfs completed with rc = %d \n",status);
			rc = WEXITSTATUS(status);
		} else {
			rc = EINTR;
		}
	} else {
		rc = EIO;
	}

	EngFncs->engine_free(buffer);
	close(fds2[0]);
	close(fds2[1]);
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * NAME: set_fsck_options
 *
 * FUNCTION: Build options array (argv) for reiserfsck
 *
 * PARAMETERS:
 *      options   - options array passed from EVMS engine
 *      argv      - fsck options array
 *      volume    - volume on which program will be executed
 *
 */
static void set_fsck_options( option_array_t * options, char ** argv, logical_volume_t * volume )
{
	int i, opt_count = 0;
	LOG_ENTRY();

	argv[opt_count++] = "reiserfsck";
	argv[opt_count++] = "-q";

	for ( i=0; i<options->count; i++) {
		if ( !options->option[i].is_number_based ) {
			if ( !strcmp(options->option[i].name, "mode") &&
			     !EngFncs->is_mounted(volume->dev_node, NULL)) {
				options->option[i].number = FSCK_MODE_INDEX;
                        } else {
				/* unknown. ignore. */
				continue;
                        }
                }

		switch (options->option[i].number) {
			case FSCK_MODE_INDEX:
                                if (!(strcmp(options->option[i].value.s,FSCK_FIX))) {
					argv[opt_count++] = "--fix-fixable";
                                } else if (!(strcmp(options->option[i].value.s,FSCK_REBUILD_TREE))) {
					argv[opt_count++] = "--rebuild-tree";
                                } else if (!(strcmp(options->option[i].value.s,FSCK_READONLY))) {
					argv[opt_count++] = "--check";
                                }
				break;
				
			default:
				break;
			}
	}
	
	argv[opt_count++] = volume->dev_node;
	/* NULL terminate the pointer array */
	argv[opt_count] = NULL;

	LOG_EXIT_VOID();
	return;
}


/*
 * Run fsck on the volume.
 */
static int fsim_fsck(logical_volume_t * volume, option_array_t * options )
{
	int     rc = EINVAL;
	char   *argv[FSCK_OPTIONS_COUNT + 7];
	pid_t   pidf;
	int     status;
	int     fds1[2];  // pipe for stdin 0=read 1=write
	int     fds2[2];  // pipe for stderr and stdout 0=-read,1=write
	int     bytes_read;
	char    *buffer = NULL;


	LOG_ENTRY();
	rc = pipe(fds1);
	if (rc) {
		rc = errno;
		LOG_EXIT_INT(rc);
		return(rc);
	}
	rc = pipe(fds2);
	if (rc) {
		rc = errno;
		close(fds1[0]);
		close(fds1[1]);
		LOG_EXIT_INT(rc);
		return(rc);
	}
	if (!(buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN))) {
		close(fds1[0]);
		close(fds1[1]);
		close(fds2[0]);
		close(fds2[1]);
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return(rc);
	}

	set_fsck_options( options, argv, volume );

	pidf = EngFncs->fork_and_execvp(volume, argv, fds1, fds2, fds2);
	if (pidf != -1) {
		/* Fork and execute the correct program. */
		// WARNING: Do Not close read handle of stdin or
		// you will cause a SIGPIPE if you write after the
		// child process has gone away.
		write(fds1[1], "Yes\n",4);

		/* wait for child to complete */
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		while (!(waitpid( pidf, &status, WNOHANG ))) {
			bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				write(fds1[1], "\n",1);	// send newline to take default
				MESSAGE(_("fsck output: \n%s"),buffer);	// on prompt if there is one
				memset(buffer,0,bytes_read); //clear out message
			}
			usleep(10000); // don't hog all the cpu
		}
		if (WIFEXITED(status)) {
			do {
				bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
				if (bytes_read > 0) {
					MESSAGE(_("fsck output: \n\n%s"),buffer);
				}
			} while (bytes_read > 0);
			LOG_DEFAULT("fsck completed with rc = %d \n",status);
			rc = WEXITSTATUS(status);
		} else {
			rc = EINTR;
		}
	} else {
		rc = EIO;
	}

	EngFncs->engine_free(buffer);

	close(fds1[0]);
	close(fds1[1]);
	close(fds2[0]);
	close(fds2[1]);
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * NAME: fsim_get_reiser_reiserfs_super_block
 *
 * FUNCTION: Get and validate a JFS reiserfs_super_block
 *
 * PARAMETERS:
 *      vol_name   - volume name from which to get the reiserfs_super_block
 *      sb_ptr     - pointer to reiserfs_super_block
 *
 * RETURNS:
 *      (0) for success
 *      != 0 otherwise
 *
 */
static int fsim_get_reiserfs_super_block( logical_volume_t *volume, struct reiserfs_super_block *sb_ptr )
{
	int  fd;
	int  rc = 0;
	LOG_ENTRY();

	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) {
		rc = -fd;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* get primary reiserfs_super_block */
	rc = fsim_rw_diskblocks( volume, fd, REISERFS_DISK_OFFSET_IN_BYTES, SB_SIZE, sb_ptr, GET );

	if ( rc == 0 ) {
		/* see if primary reiserfs_super_block is Reiser */
		if (strncmp(sb_ptr->s_v1.s_magic, REISERFS_SUPER_MAGIC_STRING,
			    strlen(REISERFS_SUPER_MAGIC_STRING)) != 0 &&
		    strncmp(sb_ptr->s_v1.s_magic, REISER2FS_SUPER_MAGIC_STRING,
			    strlen(REISER2FS_SUPER_MAGIC_STRING)) != 0) {

			/* check old SB location */
			rc = fsim_rw_diskblocks( volume, fd, REISERFS_OLD_DISK_OFFSET_IN_BYTES, SB_SIZE, sb_ptr, GET);

			if ( rc == 0 ) {
				/* see if primary reiserfs_super_block is Reiser */
				if (strncmp(sb_ptr->s_v1.s_magic, REISERFS_SUPER_MAGIC_STRING,
					    strlen(REISERFS_SUPER_MAGIC_STRING)) != 0 &&
				    strncmp(sb_ptr->s_v1.s_magic, REISER2FS_SUPER_MAGIC_STRING,
					    strlen(REISER2FS_SUPER_MAGIC_STRING)) != 0) {
					rc = EINVAL;
				}
			}
		}
	}
	EngFncs->close_volume(volume, fd);
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * NAME: fsim_rw_diskblocks
 *
 * FUNCTION: Read or write specific number of bytes for an opened device.
 *
 * PARAMETERS:
 *      dev_ptr         - file handle of an opened device to read/write
 *      disk_offset     - byte offset from beginning of device for start of disk
 *                        block read/write
 *      disk_count      - number of bytes to read/write
 *      data_buffer     - On read this will be filled in with data read from
 *                        disk; on write this contains data to be written
 *      mode            - GET (read) or PUT (write)
 *
 * RETURNS:
 *      FSIM_SUCCESS (0) for success
 *      ERROR       (-1) can't lseek
 *      EINVAL
 *      EIO
 *
 */
static int fsim_rw_diskblocks( logical_volume_t * volume,
			       int      dev_ptr,
			       int64_t  disk_offset,
			       int32_t  disk_count,
			       void     *data_buffer,
			       int      mode )
{
	int     rc = 0;
	size_t  Bytes_Transferred;

	LOG_ENTRY();

	switch (mode) {
	case GET:
		Bytes_Transferred = EngFncs->read_volume(volume,dev_ptr,data_buffer,disk_count,disk_offset);
		break;
	case PUT:
		Bytes_Transferred = EngFncs->write_volume(volume,dev_ptr,data_buffer,disk_count,disk_offset);
		break;
	default:
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	if ( Bytes_Transferred != disk_count ) {
		rc = EIO;
		LOG_EXIT_INT(rc);
		return rc;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Test mkfs.reiser version.
 *    reiserutils must be at version 3.x.0 or greater to function properly
 *    with this FSIM.  That also happens to be the first version in which
 *    mkfs.reiser supported the -V (version) option.  So, run mkfs.reiser with
 *    the -V option,and if it succeeds, the installed version is OK.  If
 *    it fails, pass back the exit code (should be EINVAL) of the failure.
 */
static int reiser_test_version(void)
{
	int     rc = 0;
	char   *argv[3];
	pid_t   pidm;
	int     fds2[2];  // pipe for stderr and stdout 0=-read,1=write
	int     status;
	char    *buffer;
	int     bytes_read;
	char    *ver,*end;

	LOG_ENTRY();

	rc = pipe(fds2);
	if (rc) {
		rc = errno;
		LOG_EXIT_INT(rc);
		return(rc);
	}

	buffer = EngFncs->engine_alloc(1024);
	if (!buffer) {
		close(fds2[0]);
		close(fds2[1]);
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return(rc);
	}

	argv[0] = "mkreiserfs";
	argv[1] = "-V";
	argv[2] = NULL;

	pidm = EngFncs->fork_and_execvp(NULL, argv, NULL, fds2, fds2);
	if (pidm != -1) {
		LOG_DEBUG("Waiting for process %d to exit.\n", pidm);
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL, 0) | O_NONBLOCK);
		waitpid(pidm, &status, 0);
		if (WIFEXITED(status) && (WEXITSTATUS(status) != ENOENT)) {
			LOG_DEBUG("Reading output from process %d on fd %d.\n",
				  pidm, fds2[0]);
			bytes_read = read(fds2[0], buffer, 1024);
			if (bytes_read > 0) {
				if ((ver = strstr(buffer, "3.x.0"))) {
					reiser_version = 1;
				} else if ((ver = strstr(buffer, "3.x.1"))) {
					reiser_version = 2;
				} else if ((ver = strstr(buffer, "3.6."))) {
					reiser_version = 3;
				}
				if (ver) {
					end = strstr(ver, "\n");
					if (end) {
						strncpy(reiser_version_text, ver, min(end-ver, 31));
					} else {
						strncpy(reiser_version_text, ver, 6);
					}
					rc = 0;
				}
			}
		}
	} else 	{
		rc = EIO;
	}

	EngFncs->engine_free(buffer);
	close(fds2[0]);
	close(fds2[1]);
	LOG_EXIT_INT(rc);
	return rc;
}


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                            Start Of EVMS Plugin Functions                            +
+                        (exported to engine via function table)                       +
+                                                                                      +
+-------------------------------------------------------------------------------------*/


static int reiser_setup(engine_functions_t *engine_function_table)
{
	int rc = 0;
	EngFncs = engine_function_table;

	LOG_ENTRY();

	/*
	 * reiserutils must be at version 1.0.9 or greater to work properly with this FSIM.
	 */
	rc = reiser_test_version();
	switch (reiser_version) {
	case 1:
		LOG_WARNING("ReiserFS utilities are out of date.\n");
		LOG_WARNING("They should be at 3.x.1b or higher to ensure proper functionality.\n");
		LOG_WARNING("Please get the current version of ReiserFS utils from http://www.namesys.com\n");
		break;
	case 2:
	case 3:
		break;
	case 0:
		LOG_WARNING("ReiserFS utils were either not found or not at the proper version.\n");
		LOG_WARNING("The fsck and mkfs utilities must be installed using the standard names\n");
		LOG_WARNING("of mkreiserfs and reiserfsck.\n");
		LOG_WARNING("ReiserFS utils must be version 3.x.0 or later to function properly with this FSIM.\n");
		LOG_WARNING("Please get the current version of ReiserFS utils from http://www.namesys.com\n");
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Free all of the private data item we have left on volumes.
 */
static void reiser_cleanup(void)
{
	int rc = 0;
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;
	logical_volume_t * volume;
	LOG_ENTRY();

	rc = EngFncs->get_volume_list(my_plugin_record, NULL, 0, &global_volumes);
	if (!rc) {
		LIST_FOR_EACH(global_volumes, vol_list_iter, volume) {
			if (volume->private_data) {
				EngFncs->engine_free(volume->private_data);
			}
		}
		EngFncs->destroy_list(global_volumes);
	}
	LOG_EXIT_VOID();
}


/*
 * Does this FSIM manage the file system on this volume?
 * Return 0 for "yes", else a reason code.
 */
static int reiser_probe(logical_volume_t *volume)
{
	int  rc = 0;
	struct reiserfs_super_block *sb_ptr;

	LOG_ENTRY();

	/* allocate space for copy of reiserfs_super_block in private data */
	sb_ptr = EngFncs->engine_alloc( SB_SIZE );

	if ( sb_ptr ) {

		/* get and validate Reiser reiserfs_super_block */
		rc = fsim_get_reiserfs_super_block( volume, sb_ptr );

		if ( !rc ) {
			/* store copy of valid reiserfs_super_block in private data */
			volume->private_data = (void*)sb_ptr;
		} else {
			/* could not get valid reiserfs_super_block */
			volume->private_data = NULL;
			EngFncs->engine_free( sb_ptr );
		}
	} else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Can mkfs this volume?
 */
static int reiser_can_mkfs(logical_volume_t *volume)
{
	int  rc=0;

	LOG_ENTRY();

	/*****************************************************
	 *  FUTURE - ensure mkfs.reiser exists                  *
	 *           match version with available functions  *
	 *****************************************************/

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't format. */
		rc = EBUSY;
	}
	if (volume->vol_size < MINREISERFS) {
		rc = EINVAL;
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Can unmkfs this volume?
 */
static int reiser_can_unmkfs(logical_volume_t *volume)
{
	int  rc=0;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
	}
	LOG_EXIT_INT(rc);
	return rc;

}


/*
 * Can fsck this volume?
 */
static int reiser_can_fsck(logical_volume_t *volume)
{
	int  rc=0;

	LOG_ENTRY();

	/*****************************************************
	 *  FUTURE - ensure reiserfsck exists                *
	 *           match version with available functions  *
	 *****************************************************/

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Get the current size of this volume
 */
static int reiser_get_fs_size(logical_volume_t *volume,
			      sector_count_t *size)
{
	int  rc = EINVAL;
	struct reiserfs_super_block *sb_ptr = (struct reiserfs_super_block *)volume->private_data;

	LOG_ENTRY();

	/* re-read super block and store in private data  */
	rc = fsim_get_reiserfs_super_block(volume, sb_ptr);

	if (sb_ptr) {
		*size = get_sb_block_count(sb_ptr) * (get_sb_block_size(sb_ptr)/EVMS_VSECTOR_SIZE);
		rc = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Get the size limits for this volume.
 */
static int reiser_get_fs_limits(logical_volume_t *volume,
				sector_count_t *min_size,
				sector_count_t *max_volume_size,
				sector_count_t *max_object_size)
{
	int rc = 0;
	struct reiserfs_super_block *sb_ptr = (struct reiserfs_super_block *)volume->private_data;

	LOG_ENTRY();

	/* re-read superblock  to set up private data */
	rc = fsim_get_reiserfs_super_block(volume, sb_ptr);

	if ( !rc ) {
		rc = fsim_get_volume_limits( sb_ptr, min_size, max_volume_size, max_object_size);
		LOG_EXTRA("volume:%s, min:%"PRIu64", max:%"PRIu64"\n",volume->name, *min_size, *max_volume_size);
		LOG_EXTRA("fssize:%"PRIu64", vol_size:%"PRIu64"\n",volume->fs_size,volume->vol_size );

	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Expand the volume to new_size.  If the volume is not expanded exactly to
 * new_size, set new_size to the new_size of the volume.
 */
static int reiser_expand(logical_volume_t *volume,
			 sector_count_t *new_size)
{
	int     rc = 0;
	char   *argv[7];
	pid_t   pidf;
	int     status;
	int     fds1[2];  // pipe for stdin 0=read 1=write
	int     fds2[2];  // pipe for stderr and stdout 0=-read,1=write
	int     bytes_read;
	char    *buffer = NULL;

	LOG_ENTRY();

	if (!reiser_version) {
		MESSAGE(_("ReiserFS utils were either not found or not at the "
			  "proper version. The fsck and mkfs utilities must be "
			  "installed using the standard names of mkreiserfs and "
			  "reiserfsck. ReiserFS utils must be version 3.x.0 or "
			  "later to function properly with this FSIM.Please get "
			  "the current version of ReiserFS utils from "
			  "http://www.namesys.com\n"));
		LOG_EXIT_INT(E_CANCELED);
		return E_CANCELED;
	}
	rc = pipe(fds1);
	if (rc) {
		rc = errno;
		LOG_EXIT_INT(rc);
		return(rc);
	}
	rc = pipe(fds2);
	if (rc) {
		rc = errno;
		close(fds1[0]);
		close(fds1[1]);
		LOG_EXIT_INT(rc);
		return(rc);
	}
	if (!(buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN))) {
		close(fds1[0]);
		close(fds1[1]);
		close(fds2[0]);
		close(fds2[1]);
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return(rc);
	}

	argv[0] = "resize_reiserfs";
	argv[1] = volume->dev_node;
	argv[2] = NULL;

	pidf = EngFncs->fork_and_execvp(volume, argv, fds1, fds2, fds2);
	if (pidf != -1) {
		// WARNING: Do Not close read handle of stdin or
		// you will cause a SIGPIPE if you write after the
		// child process has gone away.

		/* wait for child to complete */
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		while (!(waitpid( pidf, &status, WNOHANG ))) {
			bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				MESSAGE(_("expand output: \n\n%s"),buffer);
				memset(buffer,0,bytes_read); //clear out message
			}
			usleep(10000); // don't hog all the cpu
		}
		// do final read, just in case we missed some
		bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
		if (bytes_read > 0) {
			MESSAGE(_("expand output: \n\n%s"),buffer);
		}
		if ( WIFEXITED(status) ) {
			/* get expand exit code */
			LOG_DEFAULT("Expand completed with rc = %d \n",status);
			rc = WEXITSTATUS(status);
		} else {
			rc = EINTR;
		}
	} else {
		rc = EIO;
	}
	EngFncs->engine_free(buffer);
	reiser_get_fs_size(volume, new_size);

	close(fds1[0]);
	close(fds1[1]);
	close(fds2[0]);
	close(fds2[1]);
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * "unmkfs" the volume
 */
static int reiser_unmkfs(logical_volume_t *volume)
{
	int rc = EINVAL;
	int  fd;
	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
	} else {

		fd = EngFncs->open_volume(volume, O_RDWR|O_EXCL, 0);
		if (fd < 0) {
			rc = -fd;
                        LOG_EXIT_INT(rc);
			return rc;
		}

		if ( volume->private_data ) {
			memset(volume->private_data, 0, SB_SIZE);
			/* zero primary reiserfs_super_block */
			rc =  fsim_rw_diskblocks( volume, fd, REISERFS_DISK_OFFSET_IN_BYTES, SB_SIZE, volume->private_data, PUT );
			/* zero secondary reiserfs_super_block */
			rc |= fsim_rw_diskblocks( volume, fd, REISERFS_OLD_DISK_OFFSET_IN_BYTES, SB_SIZE, volume->private_data, PUT );
			/* clear private data */
			EngFncs->engine_free(volume->private_data);
		} else {
			rc = ERROR;
		}

		fd = EngFncs->close_volume(volume, fd);
		volume->private_data = NULL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Shrink the volume to new_size.  If the volume is not shrunk exactly to
 * new_size, set new_size to the new_size of the volume.
 */
static int reiser_shrink(logical_volume_t *volume,
			 sector_count_t requested_size,
			 sector_count_t *new_size)
{
	int     rc = 0;
	char   *argv[7];
	pid_t   pidf;
	int     status;
	int     fds1[2];  // pipe for stdin 0=read 1=write
	int     fds2[2];  // pipe for stderr and stdout 0=-read,1=write
	int     bytes_read;
	char    *buffer = NULL;
	char    size_buf[128];

	LOG_ENTRY();

	if (!reiser_version) {
		MESSAGE(_("ReiserFS utils were either not found or not at the "
			  "proper version. The fsck and mkfs utilities must be "
			  "installed using the standard names of mkreiserfs and "
			  "reiserfsck. ReiserFS utils must be version 3.x.0 or "
			  "later to function properly with this FSIM. Please get "
			  "the current version of ReiserFS utils from "
			  "http://www.namesys.com\n"));
		LOG_EXIT_INT(E_CANCELED);
		return E_CANCELED;
	}
	/* don't shrink if mounted */
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		rc = EBUSY;
		LOG_EXIT_INT(rc);
		return rc;
	}

	rc = pipe(fds1);
	if (rc) {
		rc = errno;
		LOG_EXIT_INT(rc);
		return(rc);
	}
	rc = pipe(fds2);
	if (rc) {
		rc = errno;
		close(fds1[0]);
		close(fds1[1]);
		LOG_EXIT_INT(rc);
		return(rc);
	}
	if (!(buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN))) {
		close(fds1[0]);
		close(fds1[1]);
		close(fds2[0]);
		close(fds2[1]);
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return(rc);
	}

	argv[0] = "resize_reiserfs";
	argv[1] = "-q";
	sprintf(size_buf,"-s%"PRIu64"", (sector_count_t)requested_size * (sector_count_t)EVMS_VSECTOR_SIZE);	//set newsize in bytes
	argv[2] = size_buf;
	argv[3] = volume->dev_node;
	argv[4] = NULL;

	pidf = EngFncs->fork_and_execvp(volume, argv, fds1, fds2, fds2);
	if (pidf != -1) {
		// WARNING: Do Not close read handle of stdin or
		// you will cause a SIGPIPE if you write after the
		// child process has gone away.
		write(fds1[1], "Yes\n",4);

		/* wait for child to complete */
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		while (!(waitpid( pidf, &status, WNOHANG ))) {
			bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				MESSAGE(_("Shrink output: \n%s"),buffer);
				memset(buffer,0,bytes_read); //clear out message
			}
			usleep(10000); // don't hog all the cpu
		}
		// do final read, just in case we missed some
		bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
		if (bytes_read > 0) {
			MESSAGE(_("Shrink output: \n%s"),buffer);
		}
		if ( WIFEXITED(status) ) {
			/* get shrink exit code */
			LOG_DEFAULT("Shrink completed with rc = %d \n",status);
			rc = WEXITSTATUS(status);
		} else {
			rc = EINTR;
		}
	} else {
		rc = EIO;
	}
	EngFncs->engine_free(buffer);
	reiser_get_fs_size(volume, new_size);

	close(fds1[0]);
	close(fds1[1]);
	close(fds2[0]);
	close(fds2[1]);
	LOG_EXIT_INT(rc);
	return rc;
}



/*
 * Format the volume.
 */
static int reiser_mkfs(logical_volume_t *volume, option_array_t *options)
{
	int  rc = 0;

	LOG_ENTRY();

	/* don't format if mounted */
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		rc = EBUSY;
		LOG_EXIT_INT(rc);
		return rc;
	}
	if (!reiser_version) {
		MESSAGE(_("ReiserFS utils were either not found or not at the "
			  "proper version. The fsck and mkfs utilities must be "
			  "installed using the standard names of mkreiserfs and "
			  "reiserfsck. ReiserFS utils must be version 3.x.0 or "
			  "later to function properly with this FSIM. Please get "
			  "the current version of ReiserFS utils from "
			  "http://www.namesys.com\n"));
		LOG_EXIT_INT(E_CANCELED);
		return E_CANCELED;
	}

	rc = fsim_mkfs(volume, options);

	/* probe to set up private data */
	if ( !rc ) {
		rc = reiser_probe(volume);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Forget about this volume.  Don't remove the file system.  Just clean
 * up any data structures you may have associated with it.
 */
static int reiser_discard(logical_volume_t *volume)
{
	LOG_ENTRY();

	if (volume->private_data) {
		EngFncs->engine_free(volume->private_data);
		volume->private_data = NULL;
	}

	LOG_EXIT_INT(0);
	return 0;
}

/*
 * Run fsck on the volume.
 */
static int reiser_fsck(logical_volume_t *volume, option_array_t *options)
{
	int rc = EINVAL;

	LOG_ENTRY();
	if (!reiser_version) {
		MESSAGE(_("ReiserFS utils were either not found or not at the "
			  "proper version. The fsck and mkfs utilities must be "
			  "installed using the standard names of mkreiserfs and "
			  "reiserfsck. ReiserFS utils must be version 3.x.0 or "
			  "later to function properly with this FSIM. Please get "
			  "the current version of ReiserFS utils from "
			  "http://www.namesys.com\n"));
		LOG_EXIT_INT(E_CANCELED);
		return E_CANCELED;
	}

	rc = fsim_fsck( volume, options );

	/*
	 * If reiserfsck returns FSCK_CORRECTED, the
	 * file system is clean, so return FSCK_OK.
	 */
	if ( rc == FSCK_CORRECTED ) {
		rc = FSCK_OK;
		/*
		 * The value of FSCK_CORRECTED is the same as
		 * EPERM, so fsim_fsck will return -1 for EPERM.
		 */
	} else if (rc == -1) {
		rc = EPERM;
	}

	/*
	 * If the volume is mounted, reiserfsck checked READ ONLY
	 * regardless of options specified.  If the check was READ ONLY
	 * and errors were found, let the user know how to fix them.
	 */
	if (EngFncs->is_mounted(volume->dev_node, NULL) && rc) {
		MESSAGE( _("%s is mounted."
			   "reiserfsck (fsck.reiserfs) checked the volume READ ONLY and found errors but did not fix them."
			   "Unmount %s and run reiserfsck (fsck.reiserfs) again to repair the file system."),
			 volume->dev_node, volume->dev_node );
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Return the total number of supported options for the specified task.
 */
static int reiser_get_option_count(task_context_t *context)
{
	int count = 0;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_mkfs:
		count = MKFS_OPTIONS_COUNT;
		break;
	case EVMS_Task_fsck:
		count = FSCK_OPTIONS_COUNT;
		break;
	case EVMS_Task_Expand:
		count = 0;
		break;
	default:
		count = -1;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}


/*
 * Initialize mkfs task acceptable objects by enumerating volumes, finding
 * those that have no FSIM claiming them and are of the proper size and
 * adding them to the acceptable objects list.
 */
static int fs_init_mkfs_acceptable_objects(task_context_t * context)
{
	int rc;
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;
	logical_volume_t * volume;

	LOG_ENTRY();

	rc = EngFncs->get_volume_list(NULL, NULL, 0, &global_volumes);
	if (!rc) {
		LIST_FOR_EACH(global_volumes, vol_list_iter, volume) {
			/* only mkfs unformatted volumes */
			if (volume->file_system_manager == NULL &&
			    !(EngFncs->is_mounted(volume->dev_node, NULL)) &&
			    (volume->vol_size > MINREISERFS)) {
				EngFncs->insert_thing(context->acceptable_objects,
						      volume,
						      INSERT_BEFORE,
						      NULL);
			}
		}
		EngFncs->destroy_list(global_volumes);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Fill in the initial list of acceptable objects.  Fill in the minimum and
 * maximum nuber of objects that must/can be selected.  Set up all initial
 * values in the option_descriptors in the context record for the given
 * task.  Some fields in the option_descriptor may be dependent on a
 * selected object.  Leave such fields blank for now, and fill in during the
 * set_objects call.
 */
static int reiser_init_task(task_context_t *context)
{
	int rc = 0;
	value_list_t * value_list;

	LOG_ENTRY();

	context->min_selected_objects = 0;
	context->max_selected_objects = 0;

	/* Parameter check */
	if (!context) {
		rc = EFAULT;
                LOG_EXIT_INT(rc);
		return rc;
	}

	switch (context->action) {
	
	case EVMS_Task_mkfs:

		rc = fs_init_mkfs_acceptable_objects(context);

		if (rc) {
			break;
		}

		if (!reiser_version) {
			MESSAGE(_("ReiserFS utils were either not found or not "
				  "at the proper version. The fsck and mkfs "
				  "utilities must be installed using the "
				  "standard names of mkreiserfs and reiserfsck. "
				  "ReiserFS utils must be version 3.x.0 or "
				  "later to function properly with this FSIM. "
				  "Please get the current version of ReiserFS "
				  "utils from http://www.namesys.com\n"));
			rc = E_CANCELED;
			break;
		}
		context->min_selected_objects = 1;
		context->max_selected_objects = 1;

		context->option_descriptors->count = MKFS_OPTIONS_COUNT;
		/* Set version display option */
		context->option_descriptors->option[MKFS_VERSION_INDEX].name = EngFncs->engine_strdup("version" );
		context->option_descriptors->option[MKFS_VERSION_INDEX].title = EngFncs->engine_strdup(_("Version of mkreiserfs"));
		context->option_descriptors->option[MKFS_VERSION_INDEX].tip = EngFncs->engine_strdup(_("Informational only: this is the current version of ResierFS utitlities found on your system."));
		context->option_descriptors->option[MKFS_VERSION_INDEX].help = NULL;
		context->option_descriptors->option[MKFS_VERSION_INDEX].type = EVMS_Type_String;
		context->option_descriptors->option[MKFS_VERSION_INDEX].unit = EVMS_Unit_None;
		context->option_descriptors->option[MKFS_VERSION_INDEX].min_len = 1;
		context->option_descriptors->option[MKFS_VERSION_INDEX].max_len = 32;
		context->option_descriptors->option[MKFS_VERSION_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		context->option_descriptors->option[MKFS_VERSION_INDEX].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[MKFS_VERSION_INDEX].value.s = EngFncs->engine_alloc(32 + 1);
		strncpy(context->option_descriptors->option[MKFS_VERSION_INDEX].value.s,reiser_version_text,31);


		/* Set Volume Label option */
		context->option_descriptors->option[MKFS_SETVOL_INDEX].name = EngFncs->engine_strdup("vollabel" );
		context->option_descriptors->option[MKFS_SETVOL_INDEX].title = EngFncs->engine_strdup(_("Volume Label"));
		context->option_descriptors->option[MKFS_SETVOL_INDEX].tip = EngFncs->engine_strdup(_("Set the volume label for the file system."));
		context->option_descriptors->option[MKFS_SETVOL_INDEX].help = NULL;
		context->option_descriptors->option[MKFS_SETVOL_INDEX].type = EVMS_Type_String;
		context->option_descriptors->option[MKFS_SETVOL_INDEX].unit = EVMS_Unit_None;
		context->option_descriptors->option[MKFS_SETVOL_INDEX].min_len = 1;
		context->option_descriptors->option[MKFS_SETVOL_INDEX].max_len = 16;
		context->option_descriptors->option[MKFS_SETVOL_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED | EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		context->option_descriptors->option[MKFS_SETVOL_INDEX].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[MKFS_SETVOL_INDEX].value.s = EngFncs->engine_alloc(16 + 1);
		if (reiser_version < 2) { //
			context->option_descriptors->option[MKFS_SETVOL_INDEX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
		}

		/* Log size option */
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].name = EngFncs->engine_strdup("logsize" );
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].title = EngFncs->engine_strdup(_("Log Size"));
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].tip = EngFncs->engine_strdup(_("Set log size (in kilobytes).  Default log size is 8193 4k Blocks."));
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].help = NULL;
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].type = EVMS_Type_Unsigned_Int32;
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].unit = EVMS_Unit_Kilobytes;
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED | EVMS_OPTION_FLAGS_INACTIVE;
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint_type = EVMS_Collection_Range;
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
		if (context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint.range == NULL) {
			rc = ENOMEM;
			LOG_EXIT_INT(rc);
			return rc;
		}
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint.range->min.ui32 = 2052;
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint.range->max.ui32 = 130996;
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint.range->increment.ui32 = 4;
		context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].value.ui32 = 32772;

		context->min_selected_objects = 1;
		context->max_selected_objects = 1;
		break;

	case EVMS_Task_fsck:
		if (!reiser_version) {
			MESSAGE(_("ReiserFS utils were either not found or not "
				  "at the proper version. The fsck and mkfs "
				  "utilities must be installed using the "
				  "standard names of mkreiserfs and "
				  "reiserfsck. ReiserFS utils must be version "
				  "3.x.0 or later to function properly with this "
				  "FSIM. Please get the current version of "
				  "ReiserFS utils from http://www.namesys.com\n"));
			rc = E_CANCELED;
			break;
		}

		context->min_selected_objects = 0;
		context->max_selected_objects = 0;

		context->option_descriptors->count = FSCK_OPTIONS_COUNT;

		context->option_descriptors->option[0].constraint.list =
		EngFncs->engine_alloc(3 * sizeof(value_t) + sizeof(value_list_t));
		value_list = context->option_descriptors->option[0].constraint.list;
		if (value_list) {
			if (EngFncs->is_mounted(context->volume->dev_node, NULL)) {
				value_list->count = 1;
				value_list->value[0].s = EngFncs->engine_strdup(_("Check Read-Only"));
			} else {
				value_list->count = 3;
				value_list->value[0].s = EngFncs->engine_strdup(_("Check Read-Only"));
				value_list->value[1].s = EngFncs->engine_strdup(_("Fix"));
				value_list->value[2].s = EngFncs->engine_strdup(_("Rebuild Tree"));
			}
		} else {
			LOG_EXIT_INT(rc);
			return(rc);
		}

		/* MODE option */
		context->option_descriptors->option[FSCK_MODE_INDEX].name = EngFncs->engine_strdup("mode");
		context->option_descriptors->option[FSCK_MODE_INDEX].title = EngFncs->engine_strdup(_("Mode"));
		context->option_descriptors->option[FSCK_MODE_INDEX].tip = EngFncs->engine_strdup(_("Mode of operation"));
		context->option_descriptors->option[FSCK_MODE_INDEX].help = NULL;
		context->option_descriptors->option[FSCK_MODE_INDEX].max_len = EVMS_VOLUME_NAME_SIZE;
		context->option_descriptors->option[FSCK_MODE_INDEX].min_len = 1;
		context->option_descriptors->option[FSCK_MODE_INDEX].type = EVMS_Type_String;
		context->option_descriptors->option[FSCK_MODE_INDEX].unit = EVMS_Unit_None;
		/* if volume is mounted, only possible reiserfsck options is READONLY */
		context->option_descriptors->option[FSCK_MODE_INDEX].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[FSCK_MODE_INDEX].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);
		strcpy(context->option_descriptors->option[FSCK_MODE_INDEX].value.s,FSCK_READONLY);
		context->option_descriptors->option[FSCK_MODE_INDEX].flags = 0;

		break;

	case EVMS_Task_Expand:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);

	return rc;

}



/*
 * Examine the specified value, and determine if it is valid for the task
 * and option_descriptor index. If it is acceptable, set that value in the
 * appropriate entry in the option_descriptor. The value may be adjusted
 * if necessary/allowed. If so, set the effect return value accordingly.
 */
static int reiser_set_option(task_context_t *context,
			     u_int32_t index,
			     value_t *value,
			     task_effect_t *effect)
{
	int  rc= 0;

	LOG_ENTRY();

	/* Parameter check */
	if (!context || !value || !effect) {
		rc = EFAULT;
		LOG_EXIT_INT(rc);
		return rc;
	}

	switch (context->action) {
	
	case EVMS_Task_mkfs:
		switch (index) {
		
		case MKFS_VERSION_INDEX:
			/* user typed over version field option set? */
			// ignore and reload from the option descriptor.
			*effect |= EVMS_Effect_Reload_Options;
			break;

		case MKFS_SETVOL_INDEX:
			/* 'set volume label' option set? */
			strncpy(context->option_descriptors->option[index].value.s, value->s, 16);
			break;

		case MKFS_SETLOGSIZE_INDEX:
			/* 'set log size' option set? */
			context->option_descriptors->option[index].value.ui32 = value->ui32;
			break;

		default:
			break;
		}
		break;

	case EVMS_Task_fsck:
		switch (index) {
		
		case FSCK_MODE_INDEX:
			/* if mounted, only allow read-only */
			if (EngFncs->is_mounted(context->volume->dev_node, NULL)) {
				if (!(strcmp(value->s,FSCK_FIX)) || !(strcmp(value->s,FSCK_REBUILD_TREE))) {
					strcpy(context->option_descriptors->option[FSCK_MODE_INDEX].value.s,FSCK_READONLY);
					MESSAGE(_("Only Check mode is alllowed when the volume is mounted."));
					*effect |= EVMS_Effect_Reload_Options;
				}
			} else {
				if (!(strcmp(value->s,FSCK_FIX)) || !(strcmp(value->s,FSCK_REBUILD_TREE)) ||
				    !(strcmp(value->s,FSCK_READONLY))) {
					strcpy(context->option_descriptors->option[FSCK_MODE_INDEX].value.s,value->s);
				}
				if (strcmp(value->s,FSCK_READONLY)) {
					MESSAGE(_("Selecting a mode other than read only will cause "
						  "default responses to be given to any questions asked by the reiserfsck (fsck.reiserfs) "
						  "utitlity.  If you do not wish to run in this mode, you must run reiserfsck (fsck.reiserfs) "
						  "from the command line."));
				}
			}
			break;
		default:
			break;
		}
		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Validate the volumes in the selected_objects list in the task context.
 * Remove from the selected objects lists any volumes which are not
 * acceptable.  For unacceptable volumes, create a declined_handle_t
 * structure with the reason why it is not acceptable, and add it to the
 * declined_volumes list.  Modify the accepatble_objects list in the task
 * context as necessary based on the selected objects and the current
 * settings of the options.  Modify any option settings as necessary based
 * on the selected objects.  Return the appropriate task_effect_t settings
 * if the object list(s), minimum or maximum objects selected, or option
 * settings have changed.
 */
static int reiser_set_volumes(task_context_t *context,
			      list_anchor_t declined_volumes,
			      task_effect_t *effect)
{
	int  rc = 0;
	logical_volume_t * vol;

	LOG_ENTRY();

	if (context->action == EVMS_Task_mkfs) {

		/* get the selected volume */
		vol = EngFncs->first_thing(context->selected_objects, NULL);

		if (vol) {
			if (EngFncs->is_mounted(vol->dev_node, NULL)) {
				/* If mounted, can't mkfs.reiser. */
				rc = EBUSY;
			} else {
				if ( vol->vol_size < MINREISERFS) {
					char number_buffer[64];

					sprintf(number_buffer, "%"PRIu64, vol->vol_size << EVMS_VSECTOR_SIZE_SHIFT );
					MESSAGE( _("The size of volume %s is %s bytes.  "
						   "mkfs.reiserfs requires a minimum of %u bytes to build the ReiserFS file system."),
						   vol->name, number_buffer, MINREISERFS );
					rc = EPERM;
				}
			}
		} else {
			rc = ENODATA;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Return any additional information that you wish to provide about the
 * volume.  The Engine privides an external API to get the information
 * stored in the logical_volume_t.  This call is to get any other
 * information about the volume that is not specified in the
 * logical_volume_t.  Any piece of information you wish to provide must be
 * in an extended_info_t structure.  Use the Engine's engine_alloc() to
 * allocate the memory for the extended_info_t.  Also use engine_alloc() to
 * allocate any strings that may go into the extended_info_t.  Then use
 * engine_alloc() to allocate an extended_info_array_t with enough entries
 * for the number of exteneded_info_t structures you are returning.  Fill
 * in the array and return it in *info.
 * If you have extended_info_t descriptors that themselves may have more
 * extended information, set the EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag
 * in the extended_info_t flags field.  If the caller wants more information
 * about a particular extended_info_t item, this API will be called with a
 * pointer to the sotrage_object_t and with a pointer to the name of the
 * extended_info_t item.  In that case, return an extended_info_array_t with
 * further information about the item.  Each of those items may have the
 * EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag set if you desire.  It is your
 * resposibility to give the items unique names so that you know which item
 * the caller is asking additional information for.  If info_name is NULL,
 * the caller just wants top level information about the object.
 */
static int reiser_get_volume_info(logical_volume_t *volume,
				  char *info_name,
				  extended_info_array_t **info)
{
	int rc = EINVAL;
	extended_info_array_t  *Info;
	struct reiserfs_super_block      *sb_ptr = (struct reiserfs_super_block *)volume->private_data;

	LOG_ENTRY();

	if (!sb_ptr) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	if (info_name == NULL) {
		reiser_get_fs_limits(volume,		   // reset limits.
				     &volume->min_fs_size,
				     &volume->max_vol_size,
				     &volume->max_fs_size);

		Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( 5 * sizeof(extended_info_t) ) );
		if (Info) {

			Info->count = 5;

			Info->info[0].name = EngFncs->engine_strdup( "MagicNumber" );
			Info->info[0].title = EngFncs->engine_strdup( _("Magic Number") );
			Info->info[0].desc = EngFncs->engine_strdup( _("Magic Number") );
			Info->info[0].type               = EVMS_Type_String;
			Info->info[0].unit               = EVMS_Unit_None;
			Info->info[0].value.s = EngFncs->engine_strdup(sb_ptr->s_v1.s_magic);
			Info->info[0].collection_type    = EVMS_Collection_None;
			memset( &Info->info[0].group, 0, sizeof(group_info_t));

			Info->info[1].name = EngFncs->engine_strdup( "Version" );
			Info->info[1].title = EngFncs->engine_strdup( _("Version Number") );
			Info->info[1].desc = EngFncs->engine_strdup( _("Version Number") );
			Info->info[1].type               = EVMS_Type_Unsigned_Int16;
			Info->info[1].unit               = EVMS_Unit_None;
			Info->info[1].value.ui16         = get_sb_version(sb_ptr);
			Info->info[1].collection_type    = EVMS_Collection_None;
			memset( &Info->info[1].group, 0, sizeof(group_info_t));

			Info->info[2].name = EngFncs->engine_strdup( "VolLabel" );
			Info->info[2].title = EngFncs->engine_strdup( _("Volume Label") );
			Info->info[2].desc = EngFncs->engine_strdup( _("File system volume label") );
			Info->info[2].type               = EVMS_Type_String;
			Info->info[2].unit               = EVMS_Unit_None;
			Info->info[2].value.s = EngFncs->engine_strdup(sb_ptr->s_label);
			Info->info[2].collection_type    = EVMS_Collection_None;
			memset( &Info->info[2].group, 0, sizeof(group_info_t));

			Info->info[3].name = EngFncs->engine_strdup( "UsableSize" );
			Info->info[3].title = EngFncs->engine_strdup( _("File System Size") );
			Info->info[3].desc = EngFncs->engine_strdup( _("Size of the file system (not including journal)") );
			Info->info[3].type               = EVMS_Type_Unsigned_Int64;
			Info->info[3].unit               = EVMS_Unit_Sectors;
			Info->info[3].value.ui64         = (get_sb_block_count(sb_ptr)
							    - get_jp_journal_size(sb_ptr)) * (get_sb_block_size(sb_ptr)/EVMS_VSECTOR_SIZE);
			Info->info[3].collection_type    = EVMS_Collection_None;
			memset( &Info->info[3].group, 0, sizeof(group_info_t));

			Info->info[4].name = EngFncs->engine_strdup( "LogSize" );
			Info->info[4].title = EngFncs->engine_strdup( _("Journal Size") );
			Info->info[4].desc = EngFncs->engine_strdup( _("Size of the file system journal") );
			Info->info[4].type               = EVMS_Type_Unsigned_Int64;
			Info->info[4].unit               = EVMS_Unit_Sectors;
			Info->info[4].value.ui64         = get_jp_journal_size(sb_ptr) * (get_sb_block_size(sb_ptr)/EVMS_VSECTOR_SIZE);
			Info->info[4].collection_type    = EVMS_Collection_None;
			memset( &Info->info[4].group, 0, sizeof(group_info_t));

			*info = Info;

			rc = 0;
		} else {
			rc = ENOMEM;
		}

	} else {
		/* There is no more information about any of the descriptors. */
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Returns Plugin specific information ...
 */
static int reiser_get_plugin_info(char *descriptor_name,
				  extended_info_array_t **info)
{
	int                      rc = EINVAL;
	extended_info_array_t   *Info;
	char                     version_string[64];
	char                     required_engine_api_version_string[64];
	char                     required_fsim_api_version_string[64];


	LOG_ENTRY();

	if (info) {

		if (descriptor_name == NULL) {
			*info = NULL;	  // init to no info returned

			Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (6*sizeof(extended_info_t))  );
			if (Info) {

				Info->count = 6;

				sprintf(version_string, "%d.%d.%d",
					MAJOR_VERSION,
					MINOR_VERSION,
					PATCH_LEVEL );

				sprintf(required_engine_api_version_string, "%d.%d.%d",
					my_plugin_record->required_engine_api_version.major,
					my_plugin_record->required_engine_api_version.minor,
					my_plugin_record->required_engine_api_version.patchlevel );

				sprintf(required_fsim_api_version_string, "%d.%d.%d",
					my_plugin_record->required_plugin_api_version.fsim.major,
					my_plugin_record->required_plugin_api_version.fsim.minor,
					my_plugin_record->required_plugin_api_version.fsim.patchlevel );

				Info->info[0].name = EngFncs->engine_strdup( "Short Name" );
				Info->info[0].title = EngFncs->engine_strdup( _("Short Name") );
				Info->info[0].desc = EngFncs->engine_strdup( _("A short name given to this plug-in") );
				Info->info[0].type               = EVMS_Type_String;
				Info->info[0].unit               = EVMS_Unit_None;
				Info->info[0].value.s = EngFncs->engine_strdup( my_plugin_record->short_name );
				Info->info[0].collection_type    = EVMS_Collection_None;
				memset( &Info->info[0].group, 0, sizeof(group_info_t));

				Info->info[1].name = EngFncs->engine_strdup( "Long Name" );
				Info->info[1].title = EngFncs->engine_strdup( _("Long Name") );
				Info->info[1].desc = EngFncs->engine_strdup( _("A longer, more descriptive name for this plug-in") );
				Info->info[1].type               = EVMS_Type_String;
				Info->info[1].unit               = EVMS_Unit_None;
				Info->info[1].value.s = EngFncs->engine_strdup( my_plugin_record->long_name );
				Info->info[1].collection_type    = EVMS_Collection_None;
				memset( &Info->info[1].group, 0, sizeof(group_info_t));

				Info->info[2].name = EngFncs->engine_strdup( "Type" );
				Info->info[2].title = EngFncs->engine_strdup( _("Plug-in Type") );
				Info->info[2].desc = EngFncs->engine_strdup( _("There are various types of plug-ins, each responsible for some kind of storage object or logical volume.") );
				Info->info[2].type               = EVMS_Type_String;
				Info->info[2].unit               = EVMS_Unit_None;
				Info->info[2].value.s = EngFncs->engine_strdup( _("File System Interface Module") );
				Info->info[2].collection_type    = EVMS_Collection_None;
				memset( &Info->info[2].group, 0, sizeof(group_info_t));

				Info->info[3].name = EngFncs->engine_strdup( "Version" );
				Info->info[3].title = EngFncs->engine_strdup( _("Plug-in Version") );
				Info->info[3].desc = EngFncs->engine_strdup( _("This is the version number of the plug-in."));
				Info->info[3].type               = EVMS_Type_String;
				Info->info[3].unit               = EVMS_Unit_None;
				Info->info[3].value.s = EngFncs->engine_strdup( version_string );
				Info->info[3].collection_type    = EVMS_Collection_None;
				memset( &Info->info[3].group, 0, sizeof(group_info_t));

				Info->info[4].name = EngFncs->engine_strdup( "Required Engine Services Version" );
				Info->info[4].title = EngFncs->engine_strdup( _("Required Engine Services Version") );
				Info->info[4].desc = EngFncs->engine_strdup( _("This is the version of the Engine services that this plug-in requires.  It will not run on older versions of the Engine services.") );
				Info->info[4].type               = EVMS_Type_String;
				Info->info[4].unit               = EVMS_Unit_None;
				Info->info[4].value.s = EngFncs->engine_strdup( required_engine_api_version_string );
				Info->info[4].collection_type    = EVMS_Collection_None;
				memset( &Info->info[4].group, 0, sizeof(group_info_t));

				Info->info[5].name = EngFncs->engine_strdup( "Required Engine FSIM API Version" );
				Info->info[5].title = EngFncs->engine_strdup( _("Required Engine FSIM API Version") );
				Info->info[5].desc = EngFncs->engine_strdup( _("This is the version of the Engine FSIM API that this plug-in requires.  It will not run on older versions of the Engine FSIM API.") );
				Info->info[5].type               = EVMS_Type_String;
				Info->info[5].unit               = EVMS_Unit_None;
				Info->info[5].value.s = EngFncs->engine_strdup( required_fsim_api_version_string );
				Info->info[5].collection_type    = EVMS_Collection_None;
				memset( &Info->info[5].group, 0, sizeof(group_info_t));

				*info = Info;

				rc = 0;
			} else {
				rc = ENOMEM;
			}

		} else {
			/* There is no more information on any of the descriptors. */
			rc = EINVAL;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * How much can file system expand?
 */
static int reiser_can_expand_by(logical_volume_t *volume,
				sector_count_t *delta)
{
	int  rc = 0;

	LOG_ENTRY();

	if (!reiser_version) {
		rc = ENOSYS;
	} else {
		reiser_get_fs_limits(volume,		   // reset limits.
				     &volume->min_fs_size,
				     &volume->max_vol_size,
				     &volume->max_fs_size);
		if (*delta > volume->max_fs_size - volume->fs_size) {
			*delta = volume->max_fs_size - volume->fs_size;
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * How much can file system shrink?
 */
static int reiser_can_shrink_by(logical_volume_t *volume,
				sector_count_t *delta)
{
	int  rc = 0;

	LOG_ENTRY();
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		rc = EBUSY; // If mounted, can't shrink
	} else if (!reiser_version) {
		rc = ENOSYS;
	} else {
		reiser_get_fs_limits(volume,		   // reset limits.
				     &volume->min_fs_size,
				     &volume->max_vol_size,
				     &volume->max_fs_size);
		if (*delta > volume->fs_size - volume->min_fs_size) {
			*delta = volume->fs_size - volume->min_fs_size;
		}
		if (volume->min_fs_size >= volume->fs_size) {
			rc = ENOSPC;
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}

static fsim_functions_t fsim_ops = {
	.setup_evms_plugin	= reiser_setup,
	.cleanup_evms_plugin	= reiser_cleanup,
	.probe			= reiser_probe,
	.can_mkfs		= reiser_can_mkfs,
	.can_unmkfs		= reiser_can_unmkfs,
	.can_fsck		= reiser_can_fsck,
	.get_fs_size		= reiser_get_fs_size,
	.get_fs_limits		= reiser_get_fs_limits,
	.can_expand_by		= reiser_can_expand_by,
	.can_shrink_by		= reiser_can_shrink_by,
	.expand			= reiser_expand,
	.shrink			= reiser_shrink,
	.mkfs			= reiser_mkfs,
	.discard		= reiser_discard,
	.fsck			= reiser_fsck,
	.unmkfs			= reiser_unmkfs,
	.get_option_count	= reiser_get_option_count,
	.init_task		= reiser_init_task,
	.set_option		= reiser_set_option,
	.set_volumes		= reiser_set_volumes,
	.get_volume_info	= reiser_get_volume_info,
	.get_plugin_info	= reiser_get_plugin_info
};

plugin_record_t reiser_plugin_record = {
	.id = SetPluginID(EVMS_OEM_IBM,
			  EVMS_FILESYSTEM_INTERFACE_MODULE,
			  FS_TYPE_REISERFS),
	.version= {
		.major		= MAJOR_VERSION,
		.minor		= MINOR_VERSION,
		.patchlevel	= PATCH_LEVEL
	},
	.required_engine_api_version = {
		.major		= 15,
		.minor		= 0,
		.patchlevel	= 0
	},
	.required_plugin_api_version = {
		.fsim = {
			.major		= 11,
			.minor		= 0,
			.patchlevel	= 0
		}
	},
	.short_name = "ReiserFS",
	.long_name = "ReiserFS File System Interface Module",
	.oem_name = "IBM",
	.functions = {
		.fsim = &fsim_ops
	},
	.container_functions = NULL
};

plugin_record_t *evms_plugin_records[] = {
	&reiser_plugin_record,
	NULL
};

