/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef __SNAPSHOT_H__
#define __SNAPSHOT_H__ 1

#define SNAPSHOT_FEATURE_ID		104
#define SNAPSHOT_SIGNATURE		0x536e4170	/* SnAp */
#define SNAPSHOT_ORIGIN_SIGNATURE	0x4f724967	/* OrIg */

/* Status flags */
#define SNAPSHOT			(1 << 0)
#define SNAPSHOT_ORIGIN			(1 << 1)
#define SNAPSHOT_INVALID		(1 << 2)
#define SNAPSHOT_WRITEABLE 		(1 << 3)
#define SNAPSHOT_DELETE_PENDING		(1 << 4)
#define SNAPSHOT_ROLLBACK		(1 << 5)

/* Private ioctl commands */
#define SNAPSHOT_QUERY_PERCENT_FULL	1
#define SNAPSHOT_START_ROLLBACK		2
#define SNAPSHOT_CHECK_STATE		3

/* Snapshot chunk sizes can range from 8kB to 1MB. The
 * default size is 64kB. These values are in sectors.
 */
#define SNAPSHOT_DEFAULT_CHUNK_SIZE	128
#define SNAPSHOT_MIN_CHUNK_SIZE		16
#define SNAPSHOT_MAX_CHUNK_SIZE		2048

/* These are dependent on the definitions from
 * drivers/md/dm-exception-store.c in the kernel.
 */
#define SNAPSHOT_DM_MAGIC		0x70416e53
#define SNAPSHOT_DM_DISK_VERSION	1
#define SIZEOF_DISK_EXCEPTION		16

#define EXCEPTION_TABLE_ENTRIES(chunk_size) \
	((chunk_size) * EVMS_VSECTOR_SIZE / SIZEOF_DISK_EXCEPTION)

/**
 * struct snapshot_metadata
 * @signature:		0  : SNAPSHOT_SIGNATURE
 * @CRC:		4  :
 * @version:		8  : Major, minor, patchlevel
 * @flags:		20 : SNAPSHOT_*
 * @original_volume:	24 : Name of volume being snapshotted
 * @original_size:	152: In sectors
 * @total_chunks:	160: Number of data chunks this snapshot can hold
 * @chunk_size:		168: In sectors
 * @current_table_index	172: During rollback, the index of the current snapshot
 *			     exception table.
 * @current_table_entry	176: During rollback, the current entry within the
 *			     current exception table.
 *
 * On-disk metadata sector for EVMS Snapshot feature.
 */
typedef struct snapshot_metadata {
	u_int32_t		signature;
	u_int32_t		crc;
	struct evms_version	version;
	u_int32_t		flags;
	u_int8_t		origin_volume[128];
	u_int64_t		origin_size;
	u_int64_t		total_chunks;
	u_int32_t		chunk_size;
	u_int32_t		current_table_index;
	u_int32_t		current_table_entry;
} snapshot_metadata_t;

/*
 * Device-Mapper Snapshot Metadata
 *
 * The metadata for Device-Mapper snapshots is as follows:
 * Chunk   Contents
 * -----   --------
 *   0     dm_snapshot_header
 *   1     dm_disk_exception structures
 *            enough to fill one chunk: EXCEPTION_TABLE_ENTRIES(chunk_size)
 *  2-n    data chunks
 */

/**
 * struct dm_snapshot_header
 * @magic:	SNAPSHOT_DM_MAGIC
 * @valid:	Non-zero if the snapshot is valid.
 * @version:	SNAPSHOT_DM_DISK_VERSION
 * @chunk_size:	In sectors
 *
 * Header defined in drivers/md/dm-exception-store.c in the kernel. This
 * structure is found on the first sector of the snapshot object (only after
 * the snapshot has been initialized by Device-Mapper).
 **/
typedef struct dm_snapshot_header {
	u_int32_t magic;
	u_int32_t valid;
	u_int32_t version;
	u_int32_t chunk_size;
} dm_snapshot_header_t;

/**
 * struct dm_disk_exception
 * @old_chunk: Chunk index on origin
 * @new_chunk: Chunk index on snapshot
 *
 * Structure to describe one chunk that has been copied from the origin to
 * the snapshot.
 **/
typedef struct dm_disk_exception {
	u_int64_t old_chunk;
	u_int64_t new_chunk;
} dm_disk_exception_t;

/**
 * struct snapshot_volume
 * @signature:		SNAPSHOT or SNAPSHOT_ORIGIN
 * @parent:		Our exported object
 * @child:		Our imported object
 * @sibling:		A secondary child object. Hidden from the engine.
 * @origin:		The volume being snapshotted
 * @next:		List of snapshots for one original
 * @metadata:		Copy of the on-disk metadata
 * @flags:		SNAPSHOT_*
 * @count:		Number of snapshots for this origin.
 * @active_count:	Number of activated snapshots for this origin.
 *
 * The in-memory representation for both a snapshot and an original.
 */
typedef struct snapshot_volume {
	storage_object_t	* parent;
	storage_object_t	* child;
	storage_object_t	* sibling;
	struct snapshot_volume	* next;
	struct snapshot_volume	* origin;
	snapshot_metadata_t	* metadata;
	unsigned int		flags;
	unsigned int		count;
	unsigned int		active_count;
	int			percent_full;
} snapshot_volume_t;

/* Is this volume a snapshot or an origin? */
static inline int is_origin(snapshot_volume_t * volume)
{
	return (volume->flags & SNAPSHOT_ORIGIN);
}

/* Is this volume active (snapshot or origin)? */
static inline int is_active(snapshot_volume_t * volume)
{
	return (volume->parent->flags & SOFLAG_ACTIVE);
}

/* Mark this volume to be activated (snapshot or origin). */
static inline void schedule_for_activate(snapshot_volume_t * volume)
{
	volume->parent->flags |= SOFLAG_NEEDS_ACTIVATE;
}

static inline void activate_complete(snapshot_volume_t * volume)
{
	volume->parent->flags &= ~SOFLAG_NEEDS_ACTIVATE;
}

/* Is this volume waiting to be activated? */
static inline int activate_is_pending(snapshot_volume_t * volume)
{
	return (volume->parent->flags & SOFLAG_NEEDS_ACTIVATE);
}

/* Mark this volume to be deactivated. */
static inline void schedule_for_deactivate(snapshot_volume_t * volume)
{
	volume->parent->flags |= SOFLAG_NEEDS_DEACTIVATE;
}

static inline void deactivate_complete(snapshot_volume_t * volume)
{
	volume->parent->flags &= ~SOFLAG_NEEDS_DEACTIVATE;
}

/* Is this volume waiting to be deactivated? */
static inline int deactivate_is_pending(snapshot_volume_t * volume)
{
	return (volume->parent->flags & SOFLAG_NEEDS_DEACTIVATE);
}

/* Will this volume be deleted on deactivation? */
static inline int delete_is_pending(snapshot_volume_t * volume)
{
	return (volume->flags & SNAPSHOT_DELETE_PENDING);
}

/* Mark this volume to be deleted on deactivation. */
static inline void schedule_for_delete(snapshot_volume_t * volume)
{
	volume->flags |= SNAPSHOT_DELETE_PENDING;
}

/* Is this volume waiting to be committed? */
static inline int commit_is_pending(snapshot_volume_t * volume)
{
	return (volume->parent->flags & SOFLAG_DIRTY);
}

/* Mark this volume to be committed. */
static inline void schedule_for_commit(snapshot_volume_t * volume)
{
	volume->parent->flags |= SOFLAG_DIRTY;
}

static inline void commit_complete(snapshot_volume_t * volume)
{
	volume->parent->flags &= ~SOFLAG_DIRTY;
}

/* Is this snapshot waiting to be rolled-back? */
static inline int rollback_is_pending(snapshot_volume_t * volume)
{
	return (volume->flags & SNAPSHOT_ROLLBACK);
}

/* Mark this snapshot to be rolled-back to its origin. */
static inline void schedule_for_rollback(snapshot_volume_t * volume)
{
	volume->flags |= SNAPSHOT_ROLLBACK;
	volume->metadata->flags |= SNAPSHOT_ROLLBACK;
	volume->metadata->current_table_index = 0;
	volume->metadata->current_table_entry = 0;
}

static inline void rollback_complete(snapshot_volume_t * volume)
{
	volume->flags &= ~SNAPSHOT_ROLLBACK;
}

/* Is this snapshot invalid, full, or disabled? */
static inline int is_invalid(snapshot_volume_t * volume)
{
	return (volume->flags & SNAPSHOT_INVALID);
}

static inline void mark_invalid(snapshot_volume_t * volume)
{
	volume->flags |= SNAPSHOT_INVALID;
}

static inline void mark_valid(snapshot_volume_t * volume)
{
	volume->flags &= ~SNAPSHOT_INVALID;
}

extern engine_functions_t * EngFncs;
extern plugin_record_t * my_plugin_record;
extern plugin_record_t Snapshot_Plugin;

#include "snap_activate.h"
#include "snap_create.h"
#include "snap_discover.h"
#include "snap_options.h"
#include "snap_rollback.h"

#endif

