/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM Region Manager
 * File: evms2/engine/plugins/lvm/lvm_list.c
 *
 * Description: This file contains all functions related to manipulating the
 *              lists that are used by the LVM region manager.
 */ 

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"

/**
 * lvm_append_segment_to_container
 *
 * Append the specified new_segment to the end of the "consumed" list in
 * the specified container.
 **/
int lvm_append_segment_to_container(storage_object_t * new_segment,
				    storage_container_t * container)
{
	list_element_t itr;
	int rc = 0;

	LOG_ENTRY();

	itr = EngFncs->insert_thing(container->objects_consumed, new_segment,
				    INSERT_AFTER, NULL);
	if (itr) {
		/* Mark the segment as belonging to this container. */
		new_segment->consuming_container = container;
	} else {
		LOG_SERIOUS("Error adding object %s to container %s\n",
			    new_segment->name, container->name );
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_remove_segment_from_container
 *
 * Remove the specified segment from its "consuming" container.
 **/
int lvm_remove_segment_from_container(storage_object_t * segment)
{
	int rc = 0;

	LOG_ENTRY();

	if (segment->consuming_container) {
		/* Mark the segment as "unconsumed". */
		EngFncs->remove_thing(segment->consuming_container->objects_consumed,
				      segment);
		segment->consuming_container = NULL;
	} else {
		LOG_WARNING("Object %s is not in a container\n", segment->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_append_region_to_container
 *
 * Append the specified new_region at the end of the "producing" list in
 * the specified container.
 **/
int lvm_append_region_to_container(storage_object_t * new_region,
				   storage_container_t * container)
{
	list_element_t itr;
	int rc = 0;

	LOG_ENTRY();

	itr = EngFncs->insert_thing(container->objects_produced, new_region,
				    INSERT_AFTER, NULL);
	if (itr) {
		/* Mark the region as belonging to this container. */
		new_region->producing_container = container;
	} else {
		LOG_SERIOUS("Error adding region %s to container %s\n",
			    new_region->name, container->name);
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_remove_region_from_container
 *
 * Remove the specified region from its "producing" container.
 **/
int lvm_remove_region_from_container(storage_object_t * region)
{
	int rc = EINVAL;

	LOG_ENTRY();

	if (region->producing_container) {
		/* Mark the region as "unproduced". */
		EngFncs->remove_thing(region->producing_container->objects_produced,
				      region);
		region->producing_container = NULL;
	} else {
		LOG_WARNING("Region %s not in a container\n", region->name);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_append_region_to_segment
 *
 * Associate the specified region and segment as parent/child. Add the
 * region to the segment's "parent" list, and add the segment to the
 * region's "child" list. Also need to check for duplicates, so the
 * same region/segment don't get associated more than once.
 **/
int lvm_append_region_to_segment(storage_object_t * region,
				 storage_object_t * segment)
{
	list_element_t itr1, itr2;

	/* No LOG_ENTRY or LOG_EXIT calls. Writes too many messages to the log. */

	itr1 = EngFncs->insert_thing(segment->parent_objects, region,
				     INSERT_AFTER | EXCLUSIVE_INSERT, NULL);
	if (!itr1) {
		LOG_SERIOUS("Error adding region %s as a parent to object %s\n",
			    region->name, segment->name);
		return ENOMEM;
	}

	itr2 = EngFncs->insert_thing(region->child_objects, segment,
				     INSERT_AFTER | EXCLUSIVE_INSERT, NULL);
	if (!itr2) {
		LOG_SERIOUS("Error adding object %s as a child to region %s\n",
			    segment->name, region->name);
		EngFncs->delete_element(itr1);
		return ENOMEM;
	}

	return 0;
}

/**
 * lvm_remove_region_from_segment
 *
 * Remove the parent/child association between the specified region
 * and segment. Remove the region from the segment's "parent" list,
 * and remove the segment from the region's "child" list.
 **/
void lvm_remove_region_from_segment(storage_object_t * region,
				    storage_object_t * segment)
{
	LOG_ENTRY();

	EngFncs->remove_thing(segment->parent_objects, region);
	EngFncs->remove_thing(region->child_objects, segment);

	LOG_EXIT_VOID();
}

/**
 * lvm_clear_child_list
 *
 * Remove all segments from this region's child list.
 **/
void lvm_clear_child_list(storage_object_t * region)
{
	storage_object_t * segment;
	list_element_t itr1, itr2;

	LOG_ENTRY();

	LIST_FOR_EACH_SAFE(region->child_objects, itr1, itr2, segment) {
		lvm_remove_region_from_segment(region, segment);
	}

	LOG_EXIT_VOID();
}

